/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-exec.h>
#include <libgnome/gnome-mime.h>
#include <libgnome/gnome-mime-info.h>
#include <libgnome/gnome-i18n.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <pthread.h>

#include "../uulib/uudeview.h"
#include "../uulib/fptools.h"

#include "acache.h"
#include "debug.h"
#include "decode.h"
#include "log.h"
#include "prefs.h" /* for download_dir */
#include "util.h"
//#include "status-item.h"
//#include "queue-item-decode.h"

extern int queue_stop_all;

static pthread_mutex_t uulib_lock = PTHREAD_MUTEX_INITIALIZER;

/***
 *** Private Function Prototypes
 ***/

static char* UUD_file_name_filter (void *opaque, char *fname);
static int   UUD_busy_callback (void *opaque, uuprogress *progress);
static void  UUD_message_callback (void *opaque, char *msg, int level);
static void  decode_open (const char *fname);

/***
 *** Public Routines
 ***/

void
decode_article (decode_data *dd)
{
	uulist *item;
	GSList *p;
	gchar *ddir = NULL;
	gint i = 0;
	gint part_qty = 0;
	gint success = 0;
	gchar *filename = NULL;

	/* sanity checking */
	g_return_if_fail (dd!=NULL);
	g_return_if_fail (dd->adata!=NULL);
	g_return_if_fail (dd->sdata!=NULL);
	g_return_if_fail (dd->list!=NULL);

	status_item_emit_progress (dd->item, 0);
	status_item_emit_status (dd->item, _("Decoding: %s"), dd->adata->subject);

	/* debug... */
	part_qty = g_slist_length(dd->list);
	debug(DEBUG_DECODE,
		"starting to decode %d-part article '%s'",
		part_qty,
		((dd->adata->subject && *dd->adata->subject)
		 ? dd->adata->subject
		 : dd->adata->message_id));

	/* decide where to save the file */

	ddir = NULL;
	if (dd->gdata)
		ddir = group_get_attrib_string (
			dd->sdata, dd->gdata, "download_dir");

	if (ddir!=NULL)
	{
		directory_check (ddir);
		chdir (ddir);
		g_free (ddir);
	}
	else if (dd->gdata != NULL)
	{
		ddir = download_dir;
		directory_check (ddir);
		chdir (ddir);
	}
	
	status_item_emit_status (
		dd->item, _("Preparing: %s"), dd->adata->subject);

	/* set up the decode... */
	pthread_mutex_lock (&uulib_lock);
	UUInitialize ();
	/* setting this option can make things a whole lot faster,
	 * but it's off by default because certaain decode situations will fail
	 * with this off.. see source/docs -JEL
	 */
#ifdef UULIB_USE_FAST_DECODE
	UUSetOption (UUOPT_FAST, 1, NULL);
#endif
	UUSetMsgCallback (dd->adata, UUD_message_callback);
	UUSetBusyCallback (dd->item, UUD_busy_callback, 1);
	UUSetFNameFilter (dd->adata, UUD_file_name_filter);

	for (p=dd->list, i=0; p; p=p->next, ++i)
	{
		gchar* path = g_strdup_printf ("/%s/%s.body", data_dir, (char*)p->data);
		debug (DEBUG_DECODE,"Decoding part %d of %d: %s...", i+1, part_qty, path);
		if ((UULoadFile (path, NULL, 0)) != UURET_OK)
		{
			/* cleanup */
			UUCleanUp();
			pthread_mutex_unlock (&uulib_lock);
			g_free (path);
			
			debug(DEBUG_DECODE,"Decode couldn't load temporary file '%s'", path);
			/* FIXME: I think this is always redundant due to the message_callback
			 * pan_error_dialog(_("Decode couldn't load temporary file:\n%s"), path);
			 */
			return;
		}
		g_free (path);
	}
	
	status_item_emit_status (dd->item, _("Decoding: %s"), dd->adata->subject);

	/* decode */
	if ((item = UUGetFileListItem(0)) != NULL)
	{
		if ((item->state & UUFILE_OK) != 0)
		{
			if ((UUDecodeFile (item, NULL)) != UURET_OK)
			{
				log_add_va (_("Error decoding '%s'"), dd->adata->subject);
				debug(DEBUG_DECODE,"Error decoding '%s'", dd->adata->subject);
			}
			else
			{
				filename = g_strdup(UUFNameFilter(item->filename));
				log_add_va (
					"Successfully decoded '%s' from group '%s', article '%s'",
					filename,
					dd->gdata?dd->gdata->name:"Unknown",
					dd->adata->subject);
				debug (DEBUG_DECODE,
					"Successfully decoded '%s' from group '%s', article '%s'",
					filename,
					dd->gdata?dd->gdata->name:"Unknown",
					dd->adata->subject);
				++success;

				if (dd->filename != NULL)
				{
					rename (item->filename, dd->filename);
				}
				else if (use_subject_as_filename)
				{
					gchar* pch = NULL;
					pch = g_strdup (dd->adata->subject);
					pch = g_strdelimit (pch, "/\\,;~\t ", '_');
					g_free (pch);
				}
			}
		}
		else
		{
			log_add_va (_("Error decoding message '%s'"), dd->adata->subject);
			debug(DEBUG_DECODE,"Error decoding '%s'", dd->adata->subject);
		}
	}

	/* start cleaning up after ourselves */
	UUCleanUp ();
	pthread_mutex_unlock (&uulib_lock);
				
	if (dd->open && filename) {
		decode_open (filename);
	}
	g_free (filename);

	/* update the node */
	dd->adata->state &= ~STATE_DECODE_QUEUED;
	article_add_flag (dd->sdata, dd->gdata, dd->adata,
		(success ? STATE_DECODED : STATE_DECODE_FAILED), TRUE);
}

/***
**** Private Routines
***/

/**
 * This is the function that the decode thread runs.
 */

static void
decode_open (const char *fname)
{
	const char *mime_type = NULL;
	const char *mime_prog = NULL;
	char *quoted_fname = NULL;
	char *exec = NULL;

	if (!((mime_type = gnome_mime_type (fname))))
		return;
	if (!((mime_prog = gnome_mime_program (mime_type))))
		return;
	if (!strstr (mime_prog, "%f"))
		return;

	quoted_fname = g_strdup_printf ("\"%s\"", fname);
	exec = pan_substitute (mime_prog, "%f", quoted_fname);
	g_free (quoted_fname);
	gnome_execute_shell (NULL, exec);
	g_free (exec);
}

/***************************************************************
* UUdeview Lib Stuff 
***************************************************************/

static void
UUD_message_callback (void *opaque, char *msg, int level)
{
	debug(DEBUG_DECODE,"UUDeview says: '%s' (level %d)", msg, level);

	if ((level == UUMSG_WARNING) || (level == UUMSG_ERROR)) {
		pan_error_dialog (_("An error occured during decoding.\n%s"), msg);
	}
}


static int
UUD_busy_callback (void *opaque, uuprogress *progress)
{
	StatusItem *item = STATUS_ITEM(opaque);

	const int i = (progress->percent + 100.0 * (progress->partno-1))
		/ progress->numparts;
	if ( 0<=i && i<=100 )
		status_item_emit_progress (item, i);

	return queue_stop_all ? -1 : 0;
}


static char *
UUD_file_name_filter (void *opaque, char *fname)
{
	char *ptr;

	if (fname == NULL)
		return NULL;

	/*
	 * strip directory information
	 */

	if ((ptr = _FP_strrchr(fname, '/')) != NULL)
		return ptr + 1;
	else if ((ptr = _FP_strrchr(fname, '\\')) != NULL)
		return ptr + 1;

	/* increment UNKNOWN.??? if necessary */
	if (strncmp(fname, "UNKNOWN", 7) == 0) {
		struct stat finfo;
		int fnum;

		if (lstat(fname, &finfo) == -1)
			return fname;

		do {
			fnum = atoi (&fname[8]);
			fnum++;
			sprintf(&fname[8], "%.3u", fnum);
		} while (lstat(fname, &finfo) == 0);
	}

	return fname;
}
