/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>

#include <pthread.h>

#include <gnome.h>

#include "articlelist.h"
#include "article-thread.h"
#include "article-toolbar.h"
#include "globals.h"
#include "gui.h"
#include "status-item.h"
#include "util.h"

#include "xpm/article_read.xpm"
#include "xpm/article_unread.xpm"
#include "xpm/favorite.xpm"
#include "xpm/bomb.xpm"
#include "xpm/binary_complete.xpm"
#include "xpm/binary_incomplete.xpm"
#include "xpm/binary.xpm"
#include "xpm/bluecheck.xpm"
#include "xpm/mini_page.xpm"
#include "xpm/thread.xpm"

static GtkWidget* toolbar = NULL;

static GtkWidget* show_unread_button = NULL;
static GtkWidget* show_read_button = NULL;
static GtkWidget* show_complete_binary_button = NULL;
static GtkWidget* show_incomplete_binary_button = NULL;
static GtkWidget* show_nonbinary_button = NULL;
static GtkWidget* show_watched_button = NULL;
static GtkWidget* show_killfile_button = NULL;
static GtkWidget* show_normal_rank_button = NULL;
static GtkWidget* show_saved_button = NULL;
static GtkWidget* show_queued_button = NULL;
static GtkWidget* show_idle_button = NULL;

static GtkWidget* filter_bar = NULL;

static GtkWidget* threaded_tb = NULL;

/*****
******  SUBJECT FILTER
*****/

static gchar *subject_filter_str = NULL;

const gchar*
article_toolbar_get_filter (void)
{
	return subject_filter_str;
}

static gchar*
filter_selected_describe (const StatusItem* item)
{
	return g_strdup ("Filtering Articles");
}

static void
filter_selected_thread (void* data)
{
	const server_data *server = articlelist_get_current_server ();
	group_data *group = articlelist_get_current_group ();
	StatusItem *item = NULL;

	if (!server || !group) /* articlelist is idle */
		return;

	/* create a new status item to get thread messages */
	item = STATUS_ITEM(status_item_new(filter_selected_describe));
	pan_object_sink(PAN_OBJECT(item));
	gui_add_status_item (item);

	/* update the articlelist */
	articlelist_refresh (item);

	/* clean out the status item */
	gui_remove_status_item (item);
	pan_object_unref(PAN_OBJECT(item));
}
static void
article_filter_cb (GtkWidget* w, gpointer data)
{
	const gchar * text = NULL;
	pthread_t thread;

	/* update subject_filter_str */
	if (subject_filter_str)
		g_free (subject_filter_str);

	/* wrap the key in wildcards to make it a substring search...
	   unless there are already wildcards present, indicating the
	   user already knows what he wants */
	text = gtk_entry_get_text (GTK_ENTRY(w));
	if (strchr(text,'*'))
		subject_filter_str = g_strdup (text);
	else
		subject_filter_str = g_strdup_printf("*%s*",text);

	/* update the article list */
	pthread_create (&thread, NULL, (void*)filter_selected_thread, NULL);
	pthread_detach (thread);
}


/*****
******  SORT ARTICLES
*****/

#if 0
/**
 * Describe function for our "sorting article" StatusItem.
 * @param StatusItem unused
 */
static gchar*
sort_selected_describe (const StatusItem* item)
{
	return g_strdup ("Sorting Articles");
}

/**
 * This function is invoked in its own thread to keep the sorting work out of the gui thread.
 * @param data a gpointer representing an int of the new sort_type
 */
static void
sort_selected_thread (void* data)
{
	const server_data *server = articlelist_get_current_server ();
	group_data *group = articlelist_get_current_group ();
	StatusItem *item = NULL;

	if (!server || !group) /* articlelist is idle */
		return;

	/* create a new status item to get sort/thread messages */
	item = STATUS_ITEM(status_item_new(sort_selected_describe));
	pan_object_sink(PAN_OBJECT(item));
	gui_add_status_item (item);

	/* sort the articles & update the list */
	articlelist_set_sort_type (item, GPOINTER_TO_INT(data));

	/* clean out the status item */
	gui_remove_status_item (item);
	pan_object_unref(PAN_OBJECT(item));
}
static void
sort_selected_cb (GtkWidget* w, gpointer data)
{
	pthread_t thread;
	pthread_create (&thread, NULL, (void*)sort_selected_thread, data);
	pthread_detach (thread);
}


void
article_toolbar_set_sort_type (int sort_mode, gboolean do_sort)
{
	GtkWidget *menu = NULL;
	int menu_item_index = GPOINTER_TO_INT(g_hash_table_lookup (
			sort_type_to_menu_index, GINT_TO_POINTER(sort_mode)));

	/* why the fsck doesn't gtk have a way to select an option menu item
	   without the activation event?  garrggh. */
	pan_lock();
	menu = gtk_option_menu_get_menu (GTK_OPTION_MENU(sort_option_menu));
	gtk_object_ref(GTK_OBJECT(menu));
	gtk_option_menu_remove_menu (GTK_OPTION_MENU(sort_option_menu));
	gtk_menu_set_active (GTK_MENU(menu), menu_item_index);
	gtk_option_menu_set_menu (GTK_OPTION_MENU(sort_option_menu), menu);
	gtk_object_unref(GTK_OBJECT(menu));
	pan_unlock();

	if (do_sort)
		sort_selected_cb (NULL, GINT_TO_POINTER(sort_mode));
}
#endif

/*****
******  THREAD / UNTHREAD ARTICLES
*****/

static void
threaded_toggled_cb (GtkToggleButton *tb, gpointer user_data)
{
	articlelist_set_threaded (gtk_toggle_button_get_active (tb));
}

static int
threaded_changed_cb (gpointer obj, gpointer arg, gpointer user)
{
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(threaded_tb), arg!=NULL);
	return 0;
}

/*****
******  STATE FILTER
*****/

static int
state_filter_changed_cb (
	gpointer call_object,
	gpointer call_arg,
	gpointer user_data)
{
	guint state_filter = GPOINTER_TO_UINT(call_arg);

	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(show_unread_button),
		state_filter & STATE_FILTER_UNREAD);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(show_read_button),
		state_filter & STATE_FILTER_READ);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(show_complete_binary_button),
		state_filter & STATE_FILTER_COMPLETE_BINARIES);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(show_incomplete_binary_button),
		state_filter & STATE_FILTER_INCOMPLETE_BINARIES);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(show_nonbinary_button),
		state_filter & STATE_FILTER_NONBINARIES);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(show_watched_button),
		state_filter & STATE_FILTER_WATCHED);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(show_killfile_button),
		state_filter & STATE_FILTER_KILLFILE);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(show_normal_rank_button),
		state_filter & STATE_FILTER_NORMAL_RANK);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(show_saved_button),
		state_filter & STATE_FILTER_SAVED);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(show_queued_button),
		state_filter & STATE_FILTER_QUEUED);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(show_idle_button),
		state_filter & STATE_FILTER_IDLE);

	return 0;
}


/*****
******  CREATE TOOLBAR
*****/

extern GtkTooltips *ttips;

static void
state_filter_button_toggled_cb (GtkToggleButton* tb, gpointer user_data)
{
        const guint flag = GPOINTER_TO_UINT(user_data);
	const gboolean on = gtk_toggle_button_get_active (tb);
	articlelist_poke_state_filter (flag, on);
}

void
article_toolbar_show_state_filter (gboolean show_state_filter)
{
	if (show_state_filter)
		gtk_widget_show_all (filter_bar);
	else
		gtk_widget_hide_all (filter_bar);
	gtk_widget_queue_resize (toolbar);
}

GtkWidget*
article_toolbar_new (void)
{
	GtkWidget *w = NULL;
	GtkWidget *w2 = NULL;
	GtkWidget *frame = NULL;
	GtkWidget *hbox;

	/* create the toolbar... */
	toolbar = gtk_toolbar_new (
		GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_TEXT);


	filter_bar = gtk_hbox_new (FALSE, 5);

	/* filter label */
	w = gtk_label_new ( _("Filters:") );
	gtk_box_pack_start (GTK_BOX(filter_bar), w, FALSE, FALSE, 0);

	/* filter one: read/unread */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	hbox = gtk_hbox_new (FALSE, 0);
	show_unread_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Unread Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_UNREAD));
	w2 = gnome_pixmap_new_from_xpm_d (article_unread_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_read_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Read Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_READ));
	w2 = gnome_pixmap_new_from_xpm_d (article_read_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame), hbox);
	gtk_box_pack_start (GTK_BOX(filter_bar), frame, FALSE, FALSE, 0);

	/* filter two: watched/killfile/other */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	hbox = gtk_hbox_new (FALSE, 0);
	show_watched_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Watched Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_WATCHED));
	w2 = gnome_pixmap_new_from_xpm_d (favorite_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_killfile_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Articles in Killfile"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_KILLFILE));
	w2 = gnome_pixmap_new_from_xpm_d (bomb_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_normal_rank_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Articles neither Watched nor Killed"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_NORMAL_RANK));
	w2 = gnome_pixmap_new_from_xpm_d (mini_page_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame), hbox);
	gtk_box_pack_start (GTK_BOX(filter_bar), frame, FALSE, FALSE, 0);

	/* filter three: complete/uncomplete/text */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	hbox = gtk_hbox_new (FALSE, 0);
	show_complete_binary_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Complete Binary Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_COMPLETE_BINARIES));
	w2 = gnome_pixmap_new_from_xpm_d (binary_complete_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_incomplete_binary_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Incomplete Binary Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_INCOMPLETE_BINARIES));
	w2 = gnome_pixmap_new_from_xpm_d (binary_incomplete_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_nonbinary_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Non-Binary Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_NONBINARIES));
	w2 = gnome_pixmap_new_from_xpm_d (mini_page_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame), hbox);
	gtk_box_pack_start (GTK_BOX(filter_bar), frame, FALSE, FALSE, 0);

	/* filter four: saved/queued/neither */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	hbox = gtk_hbox_new (FALSE, 0);
	show_saved_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Saved Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_SAVED));
	w2 = gnome_pixmap_new_from_xpm_d (binary_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_queued_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Queued Articles"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_QUEUED));
	w2 = gnome_pixmap_new_from_xpm_d (bluecheck_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	show_idle_button = w = gtk_toggle_button_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Show Articles not Saved nor in Queue"), "");
	gtk_signal_connect (GTK_OBJECT(w), "toggled",
		state_filter_button_toggled_cb, GUINT_TO_POINTER(STATE_FILTER_IDLE));
	w2 = gnome_pixmap_new_from_xpm_d (mini_page_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame), hbox);
	gtk_box_pack_start (GTK_BOX(filter_bar), frame, FALSE, FALSE, 0);

	gtk_toolbar_append_widget (GTK_TOOLBAR(toolbar), filter_bar, "", "");

	/* Thread? */
	hbox = gtk_hbox_new (FALSE, 0);
	w = threaded_tb = gtk_toggle_button_new ();
	w2 = gnome_pixmap_new_from_xpm_d (thread_xpm);
	gtk_container_add (GTK_CONTAINER(w), w2);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w),  TRUE);
	gtk_signal_connect (
		GTK_OBJECT(w), "toggled", threaded_toggled_cb, NULL);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Turn article threading on/off"), "");
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);


	w = gtk_entry_new();
	gtk_signal_connect (GTK_OBJECT (w), "activate",
			    GTK_SIGNAL_FUNC(article_filter_cb), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w,
				   _("Type in a subject search string and press ENTER.  "
				     "Wildcards are allowed."), "");
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	gtk_toolbar_append_widget (GTK_TOOLBAR(toolbar), hbox, "", "");

	/* show the toolbar */
	gtk_widget_show_all (toolbar);
	if (!gnome_config_get_bool ("/Pan/Display/Show_Filter_In_Article_Toolbar=true"))
		gtk_widget_hide_all (filter_bar);
	gnome_app_add_toolbar (
		GNOME_APP(Pan.window),
		GTK_TOOLBAR(toolbar), "article_toolbar", 0, GNOME_DOCK_TOP, 2, 0, 0);

	/* listen for changes to the articlelist "state filter"
	   so that we can update the menus accordingly */
	pan_callback_add (
		articlelist_state_filter_changed,
		state_filter_changed_cb,
		NULL);
	pan_callback_add (
		articlelist_thread_changed,
		threaded_changed_cb,
		NULL);

	return toolbar;
}
