/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "article-db.h"
#include "log.h"
#include "prefs.h" /* for data_dir */
#include "util.h"

//=========================================================================

typedef struct
{
	pan_db db;
	gchar* filename;
	const server_data* sdata;
	const group_data* gdata;
}
AHDB;

//=========================================================================

static gchar*
get_filename (
	const server_data* s,
	const group_data* g )
{
	g_return_val_if_fail(s!=NULL, NULL);
	g_return_val_if_fail(g!=NULL, NULL);

	return g_strdup_printf ( "/%s/%s/%s.db", data_dir, s->name, g->name );
}

article_db
ahdb_ref (
	const server_data* s,
	const group_data* g )
{
	AHDB* ahdb = g_malloc0 ( sizeof(AHDB) );
	ahdb->filename = get_filename ( s, g );
	ahdb->db = pan_db_ref ( ahdb->filename );
	ahdb->sdata = s;
	ahdb->gdata = g;
	return (article_db)ahdb;
}

gboolean
ahdb_exists (
	const server_data* s,
	const group_data* g )
{
	gchar* filename = get_filename ( s, g );
	gboolean exists = file_exists ( filename );
	g_free ( filename );
	return exists;
}

void
ahdb_delete_db_file (
	const server_data* s,
	const group_data* g )
{
	gchar* filename = get_filename ( s, g );
	remove ( filename );
	g_free ( filename );
}

void ahdb_unref (
	article_db db )
{
	AHDB* ahdb = (AHDB*)db;
	g_free ( ahdb->filename );
	ahdb->filename = 0;
	pan_db_unref ( ahdb->db );
	ahdb->db = 0;
	g_free ( ahdb );
}

//=========================================================================

const char *
ahdb_field (const char *flat_adata, int offset)
{
	int i = 0;
	const char *p = flat_adata;

	if (offset == 0)
		return p;
	while (*p != '\0') {
		if (*p == '\n')
			i++;
		if (i == offset)
			return (p+1);
		p++;
	}
	return NULL;
}

void ahdb_save (
	article_db db,
	const article_data* adata,
	gboolean sync )
{
	AHDB* ahdb = (AHDB*)db;

	gchar* pch = g_strdup_printf (
		"%d\n" "%lu\n"
		"%d\n" "%d\n"
		"%d\n" "%s\n"

		"%s\n" "%s\n"
		"%s\n"
		"%s\n"
		"%s\n"
		"%s\n",

		(int) adata->date, (unsigned long)(adata->state & ~STATE_NEW),
		adata->linecount, adata->parts,
		adata->part, adata->number,

		adata->subject, adata->author,
		(adata->references ? adata->references : ""),
		(adata->newsgroups ? adata->newsgroups : ""),
		(adata->reply_to ? adata->reply_to : ""),
		(adata->followup_to ? adata->followup_to : "") );

	pan_db_put_value_str (ahdb->db, adata->message_id, pch);

	g_free (pch);

	if (sync)
		ahdb_sync (db);
}

void
ahdb_save_all (
	article_db db,
	GSList* list,
	GFunc gfunc,
	gpointer user_data )
{
	while (list != NULL)
	{
		article_data* adata = (article_data*) list->data;
		if (gfunc)
			(*gfunc)(adata,user_data);
		ahdb_save (db, adata, FALSE);
		list = list->next;
	}

	ahdb_sync (db);
}

void
ahdb_sync (
	article_db db )
{
	AHDB* ahdb = (AHDB*)db;
	pan_db_sync ( ahdb->db );
}

//=========================================================================

static article_data*
article_deserialize (
	const group_data* gdata,
	const char* message_id,
	const char* db_text )
{
	gchar** sd;
	article_data* adata;
	gboolean has_pan_066_fields;

	if ( !gdata )
		return NULL;

	/* break up the db_text */
	g_assert ( message_id!=NULL );
	g_return_val_if_fail ( db_text!=NULL, NULL );
	sd = g_strsplit ( db_text, "\n", 12 );
	g_return_val_if_fail ( sd!=NULL, NULL );

	/* build the article data */
	adata = article_new ();
	adata->message_id = g_strdup (message_id);
	g_assert (sd[0] != NULL);

	/* populate the article data */
	adata->date = strtol (sd[0], NULL, 10);
	adata->state = (unsigned long) strtol (sd[1], NULL, 10);
	adata->linecount = atoi (sd[2]);
	adata->parts = atoi (sd[3]);
	adata->part = atoi (sd[4]);
	adata->number = g_strdup (sd[5]);
	adata->subject = g_strdup (sd[6]);
	adata->author = g_strdup (sd[7]);
	adata->references = g_strdup (sd[8] ? sd[8] : "");
	adata->threads = NULL;
	has_pan_066_fields = sd[8]!=NULL && sd[9]!=NULL;
	adata->newsgroups = g_strdup (has_pan_066_fields ? sd[9] : "");
	adata->reply_to = g_strdup (has_pan_066_fields ? sd[10] : "");
	adata->followup_to = g_strdup (has_pan_066_fields ? sd[11] : "");

	/* cleanup */
	g_strfreev (sd);

	return adata;
}

article_data*
ahdb_get (
	article_db db,
	const char* message_id )
{
	AHDB* ahdb = (AHDB*)db;
	gchar* s = pan_db_get_value_str ( ahdb->db, message_id );
	article_data* a = article_deserialize ( ahdb->gdata, message_id, s );
	g_free ( s );
	return a;
}

typedef struct
{
	AHDB* ahdb;
	GSList* l;
	GSList* delete_keys;
	GFunc gfunc;
	gpointer user_data;
	unsigned int first;
        int total_qty;
	int read_qty;
}
GetArticlesStruct;

static void
ahdb_get_all_dbfunc (
	const char* key,
	const char* val,
	gpointer user_data )
{
	GetArticlesStruct* gas = (GetArticlesStruct*) user_data;
	article_data* a = article_deserialize (gas->ahdb->gdata, key, val);

	if (a != NULL)
	{
		/* Expire if it's not on server anymore */
		if (atoi(a->number) < gas->first)
		{
			if (article_flag_on (a, STATE_READ))
				gas->read_qty++;
			gas->total_qty++;
			gas->delete_keys = g_slist_prepend (
				gas->delete_keys, g_strdup(key));
			article_free (a);
		}
		else
		{
			gas->l = g_slist_prepend (gas->l, a);
			if (gas->gfunc)
				(*gas->gfunc)(a, gas->user_data);
		}
	}
}

GSList*
ahdb_get_all (
	article_db db,
	GFunc gfunc,
	gpointer user_data )
{
	GSList *l = NULL;
	AHDB* ahdb = (AHDB*)db;
	GetArticlesStruct gas;

	/* load the articles... */
	gas.l = NULL;
	gas.ahdb = ahdb;
	gas.gfunc = gfunc;
	gas.user_data = user_data;
	gas.read_qty = 0;
	gas.total_qty = 0;
	gas.delete_keys = NULL;
	gas.first = group_get_attrib_i (ahdb->sdata, ahdb->gdata, "First");
	pan_db_foreach (ahdb->db, ahdb_get_all_dbfunc, (gpointer)&gas);

	/* delete expired articles.  pan_db_foreach locks the db, so we have
	   to do it here instead of in ahdb_get_all_dbfunc. */
	for (l=gas.delete_keys; l!=NULL; l=l->next) {
		pan_db_erase (gas.ahdb->db, (const char*)l->data);
		g_free (l->data);
	}
	g_slist_free (gas.delete_keys);

	group_add_attrib_i (ahdb->sdata, ahdb->gdata, "Read", -gas.read_qty);
	group_add_attrib_i (ahdb->sdata, ahdb->gdata, "Total", -gas.total_qty);
	log_add_va (_("Expired %d articles (%d were marked read) in group %s"), gas.total_qty, gas.read_qty, ahdb->gdata->name );

	return gas.l;
}

//=========================================================================

static void
ahdb_length_dbfunc (
	const char* key,
	const char* val,
	gpointer user_data)
{
	++(*((int*)user_data));
}
int
ahdb_length (
	article_db db)
{
	int i = 0;
	pan_db_foreach (((AHDB*)db)->db, ahdb_length_dbfunc, (gpointer)&i);
	return i;
}

//=========================================================================


void
ahdb_erase (
	article_db db,
	const article_data* adata,
	const char *message_id)
{
	AHDB* ahdb = (AHDB*)db;
	if (adata)
		pan_db_erase ( ahdb->db, adata->message_id );
	else if (message_id)
		pan_db_erase ( ahdb->db, message_id );
}

void
ahdb_erase_all (
	article_db db )
{
	AHDB* ahdb = (AHDB*)db;
	pan_db_erase_all ( ahdb->db );
}

