/*
texpire -- expire old articles

Written by Arnt Gulbrandsen <agulbra@troll.no> and copyright 1995
Troll Tech AS, Postboks 6133 Etterstad, 0602 Oslo, Norway, fax +47
22646949.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>
and Randolf Skerka <Randolf.Skerka@gmx.de>.
Copyright of the modifications 1997.
Modified by Kent Robotti <robotti@erols.com>. Copyright of the
modifications 1998.
Modified by Markus Enzenberger <enz@cip.physik.uni-muenchen.de>.
Copyright of the modifications 1998.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>.
Copyright of the modifications 1998, 1999.
Modified by Kazushi (Jam) Marukawa <jam@pobox.com>.
Copyright of the modifications 1998, 1999.

See file COPYING for restrictions on the use of this software.
*/

#include "leafnode.h"

#ifdef SOCKS
#include <socks.h>
#endif

#include <ctype.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>

time_t now;

time_t default_expire;

int verbose = 0;
int debug = 0;

int use_atime = 1;	/* look for atime on articles to expire */

struct exp {
    char* xover;
    int kill;
    int exists;
};

void free_expire( void ) {
    struct expire_entry *a, *b;

    b = expire_base;
    while ((a = b) != NULL) {
        b = a->next;
        free(a);
    }
}    

/*
05/27/97 - T. Sweeney - Find a group in the expireinfo linked list and return
                        its expire time. Otherwise, return zero.
*/

static time_t lookup_expire(char* group) {
    struct expire_entry *a;
   
    a = expire_base;
    while (a != NULL) {
	if (ngmatch(a->group, group) == 0)
	    return a->xtime;
        a = a->next;
    }
    return 0;
}

/*
 * return 1 if xover is a legal overview line, 0 else
 */
static int legalxoverline ( char * xover, unsigned long artno ) {
    char * p;
    char * q;

    if ( !xover )
	return 0;

    /* anything that isn't tab, printable ascii, or latin-* ? then killit */

    p = xover;
    while ( *p ) {
	int c = (unsigned char)*p++;

	if ( ( c != '\t' && c < ' ' ) || ( c > 126 && c < 160 ) ) {
	    if ( debugmode )
		syslog( LOG_DEBUG,
			"%lu xover error: non-printable chars.", artno );
	    return 0;
	}
    }

    p = xover;
    q = strchr( p, '\t' );
    if ( !q ) {
	if ( debugmode )
	    syslog( LOG_DEBUG, "%lu xover error: no Subject: header.", artno );
	return 0;
    }

    /* article number */

    while( p != q ) {
	if ( !isdigit((unsigned char)*p) ) {
	    if ( debugmode )
        	syslog( LOG_DEBUG, "%lu xover error: article "
			"number must consists of digits.", artno );
	    return 0;
	}
	p++;
    }

    p = q+1;
    q = strchr( p, '\t' );
    if ( !q ) {
	if ( debugmode )
	    syslog( LOG_DEBUG, "%lu xover error: no From: header.", artno );
	return 0;
    }

    /* subject: no limitations */

    p = q+1;
    q = strchr( p, '\t' );
    if ( !q ) {
	if ( debugmode )
            syslog( LOG_DEBUG, "%lu xover error: no Date: header.", artno );
	return 0;
    }

    /* from: no limitations */

    p = q+1;
    q = strchr( p, '\t' );
    if ( !q ) {
	if ( debugmode )
            syslog( LOG_DEBUG,
		    "%lu xover error: no Message-ID: header.", artno );
	return 0;
    }

    /* date: no limitations */

    p = q+1;
    q = strchr( p, '\t' );
    if ( !q ) {
	if ( debugmode )
	    syslog( LOG_DEBUG,
		    "%lu xover error: no References: or Bytes: header.",
		    artno );
	return 0;
    }

    /* message-id: <*@*> */

    if ( *p != '<' ) {
	if ( debugmode )
	    syslog( LOG_DEBUG,
		    "%lu xover error: Message-ID does not start with <.",
		    artno );
	return 0;
    }
    while ( p != q && *p != '@' && *p != '>' && *p != ' ' )
	p++;
    if ( *p != '@' ) {
	if ( debugmode )
	    syslog( LOG_DEBUG,
		    "%lu xover error: Message-ID does not contain @.", artno );
	return 0;
    }
    while ( p != q && *p != '>' && *p != ' ' )
	p++;
    if ( *p != '>' ) {
	if ( debugmode )
	    syslog( LOG_DEBUG,
		    "%lu xover error: Message-ID does not end with >.", artno );
	return 0;
    }
    if ( ++p != q ) {
	if ( debugmode )
	    syslog( LOG_DEBUG,
		    "%lu xover error: Message-ID does not end with >.", artno );
	return 0;
    }

    p = q+1;
    q = strchr( p, '\t' );
    if ( !q ) {
	if ( debugmode )
	    syslog( LOG_DEBUG, "%lu xover error: no Bytes: header.", artno );
	return 0;
    }

    /* references: a series of <*@*> separated by space */

    while ( p != q ) {
	if ( *p != '<' ) {
	    if ( debugmode )
        	syslog( LOG_DEBUG, "%lu xover error: "
			"Reference does not start with <.", artno );
	    return 0;
	}
	while ( p != q && *p != '@' && *p != '>' && *p != ' ' )
	    p++;
	if ( *p != '@' ) {
	    if ( debugmode )
        	syslog( LOG_DEBUG,
			"%lu xover error: Reference does not contain @.",
			artno );
	    return 0;
	}
	while ( p != q && *p != '>' && *p != ' ' )
	    p++;
	if ( *p++ != '>' ) {
	    if ( debugmode )
        	syslog( LOG_DEBUG,
			"%lu xover error: Reference does not end with >.",
			artno );
	    return 0;
	}
	while ( p != q && *p == ' ' )
	    p++;
    }

    p = q+1;
    q = strchr( p, '\t' );
    if ( !q ) {
	if ( debugmode )
	    syslog( LOG_DEBUG, "%lu xover error: no Lines: header.", artno );
	return 0;
    }

    /* byte count */

    while( p != q ) {
	if ( !isdigit((unsigned char)*p) ) {
	    if ( debugmode )
		syslog( LOG_DEBUG, "%lu xover error: illegal digit "
			"in Bytes: header.", artno );
	    return 0;
	}
	p++;
    }

    p = q+1;
    q = strchr( p, '\t' );
    if ( q )
        *q = '\0'; /* kill any extra fields */

 /* line count */

    while( p && *p && p != q ) {
	if ( !isdigit((unsigned char)*p) ) {
	    if ( debugmode )
        	syslog( LOG_DEBUG, "%lu xover error: illegal digit "
			"in Lines: header.", artno );
	    return 0;
	}
	p++;
    }

    return 1;
}

static void dogroup(struct newsgroup* g) {
    char gdir[PATH_MAX];
    char *p;
    char *q;
    DIR *d;
    struct dirent * de;
    struct stat st;
    unsigned long first, last, art;
    struct exp* articles;
    int n;
    int fd;
    char * overview; /* xover: read then free */

    int deleted,kept;

    deleted = kept = 0;
    clearidtree();

    /* eliminate empty groups */
    if ( !chdirgroup( g->name, FALSE ) )
	return;
    getcwd(gdir, PATH_MAX);

    /* find low-water and high-water marks */

    d = opendir(".");
    if (!d) {
	syslog(LOG_ERR, "opendir in %s: %m", gdir);
	return;
    }

    first = ULONG_MAX;
    last = 0;
    while ( ( de = readdir(d) ) != 0 ) {
	if ( !isdigit((unsigned char)de->d_name[0]) ||
	     stat( de->d_name, &st ) || !S_ISREG( st.st_mode ) )
	    continue;
	art = strtoul(de->d_name, &p, 10);
	if (p && !*p) {
	    if (art < first)
		first = art;
	    if (art > last)
		last = art;
	}
    }

    if (last < first) {
	closedir(d);
	return;
    }
    if ( verbose > 1 )
        printf("%s: low water mark %lu, high water mark %lu\n",
	        g->name, first, last );
    if ( debugmode )
	syslog( LOG_DEBUG,
		"%s: expire %lu, low water mark %lu, high water mark %lu",
		g->name, expire, first, last );

    if ( expire <= 0 )
	return;

    rewinddir( d );

    /* allocate and clear article array */

    articles = (struct exp*)critmalloc((last - first + 1) * sizeof(struct exp),
				       "Reading articles to expire");
    for (art = 0; art <= last - first; art++) {
	articles[art].xover = NULL;
	articles[art].kill = 0;
	articles[art].exists = 0;
    }

    /* read in overview info, to be purged and written back */

    overview = NULL;

    if (stat(".overview", &st) == 0) {
	/* could use mmap() here but I don't think it would help */
	overview = critmalloc(st.st_size + 1, "Reading article overview info");
	if ((fd = open(".overview", O_RDONLY)) < 0 ||
	    (read(fd, overview, st.st_size) < st.st_size)) {
	    syslog(LOG_ERR, "can't open/read %s/.overview: %m", gdir);
	    *overview = '\0';
	    if (fd > -1)
		close(fd);
	} else {
	    close(fd);
	    overview[st.st_size] = '\0'; /* 0-terminate string */
	}

	p = overview;
	while (p && *p) {
	    while (p && isspace((unsigned char)*p))
		p++;
	    art = strtoul(p, NULL, 10);
	    if (art >= first && art <= last &&
		!articles[art - first].xover) {
		articles[art - first].xover = p;
		articles[art - first].kill = 1;
	    }
	    p = strchr(p, '\n');
	    if (p) {
		*p = '\0';
		if (p[-1] == '\r')
		    p[-1] = '\0';
		p++;
	    }
	}
    }

    /* check the syntax of the .overview info, and delete all illegal stuff */

    for (art = first; art <= last; art++) {
	if (articles[art - first].xover &&
	    !legalxoverline(articles[art - first].xover, art)) {
	    articles[art - first].xover = NULL;
	}
    }

    /* insert articles in tree, and clear 'kill' for new or read articles */

    while ((de = readdir(d)) != 0) {
	art = strtoul(de->d_name, &p, 10);
	if (p && !*p) {
	    articles[art - first].exists = 1;
	    /* mark all articles as to-be-deleted and rescue those
	     * which fulfill certain criteria
	     */
	    articles[art - first].kill = 1;
	    if (stat(de->d_name, &st) == 0 &&
		(S_ISREG(st.st_mode)) &&
		((st.st_mtime > expire) ||
		 (use_atime && (st.st_atime > expire)))) {
		articles[art - first].kill = 0;
		p = articles[art - first].xover;
		for (n = 0; n < 4; n++)
		    if (p && (p = strchr(p + 1, '\t')))
			p++;
		q = p ? strchr(p, '\t') : NULL;
		if (p && q) {
		    *q = '\0';
		    if (findmsgid(p)) { /* another file with same msgid? */
			articles[art - first].kill = 1;
		    } else {
			insertmsgid(p, art);
			if (st.st_nlink < 2) { /* repair fs damage */
			    if (link(de->d_name, lookup(p))) {
				if (errno == EEXIST)
				    /* exists, but points to another file */
				    articles[art-first].kill = 1;
				else
				    syslog(LOG_ERR,
					   "relink of %s failed: %m (%s)",
					   p, lookup(p));
			    }
			    else
				syslog(LOG_INFO, "relinked message %s", p);
			}
			*q = '\t';
		    }
		} else if (articles[art-first].xover) {
		    /* data structure inconsistency: delete and be rid of it */
		    articles[art-first].kill = 1;
		} else {
		    /* possibly read the xover line into memory? */
		}
	    }
	}
    }
    closedir(d);

    /* compute new low-water mark */

    art = first;
    while (art<=last && articles[art-first].kill)
	art++;
    g->first = art;

    /* remove old postings */

    for (art = first; art <= last; art++) {
	char name[20];
	if (articles[art-first].exists) {
	    if (articles[art-first].kill) {
		sprintf(name, "%lu", art);
		if (!unlink(name)) {
		    if ( debugmode )
			syslog( LOG_DEBUG, "deleted article %s/%ld", gdir, art );
		    deleted++;
		} else if (errno != ENOENT && errno != EEXIST) {
		    /* if file was deleted alredy or it was not a file */
		    /* but a directory, skip error message */
		    kept++;
		    syslog( LOG_ERR, "unlink %s/%ld: %m", gdir, art );
	        } else {
		    /* deleted by someone else */
		}
	    } else {
		kept++;
	    }
	}
    }
    free( (char*)articles );
    free( overview );

    if ( last > g->last ) /* try to correct insane newsgroup info */
	g->last = last;

    if ( deleted || kept ) {
	printf("%s: %d articles deleted, %d kept\n", g->name, deleted, kept);
	syslog( LOG_INFO,
		"%s: %d articles deleted, %d kept", g->name, deleted, kept);
    }

    if ( !kept ) {
	if ( unlink( ".overview" ) < 0 )
	    syslog( LOG_ERR, "unlink %s/.overview: %m", gdir );
	if ( !chdir("..") && ( isinteresting(g->name) == 0 ) ) {
	    /* delete directory and empty parent directories */
	    while ( rmdir( gdir ) == 0 ) {
		getcwd( gdir, PATH_MAX );
		chdir("..");
	    }
	}
    }
}
    

static void expiregroup(struct newsgroup* g)
{
    if ( g ) {
	expiregroup( g->right );
	if (!(expire = lookup_expire(g->name)))
	    expire = default_expire;
	dogroup( g );
	expiregroup( g->left );
    }
}


static void expiremsgid(void)
{
    int n;
    DIR * d;
    struct dirent * de;
    struct stat st;
    int deleted, kept;

    deleted = kept = 0;

    for ( n=0; n<1000; n++ ) {
	sprintf( s, "%s/message.id/%03d", spooldir, n );
	if ( chdir( s ) ) {
	    if ( errno == ENOENT )
		mkdir( s, 0755 ); /* file system damage again */
	    if ( chdir( s ) ) {
		syslog( LOG_ERR, "chdir %s: %m", s );
		continue;
	    }
	}

	d = opendir( "." );
	if ( !d )
	    continue;
	while ((de = readdir(d)) != 0) {
	    if (stat(de->d_name, &st) == 0) {
		if (st.st_nlink < 2 && !unlink(de->d_name))
		    deleted++;
		else if (S_ISREG(st.st_mode))
		    kept++;
	    }
	}
	closedir( d );
    }

    if ( kept || deleted ) {
	printf("total: %d articles deleted, %d kept\n", deleted, kept);
	syslog( LOG_INFO, "%d articles deleted, %d kept", deleted, kept );
    }
}


int main(int argc, char** argv)
{
    int option;

    if ( !initvars( argv[0] ) )
	exit( 1 );

    openlog( "texpire", LOG_PID|LOG_CONS, LOG_NEWS );
    while ( (option=getopt( argc, argv, "vf" )) != -1 ) {
	if ( option == 'v' ) {
            verbose++;
	}
	else if ( option == 'f' ) {
	    use_atime = 0;
	}
	else {
	    fprintf( stderr, "Usage: texpire [-v] [-f]\n"
			     "  -v: more verbose (may be repeated)\n"
			     "  -f: force expire irrespective of access time\n");
	    exit( 1 );
	}
    }

    expire = 0;
    expire_base = NULL;

    if ( lockfile_exists() )
	exit( 1 );
    if ( !readconfig() ) {
	printf( "Reading configuration failed, exiting "
		"(see syslog for more information).\n");
	unlink( lockfile );
	exit( 2 );
    }
    readactive();

    if ( verbose ) {
	printf( "texpire %s: ", version );
	if ( use_atime )
	    printf( "check mtime and atime\n" );
	else
	    printf( "check mtime only\n" );
    }
    if ( debugmode )
	syslog( LOG_DEBUG, "texpire %s: use_atime is %d", version, use_atime );

    if ( expire == 0 ) {
	fprintf( stderr, "%s: no expire time\n", argv[0] );
	exit( 2 );
    }

    now = time( NULL );
    default_expire = expire;

    expiregroup( active );
    writeactive( 0 );
    unlink( lockfile );
    expiremsgid();
    free_expire();
    return 0;
}
