/*****************************************************************************
* fpipemon.c	Frame Relay Monitor.
*
* Authors:	Jaspreet Singh	
*
* Copyright:	(c) 1995-1997 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ----------------------------------------------------------------------------
* Mar 22, 1997	Jaspreet Singh	Improved Error handling
* Nov 24, 1997	Jaspreet Singh	Added new stats for driver statistics
* Nov 13, 1997	Jaspreet Singh	Fixed descriptions of Global Error Statistics
* Oct 20, 1997 	Jaspreet Singh	Added new commands for driver specific stats
*				and router up time.
* Jul 28, 1997	Jaspreet Singh	Added a new command for running line trace 
*				displaying RAW data.
* Jul 25, 1997	Jaspreet Singh	Added commands for viewing specific DLCI data 
*				including FECN and BECN. 
* Jun 24, 1997	Jaspreet Singh	S508/FT1 test commands		
* Apr 25, 1997	Farhan Thawar	Initial version based on fpipemon for WinNT.
*****************************************************************************/
#include <stdio.h>
#include <ctype.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <string.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <linux/wanpipe.h>
#include <linux/sdla_fr.h>

#define TIMEOUT 1
#define MDATALEN 2024
#define TRUE 1
#define FALSE 0

#pragma pack(1)
typedef struct {
   unsigned char signature[8];
   unsigned char requestreply;
   unsigned char id;
   unsigned char reserved2[6];
   unsigned char opp_flag;
   unsigned char command;
   unsigned short buffer_length;
   unsigned char return_code;
   unsigned short dlci;
   unsigned char FECN_BECN_DE_CR_bits;
   unsigned short rx_frm_disc_int;
   unsigned long rx_frm_disc_app;
   unsigned char reserved[2];
   unsigned char data[MDATALEN];
} CBLOCK;
#pragma pack()

/******************************************************************************
 * Structure for tracing
 *
 * The one frame trace;
 */

/* Prototypes */
void 	ResetCB(CBLOCK *c1);
int 	MakeConnection( void );
int 	ObtainConfiguration( void );
unsigned char DoCommand( CBLOCK *cb );
void 	SaveCB( CBLOCK *dest, CBLOCK *src );
void 	init( int , char **);
void 	error( char return_code);
char 	* get_ip_addr(char *);
void	banner(char *, int);



/* global for now */
int sock;
struct sockaddr_in soin;
CBLOCK cb, cbconfig;
char codeversion[10];
unsigned int frame_count;
unsigned char station_config;
int raw_data = FALSE;
int fail;
unsigned char par_port_A_byte, par_port_B_byte;
int off_counter, green_counter, red_counter;
int loop_counter;

/* defines for now */
char ipaddress[16];
int udp_port = 9000;
char cmd[5];
unsigned short dlci_number = 0;

void ResetCB(CBLOCK *c1)
{
	memset((void *)&c1->opp_flag, 0, sizeof(CBLOCK)-16);
}; //ResetCB

int MakeConnection( void ) 
{
   	sock = socket( PF_INET, SOCK_DGRAM, 0 );
   	if( sock < 0 ) {
      		printf("Error: Unable to open socket!\n");
      		return( FALSE );
   	} /* if */
   	
	soin.sin_family = AF_INET;
   	soin.sin_addr.s_addr = inet_addr(ipaddress);
	if(soin.sin_addr.s_addr < 0){
		printf("Error: Invalid IP address!\n");
		return( FALSE );
	}
   	soin.sin_port = htons((u_short)udp_port);
   
	if( !connect( sock, (struct sockaddr *)&soin, sizeof(soin)) == 0 ) {
      		printf("Error: Cannot make connection!\nMake sure the IP address is correct\n");
      		return( FALSE );
   	} /* if */
   	if( !ObtainConfiguration() ) {
      		printf("Error: Unable to obtain frame relay information.\nMake sure the IP and UDP port are correct.\n");
      		close( sock );
      		return( FALSE );
   	} /* if */   
   	return( TRUE );
}; /* MakeConnection */

int ObtainConfiguration( void ) 
{
   	unsigned char x;
   
   	x = 0;
   	cb.command = FR_READ_CONFIG;
   	cb.buffer_length = 0;
   	cb.dlci = 0;
   	while (DoCommand(&cb) != 0 && ++x < 4) {
      		if (cb.return_code == 0xaa) {
	 		printf("Error: Command timeout occurred\n"); 
	 		return(FALSE);
      		}
      		if (cb.return_code == 0xCC ) return(FALSE);
   	}
   
	if (x >= 4) return(FALSE);
   	station_config = cb.data[0];
   
   	strcpy(codeversion, "?.??");
   
   	cb.command = FR_READ_CODE_VERSION;
  	cb.buffer_length = 0;
   	DoCommand(&cb);

   	if (cb.return_code == 0) {
      		cb.data[cb.buffer_length] = 0;
      		strcpy(codeversion, cb.data);
   	}
   	return(TRUE);
}; /* ObtainConfiguration */

unsigned char DoCommand( CBLOCK *cb ) 
{
   	static unsigned char id = 0;
   	fd_set ready;
   	struct timeval to;
   	int x;

   	for( x = 0; x < 4; x += 1 ) {
      		cb->opp_flag = 0;
      		cb->requestreply = 0x01;
      		cb->id = id;
      		cb->return_code = 0xaa;	// 0xAA is our special return code indicating packet timeout
      		send(sock, cb, 32+cb->buffer_length,0);
      		FD_ZERO(&ready);
      		FD_SET(sock,&ready);
      		to.tv_sec = 5;
      		to.tv_usec = 0;
      		if(select(sock + 1,&ready, NULL, NULL, &to)) {
	 		recv(sock, cb, 32+MDATALEN,0);
			break;
      		} else {
	 		printf("Error: No Data received from the connected socket.\n"); 
      		} /* if */
   	} 
   	/* make sure the id is correct (returning results from a previous
   	   call may be disastrous if not recognized)
   	   also make sure it is a reply
	 */	
   	if (cb->id != id || cb->requestreply != 0x02) cb->return_code = 0xbb;
   	id++;
   	if (cb->return_code == 0xCC) {
      		printf("Error: Code is not running on the board!\n");
      		exit(1);
   	}
   	return(cb->return_code);
   
}; /* DoCommand */

void init( int argc, char *argv[]){

	int i, i_cnt=0, d_cnt=0, u_cnt=0, c_cnt=0, if_found=0;
	char * ip_addr;
	struct in_addr ip_str;
	cb.id = 0;

	for (i=0;i<argc;i++){
		if (!strcmp(argv[i],"-i")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid IP or Interface Name, Type fpipemon <cr> for help\n\n");
				exit(0);
			}

			strcpy(ipaddress,argv[i+1]);
			if (inet_aton(ipaddress,&ip_str) != 0 ){
			}else{
				if_found=1;
			}
			i_cnt=1;
		}else if (!strcmp(argv[i],"-u")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid UDP PORT, Type fpipemon <cr> for help\n\n");
				exit(0);
			}

		 	if(isdigit(argv[i+1][0])){
				udp_port = atoi(argv[i+1]);
			}else{
				printf("ERROR: Invalid UDP Port, Type fpipemon <cr> for help\n\n");
				exit(0);
			}
			u_cnt=1;
		}else if (!strcmp(argv[i],"-c")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid Command, Type fpipemon <cr> for help\n\n");
				exit(0);
			}

			strcpy(cmd,argv[i+1]);
			c_cnt=1;
		}else if (!strcmp(argv[i], "-d")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid Command, Type fpipemon <cr> for help\n\n");
				exit(0);
			}
			if (isdigit(argv[i+1][0])){
				dlci_number = atoi(argv[i+1]);
			}else{
				printf("ERROR: DLCI must be an integer,Type fpipemon <cr> for help\n\n");
				exit(0);
			} 
			d_cnt=1;
		
		}
	}

	if (!i_cnt){
		printf("ERROR: No IP address or Interface Name, Type fpipemon <cr> for help\n\n");
		exit(0);
	}
	if (!u_cnt){
		printf("ERROR: No UDP Port, Type fpipemon <cr> for help\n\n");
		exit(0);
	}
	if (!c_cnt){
		printf("ERROR: No Command, Type fpipemon <cr> for help\n\n");
		exit(0);
	}
	if (!d_cnt){
		/* Default DLCI Number to 0 */
		dlci_number = 0;
	}
	
	if (if_found){
		ip_addr = get_ip_addr(ipaddress);
		strcpy(ipaddress,ip_addr);
	}

	strcpy( cb.signature, "FPIPE8ND");
 
};

char * get_ip_addr(char *if_name){

	int skfd;
	struct sockaddr_in *sin;
	struct ifreq ifr;

    	if ((skfd = socket(AF_INET, SOCK_STREAM, IPPROTO_IP)) < 0) {
		perror("socket");
		exit(1);
    	}

	strncpy(ifr.ifr_ifrn.ifrn_name, if_name, sizeof(ifr.ifr_ifrn.ifrn_name));

        if (ioctl(skfd, SIOCGIFDSTADDR , &ifr) < 0) {
		fprintf(stderr, "Error: Unknown interface: %s\n",if_name);
		exit(0);
    	}

	sin = (struct sockaddr_in *)&ifr.ifr_ifru.ifru_dstaddr;
	sin->sin_family = AF_INET;
	sin->sin_port = 0;

	close (skfd);

	return (inet_ntoa(sin->sin_addr));

}

void error( char return_code ) 
{
	switch( return_code ){
		case 0x04:
			printf("Error: An invalid DLCI was selected\n");
			break;
		case 0x10:
			printf("Error: A modem failure occurred - DCD and/or CTS were found to be unexpectedly low\n");
			break;
		case 0x11:
			printf("Error: The Channel moved from Operative to being Inoperative\n");
			break;
		case 0x12:
			printf("Error: The Channel moved from Inoperative to being Operative\n");
			break;
		case 0x13:
			printf("Error: The Access Node has reported a change in the status of a DLCI or a number of DLCIs\n");
			break;
		case 0x14:
			printf("Error: A Full Status Report included a DLCI or a number of DLCIis which were not included before\n"); 
			break;
		case 0x1F:
			printf("Error: The frame relay command is invalid\n");
			break;
		default:
			break;
   	}		
}; /* error */

void link_status( void ) 
{
	cb.command = FR_READ_STATUS;
	cb.buffer_length = 0;
	cb.dlci = 0;
	DoCommand(&cb);
	if (cb.return_code == 0) {

		banner("PHYSICAL LINK STATUS", 0);

		if (cb.data[0])
			printf("Channel status: OPERATIVE\n");
		else
	 		printf("Channel status: INOPERATIVE\n");
	} else {
		error(cb.return_code);	  	
	} 
}; /* link_status */
 
void modem_status( void ) 
{
   	cb.command = FR_READ_MODEM_STATUS;
   	cb.buffer_length = 0;
   	DoCommand(&cb);
   
	if (cb.return_code == 0) {

		banner("MODEM STATUS", 0);

      		if (cb.data[0] & 0x08) 
	 		printf("DCD is HIGH\n");
      		else
	 		printf("DCD is LOW\n");
      	
      		if( cb.data[0] & 0x20) 
	 		printf("CTS is HIGH\n");
      		else 
	 		printf("CTS is LOW\n");
      	} else {
      		error(cb.return_code);
   	} 

}; //modem_status
 

void comm_err() 
{
	ResetCB(&cb);
   	cb.command = FR_READ_ERROR_STATS;
   	cb.buffer_length = 0;
   	cb.dlci = 0;	// for supervisor display
   
	DoCommand(&cb);
   
	if (cb.return_code == 0 && cb.buffer_length == 0x0A) {

		banner("COMMUNICATION ERROR STATISTICS",0);

      		printf("               Number of receiver overrun errors: %d\n",cb.data[0]);
      		printf("                   Number of receiver CRC errors: %d\n",cb.data[1]);
      		printf("                 Number of abort frames received: %d\n",cb.data[2]);
      		printf("Number of times receiver disabled (buffers full): %d\n",cb.data[3]);
      		printf("              Number of abort frames transmitted: %d\n",cb.data[5]);
      		printf("   Number of transmit underrun interrupts missed: %d\n",cb.data[6]);
      		printf("        Number of times DCD dropped unexpectedly: %d\n",cb.data[8]);
      		printf("        Number of times CTS dropped unexpectedly: %d\n",cb.data[9]);
   	} else {
      		error(cb.return_code);
   	} 
}; /* comm_err(); */

void flush_comm_err( void ) 
{
	cb.command = FR_FLUSH_ERROR_STATS;
	cb.buffer_length = 0;
	cb.dlci = 0;
	DoCommand(&cb);
	if (cb.return_code != 0) 
		error(cb.return_code);
}; /* flush_comm_err */

void global_stats( void ) 
{
   	ResetCB(&cb);
   	cb.command = FR_READ_STATISTICS;
   	cb.buffer_length = 0;
   	cb.dlci = 0;
   	DoCommand(&cb);
   
	if (cb.return_code == 0) {

		banner("GLOBAL STATISTICS",0);

      		if( station_config == 0 ) {
	 		printf("                       Full Status Enquiry messages sent: %d\n", *(unsigned short*)&cb.data[12]);
	 		printf("Link Integrity Verification Status Enquiry messages sent: %d\n", *(unsigned short*)&cb.data[14]);
	 		printf("                           Full Status messages received: %d\n", *(unsigned short*)&cb.data[16]);
	 		printf("    Link Integrity Verification Status messages received: %d\n", *(unsigned short*)&cb.data[18]);
      		} else {
	 		printf("                       Full Status Enquiry messages sent: %d\n", *(unsigned short*)&cb.data[20]);
	 		printf("Link Integrity Verification Status Enquiry messages sent: %d\n", *(unsigned short*)&cb.data[22]);
	 		printf("                           Full Status messages received: %d\n", *(unsigned short*)&cb.data[24]);
	 		printf("    Link Integrity Verification Status messages received: %d\n", *(unsigned short*)&cb.data[26]);
      		} //if
      		printf("                                     CPE initializations: %d\n", *(unsigned short*)&cb.data[42]);
      		printf("                            current Send Sequence Number: %d\n", *(unsigned short*)&cb.data[44]);
      		printf("                         current Receive Sequence Number: %d\n", *(unsigned short*)&cb.data[46]);
      		printf("                                      current N392 count: %d\n", *(unsigned short*)&cb.data[52]);
      		printf("                                      current N393 count: %d\n", *(unsigned short*)&cb.data[54]);
   	} else {
		error(cb.return_code);	  	
  	 } 
}; /* global_stats */

void flush_global_stats( void ) 
{
   	cb.command = FR_FLUSH_STATISTICS;
   	cb.buffer_length = 1;
   	cb.dlci = 0;
	cb.data[0] = 0x01;
   	DoCommand(&cb);
   
	if (cb.return_code != 0) { 
		switch(cb.return_code){
			case 0x06:
				printf("Error: Global Statistics not flushed\n");
				break;
			default:
				error(cb.return_code);
				break;	  
		}
	}	
}; /* flush_global_stats */

void error_stats( void ) 
{
	ResetCB(&cb);
	cb.command = FR_READ_STATISTICS;
   	cb.buffer_length = 0;
   	cb.dlci = 0;
   	DoCommand(&cb);

	if (cb.return_code == 0) {

		banner("ERROR STATISTICS",0);

      		printf("  I-frames not transmitted after a tx. int. due to exessive frame length: %d\n",*(unsigned short*)&cb.data[0]);
      		printf("   I-frames not transmitted after a tx. int. due to excessive throughput: %d\n",*(unsigned short*)&cb.data[2]);
      		printf("     Received frames discarded as they were either too short or too long: %d\n",*(unsigned short*)&cb.data[4]);
      		printf("                               discarded I-frames with unconfigured DLCI: %d\n",*(unsigned short*)&cb.data[6]);
      		printf("                                discarded I-frames due to a format error: %d\n",*(unsigned short*)&cb.data[8]);
      		printf("App. didn't respond to the triggered IRQ within the given timeout period: %d\n",*(unsigned short*)&cb.data[10]);
      		printf("            discarded In-channel Signalling frames due to a format error: %d\n",*(unsigned short*)&cb.data[28]);
      		printf("   In-channel frames received with an invalid Send Seq. Numbers received: %d\n",*(unsigned short*)&cb.data[32]);
      		printf("In-channel frames received with an invalid Receive Seq. Numbers received: %d\n",*(unsigned short*)&cb.data[34]);
      		if( station_config == 0 ) {
	 		printf("                    Number of unsolicited responses from the Access Node: %d\n",*(unsigned short*)&cb.data[30]);
	 		printf("                                              timeouts on the T391 timer: %d\n",*(unsigned short*)&cb.data[36]);
	 		printf("                                  consecutive timeouts on the T391 timer: %d\n",*(unsigned short*)&cb.data[48]);
      		} else {
	 		printf("                                              timeouts on the T392 timer: %d\n",*(unsigned short*)&cb.data[38]);
	 		printf("                                  consecutive timeouts on the T392 timer: %d\n",*(unsigned short*)&cb.data[50]);
      		} 
      		printf("times that N392 error threshold was reached during N393 monitored events: %d\n",*(unsigned short*)&cb.data[40]);
   	} 
}; /* error_stats */


void line_trace(int trace_mode) 
{
	unsigned char num_frames, num_chars;
   	unsigned short curr_pos = 0, dlci;
	fpipemon_trc_t *fpipemon_trc;
   	unsigned int i, j;
   	unsigned char outstr[220];
   	int recv_buff = sizeof(CBLOCK) + 100;
  	fd_set ready;
   	struct timeval to;
   
   	setsockopt( sock, SOL_SOCKET, SO_RCVBUF, &recv_buff, sizeof(int) );

	cb.command = FPIPE_DISABLE_TRACING;
	cb.buffer_length = 0;
	cb.data[0] = 0;
	DoCommand(&cb);

   	cb.command = FPIPE_ENABLE_TRACING;
	cb.buffer_length = 1;
	cb.data[0] = 0;
	if(trace_mode == TRACE_PROT){
		cb.data[0] |= TRC_SIGNALLING_FRMS | TRC_ACTIVE; 
	}else if(trace_mode == TRACE_DATA){
		cb.data[0] = TRC_INFO_FRMS | TRC_ACTIVE; 
	}else{
		cb.data[0] |= TRC_SIGNALLING_FRMS | TRC_INFO_FRMS | TRC_ACTIVE;
	}
   	DoCommand(&cb);

   	if( cb.return_code == 0 ) { 
      		printf("Starting trace...(Press ENTER to exit)\n");
   	} else if (cb.return_code == 0x1F) {
      		printf("Line Tracing is possible only with S508 board.\n");
      		return;
   	} else if( cb.return_code == 0xCD ) {
      		printf("Cannot Enable Line Tracing from Underneath.\n");
      		return;
   	} else if( cb.return_code == 0x01 ) {
      		printf("Starting trace...(although it's already enabled!)\n");
      		printf("Press ENTER to exit.\n");
   	} else {
      		printf("Failed to Enable Line Tracing. Return code: 0x%02X\n", cb.return_code );
      		return;
   	}
   
	for(;;) {
      	    FD_ZERO(&ready);
      	    FD_SET(0,&ready);
      	    to.tv_sec = 1;
      	    to.tv_usec = 0;
     
	    if(select(1,&ready, NULL, NULL, &to)) {
	 	break;
      	    }
      
	    cb.command = FPIPE_GET_TRACE_INFO;
      	    cb.buffer_length = 0;
      	    DoCommand(&cb);
    
	    if (cb.return_code == 0 && cb.buffer_length) { 

		/*  get number of frames */
 		num_frames = (cb.data[0] >> 1) & MAX_FRMS_TRACED;

		for (i = 0; i < num_frames; i++) {

			fpipemon_trc = (fpipemon_trc_t *)(&cb.data[curr_pos]);
    		   /*  frame type */
    		   if (fpipemon_trc->fpipemon_trc_hdr.status & TRC_OUTGOING_FRM) {
       		   	sprintf(outstr,"OUTGOING\t");
    		   } else {
       		   	sprintf(outstr, "INCOMING\t");
    		   }

    		   /* real length and time stamp */
    		   sprintf(outstr+strlen(outstr), "%d\t%d\t\0", fpipemon_trc->fpipemon_trc_hdr.length, fpipemon_trc->fpipemon_trc_hdr.tmstamp);
	    			
		   /*  first update curr_pos */
	    	   curr_pos += sizeof(fpipemon_trc_hdr_t);

			if(fpipemon_trc->fpipemon_trc_hdr.status &
			(TRC_ABORT_ERROR | TRC_CRC_ERROR | TRC_OVERRUN_ERROR)) {
				sprintf(outstr+strlen(outstr), "Receive error - ");

				if(fpipemon_trc->fpipemon_trc_hdr.status & TRC_ABORT_ERROR)
					sprintf(outstr+strlen(outstr), "Abort");
				else
					sprintf(outstr+strlen(outstr),(fpipemon_trc->fpipemon_trc_hdr.status & TRC_CRC_ERROR) ? "Bad CRC" : "Overrun");

			}	
	    	   else if (fpipemon_trc->fpipemon_trc_hdr.length == 0) {
	       	   	sprintf( outstr+strlen(outstr), "the frame data is not available" );
	    	   } else {
	      		/*  update curr_pos again */
	       	  	curr_pos += fpipemon_trc->fpipemon_trc_hdr.length;
	       		num_chars = (unsigned char)((fpipemon_trc->fpipemon_trc_hdr.length <= 25)? fpipemon_trc->fpipemon_trc_hdr.length:25);

       			if (raw_data) {
			   /* show raw data */
	 		   for( j=0; j<num_chars; j++ ) {
	     			sprintf(outstr+strlen(outstr), "%02X \0", (unsigned char)fpipemon_trc->data[j]);
	  		   }
	  		   outstr[strlen(outstr)-1] = '\0';
	      	  	} else { 
			   /* show intrptd. data */
	  		   dlci = (unsigned short)fpipemon_trc->data[0];
	  		   dlci = (dlci << 2) | (fpipemon_trc->data[1] >> 4);
	  		   if (dlci != 0) {
				/* An Inverse Arp packet */
				if (fpipemon_trc->data[2] == 0x03 &&
				    fpipemon_trc->data[3] == 0x00 &&
				    fpipemon_trc->data[4] == 0x80 &&
				    fpipemon_trc->data[8] == 0x08 &&
				    fpipemon_trc->data[9] == 0x06) {
					/* Request packet */
					if(fpipemon_trc->data[17] == 0x08)
						sprintf(outstr+strlen(outstr), "Inverse Arp \tRequest on DLCI %d", dlci);
					else if(fpipemon_trc->data[17] == 0x09)
						sprintf(outstr+strlen(outstr), "Inverse Arp \tReply on DLCI %d", dlci);
				} else 
	     				sprintf(outstr+strlen(outstr), "Data Frame on DLCI %d", dlci);
	  		   } else {
	     			if(fpipemon_trc->data[9])
		 		    sprintf(outstr+strlen(outstr), "Link Verification");
	     			else
		 	 	    sprintf(outstr+strlen(outstr), "Full Status");
	     			if (fpipemon_trc->data[5] == 0x75)
		 		    sprintf(outstr+strlen(outstr), "\tRequest");
	    	 		else
		 		    sprintf(outstr+strlen(outstr), "\tReply");
	     			
				sprintf(outstr+strlen(outstr), "\tSx %02X  Rx %02X  ", (unsigned char)fpipemon_trc->data[0x0C], (unsigned char)fpipemon_trc->data[0x0D] );
	     			if ( !fpipemon_trc->data[9] && (fpipemon_trc->data[5]==0x7D) ) {
				   /* full status reply */
				   for( j=0x0E; ( ((j+5)<fpipemon_trc->fpipemon_trc_hdr.length) && (j<18)); j+=5 ) {
		   			dlci = (unsigned short)fpipemon_trc->data[j+2];
		   			dlci = (dlci << 4) | ((fpipemon_trc->data[j+3]>>3) & 0x0F);
		   			sprintf(outstr+strlen(outstr), "DLCI %d ", dlci);
		   			if (fpipemon_trc->data[j+4] & 0x08)
		       			   sprintf(outstr+strlen(outstr), "NEW %d", dlci);
		   			else
		       			   sprintf(outstr+strlen(outstr), "PRS %d", dlci);
		   			if (fpipemon_trc->data[j+4] & 0x02)
		       			   sprintf(outstr+strlen(outstr), "ACT\t%d", dlci);
		   			else
		       		  	   sprintf(outstr+strlen(outstr), "INACT\t%d", dlci);
				  }
	     			}
	  		   }
		 	}
	  	   }

		printf("%s\n", outstr);

	 	} /* end of for */
      	   } else if (cb.return_code != 0) 
		error(cb.return_code);
      	
	   curr_pos = 0;
      	   if (!(cb.data[0] & MORE_TRC_DATA)) {
		 sleep(TIMEOUT);
      	   } 
  	}
   	cb.command = FPIPE_DISABLE_TRACING;
   	cb.buffer_length = 0;
   	DoCommand(&cb);
}; //line_trace

void list_dlcis( void )
{
   	int i,cnt;

   	cb.command = FR_LIST_ACTIVE_DLCI;
   	cb.buffer_length = 0;
   	DoCommand(&cb);
  
	banner("LIST OF ACTIVE DLCIs",0);
 
	if( cb.return_code == 0){	
     		cnt = cb.buffer_length;
     		if( cnt != 0 ){
       			printf("ACTIVE DLCIs:\n");
       			for(i=0; i< cnt; i+=2){
       	  			printf("DLCI %u\n", *(unsigned short*)&cb.data[i]);	
       			}
     		}else{
       			printf("There are NO ACTIVE DLCIs\n"); 
     		}
   	} else {
      		error(cb.return_code);
   	}	
} /* list_dlcis */

void read_dlci_stat( void )
{
     	cb.command = FR_READ_STATISTICS;
     	cb.dlci = dlci_number;
     	cb.buffer_length = 1;
     	cb.data[0] = 0;
     
	DoCommand(&cb);
     	if( cb.return_code != 0 ){
		error(cb.return_code);
		return;
     	}

	banner("STATISTICS FOR DLCI",dlci_number);

     	if( (cb.return_code == 0) && (cb.buffer_length == 0x20)){
		printf("                                Information frames transmitted: %ld\n", *(unsigned long*)&cb.data[0]);
		printf("                                 Information bytes transmitted: %ld\n", *(unsigned long*)&cb.data[4]);
		printf("                                   Information frames received: %ld\n", *(unsigned long*)&cb.data[8]);
		printf("                                    Information bytes received: %ld\n", *(unsigned long*)&cb.data[12]);
		printf("              Received I-frames discarded due to inactive DLCI: %ld\n", *(unsigned long*)&cb.data[20]);
		printf(" I-frames received with Discard Eligibility (DE) indicator set: %ld\n", *(unsigned long*)&cb.data[28]); 
     	}
     
	cb.command = FR_READ_ADD_DLC_STATS;
     	cb.dlci = dlci_number;
     	cb.buffer_length = 0;
     	cb.data[0] = 0;
     	DoCommand(&cb);
     
	if( cb.return_code == 0 ){
		printf("                       I-frames received with the FECN bit set: %d\n", *(unsigned short*)&cb.data[0]);
		printf("                       I-frames received with the BECN bit set: %d\n", *(unsigned short*)&cb.data[2]);
	
     	} else  
		printf("Error: Please enter a non-zero DLCI\n");

} /* read_dlci_stat */

void flush_dlci_stats( void )
{
     	cb.command = FR_FLUSH_STATISTICS;
     	cb.dlci = dlci_number;
     	cb.buffer_length = 0;
     	DoCommand(&cb);
     	
	if( cb.return_code != 0 ) {
		switch( cb.return_code ){
			case 0x06:
				printf("DLCI Statistics are not flushed\n");
				break;
			default:
				error(cb.return_code);
		}
	}
} /* flush_dlci_stats */

void set_FT1_monitor_status( unsigned char status) 
{
	fail = 0;
     	cb.command = FR_FT1_STATUS_CTRL;
      	cb.buffer_length = 1;
      	cb.data[0] = status; 	
      	DoCommand(&cb);
      
	if( cb.return_code != 0 && status){
		fail = 1;
		if( cb.return_code == 0xCD )
                	printf("Error:  Cannot run this command from Underneath.\n");
		else 
			printf("Error:  This command is only possible with S508/FT1 board!");
      	}

} /* set_FT1_monitor_status */

void set_FT1_mode( void ){
 
	for(;;){
		cb.command = FR_SET_FT1_MODE;
		cb.buffer_length = 0;
		DoCommand(&cb);
		if(cb.return_code == 0)
			break;
		else if( cb.return_code == 0xCD ){
                	printf("Error: Cannot run this command from Underneath.\n");
			exit(1);
		}

	}
} /* set_FT1_mode */

void read_FT1_status( void ){
     int i;
     long delay;
     struct timeval tv;

    
     i = gettimeofday(&tv,NULL);
     delay = tv.tv_usec;

     for(;;){
      i = gettimeofday(&tv,NULL);
      if((abs(tv.tv_usec - delay)) > 90000)
	break;  
     }

     cb.command = FPIPE_FT1_READ_STATUS;
     cb.buffer_length = 0;
     DoCommand(&cb); 
     if( cb.return_code == 0 ){
	par_port_A_byte = cb.data[0];
	par_port_B_byte = cb.data[1];
     }
} /* read_FT1_status */

void view_FT1_status( void ){
     
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     loop_counter = 0;
     /* check for INS light */
     for(;;){
     	read_FT1_status();
     	if((par_port_B_byte & 0x30) == 0x30)
 		off_counter++;
     	if((par_port_B_byte & 0x10) == 0x00)
		red_counter++;
     	if((par_port_B_byte & 0x20) == 0x00)
		green_counter++;
	if(red_counter != 0 && off_counter != 0 && loop_counter == 30 ) {
		printf("Unit is not IN SERVICE\n");
		break;
	}
	if(green_counter != 0 && off_counter == 0 && loop_counter == 30){
		printf("Unit is IN SERVICE\n");
		break;
	}
	if(green_counter != 0 && off_counter != 0 && loop_counter == 30){
		printf("INS is flashing green\n");
		break;
	}
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 30){
		printf("INS is off\n");
		break;
	}
	loop_counter++;
     }	
     /* check for ERR light */
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     loop_counter = 0;
     for(;;){
	read_FT1_status();
	if((par_port_B_byte & 0x0C) == 0x0C)
		off_counter++;
	if((par_port_B_byte & 0x08) == 0x00)
		red_counter++;
	if((par_port_B_byte & 0x04) == 0x00)
		green_counter++;
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 30){
		printf("No Line Errors: Valid Line\n");
		break;
	}
	if(red_counter != 0 && off_counter == 0 && green_counter == 0 && loop_counter == 30){
		printf("Line Errors: Invalid Line\n");
		break;
	}

	loop_counter++;
     }

     /* check TXD light */
     loop_counter = 0;
     off_counter = 0;
     green_counter = 0;
     for(;;){
	read_FT1_status();
	if((par_port_B_byte & 0x02) == 0x02)
		off_counter++;
	if((par_port_B_byte & 0x02) == 0x00)
		green_counter++;
	if(off_counter != 0 && green_counter == 0 && loop_counter == 20){
		printf("Transmit data is not present\n");	
		break;
	}
	if(off_counter != 0 && green_counter != 0 && loop_counter == 20){
		printf("Transmit data is present \n");
		break;
	}
	loop_counter++;
     }

     /* check RXD light */
     loop_counter = 0;
     off_counter = 0;
     green_counter = 0;
     for(;;){
	read_FT1_status();
	if((par_port_B_byte & 0x01) == 0x01)
		off_counter++;
	if((par_port_B_byte & 0x01) == 0x00)
		green_counter++;
	if(off_counter != 0 && green_counter == 0 && loop_counter == 20){
		printf("Receive data is not present\n");	
		break;
	}
	if(off_counter != 0 && green_counter != 0 && loop_counter == 20){
		printf("Receive data is present\n");
		break;
	}
	loop_counter++;
     }

}/* view_FT1_status */


void FT1_operational_mode(void){
     
     printf("Operational Mode has been selected\n"); 
     printf("Putting S508/FT1 in operational mode....");
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
     	read_FT1_status();
     	/* ST light is OFF */
     	if((par_port_B_byte & 0xc0) == 0xc0 ){
		off_counter++;
     	}
     	/* ST light is GREEN */
     	if((par_port_B_byte & 0x40) == 0x00){
		green_counter++;
		red_counter = 0;
     	}
     	/* ST light is RED */
     	if((par_port_B_byte & 0x80) == 0x00){
		red_counter++;
		green_counter = 0;
   	}
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 20){
		break;
	}
	if((red_counter != 0 || green_counter != 0) && loop_counter == 20){
		set_FT1_mode();
		break;
	}

   	loop_counter++;
     } /* end of for */
       
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
   	read_FT1_status();
   	/* DL light is OFF */
   	if((par_port_A_byte & 0x08) == 0x08){
		off_counter++;
	}
        /* DL light is RED */
   	if((par_port_A_byte & 0x08) == 0x00){
		red_counter++;
	}
	if(off_counter != 0 && red_counter == 0 && loop_counter == 20){
		break;
	}
	if(red_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		break;
	}
        loop_counter++; 
     } 
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
   	read_FT1_status();
   	/* LL light is off */
   	if((par_port_A_byte & 0x04) == 0x04){
		off_counter++;
	}
   	if((par_port_A_byte & 0x04) == 0x00){
		red_counter++;
	}
	if(off_counter != 0 && red_counter == 0 && loop_counter == 20){
		break;
	}
	if(red_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		break;
	}
        loop_counter++;
     }
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
   	read_FT1_status();
	/* RT light is OFF */
   	if((par_port_A_byte & 0x03) == 0x03){
		off_counter++;
	}
	/* RT light is RED */
   	if((par_port_A_byte & 0x01) == 0x00){
		red_counter++;
	}
	/* RT light is GREEN */
   	if((par_port_A_byte & 0x02) == 0x00){
   		green_counter++;
	}
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 20){
		printf("Done!\n");
		break;
	}
	if((red_counter != 0 || green_counter != 0) && off_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		printf("Done!\n");
		break;
	}
	if(off_counter == 0 && green_counter != 0 && loop_counter == 20){
		printf("Failed!\n");
		printf("Remote End is running Remote Test\n");
		printf("Exit Remote Test at remote end\n");
		break;
	}
	loop_counter++;
     }        	

} /* FT1_operational_mode */


void FT1_self_test(void){
     int started = 0; 
     int selftest = 0;
 
     set_FT1_mode();
     off_counter = 0;
     green_counter = 0;
     red_counter = 0;
     loop_counter = 0;
     printf("Self Test has been selected\n");
     for(;;){
	  read_FT1_status();
	  /* ST light is OFF */
	  if((par_port_B_byte & 0xc0) == 0xc0){
		off_counter++;
	  }
	  /* ST light is GREEN */
	  if((par_port_B_byte & 0x40) == 0x00){
		green_counter++;
	  }
	  /* ST light is RED */
	  if((par_port_B_byte & 0x80) == 0x00){
		red_counter++;
	  }
          if(red_counter == 0 && loop_counter == 3){
		set_FT1_mode();    
		off_counter = 0;
		red_counter = 0;
		green_counter = 0;
		loop_counter = -1;
     		printf("Selecting Self Test....\r");
		selftest++;
	  	if( selftest == 10){
		  printf("\t\t\tFailed!\n");
		  printf("Self Test will not operate while connected to a valid line\n");
		  FT1_operational_mode();
		  break;
	        }
	  }
	  if(red_counter != 0 && off_counter != 0 && (loop_counter % 2)){
		printf("Performing Self Test....\r");
	        started = 1;
	  }
	  if(green_counter != 0 && off_counter != 0 && started){
	   	printf("\t\t\tDone!\n");
		break;
	  }
          loop_counter++;	  
     }/* end of for */     
} /* FT1_self_test */


void FT1_dl_test( void ){
     int dl_test=0;

     set_FT1_mode();
     off_counter = 0;
     red_counter = 0;
     loop_counter = 0;
     printf("Bi-directional Digital Loop has been selected\n"); 
     for(;;){
	read_FT1_status();
	if((par_port_A_byte & 0x08) == 0x08){
		off_counter++;  
	}
	if((par_port_A_byte & 0x08) == 0x00){
		red_counter++;
	}
	if(red_counter == 0 && loop_counter == 20){
		set_FT1_mode();
		off_counter = 0;
		red_counter = 0;
		loop_counter = -1;
		printf("Selecting Digital Loop Test....\r");
		dl_test++;
		if(dl_test==10){
	 		printf("\t\t\t\tFailed\n");
			printf("Remote End might be running Remote Test\n");
			break;
		}
	}
	if(red_counter != 0){
		off_counter = 0;
		red_counter = 0;
		green_counter = 0;
		loop_counter = 0;
		printf("Performing Digital Loop Test....\r");
		for(;;){
			read_FT1_status();
			printf("Performing Digital Loop Test....\r");
			/* check INS light */
			if((par_port_B_byte & 0x30) == 0x30)
				off_counter++;
			if((par_port_B_byte & 0x10) == 0x00){
				red_counter++;
				green_counter = 0;
			}
			if((par_port_B_byte & 0x20) == 0x00){
				green_counter++;
				red_counter = 0;
			}
			if(red_counter != 0 && (par_port_B_byte & 0x08) == 0x00 ){
				printf("\t\t\t\tFailed!\n");
				printf("Bi-directional Digital Loop test has failed\n");
				printf("Either the unit is not connected or the line is Invalid\n");
				break;
			}
			if(green_counter != 0  && (par_port_B_byte & 0x0C) == 0x0C && loop_counter == 100 ){
				printf("\t\t\t\tDone!\n");
				printf("Bi-directional Digital Loop test has been successfully\n");
				printf("completed\n");
				break;
		
			}
			loop_counter++;
		} /* end of for */
		break;	
	} 
	loop_counter++;
     } /* end of for */

} /* FT1_dl_test */

void FT1_ll_test( void ){
    int ll_test = 0;

    set_FT1_mode();	
    off_counter = 0;
    red_counter = 0;
    loop_counter = 0;
    printf("Line Loop Test has been selected\n");
    for(;;){
	read_FT1_status();
	if((par_port_A_byte & 0x04) == 0x04){
		off_counter++;
	}
	if((par_port_A_byte & 0x04) == 0x00){
		red_counter++;
	}
	if(red_counter == 0 && off_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		off_counter = 0;
		red_counter = 0;
		loop_counter = -1;
		printf("Selecting Line Loop Test....\r");
		ll_test++;
		if(ll_test == 10){
			printf("\t\t\t\tFailed!\n");
			printf("Line Loop Test will not operate while connected to a valid line\n");
			FT1_operational_mode();
		 	break;	
		}
	}
	if(red_counter != 0){	
		off_counter = 0;
		red_counter = 0;
		loop_counter = 0;
		for(;;){
			printf("Performing Line Loop Test....\r");
			read_FT1_status();	
			/* check INS light */
			if((par_port_B_byte & 0x30) == 0x30)
				off_counter++;
			if((par_port_B_byte & 0x10) == 0x00){
				red_counter++;
				green_counter = 0;
			}
			if((par_port_B_byte & 0x20) == 0x00){
				green_counter++;
				red_counter = 0;
			}
			if(green_counter != 0 && red_counter == 0){
				printf("\t\t\t\tDone!\n");
				printf("Line Loop Test has been successfully completed\n");
				break;
			}
			if(red_counter != 0 && green_counter == 0 && loop_counter == 100){
				printf("\t\t\t\tFailed!\n");
				break;
			}
			loop_counter++;
		} /* end of for */
		break;
	} /* end of if */
	loop_counter++;
    } /* end of for */

} /* FT1_ll_test */

void FT1_rt_test( void ){
 
    off_counter = 0;
    red_counter = 0;
    green_counter = 0;	  	
    loop_counter = 0;
    printf("Remote Test has been selected\n");
    for(;;){
	read_FT1_status();
	if((par_port_A_byte & 0x03) == 0x03)
		off_counter++;
	if((par_port_A_byte & 0x01) == 0x00)
		red_counter++;
	if((par_port_A_byte & 0x02) == 0x00)
		green_counter++;
	if(red_counter == 0 && loop_counter == 20){
		set_FT1_mode();
		off_counter = 0;
		red_counter = 0;
		green_counter = 0;
		loop_counter = -1;
		printf("Selecting Remote Test....\r");		
	}
	if(green_counter != 0 && loop_counter == 10){
	   loop_counter = 0;
	   off_counter = 0;
	   for(;;) {
		read_FT1_status();
		if((par_port_A_byte & 0x03) == 0x03)
			off_counter++;
	        if(off_counter == 0 && loop_counter == 20){
		   printf("Remote End is currently sending Remote Test data\n");
		   printf("Exit from the Remote Test mode at the remote end\n");
		   break;
		}
		if(off_counter != 0 && loop_counter == 20) {			
		   printf("This unit is currently in Remote Testing Mode\n");
		   break;
		}
	        loop_counter++;
	   }
	   break;
	}
	if(red_counter != 0){
		printf("Waiting for a valid response from remote end....\r");
		loop_counter = 0;
		for(;;){
			read_FT1_status();
			if((par_port_A_byte & 0x02) == 0x00){
				printf("\t\t\t\t\t\tDone!\n");
				printf("Valid response has been received from remote end\n");
				printf("Remote Test has been successfully completed\n");
				break;	
			}
			if((par_port_B_byte & 0x08) == 0x00){
				printf("\t\t\t\t\t\tFailed!\n");
				printf("The test can only be successful between two Sangoma\n");
				printf("S508/FT1 units configured the SAME WAY\n");

				break;
			}
		} /* end of for */
		break;
	}
    	loop_counter++;	
    } /* end of for */	
} /* FT1_rt_test */



void fr_driver_stat_ifsend( void )
{
	if_send_stat_t *stats;
      	cb.command = FPIPE_DRIVER_STAT_IFSEND;
      	cb.buffer_length = 0;
      	cb.data[0] = 0;
      	DoCommand(&cb);
     
	stats = (if_send_stat_t *)&cb.data[0];	
 
      	banner("DRIVER IF_SEND STATISTICS",0);

      	printf("                                    Total Number of If_Send entries:  %ld\n",
		stats->if_send_entry);
      	printf("                          Number of If_Send entries with SKB = NULL:  %ld\n", 
		stats->if_send_skb_null);
      	printf("Number of If_Send entries with broadcast addressed packet discarded:  %ld\n",
		 stats->if_send_broadcast);
      	printf("Number of If_Send entries with multicast addressed packet discarded:  %ld\n",
		 stats->if_send_multicast);
      	printf("                Number of If_Send entries with CRITICAL_RX_INTR set:  %ld\n", 
		stats->if_send_critical_ISR);
      	printf("   Number of If_Send entries with Critical set and packet discarded:  %ld\n", 
		stats->if_send_critical_non_ISR);
      	printf("                     Number of If_Send entries with Device Busy set:  %ld\n", 
		stats->if_send_tbusy);
      	printf("                 Number of If_Send entries with Device Busy Timeout:  %ld\n", 
		stats->if_send_tbusy_timeout);
      	printf("               Number of If_Send entries with FPIPE MONITOR Request:  %ld\n", 
		stats->if_send_PIPE_request);
      	printf("                    Number of If_Send entries with WAN Disconnected:  %ld\n", 
		stats->if_send_wan_disconnected);
      	printf("                   Number of If_Send entries with DLCI Disconnected:  %ld\n", 
		stats->if_send_dlci_disconnected);
      	printf("            Number of If_Send entries with check for Buffers failed:  %ld\n", 
		stats->if_send_no_bfrs);
      	printf("                         Number of If_Send entries with Send failed:  %ld\n", 
		stats->if_send_adptr_bfrs_full);
      	printf("                         Number of If_Send entries with Send passed:  %ld\n", 
		stats->if_send_bfr_passed_to_adptr);
      	printf("                   Number of Consecutive send failures for a packet:  %ld\n", 
		stats->if_send_consec_send_fail);
      
} /* fr_driver_stat_ifsend */

void fr_driver_stat_intr( void )
{

	global_stats_t *global_stats;
	rx_intr_stat_t *rx_intr_stats;	 	
      	cb.command = FPIPE_DRIVER_STAT_INTR;
      	cb.buffer_length = 0;
      	cb.data[0] = 0;
      	DoCommand(&cb);

	
     	global_stats = (global_stats_t *)&cb.data[0];
	rx_intr_stats = (rx_intr_stat_t *)&cb.data[sizeof(global_stats_t)];

      	banner ("DRIVER INTERRUPT STATISTICS",0);      
 
      	printf("                                   Number of ISR entries:    %ld\n" , 
		global_stats->isr_entry);
      	printf("                 Number of ISR entries with Critical Set:    %ld\n" , 
		global_stats->isr_already_critical);
      	printf("                             Number of Receive Interrupt:    %ld\n" , 
		global_stats->isr_rx);
      	printf("                            Number of Transmit Interrupt:    %ld\n" , 
		global_stats->isr_tx);
      	printf("             Number of ISR entries for Interrupt Testing:    %ld\n" , 
		global_stats->isr_intr_test);
      	printf("                            Number of Spurious Interrupt:    %ld\n" , 
		global_stats->isr_spurious);
      	printf("      Number of Times Transmit Interrupts Enabled in ISR:    %ld\n" ,
		global_stats->isr_enable_tx_int);
      	printf("   Number of Transmit Interrupts with Device Not Started:    %ld\n" , 
		global_stats->tx_intr_dev_not_started);
      	printf("        Number of Receive Interrupts with Corrupt Buffer:    %ld\n" , 
		global_stats->rx_intr_corrupt_rx_bfr);
      	printf("         Number of Receive Interrupts with orphaned DLCI:    %ld\n" , 
		global_stats->rx_intr_on_orphaned_DLCI);
      	printf("             Number of Receive Interrupts with No socket:    %ld\n" , 
		rx_intr_stats->rx_intr_no_socket);
      	printf("    Number of Receive Interrupts with Device Not Started:    %ld\n" ,
		rx_intr_stats->rx_intr_dev_not_started);
      	printf("  Number of Receive Interrupts for FPIPE MONITOR Request:    %ld\n" , 
		rx_intr_stats->rx_intr_PIPE_request);
      	printf("      Number of Receive Interrupts with Buffer discarded:    %ld\n" , 
		rx_intr_stats->rx_intr_bfr_not_passed_to_stack);
      	printf("Number of Receive Interrupts with Buffer Passed to Stack:    %ld\n" , 
		rx_intr_stats->rx_intr_bfr_passed_to_stack);
     

} /* fr_driver_stat_intr */

void fr_driver_stat_gen( void )
{
	pipe_mgmt_stat_t *pipe_mgmt_stats;
	global_stats_t *global_stats;
      	cb.command = FPIPE_DRIVER_STAT_GEN;
      	cb.buffer_length = 0;
      	cb.data[0] = 0;
      	DoCommand(&cb);
      
	pipe_mgmt_stats = (pipe_mgmt_stat_t *)&cb.data[0];
	global_stats = (global_stats_t *)&cb.data[sizeof(pipe_mgmt_stat_t)];
	
      	banner("GENERAL DRIVER STATISTICS",0);

      	printf("            Number of FPIPE Monitor call with kmalloc error:  %ld\n", 
		pipe_mgmt_stats->UDP_PIPE_mgmt_kmalloc_err);
      	printf("       Number of FPIPE Monitor call with Adapter Type error:  %ld\n", 
		pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_type_err);
      	printf("          Number of FPIPE Monitor call with Direction Error:  %ld\n", 
		pipe_mgmt_stats->UDP_PIPE_mgmt_direction_err);
      	printf("  Number of FPIPE Monitor call with Adapter Command Timeout:  %ld\n", 
		pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_cmnd_timeout);
      	printf("       Number of FPIPE Monitor call with Adapter Command OK:  %ld\n", 
		pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_cmnd_OK);
      	printf("      Number of FPIPE Monitor call with Adapter Send passed:  %ld\n", 
		pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_send_passed);
      	printf("      Number of FPIPE Monitor call with Adapter Send failed:  %ld\n", 
		pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_send_failed);
      	printf("                Number of FPIPE Monitor call with no socket:  %ld\n", 
		pipe_mgmt_stats->UDP_PIPE_mgmt_no_socket);
      	printf("  Number of FPIPE Monitor call with pkt not passed to stack:  %ld\n", 
		pipe_mgmt_stats->UDP_PIPE_mgmt_not_passed_to_stack);
      	printf("      Number of FPIPE Monitor call with pkt passed to stack:  %ld\n", 
		pipe_mgmt_stats->UDP_PIPE_mgmt_passed_to_stack);
      	printf("                                     Number of Poll Entries:  %ld\n", 
		global_stats->poll_entry);
      	printf("                   Number of Poll Entries with Critical set:  %ld\n", 
		global_stats->poll_already_critical);
      	printf("                           Number of Poll Entries Processed:  %ld\n", 
		global_stats->poll_processed);
      	printf("             Number of times host irq left disabled in Poll:  %ld\n", 
		global_stats->poll_host_disable_irq);

} /* fr_driver_stat_gen */

void flush_driver_stats( void )
{

      cb.command = FPIPE_FLUSH_DRIVER_STATS;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);

      printf("All Driver Statistics are Flushed.\n");

} /* flush_driver_stats */

void fr_router_up_time( void )
{
     unsigned long time;
     cb.command = FPIPE_ROUTER_UP_TIME;
     cb.buffer_length = 0;
     cb.data[0] = 0;
     DoCommand(&cb);
     
     time = cb.data[0] + (cb.data[1]*256) + (cb.data[2]*65536) + 
					(cb.data[3]*16777216);

	banner("ROUTER UP TIME",0);

     if (time < 3600) {
	if (time<60) 
     		printf("    Router UP Time:  %ld seconds\n", time);
	else
     		printf("    Router UP Time:  %ld minute(s)\n", (time/60));
     }else
     		printf("    Router UP Time:  %ld hour(s)\n", (time/3600));
			
      
} /* fr_router_up_time */

void usage( void ) {

	printf("fpipemon -i <ip-addr or interface> -u <port> -c <command> -d <dlci num>\n\n");
	printf("\tOption -i: \n");
	printf("\t\tWanpipe remote IP address or\n");
	printf("\t\tWanpipe network interface name (ex: fr16)\n");   
	printf("\tOption -u: \n");
	printf("\t\tWanpipe UDPPORT specified in /etc/router.conf\n");
	printf("\tOption -d: \n");
	printf("\t\tDLCI Number: (optional) defaults to 0 \n");
	printf("\tOption -c: \n");
	printf("\t\tFpipemon Command\n"); 
	printf("\t\t\tFirst letter is a command and the rest are options:\n"); 
	printf("\t\t\tex: xm = View Modem Status\n\n");
	printf("\tSupported Commands: x=status : s=statistics : t=trace \n");
	printf("\t                    c=config : T=FT1 stats  : f=flush\n\n");
	printf("\tCommand:  Options:   Description \n");	
	printf("\t-------------------------------- \n\n");    
	printf("\tCard Status\n");
	printf("\t   x         m       Modem Status\n");
	printf("\t             l       Link Status\n");
	printf("\t             ru      Display Router UP time\n");
	printf("\tCard Configuration\n");
	printf("\t   c         l       List Active DLCIs\n");
	printf("\tCard Statistics\n");
	printf("\t   s         g       Global Statistics\n");
	printf("\t             c       Communication Error Statistics\n");
	printf("\t             e       Error Statistics\n");
	printf("\t             d       Read Statistics for a specific DLCI\n");
	printf("\tTrace Data \n");
	printf("\t   t         i       Trace and Interpert ALL frames\n");
	printf("\t             ip      Trace and Interpert PROTOCOL frames only\n");
	printf("\t             id      Trace and Interpert DATA frames only\n");
	printf("\t             r       Trace ALL frames, in RAW format\n");
	printf("\t             rp      Trace PROTOCOL frames only, in RAW format\n");
	printf("\t             rd      Trace DATA frames only, in RAW format\n");
	printf("\tFT1 Configuration\n");
	printf("\t   T         v       View Status \n");
	printf("\t             s       Self Test\n");
	printf("\t             l       Line Loop Test\n");
	printf("\t             d       Digital Loop Test\n");
	printf("\t             r       Remote Test\n");
	printf("\t             o       Operational Mode\n");
	printf("\tDriver Statistics\n");
	printf("\t   d         s       Display If_Send Driver Statistics\n");
	printf("\t             i       Display Interrupt Driver Statistics\n");
	printf("\t             g       Display General Driver Statistics\n");	
	printf("\tFlush Statistics\n");
	printf("\t   f         g       Flush Global Statistics\n");
	printf("\t             c       Flush Communication Error Statistics\n");
	printf("\t             e       Flush Error Statistics\n");
	printf("\t             i       Flush DLCI Statistics\n");
	printf("\t             d       Flush Driver Statistics\n");
	printf("\tExamples:\n");
	printf("\t--------\n\n");
	printf("\tex: fpipemon -i fr16 -u 9000 -c xm       :View Modem Status \n");
	printf("\tex: fpipemon -i 201.1.1.2 -u 9000 -c ti  :Trace and Interpert ALL frames\n");
	printf("\tex: fpipemon -i fr17 -u 9000 -c sd -d 16 :Statistics for DLCI 16 \n\n");

}; //usage

void banner (char *title, int dlci){
	
	int len,i;
	
	len = strlen(title);
	printf("\n\t");
	for (i=0;i<(len+16);i++)
		printf("-");
	if (dlci){
		printf("\n\t\t%s %i",title,dlci);
	}else{
		printf("\n\t\t%s",title);
	}
	printf("\n\t");
	for (i=0;i<(len+16);i++)
		printf("-");
	printf("\n\n");
	

}


int main(int argc, char* argv[])
{
	int proceed;
	char command;
	char *opt;

	printf("\n");
	if( argc > 2 ) {
		init( argc, argv);

		command = cmd[0];
		opt	= (char *) &cmd[1];

		proceed = MakeConnection();
		if(proceed == TRUE){
			switch(command) {
			case 'x':
				if (!strcmp(opt,"m")){
					modem_status();
				}else if (!strcmp(opt, "l")){
					link_status();
				}else if (!strcmp(opt, "ru")){
					fr_router_up_time();
				}else{
					printf("ERROR: Invalid Status Command 'x', Type fpipemon <cr> for help\n\n");
				}
				break;
			case 's':
				if (!strcmp(opt,"c")){
					comm_err();
	 			}else if (!strcmp(opt,"g")){
					global_stats();
				}else if (!strcmp(opt,"e")){
					error_stats();
				}else if (!strcmp(opt,"d")){
					read_dlci_stat();
				}else {
					printf("ERROR: Invalid Statistics Command 's', Type fpipemon <cr> for help\n\n");
				}	
				break;
      			case 't':
				if(!strcmp(opt,"i" )){
					line_trace(TRACE_ALL);
				}else if (!strcmp(opt, "ip")){
					line_trace(TRACE_PROT);
				}else if (!strcmp(opt, "id")){
					line_trace(TRACE_DATA);
				}else if (!strcmp(opt, "r")){
					raw_data = TRUE;
					line_trace(TRACE_ALL);
				}else if (!strcmp(opt, "rp")){
					raw_data = TRUE;
					line_trace(TRACE_PROT);
				}else if (!strcmp(opt, "rd")){
					raw_data = TRUE;
					line_trace(TRACE_DATA);
				}else{
					printf("ERROR: Invalid Trace Command 't', Type fpipemon <cr> for help\n\n");
				}
	 			break;

      			case 'c':
				if (!strcmp(opt, "l")){
	 				list_dlcis();
				}else{
					printf("ERROR: Invalid Configuration Command 'c', Type fpipemon <cr> for help\n\n");

				}
	 			break;
      			case 'd':
				  	/* Different signature for Driver Statistics */
	 			if(!strcmp(opt, "s")){
	 				fr_driver_stat_ifsend();
				}else if (!strcmp(opt, "i")){
					fr_driver_stat_intr();
	 			}else if (!strcmp(opt, "g")){
					fr_driver_stat_gen();
				}else{
					printf("ERROR: Invalid Driver Statistic 'd', Type fpipemon <cr> for help\n\n");
				}
	 			break;
      			case 'f':
	    			if (!strcmp(opt, "s")){
	       				flush_comm_err();
	       				comm_err();
				}else if (!strcmp(opt, "g")){
	       				flush_global_stats();
	       				global_stats();
				}else if (!strcmp(opt, "e")){
	      	 			flush_global_stats();
	       				error_stats();
				}else if (!strcmp(opt, "i")){
	       				flush_dlci_stats();
	       				read_dlci_stat();
				}else if (!strcmp(opt, "d")){
	       				flush_driver_stats();
				}else{
					printf("ERROR: Invalid Flush Command 'f', Type fpipemon <cr> for help\n\n");
				}
				break;
      			case 'T':
				if (!strcmp(opt, "v")){
	     				set_FT1_monitor_status(0x01);
	     				if(!fail){
						view_FT1_status();
	     				}
	     				set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt, "s")){
	     				set_FT1_monitor_status(0x01);
	     				if(!fail){
						FT1_self_test();
	     				}
             				set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt, "l")){
	     				set_FT1_monitor_status(0x01);
	     				if(!fail){
						FT1_ll_test();
	     				}
					set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt, "d")){
   	     				set_FT1_monitor_status(0x01);
	     				if(!fail){
	     					FT1_dl_test();
	    				}
	     				set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt, "r")){
	     				set_FT1_monitor_status(0x01);
             				if(!fail){
	     					FT1_rt_test();
	     				}
					set_FT1_monitor_status(0x00);	
				}else if (!strcmp(opt, "o")){
	    				set_FT1_monitor_status(0x01);
	     				if(!fail){
						FT1_operational_mode();
	    	 			}
					set_FT1_monitor_status(0x00);
				}else{
					printf("ERROR: Invalid FT1 Command 'T', Type fpipemon <cr> for help\n\n");

				}
				break;	
     			default:
				printf("ERROR: Invalid Command, Type fpipemon <cr> for help\n\n");

      			} //switch
     		}
     		close( sock );
   	} else {
      		usage();
   	} //if
   	printf("\n");
	return 0;
}; //main

/*
 * EOF fpipemon.c
 */
