/*
 * snmptrapd.c - receive and log snmp traps
 *
 */
/***********************************************************
	Copyright 1989 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/
#include <sys/types.h>
#include <netinet/in.h>
#include <stdio.h>
#include <sys/time.h>
#include <errno.h>
#include <syslog.h>
#include <netdb.h>
#ifdef linux
#include <stdlib.h>
#include <string.h>
#include <arpa/inet.h>
#endif

#include "asn1.h"
#include "snmp.h"
#include "snmp_impl.h"
#include "snmp_api.h"
#include "snmp_client.h"

#ifndef BSD4_3

#ifndef NFDBITS
typedef long	fd_mask;
#define NFDBITS	(sizeof(fd_mask) * NBBY)	/* bits per mask */

#define	FD_SET(n, p)	((p)->fds_bits[(n)/NFDBITS] |= (1 << ((n) % NFDBITS)))
#define	FD_CLR(n, p)	((p)->fds_bits[(n)/NFDBITS] &= ~(1 << ((n) % NFDBITS)))
#define	FD_ISSET(n, p)	((p)->fds_bits[(n)/NFDBITS] & (1 << ((n) % NFDBITS)))
#define FD_ZERO(p)	bzero((char *)(p), sizeof(*(p)))
#endif

#endif

#define USAGE "Usage: snmptrapd [-f cfgfile] [-s] [-v 1]\n"

extern int  errno;
int	snmp_dump_packet = 0;
FILE    *Print = NULL;
int     debug  = 0;		          /* debug level */

/* logfile configuration constants */
#define OIDFMT_OID   0			  /* print OIDs numerical */
#define OIDFMT_TEXT  1			  /* print OIDs symbolically */
#define NODEFMT_IP   0			  /* print nodenames as IP address */
#define NODEFMT_FQDN 1			  /* print nodenames fully qualified */
#define NODEFMT_NAME 2			  /* print short nodenames */

/* logfile configuration tokens */
#define TOK_SUBST   "SUBST"
#define TOK_OIDFMT  "OIDFMT"
#define TOK_NODEFMT "NODEFMT"
#define TOK_VARFMT  "VARFMT"
#define TOK_LOGFMT  "LOGFMT"
#define TOK_LOGFILE "LOGFILE"
#define TOK_CMUSNMP "CMUSNMP"

/* logfile configuration variables */
char  cfg_subst1  = ';';	          /* subst this char in var values */
char  cfg_subst2  = ' ';		  /* by this char */
int   cfg_oidfmt  = OIDFMT_OID;		  /* format for OID's */
int   cfg_nodefmt = NODEFMT_FQDN;	  /* format for nodenames */
char *cfg_varfmt  = ";[%s] %n (%t) : %v"; /* format string for vars in '$*' */
char *cfg_logfmt  = "$x;$A;$e;$G;$S$*";	  /* format for the logfile entry */
char *cfg_logfile = NULL;

#if 0
char *cfg_cmusnmp = NULL;	          /* configure snmp library */
#endif

static void init_syslog();
static void  lg_print_config();
static FILE *lg_parse_config();
static char *lg_format_buffer();
static void  lg_sprint_node();
static void  lg_sprint_oid();
static void  lg_sprint_int();
static void  lg_sprint_ulong();
static void  lg_sprint_ASNtype();
static void  lg_sprint_var();

char *
trap_description(trap)
    int trap;
{
    switch(trap){
	case SNMP_TRAP_COLDSTART:
	    return "Cold Start";
	case SNMP_TRAP_WARMSTART:
	    return "Warm Start";
	case SNMP_TRAP_LINKDOWN:
	    return "Link Down";
	case SNMP_TRAP_LINKUP:
	    return "Link Up";
	case SNMP_TRAP_AUTHFAIL:
	    return "Authentication Failure";
	case SNMP_TRAP_EGPNEIGHBORLOSS:
	    return "EGP Neighbor Loss";
	case SNMP_TRAP_ENTERPRISESPECIFIC:
	    return "Enterprise Specific";
	default:
	    return "Unknown Type";
    }
}

char *
uptime_string(timeticks, buf)
    u_long timeticks;
    char *buf;
{
    int	seconds, minutes, hours, days;

    timeticks /= 100;
    days = timeticks / (60 * 60 * 24);
    timeticks %= (60 * 60 * 24);

    hours = timeticks / (60 * 60);
    timeticks %= (60 * 60);

    minutes = timeticks / 60;
    seconds = timeticks % 60;

    if (days == 0){
	sprintf(buf, "%d:%02d:%02d", hours, minutes, seconds);
    } else if (days == 1) {
	sprintf(buf, "%d day, %d:%02d:%02d", days, hours, minutes, seconds);
    } else {
	sprintf(buf, "%d days, %d:%02d:%02d", days, hours, minutes, seconds);
    }
    return buf;
}

int snmp_input(op, session, reqid, pdu, magic)
     int op;
     struct snmp_session *session;
     int reqid;
     struct snmp_pdu *pdu;
     void *magic;
{
  char *buf;

  if (op == RECEIVED_MESSAGE && pdu->command == TRP_REQ_MSG)
    {
      /* allocate and format the output buffer */
      buf = lg_format_buffer(session, pdu, time(NULL));

      if ( !buf )
	{
	  printf("Error: Cannot allocate/format output buffer\n");
	}
      else if ( Print )
	{
	  fprintf(Print, "%s\n", buf);
	  fflush(Print);
	}
      else
	{
	  syslog(LOG_WARNING, "%s\n", buf);
	}
    } 
  else if (op == TIMED_OUT)
    {
      printf("Timeout: This shouldn't happen!\n");
    }
  
  return 0;
}

int
main(argc, argv)
    int	    argc;
    char    *argv[];
{
    struct snmp_session session, *ss;
    int	arg;
    int count, numfds, block;
    fd_set fdset;
    struct timeval timeout, *tvp;


    init_syslog();
    /*
     * usage: snmptrapd [-D] [-f file]
     */
    for(arg = 1; arg < argc; arg++){
	if (argv[arg][0] == '-'){
	    switch(argv[arg][1]){
		case 'd':
		    snmp_dump_packet++;
		    break;
		case 'f':
                    if ( arg + 1 < argc )
		    {
		      /* read the specification of the logfile entries */
                      Print = lg_parse_config(argv[++arg]);
		    }
		    else
		    {
		      printf(USAGE);
		      exit(1);
		    }
		    break;
  	        case 'D':
		    debug = 1;
		    break;
#if 1
		  case 's':
		    /* ignore */
		    break;

		  case 'v':
		    arg++;
		    /* ignore */
		    break;
#endif

		default:
		    printf("invalid option: -%c\n", argv[arg][1]);
		    printf(USAGE);
		    break;
	    }
	    continue;
	}
    }

#if 0
    /* initialize with configured option string */
    initmib(cfg_cmusnmp);
#endif

    /* initialize */
    init_mib();

    /* print the logfile output specification */
    if ( debug ) 
      {
	lg_print_config();
      }

    bzero((char *)&session, sizeof(struct snmp_session));
    session.peername = NULL;
    session.community = NULL;
    session.community_len = 0;
    session.retries = SNMP_DEFAULT_RETRIES;
    session.timeout = SNMP_DEFAULT_TIMEOUT;
    session.authenticator = NULL;
    session.callback = snmp_input;
    session.callback_magic = NULL;
    session.local_port = SNMP_TRAP_PORT;
    ss = snmp_open(&session);
    if (ss == NULL){
        char *p_errtxt;
	snmp_error(&session, &errno, &snmp_errno, &p_errtxt);
	printf("Couldn't open snmp: %s\n", p_errtxt);
	free(p_errtxt);
	exit(-1);
    }

    while(1){
	numfds = 0;
	FD_ZERO(&fdset);
	block = 1;
	tvp = &timeout;
	timerclear(tvp);
	snmp_select_info(&numfds, &fdset, tvp, &block);
	if (block == 1)
	    tvp = NULL;	/* block without timeout */
	count = select(numfds, &fdset, 0, 0, tvp);
	if (count > 0){
		snmp_read(&fdset);
	} else switch(count){
	    case 0:
		snmp_timeout();
		break;
	    case -1:
		if (errno == EINTR){
		    continue;
		} else {
		    perror("select");
		}
		return -1;
	    default:
		printf("select returned %d\n", count);
		return -1;
	}
    }

    /* not reached */
}

static void
init_syslog(){
/*
 * These definitions handle 4.2 systems without additional syslog facilities.
 */
#ifndef LOG_CONS
#define LOG_CONS	0	/* Don't bother if not defined... */
#endif
#ifndef LOG_LOCAL0
#define LOG_LOCAL0	0
#endif
    /*
     * All messages will be logged to the local0 facility and will be sent to
     * the console if syslog doesn't work.
     */
    openlog("snmptrapd", LOG_CONS, LOG_LOCAL0);
    syslog(LOG_INFO, "Starting snmptrapd");
}

/* Parse the configuration file:
 * - check for the tokens and set the config variables
 * - open the logfile and return the file pointer (or NULL if no logfile)
 * - syntax of configuration file (fixed text in '')
 *   comment lines begin with '#'
 *     file    ::= line | file line
 *     line    ::= 'SUBST'   '=' '\'char'\'char'\'
 *               | 'OIDFMT'  '=' ( 'text' | 'oid' )
 *               | 'NODEFMT' '=' ( 'ip' | 'fqdn' | 'name' )
 *               | 'VARFMT'  '=' stringV
 *               | 'LOGFMT'  '=' stringL
 *               | 'LOGFILE' '=' string
 *               | 'CMUSNMP' '=' string                 >>>OBSOLETE<<<
 *     string  ::= '"' { char | '\"' } '"'
 *     stringV ::= '"' { char | '%s' | '%n' | '%t' | '%v' | '%%' } '"'
 *     stringL ::= '"' { char | '$#' | '$*' | '$x' | '$r' | '$R'
 *                            | '$e' | '$A' | '$G' | '$S' | '$T'
 *                            | '$$'
 *                            | '\n' | '\r' | '\t' | '\\' } '"'
 *
 * - Tokens and associated config variables are:
 *     SUBST       ... cfg_subst1, cfg_subst2   (both char)
 *     OIDFMT      ... cfg_oidfmt               (OIDFMT_TEXT,_OID)
 *     NODEFMT     ... cfg_nodefmt              (NODEFMT_IP, _FQDN, _NAME)
 *     VARFMT      ... cfg_varfmt               (char*)
 *     LOGFMT      ... cfg_logfmt               (char*)
 *     LOGFILE     ... cfg_logfile              (char*)
 *     CMUSNMP     ... cfg_cmusnmp              (char*)    >>>OBSOLETE<<<
 *     
 */
static FILE*
lg_parse_config (cfgname)
  char* cfgname;
{
  FILE *cfg;
  FILE *log = NULL;
  char lbuf[1024];
  int  lno = 0;


  if ( !(cfg = fopen(cfgname, "r")) )
    {
      printf("Cannot open config file '%s' - %s\n", cfgname, strerror(errno));
      return NULL;
    }

  while ( fgets(lbuf, sizeof(lbuf), cfg) )
    {
      char *p = strtok(lbuf, "=\n");

      /* check for comment or empty lines */
      lno++;
      if ( !p || (*p == '#') )
	{
	  continue;
	}
      
      /* check for tokens */
      if ( strcasecmp(TOK_SUBST, p) == 0 )
	{
	  char *p1 = strtok(NULL, "\n"); /* the substitution */
	  int   ok = 1;

	  if ( !p1 || !*p1 || (*p1 != '\\') )
	    {
	      ok = 0;
	    }
	  else
	    {
	      p1++;
	      cfg_subst1 = *p1++;
	      
	      if ( *p1++ == '\\' && *p1 )
		{
		  cfg_subst2 = *p1;
		}
	      else
		{
		  ok = 0;
		}
	    }
	  if ( !ok )
	    {
  	      printf("Expected: %s=\\<char>\\<char>\\ in line %d\n", 
		     TOK_SUBST, lno);
	    }
	}
      else if ( strcasecmp(TOK_OIDFMT, p) == 0 )
	{
	  char *p1 = strtok(NULL, " \n");
	  int   ok = 1;

	  if ( !p1 || !*p1 )
	    {
	      ok = 0;
	    }
	  else if ( strcasecmp("text", p1) == 0 )
	    {
	      cfg_oidfmt = OIDFMT_TEXT;
	    }
	  else if ( strcasecmp("oid", p1) == 0 )
	    {
	      cfg_oidfmt = OIDFMT_OID;
	    }
	  else
	    {
	      ok = 0;
	    }
	  if ( !ok )
	    {
	      printf("Expected: %s='text' | 'oid' in line %d\n", 
		     TOK_OIDFMT, lno);
	    }
	}
      else if ( strcasecmp(TOK_NODEFMT, p) == 0 )
	{
	  char *p1 = strtok(NULL, " \n");
	  int   ok = 1;

	  if ( !p1 )
	    {
	      ok = 0;
	    }
	  else if ( strcasecmp("ip", p1) == 0 )
	    {
	      cfg_nodefmt = NODEFMT_IP;
	    }
	  else if ( strcasecmp("fqdn", p1) == 0 )
	    {
	      cfg_nodefmt = NODEFMT_FQDN;
	    }
	  else if ( strcasecmp("name", p1) == 0 )
	    {
	      cfg_nodefmt = NODEFMT_NAME;
	    }
	  else
	    {
	      ok = 0;
	    }
	  if ( !ok )
	    {
	      printf("Expected: %s='ip' | 'fqdn' | 'name' in line %d\n", 
		     TOK_NODEFMT, lno);
	    }
	}
      else if ( (strcasecmp(TOK_VARFMT, p) == 0)  ||
		(strcasecmp(TOK_LOGFMT, p) == 0)  ||
		(strcasecmp(TOK_LOGFILE, p) == 0) ||
		(strcasecmp(TOK_CMUSNMP, p) == 0) )
	{
	  char *p1 = strtok(NULL, "\n");
	  int   ok = 1;

	  if ( !p1 )
	    {
	      ok = 0;
	    }
	  else if ( *p1 == '\"' )
	    {
	      char *p2 = strrchr(++p1, '\"');
	      
	      if ( p2 )
		{
		  char *pval;

		  *p2  = '\0';
		  pval = strdup(p1);

		  if ( strcasecmp(TOK_VARFMT, p) == 0 )
		    {
		      cfg_varfmt = pval;
		    }
		  else if ( strcasecmp(TOK_LOGFMT, p) == 0 )
		    {
		      cfg_logfmt = pval;
		    }
		  else if ( strcasecmp(TOK_LOGFILE, p) == 0 )
		    {
		      cfg_logfile = pval;
		    }
		  else if ( strcasecmp(TOK_CMUSNMP, p) == 0 )
		    {
                      printf("The option 'CMUSNMP' is obsolete\n");
#if 0
		      cfg_cmusnmp = pval;
#endif
		    }
		}
	      else
		{
		  ok = 0;
		}
	    }
	  else
	    {
	      ok = 0;
	    }
	  if ( !ok )
	    {
	      printf("Expected: %s=\"<string>\" in line %d\n", 
		     p, lno);
	    }
	}
      else
	{
	  printf("Missing token in config file '%s', line %d\n", 
		 cfgname, lno);
	}
    }
  
  fclose(cfg);

  /* now open the log file (if any) */
  if ( cfg_logfile && *cfg_logfile )
    {
      log = fopen(cfg_logfile, "a");

      if ( !log )
	{
	  printf("Cannot append to logfile '%s' - %s\n", 
		 cfg_logfile, strerror(errno));
	}
    }

  return log;
}


/* print the logfile configuration values */
static void
lg_print_config()
{

  char *oidfmt[2];
  char *nodefmt[3]; 
  
  oidfmt[OIDFMT_OID]  = "oid";
  oidfmt[OIDFMT_TEXT] = "text";
  nodefmt[NODEFMT_IP]   = "ip";
  nodefmt[NODEFMT_FQDN] = "fqdn";
  nodefmt[NODEFMT_NAME] = "name";
  
  printf("Configuration variables:\n");
  printf(" cfg_subst1  = '%c'\n", cfg_subst1);
  printf(" cfg_subst2  = '%c'\n", cfg_subst2);
  printf(" cfg_oidfmt  = %s\n", oidfmt[cfg_oidfmt]);
  printf(" cfg_nodefmt = %s\n", nodefmt[cfg_nodefmt]);
  printf(" cfg_varfmt  = \"%s\"\n", cfg_varfmt);
  printf(" cfg_logfmt  = \"%s\"\n", cfg_logfmt);
  printf(" cfg_logfile = \"%s\"\n", 
	 (cfg_logfile && *cfg_logfile ? cfg_logfile : "<syslog>"));
#if 0
  printf(" cfg_cmusnmp = \"%s\"\n", 
	 (cfg_cmusnmp ? cfg_cmusnmp : "$CMUSNMP"));
#endif
}


/* Allocate and format the output buffer.
 * Return the pointer to the allocated area or NULL if an error 
 * occurred.
 */
static char*
lg_format_buffer (session, pdu, trap_time)
     struct snmp_session *session;
     struct snmp_pdu     *pdu;
     time_t               trap_time;
{
  static char buf[4000];	/* the output buffer */
  char *cPtr;			/* points to next character in output buf */
  int   bRem;			/* number of remaining chars in output buf */
				/* (starting with cPtr) */
  char *lPtr;			/* points to next char in log format */
  char *vPtr;			/* points to next char in variable format */
  int   seqno;			/* sequence number of variable */
  struct variable_list *vars;

  
  /* init */
  bRem = sizeof(buf) - 1;	/* leave space for EOS */
  cPtr = buf;
  
  /* process the log format string */
  for ( lPtr = cfg_logfmt; *lPtr; lPtr++ )
    {
      int l;			/* local */

      /* Assertion: bRem > 0 */
      switch ( *lPtr ) 
	{
	case '\\':
	  switch ( *++lPtr )
	    {
	    case 'n':
	      *cPtr++ = '\n';
	      break;
	    case 'r':
	      *cPtr++ = '\r';
	      break;
	    case 't':
	      *cPtr++ = '\t';
	      break;
	    default:
	      *cPtr++ = *lPtr;
	      break;
	    }
	  bRem--;
	  break;
	case '$':
	  switch ( *++lPtr )
	    {
 	    case 'x':		/* time trap was received */
	      l = strftime(cPtr, bRem, "%Y%m%d.%H%M%S", localtime(&trap_time));
	      cPtr += l;
	      bRem -= l;
	      break;
 	    case 'A':		/* agents address */
 	    case 'r':		/* implied source (agent address) */
	      lg_sprint_node(pdu->agent_addr.sa_align, 
			     cfg_nodefmt, &cPtr, &bRem);
	      break;
 	    case 'R':		/* true source (peer address) */
	      lg_sprint_node(pdu->address.sa_align, 
			     cfg_nodefmt, &cPtr, &bRem);
	      break;
 	    case 'e':		/* enterprise */
	      lg_sprint_oid(pdu->enterprise, pdu->enterprise_length,
			    cfg_oidfmt, &cPtr, &bRem);
	      break;
 	    case 'G':		/* generic trap number */
	      lg_sprint_int(pdu->trap_type, "%d", &cPtr, &bRem);
	      break;
 	    case 'S':		/* specific trap number */
	      lg_sprint_int(pdu->specific_type, "%d", &cPtr, &bRem);
	      break;
 	    case 'T':		/* agent's sysUpTime in 1/100 seconds */
	      lg_sprint_ulong(pdu->time, "%lu", &cPtr, &bRem);
	      break;
 	    case '#':		/* number of variables in trap */
	      for (l = 0, vars = pdu->variables; 
		   vars; 
		  l++,  vars = vars->next_variable)
		;
	      lg_sprint_int(l, "%d", &cPtr, &bRem);
	      break;
 	    case '*':		/* all variables in trap */
	      for (seqno = 1, vars = pdu->variables; 
		   vars; 
		   seqno++, vars = vars->next_variable)
		{
		  lg_sprint_var(vars, cfg_varfmt, seqno, &cPtr, &bRem);
		}
	      break;
 	    case '$':		/* insert '$' */
	      *cPtr++ = '$';
	      bRem--;
	      break;
	    default:		/* insert the characters verbatim */
	      *cPtr++ = '$';
	      *cPtr++ = *lPtr;
	      bRem -= 2;
	    }
	  break;
	default:
	  *cPtr++ = *lPtr;
	  bRem--;
	  break;
	}

      /* leave if no more space is left */
      if ( bRem == 0 )
	{
	  printf("No more internal space left for remaining format '%s'\n", 
		 lPtr);
	  break;
	}
    }
  *cPtr = '\0';

  /* Debug output  */
  if ( debug )
    {
      printf("%s\n", buf);
    }

  /* this is the result */
  return buf;
}

/* print a nodes ip-address, fully qualified domain name or
 * short hostname according to the specified format into the
 * output buffer
 */
static void
lg_sprint_node (ip_addrL, nodeFmt, cPPtr, bRemP)
     long           ip_addrL;
     int            nodeFmt;	/* NODEFMT_IP/_FQDN/_NAME */
     char         **cPPtr;	/* address of pointer to current outchar */
     int           *bRemP;	/* address of counter of remaining chars */
{
  char lbuf[80];		/* local buffer for result */
  int  l;
  struct hostent *hent;
  struct in_addr ip_addr;

  bcopy(&ip_addrL, &ip_addr.s_addr, sizeof(ip_addr.s_addr));

  switch ( nodeFmt )
    {
    case NODEFMT_FQDN:		/* use full qualified domain name (resolver) */
    case NODEFMT_NAME:		/* use the short name (hostname) */
      hent = gethostbyaddr((char*)&ip_addr.s_addr, 
                           sizeof(ip_addr.s_addr), AF_INET);
      if ( hent )
	{
	  sprintf(lbuf, "%s", hent->h_name);
	}
      else
	{
	  /* no result, use the ip-address instead */
	  sprintf(lbuf, "%s", inet_ntoa(ip_addr));
	}

      if ( hent && (nodeFmt == NODEFMT_NAME) )
	{
	  /* strip the domain part */
	  char *p = strchr(lbuf, '.');

	  if ( p )
	    {
	      *p = '\0';
	    }
	}
      break;
    case NODEFMT_IP:		/* use ip address */
    default:			/* shouldn't happen */
      sprintf(lbuf, "%s", inet_ntoa(ip_addr));
      break;
    }

  /* insert the local buffer into the output buffer */
  l = strlen(lbuf);
  if ( l <= *bRemP )
    {
      strncpy(*cPPtr, lbuf, l);
      (*cPPtr) += l;
      (*bRemP) -= l;
    }
  else
    {
      printf("No more internal space left for '%s'\n", lbuf);
    }
}

/* print an oid according to the specified format into the output buffer
 */
static void
lg_sprint_oid (oidP, oid_len, oidFmt, cPPtr, bRemP)
     oid *oidP;			/* the OID */
     int  oid_len;		/* it's length */
     int  oidFmt;		/* OIDFMT_TEXT/_OID */
     char **cPPtr;		/* address of pointer to current outchar */
     int   *bRemP;		/* address of counter of remaining chars */
{
  char lbuf[1024];		/* local buffer */
  int  i;
  int  l;

  switch ( oidFmt )
    {
    case OIDFMT_TEXT:		/* use textual format for the oid */
      lbuf[0] = '\0';
      sprint_objid(lbuf, oidP, oid_len);
      break;
    case OIDFMT_OID:		/* usenumerical format for the oid */
    default:			/* shouldn't happen */
      lbuf[0] = '\0';
      for (i = 0; i < oid_len; i++)
	{
	  sprintf(&lbuf[strlen(lbuf)], ".%d", oidP[i]);
	}
      break;
    }

  /* insert the local buffer into the output buffer */
  l = strlen(lbuf);
  if ( l <= *bRemP )
    {
      strncpy(*cPPtr, lbuf, l);
      (*cPPtr) += l;
      (*bRemP) -= l;
    }
  else
    {
      printf("No more internal space left for '%s'\n", lbuf);
    }
}

/* print an integer according to the specified format into the output buffer
 */
static void
lg_sprint_int (ival, iFmt, cPPtr, bRemP)
     int    ival;		/* the integer value */
     char  *iFmt;		/* the format string */
     char **cPPtr;		/* address of pointer to current outchar */
     int   *bRemP;		/* address of counter of remaining chars */
{
  char lbuf[20];		/* local buffer */
  int  l;

  /* fill the local buffer */
  sprintf(lbuf, iFmt, ival);
  
  /* insert the local buffer into the output buffer */
  l = strlen(lbuf);
  if ( l <= *bRemP )
    {
      strncpy(*cPPtr, lbuf, l);
      (*cPPtr) += l;
      (*bRemP) -= l;
    }
  else
    {
      printf("No more internal space left for '%s'\n", lbuf);
    }
}

/* print an u_long according to the specified format into the output buffer
 */
static void
lg_sprint_ulong (uval, uFmt, cPPtr, bRemP)
     unsigned long uval;	/* the integer value */
     char  *uFmt;		/* the format string */
     char **cPPtr;		/* address of pointer to current outchar */
     int   *bRemP;		/* address of counter of remaining chars */
{
  char lbuf[20];		/* local buffer */
  int  l;

  /* fill the local buffer */
  sprintf(lbuf, uFmt, uval);
  
  /* insert the local buffer into the output buffer */
  l = strlen(lbuf);
  if ( l <= *bRemP )
    {
      strncpy(*cPPtr, lbuf, l);
      (*cPPtr) += l;
      (*bRemP) -= l;
    }
  else
    {
      printf("No more internal space left for '%s'\n", lbuf);
    }
}

/* print an ASN.1 type ito the output buffer
 */
static void
lg_sprint_ASNtype (asntype, cPPtr, bRemP)
     u_char asntype;		/* ASN.1 type */
     char **cPPtr;		/* address of pointer to current outchar */
     int   *bRemP;		/* address of counter of remaining chars */
{
  char *p;
  int   l;

  switch ( asntype )
    {
    case ASN_BOOLEAN:
      p = "Boolean";
      break;
    case ASN_INTEGER:
      p = "Integer";
      break;
    case ASN_BIT_STR:
      p = "BitString";
      break;
    case ASN_OCTET_STR:
      p = "OctetString";
      break;
    case ASN_NULL:
      p = "Null";
      break;
    case ASN_OBJECT_ID:
      p = "ObjectIdentifier";
      break;
    case ASNT_OPAQUE:
      p = "Opaque";
      break;
    case TIMETICKS:
      p = "Timeticks";
      break;
    case GAUGE:
      p = "Gauge";
      break;
    case COUNTER:
      p = "Counter";
      break;
    case IPADDRESS:
      p = "IpAddress";
      break;
    case UINTEGER:
      p = "UnsignedInteger32";
      break;
    default:
      p = "BadType";
      break;
    }
  
  /* insert the local buffer into the output buffer */
  l = strlen(p);
  if ( l <= *bRemP )
    {
      strncpy(*cPPtr, p, l);
      (*cPPtr) += l;
      (*bRemP) -= l;
    }
  else
    {
      printf("No more internal space left for '%s'\n", p);
    }

  /* temporary close the string  */
  **cPPtr = '\0';
}

/* print a snmp variable according to the specified format into the 
 * output buffer
 */
static void
lg_sprint_var (var, vFmt, seqno, cPPtr, bRemP)
     struct variable_list *var;	/* the snmp variable */
     char  *vFmt;		/* the output format string */
     int    seqno;		/* current sequence number */
     char **cPPtr;		/* address of pointer to current outchar */
     int   *bRemP;		/* address of counter of remaining chars */
{
  char *vPtr;			/* current varfmt character */
  int   l;

  /* process the variable format */
  for ( vPtr = vFmt; *vPtr; vPtr++ )
    {
      switch ( *vPtr )
	{
	case '\\':
	  if ( *bRemP > 0 )
	    {
	      switch ( *++vPtr )
		{
		case 'n': 
		  *(*cPPtr)++ = '\n'; 
		  (*bRemP)--;
		  break;
		case 'r': 
		  *(*cPPtr)++ = '\r'; 
		  (*bRemP)--;
		  break;
		case 't': 
		  *(*cPPtr)++ = '\t'; 
		  (*bRemP)--;
		  break;
		default:
		  *(*cPPtr)++ = *vPtr; 
		  (*bRemP)--;
		}
	    }
	  break;
	case '%':		/* special char follows */
	  switch ( *++vPtr )
	    {
	    case 's':		/* the sequence number */
	      lg_sprint_int(seqno, "%d", cPPtr, bRemP);
	      break;
	    case 'n':		/* the name */
	      lg_sprint_oid(var->name, var->name_length,
			    cfg_oidfmt, cPPtr, bRemP);
	      break;
	    case 't':		/* the type */
	      lg_sprint_ASNtype(var->type, cPPtr, bRemP);
	      break;
	    case 'v':		/* the value */
	      /* special handling for some types */
	      switch ( var->type )
		{
		case ASN_OCTET_STR: /* substitute characters then print */
		  {
		    char *p  = var->val.string;

		    for (l = var->val_len; l > 0 ; l--, p++)
		      {
			if ( *p == cfg_subst1 )
			  {
			    *p = cfg_subst2;
			  }
		      }
		    **cPPtr = '\0';
		    sprint_value(*cPPtr, var->name, var->name_length, var);
		    l = strlen(*cPPtr);
		    (*cPPtr) += l;
		    (*bRemP) -= l;  /* if negative -> problem ... */
		  }
		  break;
		case ASN_OBJECT_ID: /* print according to config spec */
		  lg_sprint_oid((oid*)var->val.objid,
 				var->val_len / sizeof(oid),
				cfg_oidfmt, cPPtr, bRemP);
		  break;
		case IPADDRESS:	    /* print according to config spec */
		  {
		    long sin_addr;

		    bcopy(var->val.string, &sin_addr, sizeof(sin_addr));
		    lg_sprint_node(sin_addr, cfg_nodefmt, cPPtr, bRemP);
		  }
		  break;
		default:	    /* use standard printera */
		  **cPPtr = '\0';
		  sprint_value(*cPPtr, var->name, var->name_length, var);
		  l = strlen(*cPPtr);
		  (*cPPtr) += l;
		  (*bRemP) -= l;    /* if negative -> problem ... */
		  break;
		}
	      break;
	    case '%':		/* insert '%' */
	      if ( *bRemP > 0 )
		{
		  *(*cPPtr)++ = '%';
		  (*bRemP)--;
		}
	      break;
	    default:		/* insert the two chars verbatim */
	      if ( *bRemP > 1 )
		{
		  *(*cPPtr)++ = '%';
		  *(*cPPtr)++ = *vPtr;
		  (*bRemP) -= 2;
		}
	      else
		{
		  *bRemP = 0;
		}
	      break;
	    }
	  break;
	default:		/* insert char verbatim */
	  if ( *bRemP > 0 )
	    {
	      *(*cPPtr)++ = *vPtr;
	      (*bRemP)--;
	    }
	}

      if ( *bRemP <= 0 )
	{
	  printf("No more internal space left for variable %d\n",
		 seqno);
	  *bRemP = 0;
	  break;
	}
    }
}

