/*
  $Id: op.c,v 1.7 1997/01/17 01:43:36 luik Exp $

  op.c - perform op on omirrd master or slave machine.
  Copyright (C) 1996, Andreas Luik, <luik@pharao.s.bawue.de>.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#define NEED_ALLOCA
#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: op.c,v 1.7 1997/01/17 01:43:36 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <utime.h>
#include "libomirr.h"
#include "fileop.h"
#include "debug.h"
#include "error.h"
#include "cf.h"
#include "conn.h"
#include "op.h"


#ifndef S_ISLNK			/* should be in <sys/stat.h> */
#define S_ISLNK(mode)	(((mode) & S_IFMT) == S_IFLNK)
#endif

#ifndef SEEK_SET		/* should be in <fcntl.h> or <unistd.h> */
#define SEEK_SET 0
#endif


/* XXX temporary, should go into header file (conn.h?) */
#define ACK_SUCCESS	1
#define ACK_ALREADY	2
#define NACK_ENOENT	3
#define NACK_EEXIST	4
#define NACK_SYSERR	5
#define NACK_OBSOLETE	6
#define NACK_PROTERR	7

/* XXX temporary, should go into proper file (conn.c?) */
static void connAck(int code, char *msg)
{
    /* XXX send ack/nack to remote server */

#ifdef DEBUG
    switch (code) {
      case ACK_SUCCESS:
	debuglog(DEBUG_OP, (": succeeded\n"));
	break;
      case ACK_ALREADY:
	debuglog(DEBUG_OP, (": already up-to-date\n"));
	break;
      case NACK_ENOENT:
	debuglog(DEBUG_OP, (": ENOENT (no such file or directory)\n"));
	break;
      case NACK_EEXIST:
	debuglog(DEBUG_OP, (": EEXIST (file already exists)\n"));
	break;
      case NACK_SYSERR:
	debuglog(DEBUG_OP, (": SYSERR: %s\n", msg));
	break;
      case NACK_OBSOLETE:
	debuglog(DEBUG_OP, (": OBSOLETE (local version is newer)\n"));
	break;
      case NACK_PROTERR:
	debuglog(DEBUG_OP, (": PROTERR (internal protocol error)\n"));
	break;
      default:
	debuglog(DEBUG_OP, (": UNKNOWNERR (unknown internal error)\n"));
	break;
    }
#endif
}


static int symlinkcmp(const char *link, const char *path)
{
    int result;
    char *buf;
    int buf_len = 512;

    for (;;) {
	buf = alloca(buf_len);
	result = Readlink(path, buf, buf_len);
	if (result == -1)
	    return -1;
	if (result < buf_len) {
	    buf[result] = '\0';
	    break;
	}
	buf_len *= 2;
    }
    return (strcmp(link, buf) > 0); /* return 0 (equal) or 1 (not equal) */
}





static int op_stat_func(int (*statfunc)(const char *path, struct stat *statbuf),
			const char *path, struct stat *statbuf)
{
    int result;
    int errno_save;

  again:
    result = (*statfunc)(path, statbuf);
    if (result == -1) {
	if (errno == EINTR)
	    goto again;
	else if (errno != ENOENT) {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("stat(%s) failed: %s\n", path, xstrerror(errno_save));
	}
    }
    return result;
}

static int op_stat(const char *path, struct stat *statbuf)
{
    return op_stat_func(stat, path, statbuf);
}

static int op_lstat(const char *path, struct stat *statbuf)
{
    return op_stat_func(lstat, path, statbuf);
}

static int op_stat_exist_func(int (*statfunc)(const char *path,
					      struct stat *statbuf),
			      const char *path, struct stat *statbuf)
{
    int result;

    result = op_stat_func(statfunc, path, statbuf);
    if (result == -1) {
	if (errno == ENOENT) {
	    connAck(NACK_ENOENT, "");
	}
    }
    return result;
}

static int op_stat_exist(const char *path, struct stat *statbuf)
{
    return op_stat_exist_func(stat, path, statbuf);
}

static int op_lstat_exist(const char *path, struct stat *statbuf)
{
    return op_stat_exist_func(lstat, path, statbuf);
}

static int op_stat_non_exist_func(int (*statfunc)(const char *path,
						  struct stat *statbuf),
				  const char *path, struct stat *statbuf)
{
    int result;

    result = op_stat_func(statfunc, path, statbuf);
    if (result == -1) {
	if (errno == ENOENT) {
	    result = 0;		/* file should not exist */
	}
    }
    else {			/* file already exists */
	result = -1;
	errno = EEXIST;
    }
    return result;
}

static int op_stat_non_exist(const char *path, struct stat *statbuf)
{
    return op_stat_non_exist_func(stat, path, statbuf);
}

static int op_lstat_non_exist(const char *path, struct stat *statbuf)
{
    return op_stat_non_exist_func(lstat, path, statbuf);
}


static int op_chmod(const char *path, mode_t mode)
{
    int result;
    int errno_save;

  again:
    result = chmod(path, mode);
    if (result == -1) {
	if (errno == EINTR)
	    goto again;
	else if (errno == ENOENT) {
	    connAck(NACK_ENOENT, "");
	}
	else {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("chmod(%s,%#o) failed: %s\n",
		    path, mode, xstrerror(errno_save));
	}
    }
    return result;
}

static int op_chown(const char *path, uid_t owner, gid_t group)
{
    int result;
    int errno_save;

  again:
    result = chown(path, owner, group);
    if (result == -1) {
	if (errno == EINTR)
	    goto again;
	else if (errno == ENOENT) {
	    connAck(NACK_ENOENT, "");
	}
	else {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("chown(%s,%d,%d) failed: %s\n",
		    path, owner, group, xstrerror(errno_save));
	}
    }
    return result;
}

static int op_utime(const char *path, time_t actime, time_t modtime)
{
    int result;
    int errno_save;
    struct utimbuf times;

    times.actime = actime;
    times.modtime = modtime;

  again:
    result = utime(path, &times);
    if (result == -1) {
	if (errno == EINTR)
	    goto again;
	else if (errno == ENOENT) {
	    connAck(NACK_ENOENT, "");
	}
	else {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("utime(%s,%ld,%ld) failed: %s\n",
		    path, (long) actime, (long) modtime, xstrerror(errno_save));
	}
    }
    return result;
}

static int op_mkdir(const char *path, mode_t mode)
{
    int result;
    int errno_save;

  again:
    result = mkdir(path, mode);
    if (result == -1) {
	if (errno == EINTR)
	    goto again;
	else if (errno == ENOENT) {
	    connAck(NACK_ENOENT, "");
	}
	else if (errno == EEXIST) {
	    connAck(NACK_EEXIST, "");
	}
	else {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("mkdir(%s,%d) failed: %s\n",
		    path, mode, xstrerror(errno_save));
	}
    }
    return result;
}

static int op_link(const char *path, const char *newpath)
{
    int result;
    int errno_save;

  again:
    result = link(path, newpath);
    if (result == -1) {
	if (errno == EINTR)
	    goto again;
	else if (errno == ENOENT) {
	    connAck(NACK_ENOENT, "");
	}
	else if (errno == EEXIST) {
	    connAck(NACK_EEXIST, "");
	}
	else {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("link(%s,%s) failed: %s\n",
		    path, newpath, xstrerror(errno_save));
	}
    }
    return result;
}

static int op_rename(const char *path, const char *newpath)
{
    int result;
    int errno_save;

  again:
    result = rename(path, newpath);
    if (result == -1) {
	if (errno == EINTR)
	    goto again;
	else if (errno == ENOENT) {
	    connAck(NACK_ENOENT, "");
	}
	else if (errno == EEXIST) {
	    connAck(NACK_EEXIST, "");
	}
	else {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("rename(%s,%s) failed: %s\n",
		    path, newpath, xstrerror(errno_save));
	}
    }
    return result;
}

static int op_mknod(const char *path, mode_t mode, dev_t dev)
{
    int result;
    int errno_save;

  again:
    result = mknod(path, mode, dev);
    if (result == -1) {
	if (errno == EINTR)
	    goto again;
	else if (errno == ENOENT) {
	    connAck(NACK_ENOENT, "");
	}
	else if (errno == EEXIST) {
	    connAck(NACK_EEXIST, "");
	}
	else {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("mknod(%s,%d,%ld) failed: %s\n",
		    path, mode, (long) dev, xstrerror(errno_save));
	}
    }
    return result;
}

static int op_rmdir(const char *path)
{
    int result;
    int errno_save;

  again:
    result = rmdir(path);
    if (result == -1) {
	if (errno == EINTR)
	    goto again;
	else if (errno == ENOENT) {
	    connAck(NACK_ENOENT, "");
	}
	else {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("rmdir(%s) failed: %s\n",
		    path, xstrerror(errno_save));
	}
    }
    return result;
}

static int op_symlink(const char *link, const char *path)
{
    int result;
    int errno_save;

  again:
    result = symlink(link, path);
    if (result == -1) {
	if (errno == EINTR)
	    goto again;
	else if (errno == ENOENT) {
	    connAck(NACK_ENOENT, "");
	}
	else if (errno == EEXIST) {
	    connAck(NACK_EEXIST, "");
	}
	else {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("symlink(%s,%s) failed: %s\n",
		    link, path, xstrerror(errno_save));
	}
    }
    return result;
}

static int op_unlink(const char *path)
{
    int result;
    int errno_save;

  again:
    result = unlink(path);
    if (result == -1) {
	if (errno == EINTR)
	    goto again;
	else if (errno == ENOENT) {
	    connAck(NACK_ENOENT, "");
	}
	else {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("unlink(%s) failed: %s\n",
		    path, xstrerror(errno_save));
	}
    }
    return result;
}



/* save_dir_utime - saves in statbuf the atime, mtime and ctime values
   of the *directory* containing `path' (not of path itself!). If the
   stat call fails, the time values are set to 0.  */
static void save_dir_utime(const char *path, struct stat *statbuf)
{
    char *dir;
    char *cp;
    int result;

    dir = alloca(strlen(path) + 1);
    strcpy(dir, path);
    cp = strrchr(dir, '/');
    if (cp == dir)
	cp[1] = '\0';		/* root directory */
    else if (cp)
	*cp = '\0';
    else *dir = '\0';

    result = Stat(dir, statbuf);
    if (result == -1) {
	debuglog(DEBUG_OP, ("[save_dir_utime: %s]", xstrerror(errno)));
	statbuf->st_atime = 0;
	statbuf->st_mtime = 0;
	statbuf->st_ctime = 0;
    }
}

/* restore_dir_utime - restores the atime and mtime values of the
   *directory* containing `path' (not of path itself!) using the times
   saved in the statbuf, except if timestamp is later than one of the
   values in statbuf. In this latter case, the timestamp value is used
   instead.  */
static void restore_dir_utime(const char *path, struct stat *statbuf,
			      time_t timestamp)
{
    char *dir;
    char *cp;
    int result;
    struct utimbuf times;

    dir = alloca(strlen(path) + 1);
    strcpy(dir, path);
    cp = strrchr(dir, '/');
    if (cp == dir)
	cp[1] = '\0';		/* root directory */
    else if (cp)
	*cp = '\0';
    else *dir = '\0';

    times.actime = (timestamp > statbuf->st_atime
		    ? timestamp
		    : statbuf->st_atime);
    times.modtime = (timestamp > statbuf->st_mtime
		     ? timestamp
		     : statbuf->st_mtime);

    result = Utime(dir, &times);
    if (result == -1) {
	debuglog(DEBUG_OP, ("[restore_dir_utime: %s]", xstrerror(errno)));
    }
}



#define OP_ALREADY(COND) \
    do { if (COND) { connAck(ACK_ALREADY, ""); return; } } while (0)

#define OP_OBSOLETE(COND) \
    do { if (COND) { connAck(NACK_OBSOLETE, ""); return; } } while (0)


OpFile opAllocSlaveOp(void)
{
    OpFile op_file;

    op_file = xmalloc(sizeof (OpFileRec));
    op_file->f = NULL;
    op_file->path = NULL;
    op_file->tmppath = NULL;
    op_file->error = 0;
    return op_file;
}


void opFreeSlaveOp(OpFile op_file)
{
    if (op_file->f)
	fclose(op_file->f);
    if (op_file->tmppath)
	unlink(op_file->tmppath);
    xfree(op_file->tmppath);
    xfree(op_file->path);
    free(op_file);
}


void opSlave(ReqRequest r, ConnEntry e)
{
    char *path;
    char *newpath;
    char *cp;
    struct stat statbuf;
    struct stat newstatbuf;
    struct stat dirstatbuf;
    struct utimbuf times;
    int fd;
    int errno_save;
    OpFile op_file = e->slave_op;

    switch (r->op) {
      case 'D':
	if (!op_file)
	    return;

	switch (r->subop) {
	    unsigned char *cp;
	    unsigned char c;

	  case '0':
	    for (cp = r->s; *cp; cp += 2) {
		if (isdigit(*cp))
		    c = *cp - '0';
		else if (*cp >= 'A' && *cp <= 'F')
		    c = *cp - 'A' + 10;
		else if (*cp >= 'a' && *cp <= 'f')
		    c = *cp - 'a' + 10;
		else op_file->error = NACK_PROTERR;
		c = c << 4;
		if (isdigit(cp[1]))
		    c += cp[1] - '0';
		else if (cp[1] >= 'A' && cp[1] <= 'F')
		    c += cp[1] - 'A' + 10;
		else if (cp[1] >= 'a' && cp[1] <= 'f')
		    c += cp[1] - 'a' + 10;
		else op_file->error = NACK_PROTERR;
		if (fwrite(&c, 1, 1, op_file->f) != 1)
		    op_file->error = NACK_SYSERR;
	    }
	    break;

	  default:
	    op_file->error = NACK_PROTERR;
	    break;
	}
	return;

      case 'E':
	if (!op_file)
	    return;

	path = op_file->path;
	errno_save = 0;
	debuglog(DEBUG_OP, ("writeend(%s)", path));

	if (fclose(op_file->f) == EOF)
	    errno_save = errno, op_file->error = NACK_SYSERR;
	op_file->f = NULL;

	times.actime = op_file->timestamp;
	times.modtime = op_file->timestamp;
	Utime(op_file->tmppath, &times);

	if (!op_file->error) {
	    if (rename(op_file->tmppath, path) == -1) {
		errno_save = errno, op_file->error = NACK_SYSERR;
	    }
	}

	if (op_file->error) {
	    connAck(op_file->error,
		    op_file->error == NACK_SYSERR ? xstrerror(errno_save) : "");
	    warning("writeend(%s) failed: %s\n", path, xstrerror(errno_save));
	}
	else {
	    /* XXX restore dir_utime??? */
	    connAck(ACK_SUCCESS, "");
	}
	opFreeSlaveOp(op_file);
	e->slave_op = NULL;
	return;

      case 'M':			/* M ts path mode 0 - set file mode */
	path = r->path.path;
	debuglog(DEBUG_OP, ("chmod(%s, %#o)", path, r->i1));
	if (op_stat_exist(path, &statbuf) == -1)
	    return;
	OP_ALREADY(statbuf.st_mode == r->i1);
	/* XXX OP_OBSOLETE(r->timestamp < statbuf.st_ctime); */
	if (op_chmod(path, r->i1) != -1)
	    connAck(ACK_SUCCESS, "");
	return;

      case 'O':			/* O ts path owner group - set file owner/group */
	path = r->path.path;
	debuglog(DEBUG_OP, ("chown(%s, %d, %d)", path, r->i1, r->i2));
	/* XXX use op_lstat_exist if lchown is available??? */
	if (op_stat_exist(path, &statbuf) == -1)
	    return;
	OP_ALREADY(statbuf.st_uid == r->i1 && statbuf.st_gid == r->i2);
	/* XXX OP_OBSOLETE(r->timestamp < statbuf.st_ctime); */
	if (op_chown(path, r->i1, r->i2) != -1)
	    connAck(ACK_SUCCESS, "");
	return;

      case 'U':			/* U ts path actime modtime - set file times */
	path = r->path.path;
	debuglog(DEBUG_OP, ("utime(%s, %d, %d)", path, r->i1, r->i2));
	if (op_stat_exist(path, &statbuf) == -1)
	    return;
	OP_ALREADY(statbuf.st_atime == r->i1 && statbuf.st_mtime == r->i2);
	/* XXX OP_OBSOLETE(r->timestamp < statbuf.st_ctime); */
	if (op_utime(path, r->i1, r->i2) != -1)
	    connAck(ACK_SUCCESS, "");
	return;

      case 'W':			/* W ts inopath mode size - write file contents */
	path = r->path.path;
	debuglog(DEBUG_OP, ("writefile(%s, %#o, %d)", path, r->i1, r->i2));
	if (op_stat(path, &statbuf) == -1) {
	    if (errno == ENOENT) { /* create file */
		statbuf.st_size = 0;
		/* File is created below */
	    }
	    else return;
	}
	else {
	    OP_ALREADY(statbuf.st_mtime == r->timestamp
		       && statbuf.st_size == r->i2);
	    OP_OBSOLETE(r->timestamp < statbuf.st_mtime);
	}

	if (op_file) {
	    connAck(NACK_PROTERR, "");
	    error("second request op `%c' while \"%s\" still open\n",
		  r->op, op_file->path ? op_file->path : "(null)");
	    return;
	}

	e->slave_op = op_file = opAllocSlaveOp();
	/* Build temporary filename in op_file->tmppath.  */
	op_file->tmppath = xmalloc(strlen(path) + 6 + 1);
	strcpy(op_file->tmppath, path);
	if ((cp = strrchr(op_file->tmppath, '/')))
	    strcpy(cp + 1, "XXXXXX");
	if (mktemp(op_file->tmppath) == NULL) {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("mktemp(%s) failed: %s\n",
		    op_file->tmppath, xstrerror(errno_save));
	    opFreeSlaveOp(op_file);
	    e->slave_op = NULL;
	    return;
	}

	/* Fill rest of op_file.  */
	op_file->path = xstrdup(path);
	op_file->error = 0;
	op_file->timestamp = r->timestamp;
	op_file->f = fopen(op_file->tmppath, "wb");
	if (op_file->f == NULL) {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("fopen(%s, \"wb\") failed: %s\n",
		    op_file->tmppath, xstrerror(errno_save));
	    opFreeSlaveOp(op_file);
	    e->slave_op = NULL;
	    return;
	}
	fchmod(fileno(op_file->f), r->i1);
	debuglog(DEBUG_OP, (": succeeded\n"));
	return;

      case 'c':			/* c ts fpath mode 0 - creat file */
	path = r->path.path;
	debuglog(DEBUG_OP, ("creat(%s, %#o)", path, r->i1));
	if (op_stat(path, &statbuf) == -1) {
	    if (errno != ENOENT)
		return;
	}
	else {
	    OP_ALREADY(statbuf.st_mtime == r->timestamp
		       && statbuf.st_mode == r->i1
		       && statbuf.st_size == 0);
	    OP_OBSOLETE(r->timestamp < statbuf.st_mtime);
	}
	fd = creat(path, r->i1);
	if (fd == -1) {
	    errno_save = errno;
	    connAck(NACK_SYSERR, xstrerror(errno_save));
	    warning("creat(%s,%#o) failed: %s\n",
		    path, r->i1, xstrerror(errno_save));
	    return;
	}
	close(fd);
	times.actime = r->timestamp;
	times.modtime = r->timestamp;
	Utime(path, &times);
	/* XXX restore dir_utime??? */
	connAck(ACK_SUCCESS, "");
	return;

      case 'd':			/* d ts fpath mode - mkdir */
	path = r->path.path;
	debuglog(DEBUG_OP, ("mkdir(%s, %#o)", path, r->i1));
	if (op_lstat_non_exist(path, &statbuf) == -1) {
	    if (errno == EEXIST) {
		/* XXX what should we do here? */
		/* XXX OP_ALREADY(???); */
		/* XXX OP_OBSOLETE(r->timestamp < statbuf.st_ctime); */
	    }
	    else return;
	}
	if (op_mkdir(path, r->i1) != -1) {
	    times.actime = r->timestamp;
	    times.modtime = r->timestamp;
	    Utime(path, &times);
	    /* XXX restore dir_utime??? */
	    connAck(ACK_SUCCESS, "");
	}
	return;

      case 'l':			/* l ts name newname - link */
	path = r->path.path;
	newpath = r->newpath.path;
	debuglog(DEBUG_OP, ("link(%s, %s)", path, newpath));
	if (op_stat_exist(path, &statbuf) == -1)
	    return;
	if (op_stat_non_exist(newpath, &newstatbuf) == -1) {
	    if (errno == EEXIST) {
		OP_ALREADY(statbuf.st_dev == newstatbuf.st_dev
			   && statbuf.st_ino == newstatbuf.st_ino);
		/* XXX OP_OBSOLETE(r->timestamp < statbuf.st_mtime); */
		connAck(NACK_EEXIST, "");
	    }
	    else return;
	}
	if (op_link(path, newpath) != -1) {
	    /* XXX restore dir_utime??? */
	    connAck(ACK_SUCCESS, "");
	}
	return;

      case 'm':			/* m ts name newname - rename (move) */
	path = r->path.path;
	newpath = r->newpath.path;
	debuglog(DEBUG_OP, ("rename(%s, %s)", path, newpath));
	if (op_stat_exist(path, &statbuf) == -1)
	    return;
	if (op_stat(newpath, &newstatbuf) == -1) {
	    if (errno != ENOENT)
		return;
	}
	else {
	    /* XXX OP_ALREADY(???); */
	    /* XXX OP_OBSOLETE(r->timestamp < statbuf.st_mtime); */
	}
	if (op_rename(path, newpath) != -1) {
	    /* XXX restore dir_utime??? */
	    connAck(ACK_SUCCESS, "");
	}
	return;

      case 'n':			/* n ts fpath mode dev - mknod */
	path = r->path.path;
	debuglog(DEBUG_OP, ("mknod(%s, %#o, %#x)", path, r->i1, r->i2));
	if (op_lstat_non_exist(path, &statbuf) == -1) {
	    if (errno == EEXIST) {
		/* XXX what should we do here? */
		/* XXX OP_ALREADY(???); */
		/* XXX OP_OBSOLETE(r->timestamp < statbuf.st_ctime); */
	    }
	    else return;
	}
	if (op_mknod(path, r->i1, r->i2) != -1) {
	    times.actime = r->timestamp;
	    times.modtime = r->timestamp;
	    Utime(path, &times);
	    /* XXX restore dir_utime??? */
	    connAck(ACK_SUCCESS, "");
	}
	return;

      case 'r':			/* r ts fpath - rmdir */
	path = r->path.path;
	debuglog(DEBUG_OP, ("rmdir(%s)", path));
	if (op_lstat_exist(path, &statbuf) == -1)
	    return;
	OP_OBSOLETE(r->timestamp < statbuf.st_mtime);
	save_dir_utime(path, &dirstatbuf);
	if (op_rmdir(path) != -1) {
	    restore_dir_utime(path, &dirstatbuf, r->timestamp);
	    connAck(ACK_SUCCESS, "");
	}
	return;

      case 's':			/* s ts string name - symlink */
	path = r->path.path;
	debuglog(DEBUG_OP, ("symlink(%s, %s)", r->s, path));
	if (op_lstat_non_exist(path, &statbuf) == -1) {
	    if (errno == EEXIST) {
		OP_ALREADY(S_ISLNK(statbuf.st_mode)
			   && symlinkcmp(r->s, path) == 0);
		OP_OBSOLETE(r->timestamp < statbuf.st_mtime);
		connAck(NACK_EEXIST, "");
	    }
	    else return;
	}
	save_dir_utime(path, &dirstatbuf);
	if (op_symlink(r->s, path) != -1) {
	    restore_dir_utime(path, &dirstatbuf, r->timestamp);
	    /* Updating the times and modes of a symbolic link is not
	       possible.  Updating the owner/group is only possible if
	       lchown() is available. This is currently not supported.  */
	    connAck(ACK_SUCCESS, "");
	}
	return;

      case 'u':			/* u ts fpath - unlink */
	path = r->path.path;
	debuglog(DEBUG_OP, ("unlink(%s)", path));
	if (op_lstat_exist(path, &statbuf) == -1)
	    return;
	OP_OBSOLETE(r->timestamp < statbuf.st_mtime);
	save_dir_utime(path, &dirstatbuf);
	if (op_unlink(path) != -1) {
	    restore_dir_utime(path, &dirstatbuf, r->timestamp);
	    connAck(ACK_SUCCESS, "");
	}
	return;
    }

    debuglog(DEBUG_OP, ("operation %c(%s)",
			r->op, r->path.path ? r->path.path : "(null)"));
    connAck(NACK_PROTERR, "");
    error("illegal request op `%c'\n", r->op);
}




#define BUFFER_LEN 4096

void opMaster(ReqRequest r, ConnEntry e)
{
    char *targetpath;
    char *saved_path = NULL;
    char *saved_newpath = NULL;
    char *line;
    FILE *f = NULL;
    unsigned char ignore_request = 0;

    if (!cfFileMatch(r->path.path, e, &targetpath))
	return;

    if (targetpath) {
	saved_path = r->path.path;
	r->path.path = targetpath;
    }

    targetpath = NULL;
    if (r->newpath.path && !cfFileMatch(r->newpath.path, e, &targetpath)) {
	if (saved_path) {
	    free(r->path.path);
	    r->path.path = saved_path;
	}
	return;
    }

    if (targetpath) {
	saved_newpath = r->newpath.path;
	r->newpath.path = targetpath;
    }

    line = reqRequestToMessage(r);

    if (saved_path) {
	free(r->path.path);
	r->path.path = saved_path;
    }
    if (saved_newpath) {
	free(r->newpath.path);
	r->newpath.path = saved_newpath;
    }

    if (!line)
	return;

    switch (r->op) {
      case 'W':			/* W ts inopath mode size - write file contents */
	f = fopen(r->path.path, "r");
	if (!f) {		/* file no longer exists, therefore ignore */
	    ignore_request = 1;	/* this request */
	    /* XXX send unlink(r->path.path) request to fix this race condition */
	}
	break;
    }

    if (!ignore_request)
	bioWrite(bio_context, e->masterfd, line, strlen(line) + 1);

    free(line);

    if (f) {			/* send file */
	int bytes = 0;
	int c;
	char buf[BUFFER_LEN * 2 + 1];

	do {
	    c = getc(f);
	    if (c != EOF) {
	        buf[bytes * 2] = "0123456789ABCDEF"[c / 16];
		buf[bytes * 2 + 1] = "0123456789ABCDEF"[c % 16];
		bytes++;
	    }
	    if ((c == EOF && bytes != 0)
		|| bytes == BUFFER_LEN) { /* line break after BUFFER_LEN input bytes */
		bioWrite(bio_context, e->masterfd, "D0 ", 3);
		buf[bytes * 2] = '\0';
		bioWrite(bio_context, e->masterfd, buf, bytes * 2 + 1);
		bytes = 0;
	    }
	} while (c != EOF);
	fclose(f);
	bioWrite(bio_context, e->masterfd, "E", 2);
    }
}

void opStoreRequest(ReqRequest r, ConnEntry e)
{
    char filename[1024];
    FILE *f;

    switch (r->op) {
      case 'r':			/* rmdir */
      case 'u':			/* unlink */
	if (cfFileMatch(r->path.path, e, NULL)) {
	    sprintf(filename, "%s/omirr-%s-%s", store_dir, own_hostname, e->name);
	    f = fopen(filename, "a");
	    if (f == NULL) {
	        error("%s: failed to store request in redo-buffer: %s\n",
		      filename, xstrerror(errno));
		return;
	    }
	    fprintf(f, "%c %ld %s\n", r->op, r->timestamp, r->path.path);
	    if (fclose(f) == EOF) {
	        error("%s: failed to store request in redo-buffer: %s\n",
		      filename, xstrerror(errno));
	    }
	}
	break;
      case 'm':
	/* XXX */
	break;
    }
}
