/*
 *  Copyright (C) 1998-99 Luca Deri <deri@unipi.it>
 *                      
 *  			  Centro SERRA, University of Pisa
 *  			  http://www-serra.unipi.it/
 *  					
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ntop.h"

typedef struct nfsEntries {
  HostTraffic* host;
  TrafficCounter bytesSent, bytesRcvd;
  TrafficCounter lastBytesSent, lastBytesRcvd;
  float sentThpt, rcvdThpt;
} NfsEntries;

extern char* intoa(struct in_addr addr);
extern char* etheraddr_string(const u_char *ep);
extern void printHTTPheader();
extern void printHTTPtrailer();
extern HostTraffic* findHostByNumIP(char* numIPaddr);
extern char* makeHostLink(HostTraffic *el, short mode, 
			  short cutName, short addCountryFlag);
extern char* formatThroughput(float numBytes);
extern void sendString(char *theString);
extern char* formatBytes(TrafficCounter numBytes, short encodeString);
extern char* getRowColor();
extern void quicksort(void*, size_t, size_t, int(*cmp)());
extern void printBar(char *buf, unsigned short percentage,
	      unsigned short maxPercentage, unsigned short ratio);
extern char* formatThroughput(float numBytes);

static int columnSort = 0;
static NfsEntries *nfsEntries[SHORTHASHNAMESIZE];

static time_t nextNfsUpdate;
extern int webMode;

/* ********************* */

static NfsEntries* findNfsEntry(struct in_addr addr) {
  unsigned long key = addr.s_addr % SHORTHASHNAMESIZE;
  unsigned int numSearches = 0;

  while((nfsEntries[key] != NULL) 
	&& (numSearches++ < SHORTHASHNAMESIZE)
	&& (nfsEntries[key]->host->hostIpAddress.s_addr == addr.s_addr))
    break;

  if(nfsEntries[key] == NULL) {
    HostTraffic* host = findHostByNumIP(intoa(addr));

    if(host == NULL)
      return(NULL); /* This shouldn't happen */

    nfsEntries[key] = (NfsEntries*)malloc(sizeof(NfsEntries));
    memset(nfsEntries[key], 0, sizeof(NfsEntries));
    nfsEntries[key]->host = host;
    return(nfsEntries[key]);
  } else if(nfsEntries[key]->host->hostIpAddress.s_addr == addr.s_addr)
    return(nfsEntries[key]);
  else
    return(NULL);
}

/* ********************* */

static void updateNfsThpt() {
  unsigned long now, secsDiff;
  int i;


  now = time(NULL);
  secsDiff = now-nextNfsUpdate+THROUGHPUT_REFRESH_TIME;

#ifdef DEBUG
  printf("Updating NFS thpt (%d)...\n");
#endif

  for(i=0; i<SHORTHASHNAMESIZE; i++)
    if(nfsEntries[i] != NULL) {
      float diff;

      diff = (float)(nfsEntries[i]->bytesSent-nfsEntries[i]->lastBytesSent);
      nfsEntries[i]->sentThpt = diff/secsDiff;
#ifdef DEBUG
      printf("%s) %.2f/", nfsEntries[i]->host->hostSymIpAddress, nfsEntries[i]->sentThpt);
#endif

      diff = (float)(nfsEntries[i]->bytesRcvd-nfsEntries[i]->lastBytesRcvd);
      nfsEntries[i]->rcvdThpt = diff/secsDiff;
#ifdef DEBUG
      printf("%.2f\n", nfsEntries[i]->rcvdThpt);
#endif

      nfsEntries[i]->lastBytesSent = nfsEntries[i]->bytesSent;
      nfsEntries[i]->lastBytesRcvd = nfsEntries[i]->bytesRcvd;
    }

  nextNfsUpdate = now+THROUGHPUT_REFRESH_TIME;
}

/* ********************* */

static void handleNFSPacket(const struct pcap_pkthdr *h, const u_char *p) {
  struct ip ip;
  struct udphdr nfsPktHdr;
  u_int hlen, length = h->len-sizeof(struct ether_header);
#ifdef DEBUG
  u_short sport, dport;
#endif
  NfsEntries *srcHost, *dstHost;

  memcpy(&ip, (p+sizeof(struct ether_header)), sizeof(struct ip));
  hlen = (u_int)ip.ip_hl * 4;
  memcpy(&nfsPktHdr, (p+sizeof(struct ether_header)+hlen), sizeof(struct udphdr));  

#ifdef DEBUG
  sport = ntohs(nfsPktHdr.uh_sport);
  dport = ntohs(nfsPktHdr.uh_dport);

  printf("NFS %s:%d ", intoa(ip.ip_src), sport);
  printf("-> %s:%d [len=%d]\n", intoa(ip.ip_dst), dport, length);
#endif

  srcHost = findNfsEntry(ip.ip_src);
  if(srcHost != NULL) srcHost->bytesSent += length;

  dstHost = findNfsEntry(ip.ip_dst);
  if(dstHost != NULL) dstHost->bytesRcvd += length;
  
  if(time(NULL) > nextNfsUpdate)
    updateNfsThpt();
}

/* ******************************* */

static int sortNFShosts(const void *_a, const void *_b) {
  NfsEntries **a = (NfsEntries **)_a;
  NfsEntries **b = (NfsEntries **)_b;

  if((a == NULL) && (b != NULL)) {
    printf("WARNING (1)\n");
    return(1);
  } else if((a != NULL) && (b == NULL)) {
    printf("WARNING (2)\n");
    return(-1);
  } else if((a == NULL) && (b == NULL)) {
    printf("WARNING (3)\n");
    return(0);
  }

  switch(columnSort) {
  default:
  case 1:
    if((*a)->host->hostSymIpAddress == NULL) 
      (*a)->host->hostSymIpAddress = (*a)->host->hostNumIpAddress;
    if((*b)->host->hostSymIpAddress == NULL) 
      (*b)->host->hostSymIpAddress = (*b)->host->hostNumIpAddress;		
    return(strcasecmp((*a)->host->hostSymIpAddress, (*b)->host->hostSymIpAddress));
    break;

  case 2:
    if((*a)->bytesSent < (*b)->bytesSent)
      return(1);
    else if ((*a)->bytesSent > (*b)->bytesSent)
      return(-1);
    else
      return(0);
    break;

  case 3:
    if((*a)->sentThpt < (*b)->sentThpt)
      return(1);
    else if ((*a)->sentThpt > (*b)->sentThpt)
      return(-1);
    else
      return(0);
    break;

  case 4:
    if((*a)->bytesRcvd < (*b)->bytesRcvd)
      return(1);
    else if ((*a)->bytesRcvd > (*b)->bytesRcvd)
      return(-1);
    else
      return(0);
    break;

  case 5:
    if((*a)->rcvdThpt < (*b)->rcvdThpt)
      return(1);
    else if ((*a)->rcvdThpt > (*b)->rcvdThpt)
      return(-1);
    else
      return(0);
    break;

  }  
}

/* ****************************** */

static void handleNfsWatchHTTPrequest(char* url) {
  static short everUpdated = 0;
  char tmpStr[2*BUF_SIZE];
  int numEntries = 0, i, revertOrder=0;
  char *pluginName = "<A HREF=/plugins/nfsWatch";
  char *sign[16];
  NfsEntries *tmpNfsEntries[SHORTHASHNAMESIZE];
  float maxSentThpt=-1, maxRcvdThpt=-1;

  if(!everUpdated) {
    updateNfsThpt();
    everUpdated = 1;
  }

  for(i=0; i<16; i++) sign[i] = "";

  if(url[0] == '\0') {
    columnSort = 0;
  } else {
    if(url[0] == '-') {
      revertOrder = 1;
      columnSort = atoi(&url[1]);
    } else {
      columnSort = atoi(url);
    }
  }

  if(!revertOrder) sign[columnSort] = "-";

  printHTTPheader();

  for(i=0; i<SHORTHASHNAMESIZE; i++) {
    if(nfsEntries[i] != NULL) {
      tmpNfsEntries[numEntries++] = nfsEntries[i];
      if(nfsEntries[i]->sentThpt > maxSentThpt) maxSentThpt = nfsEntries[i]->sentThpt;
      if(nfsEntries[i]->sentThpt > maxRcvdThpt) maxRcvdThpt = nfsEntries[i]->rcvdThpt;
    }
  }

  sendString("<HTML><BODY BGCOLOR=#FFFFFF><FONT FACE=Helvetica>"
	     "<CENTER><H1>Welcome to nfsWatch</H1>\n<p>");

  if(numEntries > 0) {
    sendString("<TABLE BORDER><TR>");

    sprintf(tmpStr, "<TH>%s?%s1>Host</A></TH>"
	    "<TH>%s?%s2>Data&nbsp;Sent</A></TH>"
	    "<TH COLSPAN=2>%s?%s3>Actual&nbsp;Sent&nbsp;Thpt</A></TH>"
	    "<TH>%s?%s4>Data&nbsp;Rcvd</A></TH>"
	    "<TH COLSPAN=2>%s?%s5>Actual&nbsp;Rcvd&nbsp;Thpt</A></TH>"
	    "</TR>\n",
	    pluginName, sign[1],
	    pluginName, sign[2],
	    pluginName, sign[3],
	    pluginName, sign[4],
	    pluginName, sign[5]);

    sendString(tmpStr);
 
    quicksort(tmpNfsEntries, numEntries, sizeof(NfsEntries*), sortNFShosts);

    for(i=0; i<numEntries; i++) {
      NfsEntries *theEntry;
      char bar[512];

      if(revertOrder)
	theEntry = tmpNfsEntries[numEntries-i-1];
      else
	theEntry = tmpNfsEntries[i];
      sprintf(tmpStr, "<TR %s>%s"
	      "<TD ALIGN=RIGHT>%s</TD>"
	      "<TD ALIGN=RIGHT>%s</TD>",
	      getRowColor(), makeHostLink(theEntry->host, 1, 1, 0),
	      formatBytes(theEntry->bytesSent, 1),
	      formatThroughput(theEntry->sentThpt));
      sendString(tmpStr);
      printBar(bar, (unsigned short)((theEntry->sentThpt*100)/maxSentThpt), 100, 1);

      sprintf(tmpStr, "<TD ALIGN=RIGHT>%s</TD>"
	      "<TD ALIGN=RIGHT>%s</TD>\n",
	      formatBytes(theEntry->bytesRcvd, 1),
	      formatThroughput(theEntry->rcvdThpt));
      sendString(tmpStr);

      printBar(bar, (unsigned short)((theEntry->rcvdThpt*100)/maxRcvdThpt), 100, 1);

      sendString("</TR>\n");
    }

    sendString("</TABLE></CENTER><p>\n");
  } else {
    sendString("<P><FONT FACE=Helvetica><CENTER><H1>"
	       "<i>No Data To Display (yet)</i></H1>"
	       "</CENTER></FONT>\n");
  }

  printHTTPtrailer();  
}

/* ****************************** */

static void termNfsFunct() {
  if(webMode)
    printf("Thanks for using nfsWatch.\n");
}

/* ****************************** */

static PluginInfo nfsPluginInfo[] = {
  "nfsWatchPlugin",
  "This plugin handles NFS traffic",
  "1.0",           /* version */
  "<A HREF=http://jake.unipi.it/~deri/>Luca Deri</A>", 
  "nfsWatch",      /* http://<host>:<port>/plugins/nfsWatch */
  termNfsFunct,    /* TermFunc   */
  handleNFSPacket, /* PluginFunc */
  handleNfsWatchHTTPrequest,
  "udp" /* udp port 2049 */  /* BPF filter: filter all the nfs (nfsd = port 2049) packets */
};

/* Plugin entry fctn */
PluginInfo* PluginEntryFctn() {

  if(webMode)
    printf("Welcome to %s. (C) 1999 by Luca Deri.\n",  nfsPluginInfo->pluginName);

  memset(nfsEntries, 0, sizeof(NfsEntries*));
  nextNfsUpdate = time(NULL)+THROUGHPUT_REFRESH_TIME;

  return(nfsPluginInfo);
}
