/* NetLeds applet - Lights from a network device
 * Copyright (C) 1999 Michael Witrant
 *
 * Homepage: http://netleds.port5.com
 * Author: Michael Witrant
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include "netleds.h"

/* Display */
static GtkWidget *frame  = NULL;
static GtkWidget *box    = NULL;
extern GtkWidget *applet;

int spacing = DEFAULT_SPACING;
int border  = DEFAULT_BORDER;
int nb_leds = DEFAULT_NBLEDS;
orient_style ostyle = DEFAULT_ORIENT_STYLE;
int was_vertical = FALSE;

inline
void draw_led(int n, int check)
{
	led_update_value(n, proc_get_stat(leds[n].iid), check);
}

/* Draw all lights */
int draw_check(int check)
{
	int i;
	
	proc_update(device_name);
	
	for (i=0; i<nb_leds; ++i)
	{
		draw_led(i,check);
	}

	return TRUE;
}

int draw()
{
	return draw_check(TRUE);
}

int draw_all()
{
	return draw_check(FALSE);
}

inline
void add_led(int n, GtkWidget *box)
{
	gtk_box_pack_start(GTK_BOX(box), led_get_widget(n), TRUE, TRUE, 0);
	gtk_widget_show(led_get_widget(n));
}

inline
void remove_led(int n, GtkWidget *box)
{
	gtk_container_remove(GTK_CONTAINER(box), led_get_widget(n));
}

void set_nb_leds(int n)
{
	int i;

	if (n < nb_leds)
		for (i=n; i<nb_leds; i++)
			remove_led(i,box);
	if (n > nb_leds)
		for (i=nb_leds; i<n; i++)
			add_led(i,box);
	nb_leds = n;
}

void set_spacing(int s)
{
	spacing = s;
	gtk_box_set_spacing(GTK_BOX(box), s);
}

void set_border(int b)
{
	border = b;
	gtk_container_set_border_width(GTK_CONTAINER(box), b);
}

void change_box(int v)
{
	int i;
	GtkWidget *newbox;
	GtkWidget *lw;

	newbox = v ? gtk_vbox_new(FALSE, spacing) : gtk_hbox_new(FALSE, spacing);
	
	for (i=0; i<nb_leds; i++)
	{
		lw = led_get_widget(i);
		if (box)
			gtk_container_remove(GTK_CONTAINER(box), lw); 
		add_led(i,newbox);
	}

	if (box)
	{
		gtk_widget_hide(box);
		gtk_container_remove(GTK_CONTAINER(frame), box);
	}
	gtk_container_add(GTK_CONTAINER(frame),newbox);
	
	gtk_widget_show(newbox);
	box = newbox;
	set_border(border);
}

void draw_orient_changed()
{
	int vertical = (orient == ORIENT_UP || orient == ORIENT_DOWN);
	
	switch (ostyle)
	{
		case orient_auto:
			break;
		case orient_invert:
			vertical = !vertical;
			break;
		case orient_vertical:
			vertical = TRUE;
			break;
		case orient_horizontal:
			vertical = FALSE;
			break;
		default:
			g_error("unknown orient style: %d", ostyle);
	}
	change_box(vertical);
}

void set_orient_style(orient_style o)
{
	if (o != ostyle)
	{
		ostyle = o;
		draw_orient_changed();
	}
}

pixmask create_pixmask(GdkWindow *window, GtkStyle *style, char **xpm)
{
	pixmask pix;
	pix.img = gdk_pixmap_create_from_xpm_d(window, &pix.mask, 
			&style->bg[GTK_STATE_NORMAL], (gchar**)xpm);
	return pix;
}

ledpix create_ledpix(GdkWindow *window, GtkStyle *style, lcolor lc)
{
	ledpix lp;
	lp.on  = create_pixmask(window, style, lc.on);
	lp.off = create_pixmask(window, style, lc.off);
	return lp;
}

void create_pixmaps(GdkWindow *window, GtkStyle *style)
{
	int i;
	for (i=0; lcolors[i].cid != -1; i++)
		lcolors[i].pix = create_ledpix(window, style, lcolors[i]);
}


ledpix get_ledpix(int cid)
{
	int i;
	for (i=0; lcolors[i].cid != -1; i++)
		if (cid == lcolors[i].cid)
			return lcolors[i].pix;
	
	g_error("invlalid color: %d", cid);
	return lcolors[0].pix;
}

void set_widget_tooltip(GtkWidget *widget, char *str)
{
	applet_widget_set_widget_tooltip(APPLET_WIDGET(applet), widget, str);
}

gint draw_init(GtkWidget *applet)
{
	GtkStyle *style;
	GdkWindow *window;

	frame = gtk_frame_new(NULL);
	applet_widget_add(APPLET_WIDGET(applet),frame);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
	gtk_widget_realize(frame);
	
	/* orient_changed is called before the applet is displayed. We don't
	   need to create the box before that
	*/	
	/* change_box(TRUE); */

	style = gtk_widget_get_style(frame);
	window = frame->window;

	create_pixmaps(window, style);
	leds_create_widget();

	gtk_widget_show(frame);

	return TRUE;
}
