/*  Protocol compatible masqdialer server written in C
    Copyright (C) 1998 Charles P. Wright 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    Parameter Database Functions by Shane Kerr <kerr@wizard.net>
*/

#include <syslog.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "mserver.h"

/* initial number of entries in the parameter database */
#define INIT_SPACE_FOR_PARAM 32

/* boolean data type */
typedef enum { FALSE, TRUE } boolean_t;

/* release memory used by a parameter */
static void dispose_param(struct param_t* param)
{
	if (param != NULL)
	{
		if (param->key != NULL)
		{
			free(param->key);
			param->key = NULL;
		}

		if (param->val != NULL)
		{
			free(param->val);
			param->val = NULL;
		}
	}
}

/* initialize a parameter database */
void init_param_dbase(struct param_dbase_t* dbase)
{
	int i;

	dbase->num_param = 0;
	dbase->space_for_param = INIT_SPACE_FOR_PARAM;

	dbase->param = (struct param_t *) calloc(dbase->space_for_param, sizeof(struct param_t));

	if (dbase->param == NULL)
	{
		fprintf(stderr, "Out of memory initializing parameter database\n");
#ifndef UTIL
		mserver_cleanup();
#endif
		exit(1);
	}

	for (i = 0; i < dbase->num_param; i++)
	{
		((dbase->param + i)->val) = NULL;
		((dbase->param + i)->key) = NULL;
    	}
}

/* add a value with the given key to the database */
void param_dbase_setval(struct param_dbase_t* dbase, const char* key, const char *val)
{
	boolean_t found_key;
	int i;
	int mem_size;
	int origsize;

	found_key = false;

	for (i = 0; ((i < dbase->num_param) && (!found_key)); i++)
	{
		if (dbase->param[i].key != NULL)	
		{
			if (strcmp(dbase->param[i].key, key) == 0)
			{
				if ((dbase->param + i)->val != NULL)
				{
					free((dbase->param + i)->val);
				}
				(dbase->param + i)->val = strdup(val);
				found_key = TRUE;
			}
		}
	}

	if (!found_key)
	{
		/* expand database if necessary */
		if (dbase->num_param >= dbase->space_for_param)
		{
			origsize = dbase->space_for_param;
			dbase->space_for_param *= 2;
			mem_size = dbase->space_for_param * sizeof (struct param_t);
			dbase->param = (struct param_t *)realloc(dbase->param, mem_size);
			if (dbase->param == NULL)
			{
#ifndef UTIL
				mserver_cleanup();
#endif
				exit(1);
			}
			for (i = origsize; i < dbase->num_param; i++)
			{
				((dbase->param + i)->val) = NULL;
				((dbase->param + i)->key) = NULL;
    			}
		}

		(dbase->param + i)->val = strdup(val);
		(dbase->param + i)->key = strdup(key);
		dbase->num_param += 1;
	}

}

/* find a key with the given value */
char* param_dbase_getval(const struct param_dbase_t* dbase, const char* key)
{
	char* ret_val;
	int i;

	ret_val = NULL;
	for (i = 0; (i < dbase->num_param) && (ret_val == NULL); i++)
	{
		if (dbase->param[i].key != NULL)
		{
			if (strcmp(dbase->param[i].key, key) == 0)
			{
				ret_val = dbase->param[i].val;
			}
		}
	}

	return ret_val;
}

/* clear the database */
void clear_param_dbase(struct param_dbase_t* dbase)
{
	int i;

	for (i=0; i<dbase->num_param; i++)
	{
		dispose_param(dbase->param + i);
    	}
}
