;;; liece-url.el --- URL browsing.
;; Copyright (C) 1998, 1999 Daiki Ueno

;; Author: Daiki Ueno <ueno@ueda.info.waseda.ac.jp>
;; Created: 1998-09-28
;; Revised: 1998-11-25
;; Keywords: IRC, liece

;; This file is part of Liece.

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Code:

(eval-when-compile 
  (require 'liece-inlines)
  (require 'liece-intl)
  (require 'browse-url)
  (require 'liece-menu))

(defvar-maybe browse-url-browser-function nil)

(defgroup liece-url nil
  "URL Browsing in IRC buffer"
  :group 'liece-vars)

(defcustom liece-url-regexp  "\\b\\(s?https?\\|ftp\\|file\\|gopher\\|news\\|telnet\\|wais\\|mailto\\):\\(//[-a-zA-Z0-9_.]+:[0-9]*\\)?[-a-zA-Z0-9_=?#$@~`%&*+|\\/.,]*[-a-zA-Z0-9_=#$@~`%&*+|\\/]"
  "Regular expression that matches URLs."
  :group 'liece-url
  :type 'regexp)

(defcustom liece-url-mail-regexp "\\bmailto:\\([-a-zA-Z0-9_=?#$@~`%&*+|\\/.,]*[-a-zA-Z0-9_=#$@~`%&*+|\\/]\\)"
  "Regular expression that matches mailto:."
  :group 'liece-url
  :type 'regexp)

(defcustom liece-url-browser-name
  (let (url-func url-prog len)
    (if (featurep 'browse-url)
	(and (setq url-func browse-url-browser-function)
	     (setq url-prog (symbol-name url-func)) 
	     (stringp url-prog)
	     (setq len (length url-prog))
	     (setq url-prog (substring url-prog 11 len)))
      "netscape"))
  "Default URL Browser Name
    netscape   Netscape    1.1b1         
    mosaic     XMosaic     <= 2.4
    cci        XMosaic     2.5
    w3         w3          0
    iximosaic  IXI Mosaic  ?
    lynx-*     Lynx        0
    grail      Grail       0.3b1"
  :type 'string
  :group 'liece-url)

(defvar liece-url-add-hook nil)

(defmacro liece-url-prepare-browser-function ()
  (if (featurep 'browse-url)
      '(intern (concat "browse-url-" liece-url-browser-name))
    '(intern (concat "liece-url-browser-" liece-url-browser-name))))

(defcustom liece-url-browser-function
  (liece-url-prepare-browser-function)
  "Default URL Browser"
  :type 'function
  :group 'liece-url)

(defcustom liece-url-alist nil
  "Alist for URL completion"
  :type 'alist
  :group 'liece-url)

(define-widget 'liece-url-link 'link
  "A link to an www page." 
  :help-echo 'widget-url-link-help-echo
  :action 'liece-url-link-action)

(defun liece-url-link-action (widget &optional event)
  (let ((func (liece-url-prepare-browser-function)))
    (if (fboundp func)
        (setq liece-url-browser-function func)
      (setq func liece-url-browser-function))
    (funcall func (widget-value widget))))

(defun liece-url-add-buttons (start end)
  (save-excursion
    (goto-char start)
    (while (re-search-forward liece-url-regexp end t)
      (let* ((url-start (match-beginning 0))
	     (url-end (match-end 0))
	     (url (buffer-substring url-start url-end)))
	(if liece-highlight-mode 
	    (liece-widget-convert-button 'url-link url-start url-end url))
	(unless (assoc url liece-url-alist)
	  (push (list url) liece-url-alist)
	  (run-hook-with-args 'liece-url-add-hook url))))
    ))

(defun liece-command-browse-url (&optional url)
  (interactive
   (let (url)
     (if (and current-prefix-arg (eq current-prefix-arg '-))
	 (setq url (caar liece-url-alist))
       (setq url (liece-minibuffer-completing-default-read 
		  (_ "Open URL:") liece-url-alist)))
     (list url)))
  (let ((func (liece-url-prepare-browser-function)))
    (if (fboundp func)
	(setq liece-url-browser-function func)
      (setq func liece-url-browser-function))
    (funcall func url)))

(defun liece-url-browser-netscape (url)
  (let ((mail-regex liece-url-mail-regexp)
        (url-proc-name "netscape"))
    (cond ((string-match mail-regex url)
	   (let ((composefunc (get 'mail-user-agent 'composefunc)))
	     (funcall composefunc (match-string 1 url))))
	  (t
	   (process-kill-without-query 
	    (start-process "url-view" nil url-proc-name
			   "-noraise"                         
			   "-remote"
			   (concat "openURL(" url ")")))))
    ))

(defun url-irc-liece (host port channel user password)
  (let ((liece-server 
	 `(:host ,host :service ,port :password ,password))
	(liece-startup-channel channel))
    (liece)))

(provide 'liece-url)

;;; liece-url.el ends here.