/*
 ui-printtext.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "../nls.h"

static gboolean toggle_show_timestamps, toggle_show_msgs_timestamps, toggle_hide_text_style;
static gint printtag;
static gchar ansitab[8] = { 0, 4, 2, 6, 1, 5, 3, 7 };

void ui_beep(void)
{
    signal_emit("gui print text", 6, cur_channel, NULL, NULL,
		GINT_TO_POINTER(PRINTFLAG_BEEP), "", MSGLEVEL_NEVER);
}

/* parse ANSI color string */
static gchar *convert_ansi(gchar *str, gint *fgcolor, gint *bgcolor, gint *flags)
{
    gchar *start;
    gint fg, bg, fl, num;

    if (*str != '[') return str;

    start = str;

    fg = *fgcolor < 0 ? current_theme->default_color : *fgcolor;
    bg = *bgcolor < 0 ? -1 : *bgcolor;
    fl = *flags;

    str++; num = 0;
    for (;; str++)
    {
        if (*str == '\0') return start;

        if (isdigit((gint) *str))
        {
            num = num*10 + (*str-'0');
            continue;
        }

        if (*str != ';' && *str != 'm') return start;

        switch (num)
        {
            case 0:
                /* reset colors back to default */
                fg = current_theme->default_color;
                bg = -1;
                break;
            case 1:
                /* hilight */
                fg |= 8;
                break;
            case 5:
                /* blink */
                bg = bg == -1 ? 8 : bg | 8;
                break;
            case 7:
                /* reverse */
                fl |= PRINTFLAG_REVERSE;
                break;
            default:
                if (num >= 30 && num <= 37)
                    fg = (fg & 0xf8) + ansitab[num-30];
                if (num >= 40 && num <= 47)
                {
                    if (bg == -1) bg = 0;
                    bg = (bg & 0xf8) + ansitab[num-40];
                }
                break;
        }
        num = 0;

        if (*str == 'm')
        {
            if (!toggle_hide_text_style)
            {
                *fgcolor = fg;
                *bgcolor = bg == -1 ? -1 : bg;
                *flags = fl;
            }
            str++;
            break;
        }
    }

    return str;
}

#define IN_COLOR_CODE 2
#define IN_SECOND_CODE 4
gchar *strip_codes(gchar *input)
{
    gchar *str, *p, *out;
    gint loop_state;

    loop_state = 0;
    out = str = g_strdup(input);
    for (p = input; *p != '\0'; p++) /* Going through the string till the end k? */
    {
	if (*p == '\003')
	{
	    loop_state = IN_COLOR_CODE;
	    continue;
	}

	if (loop_state & IN_COLOR_CODE)
	{
	    if (isdigit( (gint) *p )) continue;
	    if (*p != ',' || (loop_state & IN_SECOND_CODE))
	    {
		/* we're no longer in a color code */
		*out++ = *p;
		loop_state &= ~IN_COLOR_CODE|IN_SECOND_CODE;
		continue;
	    }

	    /* we're in the second code */
	    loop_state |= IN_SECOND_CODE;
	    continue;

	}

	/* we're not in a color code that means we should add the character */
	if (*p == 4 && p[1] != '\0' && p[2] != '\0')
	{
	    p += 2;
	    continue;
	}

	if (*p == 2 || *p == 22 || *p == 27 || *p == 31 || *p == 15)
	    continue;
        *out++ = *p;
    }

    *out = '\0';
    return str;
}

static gboolean expand_styles(GString *out, gchar **format, SERVER_REC *server, gchar *channel, gint level)
{
    static gchar *backs = "01234567";
    static gchar *fores = "krgybmcw";
    static gchar *boldfores = "KRGYBMCW";
    gchar *p, fmt;

    /* p/P -> m/M */
    fmt = **format;
    if (fmt == 'p')
	fmt = 'm';
    else if (fmt == 'P')
	fmt = 'M';

    switch (fmt)
    {
        case 'U':
            /* Underline on/off */
            g_string_append_c(out, 4);
            g_string_append_c(out, -1);
            g_string_append_c(out, 2);
	    break;
	case '9':
	case '_':
            /* bold on/off */
	    g_string_append_c(out, 4);
            g_string_append_c(out, -1);
            g_string_append_c(out, 1);
	    break;
	case '8':
	    /* reverse */
	    g_string_append_c(out, 4);
            g_string_append_c(out, -1);
            g_string_append_c(out, 3);
	    break;
	case '%':
            g_string_append_c(out, '%');
            break;
	case ':':
            /* Newline */
            printtext(server, channel, level, out->str);
            g_string_truncate(out, 0);
	    break;

	case 'F':
            /* flashing - ignore */
	    break;

	case 'N':
	    /* don't put clear-color tag at the end of the output - ignore */
	    break;

	case 'n':
	    /* default color */
            g_string_append_c(out, 4);
            g_string_append_c(out, -1);
            g_string_append_c(out, -1);
	    break;

	default:
	    /* check if it's a background color */
	    p = strchr(backs, fmt);
	    if (p != NULL)
	    {
		g_string_append_c(out, 4);
		g_string_append_c(out, -2);
		g_string_append_c(out, ansitab[(gint) (p-backs)]+1);
		break;
	    }

	    /* check if it's a foreground color */
	    p = strchr(fores, fmt);
	    if (p != NULL)
	    {
		g_string_append_c(out, 4);
		g_string_append_c(out, ansitab[(gint) (p-fores)]+1);
		g_string_append_c(out, -2);
		break;
	    }

	    /* check if it's a bold foreground color */
	    p = strchr(boldfores, fmt);
	    if (p != NULL)
	    {
		g_string_append_c(out, 4);
		g_string_append_c(out, 8+ansitab[(gint) (p-boldfores)]+1);
		g_string_append_c(out, -2);
		break;
	    }
	    return FALSE;
    }

    return TRUE;
}

void output_format_text_args(GString *out, SERVER_REC *server, gchar *channel, gint level, FORMAT_REC *format, gchar *text, va_list args)
{
    gchar *str, *arg;
    gint num;
    gpointer params[10];

    str = current_theme != NULL && text != NULL ? text : format->def;

    /* read all optional arguments to params[] list so they can be used in
       any order.. */
    for (num = 0; num < format->params && num < 10; num++)
    {
        switch (format->paramtypes[num])
        {
            case FORMAT_STRING:
                params[num] = (gchar *) va_arg(args, gchar *);
		if (params[num] == NULL)
		{
		    g_warning("output_format_text_args() : parameter %d is NULL in %s", num, str);
		    params[num] = "";
		}
                break;
            case FORMAT_INT:
                {
                    gint d = (gint) va_arg(args, gint);
                    params[num] = g_new(gint, 1);
                    memcpy(params[num], &d, sizeof(gint));
                }
                break;
            case FORMAT_LONG:
                {
                    glong l = (glong) va_arg(args, glong);
                    params[num] = g_new(glong, 1);
                    memcpy(params[num], &l, sizeof(glong));
                }
                break;
            case FORMAT_FLOAT:
                {
                    gfloat f = (gfloat) va_arg(args, gfloat);
                    params[num] = g_new(gfloat, 1);
                    memcpy(params[num], &f, sizeof(gfloat));
                }
                break;
        }
    }

    for (; *str != '\0'; str++)
    {
        if (*str != '%' && *str != '$')
        {
            g_string_append_c(out, *str);
            continue;
        }

        if (*++str == '\0') break;
        if (str[-1] == '%')
        {
            if (!expand_styles(out, &str, server, channel, level))
            {
                g_string_append_c(out, '%');
                g_string_append_c(out, '%');
                g_string_append_c(out, *str);
            }
        }
        else
        {
            /* found a parameter! */
            gint num, argnum;
	    gboolean cut, rightpad;
	    gchar pad;

	    if (!isdigit(*str) && *str != '[')
	    {
		/* some buggy $ thing .. just print as is */
		g_string_append_c(out, *str);
		continue;
	    }

	    rightpad = cut = FALSE; pad = ' ';
            if (*str != '[')
		argnum = 0;
            else
	    {
		/* ! = don't cut, - = right padding */
		str++; cut = TRUE;
		while (*str != '\0' && *str != ']' && !isdigit(*str))
		{
		    if (*str == '!')
			cut = FALSE;
		    else if (*str == '-')
                        rightpad = TRUE;
                    str++;
		}
		if (!isdigit(*str)) break;

		/* get the pad size */
		argnum = 0;
		while (isdigit(*str))
		{
		    argnum = argnum*10 + *str-'0';
		    str++;
		}

		/* get the pad character */
		while (*str != '\0' && *str != ']')
		{
		    pad = *str;
		    str++;
		}

		if (*str != ']') break;
		str++;
	    }

	    /* get parameter number */
            num = *str-'1';
	    if (num < 0 || num >= format->params)
            {
                /* out of range */
                printtext(server, channel, MSGLEVEL_CLIENTERROR,
                          "Error in format '%s': tried to use parameter %d but last is %d.. Fix from setup.",
			  format->tag, num, format->params);
		continue;
            }

	    switch (format->paramtypes[num])
            {
		case FORMAT_STRING:
		    arg = g_strdup((gchar *) params[num]);
                    break;
		case FORMAT_INT:
		    arg = g_strdup_printf("%d", *((gint *) params[num]));
                    break;
                case FORMAT_LONG:
		    arg = g_strdup_printf("%ld", *((glong *) params[num]));
                    break;
		case FORMAT_FLOAT:
		    /* FIXME: floats don't work for some reason */
		    arg = g_strdup_printf("%0.2f", *((gfloat *) params[num]));
		    break;
		default:
		    arg = NULL;
		    break;
	    }

	    if (arg != NULL)
	    {
		gint len;

		/* cut */
		len = strlen(arg);
		if (cut && argnum > 0 && len > argnum)
		    arg[argnum] = '\0';

		if (!rightpad)
		{
		    /* left padding, add pad characters */
		    while (argnum > len)
		    {
			g_string_append_c(out, pad);
			argnum--;
		    }
		}

		/* add the text */
		g_string_append(out, arg);

		if (rightpad)
		{
		    /* right padding, add pad characters */
		    while (argnum > len)
		    {
			g_string_append_c(out, pad);
			argnum--;
		    }
		}
		g_free(arg);
	    }
        }
    }

    /* free params list */
    for (num = 0; num < format->params && num < 10; num++)
    {
        if (format->paramtypes[num] != FORMAT_STRING)
            g_free(params[num]);
    }
}

static void output_format_text(GString *out, SERVER_REC *server, gchar *channel, gint level, gint formatnum, ...)
{
    va_list args;

    va_start(args, formatnum);
    output_format_text_args(out, server, channel, level,
			    &default_formats[formatnum],
			    current_theme->format[formatnum], args);
    va_end(args);
}

static void add_timestamp(GString *out, SERVER_REC *server, gchar *channel, gint level)
{
    time_t t;
    struct tm *tm;
    GString *tmp;

    if (!(level != MSGLEVEL_NEVER && level != MSGLEVEL_CLIENTCRAP &&
	  (toggle_show_timestamps || (toggle_show_msgs_timestamps && (level & MSGLEVEL_MSGS) != 0))))
	return;

    tmp = g_string_new(NULL);
    t = time(NULL);
    tm = localtime(&t);

    output_format_text(tmp, server, channel, level, IRCTXT_TIMESTAMP,
                       tm->tm_year+1900, tm->tm_mon, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);

    /* insert the timestamp just after \n */
    g_string_prepend(out, tmp->str);
    g_string_free(tmp, TRUE);
}

static void new_line_stuff(GString *out, SERVER_REC *server, gchar *channel, gint level)
{
    if ((level & (MSGLEVEL_CLIENTERROR|MSGLEVEL_CLIENTNOTICE)) != 0)
        output_format_text(out, server, channel, level, IRCTXT_LINE_START_IRSSI);
    else if ((level & (MSGLEVEL_MSGS|MSGLEVEL_PUBLIC|MSGLEVEL_NOTICES|MSGLEVEL_SNOTES|MSGLEVEL_CTCPS|MSGLEVEL_ACTIONS|MSGLEVEL_DCC|MSGLEVEL_CLIENTCRAP)) == 0 && level != MSGLEVEL_NEVER)
        output_format_text(out, server, channel, level, IRCTXT_LINE_START);
}

/* Write text to channel - convert color codes */
void printtext(SERVER_REC *server, gchar *channel, gint level, gchar *str, ...)
{
    va_list args;
    GString *out;
    gchar *tmpstr;
    gint pros;

    g_return_if_fail(str != NULL);

    va_start(args, str);

    pros = 0;
    out = g_string_new(NULL);

    new_line_stuff(out, server, channel, level);
    for (; *str != '\0'; str++)
    {
        if (*str != '%')
        {
            g_string_append_c(out, *str);
            continue;
        }

        if (*++str == '\0') break;
        switch (*str)
        {
            /* standard parameters */
            case 's':
                {
                    gchar *s = (gchar *) va_arg(args, gchar *);
                    if (s) g_string_sprintfa(out, "%s", s);
                    break;
                }
            case 'd':
                {
                    gint d = (gint) va_arg(args, gint);
                    g_string_sprintfa(out, "%d", d);
                    break;
                }
            case 'f':
                {
                    gdouble f = (gdouble) va_arg(args, gdouble);
                    g_string_sprintfa(out, "%0.2f", f);
                    break;
                }
            case 'c':
                {
                    gchar c = (gchar )va_arg(args, gint);
                    g_string_append_c(out, c);
                    break;
                }
            case 'u':
                {
                    guint d = (guint) va_arg(args, guint);
                    g_string_sprintfa(out, "%u", d);
                    break;
                }
            case 'l':
                {
                    gulong d = (gulong) va_arg(args, gulong);
                    if (*++str != 'd' && *str != 'u')
                    {
                        g_string_sprintfa(out, "%ld", d);
                        str--;
                    }
                    else
                    {
                        if (*str == 'd')
                            g_string_sprintfa(out, "%ld", d);
                        else
                            g_string_sprintfa(out, "%lu", d);
                    }
                    break;
                }
            default:
                if (!expand_styles(out, &str, server, channel, level))
                {
                    g_string_append_c(out, '%');
                    g_string_append_c(out, *str);
                }
                break;
        }
    }
    va_end(args);

    /* send the plain text version for logging.. */
    tmpstr = strip_codes(out->str);
    signal_emit("print text stripped", 4, server, channel, GINT_TO_POINTER(level), tmpstr);
    g_free(tmpstr);

    signal_emit("print text", 4, server, channel, GINT_TO_POINTER(level), out->str);

    g_string_free(out, TRUE);
}

void printformat(SERVER_REC *server, gchar *channel, gint level, gint formatnum, ...)
{
    GString *out;
    va_list args;

    va_start(args, formatnum);
    out = g_string_new(NULL);

    output_format_text_args(out, server, channel, level,
			    &default_formats[formatnum],
			    current_theme->format[formatnum], args);
    printtext(server, channel, level, "%s", out->str);

    g_string_free(out, TRUE);
    va_end(args);
}

static void newline(CHANNEL_REC *channel)
{
    CHANNEL_PARENT(channel)->lines++;
    if (CHANNEL_PARENT(channel)->lines != 1)
    {
        signal_emit("gui print text", 6, channel,
                    GINT_TO_POINTER(-1), GINT_TO_POINTER(-1),
                    GINT_TO_POINTER(0), "\n", GINT_TO_POINTER(-1));
    }
}

static gboolean sig_print_text(SERVER_REC *server, gchar *channel, gpointer level, gchar *str)
{
    CHANNEL_REC *chanrec;
    WINDOW_REC *window;
    GString *out;
    gchar *dup, *ptr, type;
    gint fgcolor, bgcolor;
    gint flags;

    g_return_val_if_fail(str != NULL, FALSE);

    chanrec = channel_find_closest(server, channel, GPOINTER_TO_INT(level));

    window = CHANNEL_PARENT(chanrec);
    flags = 0; fgcolor = -1; bgcolor = -1; type = '\0';

    newline(chanrec);

    /* add timestamp */
    out = g_string_new(str);
    add_timestamp(out, server, channel, GPOINTER_TO_INT(level));
    dup = str = out->str;
    g_string_free(out, FALSE);

    while (*str != '\0')
    {
        for (ptr = str; *ptr != '\0'; ptr++)
            if (*ptr == 2 || *ptr == 3 || *ptr == 4 || *ptr == 7 || *ptr == 15 || *ptr == 22 || *ptr == 27 || *ptr == 31)
            {
                type = *ptr;
                *ptr++ = '\0';
                break;
            }

        if (type == 7)
        {
            /* bell */
            if (setup_get_bool("toggle_bell_beeps"))
                flags |= PRINTFLAG_BEEP;
        }
        if (*str != '\0' || flags & PRINTFLAG_BEEP)
        {
            signal_emit("gui print text", 6, chanrec,
                        GINT_TO_POINTER(fgcolor), GINT_TO_POINTER(bgcolor),
                        GINT_TO_POINTER(flags), str, level);
            flags &= ~PRINTFLAG_BEEP;
        }
        if (*ptr == '\0') break;

        switch (type)
        {
            case 2:
                /* bold */
                if (!toggle_hide_text_style)
                    flags ^= PRINTFLAG_BOLD;
                break;
	    case 15:
                /* remove all styling */
		flags &= PRINTFLAG_BEEP;
		fgcolor = bgcolor = -1;
		break;
	    case 22:
                /* reverse */
                if (!toggle_hide_text_style)
                    flags ^= PRINTFLAG_REVERSE;
                break;
            case 31:
                /* underline */
                if (!toggle_hide_text_style)
                    flags ^= PRINTFLAG_UNDERLINE;
            case 27:
                /* ansi color code */
                ptr = convert_ansi(ptr, &fgcolor, &bgcolor, &flags);
                break;
            case 4:
                /* user specific colors */
                if ((signed char) *ptr == -1)
                {
		    ptr++;
		    if ((signed char) *ptr == -1)
		    {
			fgcolor = bgcolor = -1;
			flags = 0;
		    }
                    else if (*ptr == 1)
                        flags ^= PRINTFLAG_BOLD;
                    else if (*ptr == 2)
                        flags ^= PRINTFLAG_UNDERLINE;
                    else if (*ptr == 3)
                        flags ^= PRINTFLAG_REVERSE;
                }
                else
		{
		    if ((signed char) *ptr != -2)
		    {
			fgcolor = (guchar) *ptr-1;
			if (fgcolor <= 7)
			    flags &= ~PRINTFLAG_BOLD;
			else
			{
			    /* bold */
			    if (fgcolor != 8) fgcolor -= 8;
			    flags |= PRINTFLAG_BOLD;
			}
		    }
                    ptr++;
		    if ((signed char) *ptr != -2)
			bgcolor = (signed char) *ptr == -1 ? -1 : *ptr-1;
                }
                ptr++;
                break;
	    case 3:
                if (*ptr < 17)
                {
		    /* mostly just for irssi's internal use.. */
		    fgcolor = (*ptr++)-1;
		    if (*ptr == 0 || *ptr >= 17)
			bgcolor = -1;
		    else
			bgcolor = (*ptr++)-1;
		    if (fgcolor & 8)
			flags |= PRINTFLAG_BOLD;
		    else
			flags &= ~PRINTFLAG_BOLD;
		    break;
		}

                /* MIRC color */
                if (toggle_hide_text_style)
                {
                    /* don't show them. */
                    if (isdigit((gint) *ptr))
                    {
                        ptr++;
                        if (isdigit((gint) *ptr)) ptr++;
                        if (*ptr == ',')
                        {
                            ptr++;
                            if (isdigit((gint) *ptr))
                            {
                                ptr++;
                                if (isdigit((gint) *ptr)) ptr++;
                            }
                        }
                    }
                    break;
                }

		flags |= PRINTFLAG_MIRC_COLOR;
		if (!isdigit((gint) *ptr) && *ptr != ',')
		{
		    fgcolor = -1;
		    bgcolor = -1;
		}
		else
		{
		    /* foreground color */
		    if (*ptr != ',')
		    {
			fgcolor = *ptr++-'0';
			if (isdigit((gint) *ptr))
			    fgcolor = fgcolor*10 + (*ptr++-'0');
		    }
		    if (*ptr == ',')
		    {
			/* back color */
			bgcolor = 0;
			if (!isdigit((gint) *++ptr))
			    bgcolor = -1;
			else
			{
			    bgcolor = *ptr++-'0';
			    if (isdigit((gint) *ptr))
				bgcolor = bgcolor*10 + (*ptr++-'0');
			}
		    }
		}
                break;
        }

        str = ptr;
    }
    g_free(dup);
    signal_emit("print text finished", 1, chanrec);

    return TRUE;
}

static gint sig_check_daychange(void)
{
    static gint lastday = -1;
    GList *tmp;
    time_t t;
    struct tm *tm;

    if (!toggle_show_timestamps)
    {
        /* display day change notice only when using timestamps */
	return TRUE;
    }

    t = time(NULL);
    tm = localtime(&t);

    if (lastday == -1)
    {
	/* First check, don't display. */
	lastday = tm->tm_mday;
	return TRUE;
    }

    if (tm->tm_mday == lastday)
	return TRUE;

    /* day changed, print notice about it to every window */
    for (tmp = g_list_first(windows); tmp != NULL; tmp = tmp->next)
    {
	WINDOW_REC *win = tmp->data;

	printformat(win->active->server, win->active->name, MSGLEVEL_NEVER,
		    IRCTXT_DAYCHANGE, tm->tm_mday, tm->tm_mon, 1900+tm->tm_year);
    }
    lastday = tm->tm_mday;
    return TRUE;
}

static gboolean sig_gui_dialog(gchar *type, gchar *text)
{
    if (g_strcasecmp(type, "warning") == 0)
	type = _("%_Warning:%_ %s");
    else if (g_strcasecmp(type, "error") == 0)
	type = _("%_Error:%_ %s");
    else
	type = "%s";

    printtext(NULL, NULL, MSGLEVEL_NEVER, type, text);
    return TRUE;
}

static gboolean sig_setup(void)
{
    toggle_show_timestamps = setup_get_bool("toggle_show_timestamps");
    toggle_show_msgs_timestamps = setup_get_bool("toggle_show_msgs_timestamps");
    toggle_hide_text_style = setup_get_bool("toggle_hide_text_style");
    return TRUE;
}

void ui_printtext_init(void)
{
    printtag = gui_timeout_add(30000, (GUITimeoutFunction) sig_check_daychange, NULL);
    signal_add("print text", (SIGNAL_FUNC) sig_print_text);
    signal_add("gui dialog", (SIGNAL_FUNC) sig_gui_dialog);
    signal_add("startup settings read", (SIGNAL_FUNC) sig_setup);
    signal_add("setup changed", (SIGNAL_FUNC) sig_setup);
}

void ui_printtext_deinit(void)
{
    gui_timeout_remove(printtag);
    signal_remove("print text", (SIGNAL_FUNC) sig_print_text);
    signal_remove("gui dialog", (SIGNAL_FUNC) sig_gui_dialog);
    signal_remove("startup settings read", (SIGNAL_FUNC) sig_setup);
    signal_remove("setup changed", (SIGNAL_FUNC) sig_setup);
}
