/*
 ui-dcc.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gboolean dcc_default_command(gchar *data)
{
    g_return_val_if_fail(data != NULL, FALSE);

    printformat(NULL, NULL, MSGLEVEL_CLIENTERROR, IRCTXT_DCC_UNKNOWN_COMMAND, data);
    return TRUE;
}

static gboolean dcc_connected(DCC_REC *dcc)
{
    gchar *str;

    g_return_val_if_fail(dcc != NULL, FALSE);

    switch (dcc->type)
    {
        case DCC_TYPE_CHAT:
            printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_CHAT_CONNECTED,
                        dcc->nick, dcc->addrstr, dcc->port);

            str = g_strconcat("=", dcc->nick, NULL);
            channel_create(dcc->server, str, CHANNEL_TYPE_DCC_CHAT);
            g_free(str);
            break;
        case DCC_TYPE_SEND:
            printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_SEND_CONNECTED,
                        dcc->arg, dcc->nick, dcc->addrstr, dcc->port);
            break;
        case DCC_TYPE_GET:
            printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_GET_CONNECTED,
                        dcc->arg, dcc->nick, dcc->addrstr, dcc->port);
            break;
    }

    return TRUE;
}

static gboolean dcc_rejected(DCC_REC *dcc)
{
    g_return_val_if_fail(dcc != NULL, FALSE);

    printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_CLOSE,
                dcc_type2str(dcc->type), dcc->nick, dcc->arg);
    return TRUE;
}

static gboolean dcc_closed(DCC_REC *dcc)
{
    time_t secs;
    glong kbs;

    g_return_val_if_fail(dcc != NULL, FALSE);

    secs = dcc->starttime == 0 ? -1 : time(NULL)-dcc->starttime;
    kbs = dcc->transfd / (secs == 0 ? 1 : secs) / 1024;

    switch (dcc->type)
    {
        case DCC_TYPE_CHAT:
            {
                /* nice kludge :) if connection was lost, close the channel.
                   after closed channel (can be done with /unquery too)
                   prints the disconnected-text.. */
                CHANNEL_REC *channel;
                gchar *str;

                str = g_strdup_printf("=%s", dcc->nick);
                printformat(dcc->server, str, MSGLEVEL_DCC,
                            IRCTXT_DCC_CHAT_DISCONNECTED, dcc->nick);

                channel = channel_find(dcc->server, str);
                if (channel != NULL)
                    channel_destroy(channel);
                g_free(str);
            }
            break;
        case DCC_TYPE_SEND:
            if (secs == -1)
            {
                /* aborted */
                printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_SEND_ABORTED,
                            dcc->arg, dcc->nick);
            }
            else
            {
                printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_SEND_COMPLETE,
                            dcc->arg, dcc->transfd/1024, dcc->nick, secs, kbs);
            }
            break;
        case DCC_TYPE_GET:
            if (secs == -1)
            {
                /* aborted */
                printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_GET_ABORTED,
                            dcc->arg, dcc->nick);
            }
            else
            {
                printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_GET_COMPLETE,
                            dcc->arg, dcc->transfd/1024, dcc->nick, secs, kbs);
            }
            break;
    }

    return TRUE;
}

static gboolean dcc_chat_in_action(gchar *msg, DCC_REC *dcc)
{
    gchar *sender;

    g_return_val_if_fail(dcc != NULL, FALSE);
    g_return_val_if_fail(msg != NULL, FALSE);

    sender = g_strconcat("=", dcc->nick, NULL);
    printformat(NULL, sender, MSGLEVEL_DCC,
                IRCTXT_ACTION_DCC, dcc->nick, msg);
    g_free(sender);
    return TRUE;
}

static gboolean dcc_chat_ctcp(gchar *msg, DCC_REC *dcc)
{
    gchar *sender;

    g_return_val_if_fail(dcc != NULL, FALSE);
    g_return_val_if_fail(msg != NULL, FALSE);

    sender = g_strconcat("=", dcc->nick, NULL);
    printformat(NULL, sender, MSGLEVEL_DCC, IRCTXT_DCC_CTCP, dcc->nick, msg);
    g_free(sender);
    return TRUE;
}

static gboolean dcc_chat_msg(DCC_REC *dcc, gchar *msg)
{
    gchar *nick;

    g_return_val_if_fail(dcc != NULL, FALSE);
    g_return_val_if_fail(msg != NULL, FALSE);

    nick = g_strconcat("=", dcc->nick, NULL);
    printformat(NULL, nick, MSGLEVEL_DCC, IRCTXT_DCC_MSG, dcc->nick, msg);
    g_free(nick);
    return TRUE;
}

static gboolean dcc_request(DCC_REC *dcc)
{
    g_return_val_if_fail(dcc != NULL, FALSE);

    switch (dcc->type)
    {
        case DCC_TYPE_CHAT:
            printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_CHAT,
                        dcc->nick, dcc->addrstr, dcc->port);
            break;
        case DCC_TYPE_GET:
            printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_SEND,
                        dcc->nick, dcc->addrstr, dcc->port, dcc->arg, dcc->size);
            break;
    }
    return TRUE;
}

static gboolean dcc_error_connect(DCC_REC *dcc)
{
    g_return_val_if_fail(dcc != NULL, FALSE);

    printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_CONNECT_ERROR, dcc->addrstr, dcc->port);
    return TRUE;
}

static gboolean dcc_error_file_create(DCC_REC *dcc, gchar *fname)
{
    g_return_val_if_fail(dcc != NULL, FALSE);

    printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_CANT_CREATE, fname);
    return TRUE;
}

static gboolean dcc_error_file_not_found(gchar *nick, gchar *fname)
{
    g_return_val_if_fail(nick != NULL, FALSE);
    g_return_val_if_fail(fname != NULL, FALSE);

    printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_SEND_FILE_NOT_FOUND, fname);
    return TRUE;
}

static gboolean dcc_error_get_not_found(gchar *nick)
{
    g_return_val_if_fail(nick != NULL, FALSE);

    printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_GET_NOT_FOUND, nick);
    return TRUE;
}

static gboolean dcc_error_send_exists(gchar *nick, gchar *fname)
{
    g_return_val_if_fail(nick != NULL, FALSE);
    g_return_val_if_fail(fname != NULL, FALSE);

    printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_SEND_EXISTS, fname, nick);
    return TRUE;
}

static gboolean dcc_error_unknown_type(gchar *type)
{
    g_return_val_if_fail(type != NULL, FALSE);

    printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_UNKNOWN_TYPE, type);
    return TRUE;
}

static gboolean dcc_error_close_not_found(gchar *type, gchar *nick, gchar *fname)
{
    g_return_val_if_fail(type != NULL, FALSE);
    g_return_val_if_fail(nick != NULL, FALSE);
    g_return_val_if_fail(fname != NULL, FALSE);

    if (fname == '\0') fname = "(ANY)";
    switch (dcc_str2type(type))
    {
        case DCC_TYPE_CHAT:
            printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_CHAT_NOT_FOUND, nick);
            break;
        case DCC_TYPE_SEND:
            printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_SEND_NOT_FOUND, nick, fname);
            break;
        case DCC_TYPE_GET:
            printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_GET_NOT_FOUND, nick, fname);
            break;
    }

    return TRUE;
}

static gboolean dcc_unknown_ctcp(gchar *data, gchar *sender)
{
    gchar *params, *type, *args;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &type, &args);
    printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_UNKNOWN_CTCP, type, sender, args);
    g_free(params);

    return TRUE;
}

static gboolean dcc_unknown_reply(gchar *data, gchar *sender)
{
    gchar *params, *type, *args;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &type, &args);
    printformat(NULL, NULL, MSGLEVEL_DCC, IRCTXT_DCC_UNKNOWN_REPLY, type, sender, args);
    g_free(params);

    return TRUE;
}

static gboolean dcc_chat_write(gchar *data)
{
    DCC_REC *dcc;
    gchar *params, *text, *target;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &target, &text);

    if (*target == '=')
    {
        /* dcc msg */
        dcc = dcc_find_item(DCC_TYPE_CHAT, target+1, NULL);
        if (dcc == NULL)
        {
            printformat(NULL, NULL, MSGLEVEL_CLIENTERROR,
                        IRCTXT_DCC_CHAT_NOT_FOUND, target+1);
            return TRUE;
        }

        printformat(NULL, target, MSGLEVEL_DCC, IRCTXT_OWN_DCC, target+1, text);
    }

    g_free(params);
    return TRUE;
}

static gboolean dcc_chat_out_action(gchar *data, SERVER_REC *server, CHANNEL_REC *cur_channel)
{
    DCC_REC *dcc;

    g_return_val_if_fail(data != NULL, FALSE);

    if (cur_channel->type != CHANNEL_TYPE_DCC_CHAT)
    {
        /* handle only DCC actions */
        return TRUE;
    }

    dcc = dcc_find_item(DCC_TYPE_CHAT, cur_channel->name+1, NULL);
    if (dcc == NULL)
    {
        printformat(NULL, NULL, MSGLEVEL_CLIENTERROR,
                    IRCTXT_DCC_CHAT_NOT_FOUND, cur_channel->name+1);
    }
    else
    {
        printformat(NULL, cur_channel->name, MSGLEVEL_DCC,
                    IRCTXT_OWN_ME, dcc->mynick, data);
    }
    return TRUE;
}

static gboolean cmd_dcc_list(gchar *data)
{
    GList *tmp;
    time_t going;

    g_return_val_if_fail(data != NULL, FALSE);

    printtext(NULL, NULL, MSGLEVEL_DCC, "%gDCC connections");
    for (tmp = g_list_first(dcc_conns); tmp != NULL; tmp = tmp->next)
    {
        DCC_REC *dcc = tmp->data;

        going = time(NULL) - dcc->starttime;
        if (going == 0) going = 1; /* no division by zeros :) */

        if (dcc->type == DCC_TYPE_CHAT)
            printtext(NULL, NULL, MSGLEVEL_DCC, "%g %s %s", dcc->nick, dcc_type2str(dcc->type));
        else
            printtext(NULL, NULL, MSGLEVEL_DCC, "%g %s %s: %luk of %luk (%d%%) - %fkB/s - %s",
                      dcc->nick, dcc_type2str(dcc->type), dcc->transfd/1024, dcc->size/1024,
                      dcc->size == 0 ? 0 : (100*dcc->transfd/dcc->size), (gdouble) dcc->transfd/going/1024, dcc->arg);
    }

    return TRUE;
}

static gboolean dcc_chat_closed(CHANNEL_REC *channel)
{
    DCC_REC *dcc;

    g_return_val_if_fail(channel != NULL, FALSE);

    if (channel->type != CHANNEL_TYPE_DCC_CHAT)
        return TRUE;

    /* check that we haven't got here from dcc_destroy() so we won't try to
       close the dcc again.. */
    dcc = dcc_find_item(DCC_TYPE_CHAT, channel->name+1, NULL);
    if (dcc != NULL && !dcc->destroyed)
    {
        /* DCC query window closed, close the dcc chat too. */
        dcc_destroy(dcc);
    }

    return TRUE;
}

void ui_dcc_init(void)
{
    signal_add("default command dcc", (SIGNAL_FUNC) dcc_default_command);

    signal_add("dcc connected", (SIGNAL_FUNC) dcc_connected);
    signal_add("dcc rejected", (SIGNAL_FUNC) dcc_rejected);
    signal_add("dcc closed", (SIGNAL_FUNC) dcc_closed);
    signal_add("dcc chat message", (SIGNAL_FUNC) dcc_chat_msg);
    signal_add("dcc ctcp action", (SIGNAL_FUNC) dcc_chat_in_action);
    signal_add("default dcc ctcp", (SIGNAL_FUNC) dcc_chat_ctcp);
    signal_add("dcc request", (SIGNAL_FUNC) dcc_request);
    signal_add("dcc error connect", (SIGNAL_FUNC) dcc_error_connect);
    signal_add("dcc error file create", (SIGNAL_FUNC) dcc_error_file_create);
    signal_add("dcc error file not found", (SIGNAL_FUNC) dcc_error_file_not_found);
    signal_add("dcc error get not found", (SIGNAL_FUNC) dcc_error_get_not_found);
    signal_add("dcc error send exists", (SIGNAL_FUNC) dcc_error_send_exists);
    signal_add("dcc error unknown type", (SIGNAL_FUNC) dcc_error_unknown_type);
    signal_add("dcc error close not found", (SIGNAL_FUNC) dcc_error_close_not_found);
    signal_add("dcc unknown ctcp", (SIGNAL_FUNC) dcc_unknown_ctcp);
    signal_add("dcc unknown reply", (SIGNAL_FUNC) dcc_unknown_reply);
    command_bind("msg", NULL, (SIGNAL_FUNC) dcc_chat_write);
    command_bind("me", NULL, (SIGNAL_FUNC) dcc_chat_out_action);
    command_bind("dcc ", NULL, (SIGNAL_FUNC) cmd_dcc_list);
    command_bind("dcc list", NULL, (SIGNAL_FUNC) cmd_dcc_list);
    signal_add("channel destroyed", (SIGNAL_FUNC) dcc_chat_closed);
}

void ui_dcc_deinit(void)
{
    signal_remove("default command dcc", (SIGNAL_FUNC) dcc_default_command);

    signal_remove("dcc connected", (SIGNAL_FUNC) dcc_connected);
    signal_remove("dcc rejected", (SIGNAL_FUNC) dcc_rejected);
    signal_remove("dcc closed", (SIGNAL_FUNC) dcc_closed);
    signal_remove("dcc chat message", (SIGNAL_FUNC) dcc_chat_msg);
    signal_remove("dcc ctcp action", (SIGNAL_FUNC) dcc_chat_in_action);
    signal_remove("default dcc ctcp", (SIGNAL_FUNC) dcc_chat_ctcp);
    signal_remove("dcc request", (SIGNAL_FUNC) dcc_request);
    signal_remove("dcc error connect", (SIGNAL_FUNC) dcc_error_connect);
    signal_remove("dcc error file create", (SIGNAL_FUNC) dcc_error_file_create);
    signal_remove("dcc error file not found", (SIGNAL_FUNC) dcc_error_file_not_found);
    signal_remove("dcc error get not found", (SIGNAL_FUNC) dcc_error_get_not_found);
    signal_remove("dcc error send exists", (SIGNAL_FUNC) dcc_error_send_exists);
    signal_remove("dcc error unknown type", (SIGNAL_FUNC) dcc_error_unknown_type);
    signal_remove("dcc error close not found", (SIGNAL_FUNC) dcc_error_close_not_found);
    signal_remove("dcc unknown ctcp", (SIGNAL_FUNC) dcc_unknown_ctcp);
    signal_remove("dcc unknown reply", (SIGNAL_FUNC) dcc_unknown_reply);
    command_unbind("msg", (SIGNAL_FUNC) dcc_chat_write);
    command_unbind("me", (SIGNAL_FUNC) dcc_chat_out_action);
    command_unbind("dcc ", (SIGNAL_FUNC) cmd_dcc_list);
    command_unbind("dcc list", (SIGNAL_FUNC) cmd_dcc_list);
    signal_remove("channel destroyed", (SIGNAL_FUNC) dcc_chat_closed);
}
