/*
 server-reconnect.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

typedef struct
{
    gchar *address;
    gchar *password;
    gint port;

    gchar *ircnet;
    gchar *nick;

    time_t next_connect;
}
RECONNECT_REC;

static GList *reconnects;
static gint reconnect_timeout;

static void server_reconnect_add(gchar *address, gint port, gchar *ircnet, gchar *password, gchar *nick, time_t next_connect)
{
    RECONNECT_REC *rec;

    rec = g_new(RECONNECT_REC, 1);
    rec->address = g_strdup(address);
    rec->password = password == NULL ? NULL : g_strdup(password);
    rec->port = port;
    rec->ircnet = ircnet == NULL ? NULL : g_strdup(ircnet);
    rec->nick = nick == NULL ? NULL : g_strdup(nick);
    rec->next_connect = next_connect;

    reconnects = g_list_append(reconnects, rec);
}

static void server_reconnect_destroy(RECONNECT_REC *rec)
{
    if (rec->password != NULL) g_free(rec->password);
    g_free(rec->ircnet);
    g_free(rec->address);
    if (rec->nick != NULL) g_free(rec->nick);
    g_free(rec);

    reconnects = g_list_remove(reconnects, rec);
}

static gint server_reconnect_timeout(void)
{
    GList *tmp, *next;
    time_t now;

    now = time(NULL);
    for (tmp = reconnects; tmp != NULL; tmp = next)
    {
        RECONNECT_REC *rec = tmp->data;

        next = tmp->next;
        if (rec->next_connect <= now)
        {
	    server_connect(rec->address, rec->port, rec->password, rec->nick);
            server_reconnect_destroy(rec);
        }
    }

    return 1;
}

static gboolean sig_reconnect(SERVER_REC *server)
{
    SETUP_SERVER_REC *sserver;
    gboolean found, through;
    GList *tmp;

    g_return_val_if_fail(server != NULL, FALSE);

    if (setup_get_int("server_reconnect_time") == -1 || !server->connection_lost)
        return TRUE;

    sserver = server_setup_find(server->address, server->port);
    if (sserver == NULL)
    {
	/* port specific record not found, try without port.. */
	sserver = server_setup_find(server->address, -1);
    }

    if (sserver != NULL)
    {
	/* don't try to connect back immediately after connect failed */
	sserver->last_connect = server->connect_time == 0 ?
	    time(NULL) : server->connect_time;
    }

    if (sserver == NULL || server->ircnet == NULL)
    {
	/* not found from setup / no ircnet, just try to reconnect
	   back to same server */
	if (server->connect_time != 0 && time(NULL)-server->connect_time > setup_get_int("server_reconnect_time"))
	{
	    /* there's been enough time since last connection, try to
	       reconnect back immediately */
	    server_connect(server->address, server->port, server->password, server->wanted_nick);
	}
	else
	{
	    /* reconnect later.. */
	    server_reconnect_add(server->address, server->port, NULL, server->password, server->wanted_nick,
				 (server->connect_time == 0 ? time(NULL) : server->connect_time) + setup_get_int("server_reconnect_time"));
	}
	return TRUE;
    }

    /* got disconnected, reconnect. */
    found = FALSE; through = FALSE;
    for (tmp = setupservers; tmp != NULL; )
    {
        SETUP_SERVER_REC *rec = tmp->data;

        if (!found && g_strcasecmp(rec->server, server->address) == 0 && server->port == rec->port)
            found = TRUE;
        else if (found && g_strcasecmp(server->ircnet, rec->ircnet) == 0)
        {
            /* got one server. */
            if (rec->last_connect > time(NULL)-setup_get_int("server_reconnect_time"))
            {
                /* can't reconnect this fast, wait.. */
		server_reconnect_add(rec->server, rec->port, rec->ircnet, rec->password, server->wanted_nick,
				     rec->last_connect+setup_get_int("server_reconnect_time"));
            }
            else
	    {
		/* connect to server.. */
		server_connect(rec->server, rec->port, rec->password, server->wanted_nick);
            }
            return TRUE;
        }

        if (tmp->next != NULL)
            tmp = tmp->next;
        else
        {
            if (through)
            {
                /* shouldn't happen unless there's no servers in this ircnet
                   in setup.. */
                return TRUE;
            }
            tmp = setupservers;
            if (!found)
            {
                /* shouldn't happen unless server is removed from setup.. */
                found = TRUE;
            }
            through = TRUE;
        }
    }

    return TRUE;
}

static gboolean sig_connected(SERVER_REC *server)
{
    GList *tmp, *next;

    g_return_val_if_fail(server != NULL, FALSE);

    if (server->ircnet == NULL)
        return TRUE;

    /* connected to somewhere, check if there's anything in reconnect queue
       waiting to connect to same ircnet.. */
    for (tmp = g_list_first(reconnects); tmp != NULL; tmp = next)
    {
        RECONNECT_REC *rec = tmp->data;

        next = tmp->next;
        if (g_strcasecmp(server->ircnet, rec->ircnet) == 0)
            server_reconnect_destroy(rec);
    }

    return TRUE;
}

/* Remove all servers from reconnect list */
static gboolean cmd_rmreconns(gchar *data)
{
    while (reconnects != NULL)
        server_reconnect_destroy(reconnects->data);

    return TRUE;
}

void servers_reconnect_init(void)
{
    reconnects = NULL;

    reconnect_timeout = gui_timeout_add(1000, (GUITimeoutFunction) server_reconnect_timeout, NULL);

    signal_add("server connect failed", (SIGNAL_FUNC) sig_reconnect);
    signal_add("server disconnected", (SIGNAL_FUNC) sig_reconnect);
    signal_add("server connected", (SIGNAL_FUNC) sig_connected);
    command_bind("rmreconns", NULL, (SIGNAL_FUNC) cmd_rmreconns);
}

void servers_reconnect_deinit(void)
{
    gui_timeout_remove(reconnect_timeout);

    while (reconnects != NULL)
        server_reconnect_destroy(reconnects->data);

    signal_remove("server connect failed", (SIGNAL_FUNC) sig_reconnect);
    signal_remove("server disconnected", (SIGNAL_FUNC) sig_reconnect);
    signal_remove("server connected", (SIGNAL_FUNC) sig_connected);
    command_unbind("rmreconns", (SIGNAL_FUNC) cmd_rmreconns);
}
