/*
 nicklist.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gint massjoin_tag;

/* nick record comparision for sort functions */
gint nicklist_compare(NICK_REC *p1, NICK_REC *p2)
{
    if (p1 == NULL) return -1;
    if (p2 == NULL) return 1;

    if (p1->op && !p2->op) return -1;
    if (!p1->op && p2->op) return 1;

    if (p1->voice && !p2->voice) return -1;
    if (!p1->voice && p2->voice) return 1;

    return g_strcasecmp(p1->nick, p2->nick);
}

/* Add new nick to list */
NICK_REC *nicklist_insert(CHANNEL_REC *channel, gchar *nick, gboolean op, gboolean voice, gboolean send_massjoin)
{
    NICK_REC *rec;

    g_return_val_if_fail(channel != NULL, NULL);
    g_return_val_if_fail(nick != NULL, NULL);

    rec = g_new0(NICK_REC, 1);

    if (op) rec->op = TRUE;
    if (voice) rec->voice = TRUE;

    rec->send_massjoin = send_massjoin;
    rec->nick = g_strdup(nick);
    rec->host = NULL;

    g_hash_table_insert(channel->nicks, rec->nick, rec);
    signal_emit("nicklist new", 2, channel, rec);
    return rec;
}

static void nicklist_destroy(CHANNEL_REC *channel, NICK_REC *nick)
{
    signal_emit("nicklist remove", 2, channel, nick);

    g_free(nick->nick);
    if (nick->realname != NULL) g_free(nick->realname);
    if (nick->host != NULL) g_free(nick->host);
    g_free(nick);
}

/* remove nick from list */
void nicklist_remove(CHANNEL_REC *channel, NICK_REC *nick)
{
    g_return_if_fail(channel != NULL);
    g_return_if_fail(nick != NULL);

    g_hash_table_remove(channel->nicks, nick->nick);
    nicklist_destroy(channel, nick);
}

static void nicklist_remove_hash(gpointer key, NICK_REC *nick, CHANNEL_REC *channel)
{
    nicklist_destroy(channel, nick);
}

/* Find nick record from list */
NICK_REC *nicklist_find(CHANNEL_REC *channel, gchar *mask)
{
    NICK_REC *nick;
    GList *nicks, *tmp;
    gchar *ptr;

    g_return_val_if_fail(channel != NULL, NULL);
    g_return_val_if_fail(mask != NULL, NULL);

    if (strchr(mask, '*') != NULL || strchr(mask, '?') != NULL)
    {
	/* wildcards used, scan through every nick in nicklist */
	nicks = nicklist_getnicks(channel);
	for (tmp = nicks; tmp != NULL; tmp = tmp->next)
	{
	    NICK_REC *rec = tmp->data;

	    if (irc_mask_match_address(mask, rec->nick, rec->host == NULL ? "" : rec->host))
		return rec;
	}

	return NULL;
    }

    /* no wildcards, first find the nick from hash table */
    mask = g_strdup(mask);
    ptr = strchr(mask, '!');
    if (ptr != NULL) *ptr++ = '\0';

    nick = g_hash_table_lookup(channel->nicks, mask);

    /* and check that host masks */
    if (nick != NULL && ptr != NULL && !match_wildcards(ptr, nick->host == NULL ? "" : nick->host))
    {
	/* hosts didn't match */
	nick = NULL;
    }
    g_free(mask);
    return nick;
}

static void get_nicks_hash(gpointer key, NICK_REC *rec, GList **list)
{
    *list = g_list_append(*list, rec);
}

/* Get list of nicks */
GList *nicklist_getnicks(CHANNEL_REC *channel)
{
    GList *list;

    list = NULL;
    g_hash_table_foreach(channel->nicks, (GHFunc) get_nicks_hash, &list);
    return list;
}

gchar *nick_strip(gchar *nick)
{
    gchar *stripped, *spos;

    g_return_val_if_fail(nick != NULL, NULL);

    spos = stripped = g_strdup(nick);
    while (*nick != '\0' && *nick != ' ' && *nick != ':' && *nick != ',')
    {
        if (isalnum((gint) *nick) || (guchar) *nick >= 128) *spos++ = *nick;
        nick++;
    }
    *spos = '\0';
    return stripped;
}

static gboolean event_names_list(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *type, *channel, *names, *ptr;

    g_return_val_if_fail(data != NULL, FALSE);

    /* type = '=' = public, '*' = private, '@' = secret */
    params = event_get_params(data, 4, NULL, &type, &channel, &names);

    chanrec = channel_find(server, channel);
    if (chanrec == NULL || chanrec->names_got)
    {
	/* unknown channel / names list already read */
        g_free(params);
        return TRUE;
    }
    server->names_coming = TRUE;

    while (*names != '\0')
    {
        while (*names == ' ') names++;
        ptr = names;
        while (*names != '\0' && *names != ' ') names++;
        if (*names != '\0') *names++ = '\0';

        if (*ptr == '@' && strcmp(server->nick, ptr+ismodeflag(*ptr)) == 0)
            chanrec->chanop = TRUE;

        nicklist_insert(chanrec, ptr+ismodeflag(*ptr), *ptr == '@', *ptr == '+', FALSE);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_end_of_names(gchar *data, SERVER_REC *server)
{
    gchar *params, *channel;
    CHANNEL_REC *chanrec;

    g_return_val_if_fail(server != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &channel);
    server->names_coming = FALSE;

    chanrec = channel_find(server, channel);
    if (chanrec != NULL && !chanrec->names_got)
    {
	/* send the query signal.. */
	chanrec->names_got = TRUE;
	signal_emit("channel query", 1, chanrec);
    }

    g_free(params);
    return TRUE;
}

static void nicklist_update_flags(SERVER_REC *server, gchar *nick, gint gone, gint ircop)
{
    NICK_REC *rec;
    GList *tmp;

    g_return_if_fail(server != NULL);
    g_return_if_fail(nick != NULL);

    /* update the gone/ircop-status for nick in every channel .. */
    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *chanrec = tmp->data;

        if (chanrec->server == server)
        {
            rec = nicklist_find(chanrec, nick);
            if (rec != NULL)
            {
                rec->last_check = time(NULL);
                if (gone != -1 && rec->gone != gone)
                {
                    rec->gone = gone;
                    signal_emit("nick gone changed", 2, chanrec, rec);
		}
		if (ircop != -1 && rec->ircop != ircop)
		{
		    rec->ircop = ircop;
                    signal_emit("nick ircop changed", 2, chanrec, rec);
		}
            }
        }
    }
}

static gboolean event_who(gchar *data, SERVER_REC *server)
{
    gchar *params, *nick, *channel, *user, *host, *stat, *realname;
    CHANNEL_REC *chanrec;
    NICK_REC *rec;
    gboolean gone, ircop;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 8, NULL, &channel, &user, &host, NULL, &nick, &stat, &realname);
    /* skip hop count */
    while (*realname != '\0' && *realname != ' ') realname++;
    while (*realname == ' ') realname++;

    gone = strchr(stat, 'G') != NULL;
    ircop = strchr(stat, '*') != NULL;

    chanrec = channel_find(server, channel);
    if (chanrec != NULL)
    {
	rec = nicklist_find(chanrec, nick);
	if (rec != NULL)
	{
	    /* check if host/realname isn't set yet */
	    if (rec->host == NULL) rec->host = g_strdup_printf("%s@%s", user, host);
	    if (rec->realname == NULL) rec->realname = g_strdup(realname);
	}
    }

    nicklist_update_flags(server, nick, gone, ircop);

    g_free(params);
    return TRUE;
}

static gboolean event_whois(gchar *data, SERVER_REC *server)
{
    gchar *params, *nick, *realname;
    CHANNEL_REC *chanrec;
    NICK_REC *nickrec;
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);

    server->whois_coming = TRUE;

    /* first remove the gone-flag, if user is gone it will be set later.. */
    params = event_get_params(data, 6, NULL, &nick, NULL, NULL, NULL, &realname);

    /* Update realname - scan through all channels the nick is in */
    for (tmp = channels; tmp != NULL; tmp = tmp->next)
    {
	chanrec = tmp->data;

	if (chanrec->server != server)
	    continue; /* different server */

	nickrec = nicklist_find(chanrec, nick);
	if (nickrec != NULL && nickrec->realname == NULL)
	    nickrec->realname = g_strdup(realname);
    }

    nicklist_update_flags(server, nick, FALSE, FALSE);
    g_free(params);
    return TRUE;
}

static gboolean event_whois_away(gchar *data, SERVER_REC *server)
{
    gchar *params, *nick, *awaymsg;

    g_return_val_if_fail(data != NULL, FALSE);

    /* set user's gone flag.. */
    params = event_get_params(data, 3, NULL, &nick, &awaymsg);
    nicklist_update_flags(server, nick, TRUE, -1);
    g_free(params);
    return TRUE;
}

static gboolean event_whois_ircop(gchar *data, SERVER_REC *server)
{
    gchar *params, *nick, *awaymsg;

    g_return_val_if_fail(data != NULL, FALSE);

    /* set user's gone flag.. */
    params = event_get_params(data, 3, NULL, &nick, &awaymsg);
    nicklist_update_flags(server, nick, -1, TRUE);
    g_free(params);
    return TRUE;
}

static gboolean event_end_of_whois(gchar *data, SERVER_REC *server)
{
    server->whois_coming = FALSE;
    return TRUE;
}

static gboolean event_nick_in_use(gchar *data, SERVER_REC *server)
{
    gchar *str;
    gint n;

    g_return_val_if_fail(data != NULL, FALSE);

    if (server->connected)
    {
        /* Already connected, no need to handle this anymore. */
        return TRUE;
    }

    /* nick already in use - need to change it .. */
    if (strcmp(server->nick, setup_get_str("default_nick")) == 0)
    {
        /* first try, so try the alternative nick.. */
        g_free(server->nick);
        server->nick = g_strdup(setup_get_str("alternate_nick"));
    }
    else if (strlen(server->nick) < 9)
    {
        /* keep adding '_' to end of nick.. */
        str = g_strdup_printf("%s_", server->nick);
        g_free(server->nick);
        server->nick = str;
    }
    else
    {
        for (n = 8; n > 0; n--)
        {
            if (server->nick[n] < '0' || server->nick[n] > '9')
            {
                server->nick[n] = '1';
                break;
            }
            else
            {
                if (server->nick[n] < '9')
                {
                    server->nick[n]++;
                    break;
                }
                else
                    server->nick[n] = '0';
            }
        }
    }

    str = g_strdup_printf("NICK %s", server->nick);
    irc_send_cmd(server, str);
    g_free(str);

    return TRUE;
}

static gboolean event_target_unavailable(gchar *data, SERVER_REC *server)
{
    gchar *params, *channel;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &channel);
    if (!ischannel(*channel))
    {
        /* nick is unavailable. */
        event_nick_in_use(data, server);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_nick(gchar *data, SERVER_REC *server, gchar *orignick)
{
    gchar *params, *nick;
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 1, &nick);

    if (g_strcasecmp(orignick, server->nick) == 0)
    {
        /* You changed your nick */
        g_free(server->wanted_nick);
        g_free(server->nick);
        server->wanted_nick = g_strdup(nick);
        server->nick = g_strdup(nick);
        signal_emit("server nick changed", 1, server);
    }

    /* Someone else changed nick */
    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *chanrec = tmp->data;

        if (chanrec->server != server) continue;

        if (chanrec->type == CHANNEL_TYPE_CHANNEL)
        {
            /* channel */
            NICK_REC *nickrec;

            nickrec = nicklist_find(chanrec, orignick);
            if (nickrec != NULL)
	    {
		/* remove old nick from hash table */
		g_hash_table_remove(chanrec->nicks, nickrec->nick);

		g_free(nickrec->nick);
		nickrec->nick = g_strdup(nick);

		/* add new nick to hash table */
		g_hash_table_insert(chanrec->nicks, nickrec->nick, nickrec);

		signal_emit("nicklist changed", 3, chanrec, nickrec, orignick);
            }
        }
        else if (chanrec->type == CHANNEL_TYPE_QUERY)
        {
            /* query */
            if (g_strcasecmp(chanrec->name, orignick) == 0)
            {
                /* change query name */
                channel_change_name(chanrec, nick);
            }
        }
    }

    g_free(params);
    return TRUE;
}

static gboolean event_userhost(gchar *data, SERVER_REC *server)
{
    gchar *params, *hosts;
    GList *list, *tmp;
    gchar *ptr;

    g_return_val_if_fail(data != NULL, FALSE);

    /* set user's gone flag.. */
    params = event_get_params(data, 2, NULL, &hosts);

    list = str2list(hosts, ' ');
    for (tmp = g_list_first(list); tmp != NULL; tmp = tmp->next)
    {
        ptr = strchr((gchar *) tmp->data, '=');
        if (ptr == NULL) continue;
        *ptr++ = '\0';

        nicklist_update_flags(server, (gchar *) tmp->data, *ptr == '-', -1);
    }

    if (list != NULL)
    {
	g_free(list->data);
	g_list_free(list);
    }

    g_free(params);
    return TRUE;
}

static gboolean sig_usermode(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    nicklist_update_flags(server, server->nick, server->usermode_away, -1);
    return TRUE;
}

static gboolean nicklist_channel_created(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    channel->nicks = g_hash_table_new((GHashFunc) g_istr_hash, (GCompareFunc) g_istr_equal);
    return TRUE;
}

static gboolean nicklist_channel_destroyed(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    g_hash_table_foreach(channel->nicks, (GHFunc) nicklist_remove_hash, channel);
    g_hash_table_destroy(channel->nicks);
    return TRUE;
}

/* Massjoin support - really useful when trying to do things (op) people after
   netjoins. It sends "massjoin #channel nick!user@host nick2!user@host ..."
   signals */
static gboolean event_join(gchar *data, SERVER_REC *server, gchar *nick, gchar *address)
{
    gchar *params, *channel, *ptr;
    CHANNEL_REC *chanrec;
    NICK_REC *nickrec, *nickrec2;
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);

    if (g_strcasecmp(nick, server->nick) == 0)
    {
	/* You joined, no need to do anything */
	return TRUE;
    }

    params = event_get_params(data, 1, &channel);
    ptr = strchr(channel, 7); /* ^G does something weird.. */
    if (ptr != NULL) *ptr = '\0';

    /* find channel */
    chanrec = channel_find(server, channel);
    g_free(params);
    if (chanrec == NULL) return TRUE;

    /* add user to nicklist */
    nickrec = nicklist_insert(chanrec, nick, FALSE, FALSE, TRUE);
    nickrec->host = g_strdup(address);

    if (chanrec->massjoins == 0)
    {
	/* no nicks waiting in massjoin queue */
	chanrec->massjoin_start = time(NULL);
	chanrec->last_massjoins = 0;
    }

    /* Check if user is already in some other channel,
       get the realname from there */
    for (tmp = channels; tmp != NULL; tmp = tmp->next)
    {
	CHANNEL_REC *rec = tmp->data;

	if (rec->server != server || rec == chanrec)
	    continue;

	nickrec2 = nicklist_find(rec, nick);
	if (nickrec2 != NULL && nickrec2->realname != NULL)
	{
	    nickrec->last_check = nickrec2->last_check;
            nickrec->realname = g_strdup(nickrec2->realname);
	    nickrec->gone = nickrec2->gone;
	    break;
	}
    }

    chanrec->massjoins++;
    return TRUE;
}

static void massjoin_send_hash(gpointer key, NICK_REC *nick, GList **list)
{
    if (nick->send_massjoin)
    {
	nick->send_massjoin = FALSE;
	*list = g_list_append(*list, nick);
    }
}

/* Send channel's massjoin list signal */
static void massjoin_send(CHANNEL_REC *channel)
{
    GList *list;

    list = NULL;
    g_hash_table_foreach(channel->nicks, (GHFunc) massjoin_send_hash, &list);

    channel->massjoins = 0;
    signal_emit("massjoin", 2, channel, list);
    g_list_free(list);
}

static gboolean event_part(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    gchar *params, *channel, *reason;
    CHANNEL_REC *chanrec;
    NICK_REC *nickrec;

    g_return_val_if_fail(data != NULL, FALSE);

    if (g_strcasecmp(nick, server->nick) == 0)
    {
	/* you left channel, no need to do anything */
	return TRUE;
    }

    params = event_get_params(data, 2, &channel, &reason);

    /* find channel */
    chanrec = channel_find(server, channel);
    if (chanrec == NULL)
    {
	g_free(params);
	return TRUE;
    }

    /* remove user from nicklist */
    nickrec = nicklist_find(chanrec, nick);
    if (nickrec != NULL)
    {
	if (nickrec->send_massjoin)
	{
	    /* quick join/part after which it's useless to send nick in
	       massjoin */
	    chanrec->massjoins--;
	}
	nicklist_remove(chanrec, nickrec);
    }
    g_free(params);
    return TRUE;
}

static gboolean event_quit(gchar *data, SERVER_REC *server, gchar *nick)
{
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);

    if (g_strcasecmp(nick, server->nick) == 0)
    {
	/* you quit, don't do anything */
	return TRUE;
    }

    /* Remove user from all channels */
    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *channel = tmp->data;
        NICK_REC *rec;

	if (channel->server == server && channel->type == CHANNEL_TYPE_CHANNEL)
	{
	    rec = nicklist_find(channel, nick);
	    if (rec != NULL)
	    {
		if (rec->send_massjoin)
		{
		    /* quick join/quit after which it's useless to send nick
		       in massjoin */
		    channel->massjoins--;
		}
		nicklist_remove(channel, rec);
	    }
	}
    }

    return TRUE;
}

static gboolean event_kick(gchar *data, SERVER_REC *server)
{
    gchar *params, *channel, *nick, *reason;
    CHANNEL_REC *chanrec;
    NICK_REC *nickrec;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 3, &channel, &nick, &reason);

    if (g_strcasecmp(nick, server->nick) == 0)
    {
	/* you were kicked, no need to do anything */
        g_free(params);
	return TRUE;
    }

    /* Remove user from nicklist */
    chanrec = channel_find(server, channel);
    if (chanrec != NULL)
    {
	nickrec = nicklist_find(chanrec, nick);
	if (nickrec != NULL)
	{
	    if (nickrec->send_massjoin)
	    {
		/* quick join/kick after which it's useless to send nick in
		   massjoin */
		chanrec->massjoins--;
	    }
	    nicklist_remove(chanrec, nickrec);
	}
    }

    g_free(params);
    return TRUE;
}

static gint sig_massjoin_timeout(void)
{
    GList *tmp;
    time_t t;

    t = time(NULL)-MAX_MASSJOIN_WAIT;

    /* Scan all channels through for massjoins */
    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
	CHANNEL_REC *rec = tmp->data;

	if (rec->massjoins > 0)
	{
	    if (rec->massjoin_start < t || /* We've waited long enough */
		rec->massjoins-5 < rec->last_massjoins) /* Less than 5 joins since last check */
	    {
		/* send the first block */
		massjoin_send(rec);
	    }
	    else
	    {
		/* Wait for some more.. */
		rec->last_massjoins = rec->massjoins;
	    }
	}
    }
    return TRUE;
}

void nicklist_init(void)
{
    massjoin_tag = gui_timeout_add(1000, (GUITimeoutFunction) sig_massjoin_timeout, NULL);

    signal_add("event nick", (SIGNAL_FUNC) event_nick);
    signal_add_first("event 352", (SIGNAL_FUNC) event_who);
    signal_add("silent event who", (SIGNAL_FUNC) event_who);
    signal_add("silent event whois", (SIGNAL_FUNC) event_whois);
    signal_add_first("event 311", (SIGNAL_FUNC) event_whois);
    signal_add_first("event 301", (SIGNAL_FUNC) event_whois_away);
    signal_add_first("event 313", (SIGNAL_FUNC) event_whois_ircop);
    signal_add("event 318", (SIGNAL_FUNC) event_end_of_whois);
    signal_add("event 353", (SIGNAL_FUNC) event_names_list);
    signal_add("event 366", (SIGNAL_FUNC) event_end_of_names);
    signal_add("event 433", (SIGNAL_FUNC) event_nick_in_use);
    signal_add("event 437", (SIGNAL_FUNC) event_target_unavailable);
    signal_add("event 302", (SIGNAL_FUNC) event_userhost);
    signal_add("userhost event", (SIGNAL_FUNC) event_userhost);
    signal_add("user mode changed", (SIGNAL_FUNC) sig_usermode);
    signal_add_first("channel created", (SIGNAL_FUNC) nicklist_channel_created);
    signal_add("channel destroyed", (SIGNAL_FUNC) nicklist_channel_destroyed);
    signal_add("event join", (SIGNAL_FUNC) event_join);
    signal_add("event part", (SIGNAL_FUNC) event_part);
    signal_add("event kick", (SIGNAL_FUNC) event_kick);
    signal_add("event quit", (SIGNAL_FUNC) event_quit);
}

void nicklist_deinit(void)
{
    gui_timeout_remove(massjoin_tag);
    signal_remove("event nick", (SIGNAL_FUNC) event_nick);
    signal_remove("event 352", (SIGNAL_FUNC) event_who);
    signal_remove("silent event who", (SIGNAL_FUNC) event_who);
    signal_remove("silent event whois", (SIGNAL_FUNC) event_whois);
    signal_remove("event 311", (SIGNAL_FUNC) event_whois);
    signal_remove("event 301", (SIGNAL_FUNC) event_whois_away);
    signal_remove("event 313", (SIGNAL_FUNC) event_whois_ircop);
    signal_remove("event 318", (SIGNAL_FUNC) event_end_of_whois);
    signal_remove("event 353", (SIGNAL_FUNC) event_names_list);
    signal_remove("event 366", (SIGNAL_FUNC) event_end_of_names);
    signal_remove("event 433", (SIGNAL_FUNC) event_nick_in_use);
    signal_remove("event 437", (SIGNAL_FUNC) event_target_unavailable);
    signal_remove("event 302", (SIGNAL_FUNC) event_userhost);
    signal_remove("userhost event", (SIGNAL_FUNC) event_userhost);
    signal_remove("user mode changed", (SIGNAL_FUNC) sig_usermode);
    signal_remove("channel created", (SIGNAL_FUNC) nicklist_channel_created);
    signal_remove("channel destroyed", (SIGNAL_FUNC) nicklist_channel_destroyed);
    signal_remove("event join", (SIGNAL_FUNC) event_join);
    signal_remove("event part", (SIGNAL_FUNC) event_part);
    signal_remove("event kick", (SIGNAL_FUNC) event_kick);
    signal_remove("event quit", (SIGNAL_FUNC) event_quit);
}
