/*
 irc.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

gboolean irc_mask_match(gchar *mask, gchar *nick, gchar *user, gchar *host)
{
    gboolean ret;
    gchar *p, *str;

    p = strchr(mask, '!');
    if (p == NULL)
        return g_strcasecmp(mask, nick) == 0;

    str = g_strdup_printf("%s!%s@%s", nick, user, host);
    ret = match_wildcards(mask, str);
    g_free(str);

    return ret;
}

gboolean irc_mask_match_address(gchar *mask, gchar *nick, gchar *address)
{
    gboolean ret;
    gchar *p, *str;

    g_return_val_if_fail(address != NULL, FALSE);

    p = strchr(mask, '!');
    if (p == NULL)
        return g_strcasecmp(mask, nick) == 0;

    str = g_strdup_printf("%s!%s", nick, address);
    ret = match_wildcards(mask, str);
    g_free(str);

    return ret;
}

gboolean irc_masks_match(gchar *masks, gchar *nick, gchar *address)
{
    GList *list, *tmp;

    list = str2list(masks, ' ');
    for (tmp = list; tmp != NULL; tmp = tmp->next)
    {
	if (irc_mask_match_address(tmp->data, nick, address))
	    return TRUE;
    }
    if (list != NULL)
    {
	g_free(list->data);
	g_list_free(list);
    }

    return FALSE;
}

gchar *irc_get_mask(gchar *nick, gchar *host, gint flags)
{
    gchar *str, *addr, *user, *mhost, *ptr;

    /* strip -, ^ or ~ from start.. */
    addr = g_strconcat("*", *host == '^' || *host == '-' || *host == '~' ? host+1 : host, NULL);

    mhost = strchr(addr, '@');
    if (mhost == NULL)
    {
        /* what??? */
        g_free(addr);
        return NULL;
    }
    *mhost++ = '\0';

    /* get host */
    if ((flags & IRC_MASK_HOST) == 0 && (flags & IRC_MASK_DOMAIN))
    {
	if (strchr(mhost, '.') == NULL)
	{
	    /* no dots, toplevel domain (??) or IPv6 address */
	    mhost = g_strdup(mhost);
	    ptr = strrchr(mhost, ':');
	    if (ptr != NULL)
	    {
		/* IPv6 address, ban the last 64k addresses */
                if (ptr[1] != '\0') strcpy(ptr+1, "*");
	    }
	}
	else
	{
	    for (ptr = mhost; *ptr != '\0'; ptr++)
		if (*ptr != '.' && !isdigit(*ptr)) break;

	    if (*ptr == '\0')
	    {
		/* it's an IP address, change last digit to * */
		mhost = g_strdup(mhost);
		ptr = strrchr(mhost, '.');
		if (ptr != NULL && isdigit(ptr[1]))
		    strcpy(ptr+1, "*");
	    }
	    else
	    {
		/* if more than one dot, skip the first (d123.blah.net -> blah.net) */
		ptr = strchr(mhost, '.');
		if (ptr != NULL && strchr(ptr+1, '.') != NULL)
		    mhost = ptr+1;
		mhost = g_strdup_printf("*%s", mhost);
	    }
	}
    }
    else if ((flags & (IRC_MASK_HOST|IRC_MASK_DOMAIN)) == 0)
        mhost = g_strdup("*");

    if ((flags & IRC_MASK_NICK) == 0) nick = "*";
    user = (flags & IRC_MASK_USER) == 0 ? "*" : addr;

    str = g_strdup_printf("%s!%s@%s", nick, user, mhost);

    g_free(mhost);
    g_free(addr);

    return str;
}

/* Send command to IRC server */
void irc_send_cmd(SERVER_REC *server, gchar *cmd)
{
    gchar str[513], *ptr;
    gint len, ret;

    g_return_if_fail(cmd != NULL);
    if (server == NULL) return;

    if (server->cmdcount == 0)
	rawlog_output(server, cmd);

    /* just check that we don't send any longer commands than 512 bytes.. */
    len = 0;
    while (*cmd != '\0' && len < 510)
    {
        str[len] = *cmd++;
        len++;
    }
    str[len++] = 13; str[len++] = 10; str[len] = '\0'; ptr = str;

    if (server->cmdcount++ == 0)
    {
	if (gettimeofday(&server->last_cmd, NULL) != 0)
	    g_warning("irc_send_cmd() : gettimeofday() failed\n");

	ret = net_transmit(server->handle, str, len);
	if (ret == len)
	    return;

	/* this really shouldn't happen unless connection has broken
	   somehow weirdly. anyway, try again later.. */
        if (len > 0) ptr += ret;
    }

    /* add to queue */
    server->cmdqueue = g_list_append(server->cmdqueue, g_strdup(ptr));
}

static void parse_alias(SERVER_REC *server, CHANNEL_REC *channel, gchar *alias, gchar *cmd, gchar *data)
{
    gchar *chr, code;
    GString *str;

    g_return_if_fail(alias != NULL);
    g_return_if_fail(cmd != NULL);
    g_return_if_fail(data != NULL);

    code = 0;

    str = g_string_new(NULL);

    for (chr = cmd; *chr != '\0'; chr++)
    {
        if (code)
        {
            if (code == '%')
            {
                if (*chr == '0')
                    g_string_append(str, alias);
                else if (*chr >= '1' && *chr <= '9')
                {
                    gchar *ptr;
                    gint num;

                    for (ptr = data, num = 0; *ptr != '\0' && num < *chr-'1'; ptr++)
                        if (*ptr == ' ') num++;

                    while (*ptr != '\0' && *ptr != ' ')
                        g_string_append_c(str, *ptr++);
                }
                else if (*chr == 'c' && channel != NULL)
                    g_string_append(str, channel->name);
                else if (*chr == 'n' && channel != NULL && channel->server != NULL)
                    g_string_append(str, channel->server->nick);
            }
            else if (code == '&')
            {
                if (*chr >= '1' && *chr <= '9')
                {
                    gchar *ptr;
                    gint num;

                    for (ptr = data, num = 0; *ptr != '\0' && num < *chr-'1'; ptr++)
                        if (*ptr == ' ') num++;
                    g_string_append(str, ptr);
                }
		if (*chr == '&')
		{
			signal_emit("send command", 3, str->str, server, channel);
			/* now start next command */
			g_string_assign(str, "");
			while(*chr == ' ') chr++;
		}
            }
	    else if (code == '\\')
	    {
		g_string_append(str, chr);
	    }

            code = 0;
            continue;
        }

        if (*chr == '\\' || *chr == '%' || *chr == '&')
            code = *chr;
        else
            g_string_append_c(str, *chr);
    }

    signal_emit("send command", 3, str->str, server, channel);
    g_string_free(str, TRUE);
}

static gboolean irc_parse_outgoing(gchar *line, SERVER_REC *server, CHANNEL_REC *channel)
{
    gchar *cmd, *str, *args;
    gboolean use_alias = TRUE;

    g_return_val_if_fail(channel != NULL, FALSE);
    g_return_val_if_fail(line != NULL, FALSE);

    if (*line == '\0')
	return FALSE; /* empty line, forget it. */

    if (*line++ != CMD_CHAR)
        return TRUE; /* this is handled in ui-common/ui-completion.c */

    /* //command ignores aliases */
    if (*line == CMD_CHAR) { line++; use_alias = FALSE; }

    cmd = str = g_strconcat("command ", line, NULL);
    args = strchr(cmd+8, ' ');
    if (args != NULL) *args++ = '\0'; else args = "";

    if (use_alias)
    {
        /* check if there's an alias for command */
        LIST_REC *alias;

        alias = list_find(aliases, cmd+8);
        if (alias != NULL)
        {
            parse_alias(server, channel, alias->key, alias->value, args);
            g_free(str);
            return TRUE;
        }
    }

    if (server != NULL)
        server_redirect_default(server, cmd);

    g_strdown(cmd);
    if (!signal_emit(cmd, 3, args, server, channel))
        signal_emit("default command", 3, line, server, channel);

    g_free(str);
    return TRUE;
}

/* Parse command line sent by server */
static gboolean irc_parse_incoming_line(SERVER_REC *server, gchar *line)
{
    gchar *nick, *address;

    g_return_val_if_fail(server != NULL, FALSE);
    g_return_val_if_fail(line != NULL, FALSE);

    nick = address = NULL;

    if (*line == ':')
    {
        /* read prefix.. */
        nick = ++line;
        while (*line != '\0' && *line != ' ')
        {
            if (*line == '!')
            {
                *line = '\0';
                address = line+1;
            }
            line++;
        }
        if (*line == ' ')
        {
            *line++ = '\0';
            while (*line == ' ') line++;
        }
    }

    if (*line != '\0')
        signal_emit("server event", 4, line, server, nick, address);

    return TRUE;
}

static gboolean irc_server_event(gchar *line, SERVER_REC *server, gchar *nick, gchar *address)
{
    gchar *event, *args, *callcmd;
    GList *list;

    g_return_val_if_fail(line != NULL, FALSE);

    /* get command.. */
    event = g_strconcat("event ", line, NULL);
    args = strchr(event+6, ' ');
    if (args != NULL) *args++ = '\0'; else args = "";
    while (*args == ' ') args++;

    list = server_redirect_getqueue(server, event, args);
    if (list == NULL)
        callcmd = g_strdup(event);
    else
    {
        /* event is redirected somewhere else.. */
        REDIRECT_REC *rec;

        rec = list->data;
	callcmd = g_strdup(rec->name);
	rawlog_redirect(server, callcmd);
        server_redirect_remove_next(server, event, list);
    }

    g_strdown(callcmd);
    if (!signal_emit(callcmd, 4, args, server, nick, address))
        signal_emit("default event", 4, line, server, nick, address);

    g_free(callcmd);
    g_free(event);

    return TRUE;
}

/* Read line from server */
static gint irc_receive_line(SERVER_REC *server, GString *str)
{
    gint ret;

    g_return_val_if_fail(server != NULL, -1);
    g_return_val_if_fail(str != NULL, -1);

    ret = read_line(TRUE, server->handle, str, server->buffer);
    if (ret == -1)
    {
        /* connection lost */
        server->connection_lost = TRUE;
        server_disconnect(server);
    }
    return ret;
}

/* input function: handle incoming server messages */
static void irc_parse_incoming(SERVER_REC *server)
{
    GString *str;

    g_return_if_fail(server != NULL);

    str = g_string_new(NULL);

    while (irc_receive_line(server, str) > 0)
    {
	rawlog_input(server, str->str);
	signal_emit("server incoming", 2, server, str->str);
    }

    g_string_free(str, TRUE);
}

static gboolean irc_init_server(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    server->readtag = gui_input_add(server->handle, GUI_INPUT_READ/* | GUI_INPUT_EXCEPTION*/,
                                    (GUIInputFunction) irc_parse_incoming, server);

    return TRUE;
}

static gboolean irc_deinit_server(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    if (server->readtag > 0)
        gui_input_remove(server->readtag);

    return TRUE;
}

void irc_init(void)
{
    signal_add("send command", (SIGNAL_FUNC) irc_parse_outgoing);
    signal_add("server event", (SIGNAL_FUNC) irc_server_event);
    signal_add("server connected", (SIGNAL_FUNC) irc_init_server);
    signal_add("server disconnected", (SIGNAL_FUNC) irc_deinit_server);
    signal_add("server incoming", (SIGNAL_FUNC) irc_parse_incoming_line);
}

void irc_deinit(void)
{
    signal_remove("send command", (SIGNAL_FUNC) irc_parse_outgoing);
    signal_remove("server event", (SIGNAL_FUNC) irc_server_event);
    signal_remove("server connected", (SIGNAL_FUNC) irc_init_server);
    signal_remove("server disconnected", (SIGNAL_FUNC) irc_deinit_server);
    signal_remove("server incoming", (SIGNAL_FUNC) irc_parse_incoming_line);
}

/* vim: set shiftwidth=4 sts=4: */
