/*
 gui-mainwindows.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "gui-menu-commands.h"
#include <gdk/gdkkeysyms.h>

#define SPECIAL_CHAR_BOLD       '*'
#define SPECIAL_CHAR_COLOR      'C'
#define SPECIAL_CHAR_BELL       '!'
#define SPECIAL_CHAR_REVERSE    '/'
#define SPECIAL_CHAR_UNDERLINE  '_'
#define SPECIAL_CHAR_NORMAL     'O'

GList *mainwindows;

/* main window signal: delete_event */
static gint sig_window_delete(GtkWidget *widget, GdkEventAny *event, MAIN_WINDOW_REC *window)
{
    g_return_val_if_fail(window != NULL, 1);

    if (g_list_length(mainwindows) == 1)
        signal_emit("command quit", 1, "");
    else
        gui_mainwindow_destroy(window);
    return 1;
}

/* main window signal: enter_notify_event */
static gint sig_window_enter(GtkWidget *widget, GdkEventCrossing *event, MAIN_WINDOW_REC *window)
{
    g_return_val_if_fail(window != NULL, 0);

    if (cur_channel != window->active->active)
    {
        cur_channel = window->active->active;
        signal_emit("channel changed", 1, cur_channel);
    }

    return 0;
}

/* notebook signal: switch_page */
static void sig_switch_page(GtkWidget *notebook, GtkNotebookPage *child, guint pagenum, MAIN_WINDOW_REC *window)
{
    GUI_WINDOW_REC *gui;

    g_return_if_fail(child != NULL);
    g_return_if_fail(window != NULL);

    gui = gtk_object_get_data(GTK_OBJECT(child->child), "window");
    if (gui == NULL || gui->active->window->active == NULL)
    {
        /* couldn't find for some reason, probably just being destroyed, just
           use any channel for a while.. */
	cur_channel = channels == NULL ? NULL : channels->data;
        return;
    }

    window->active = gui->active->window;
    cur_channel = window->active->active;

    signal_emit("channel changed", 1, cur_channel);
    signal_emit("window focused", 1, window->active);
}

static gboolean gui_window_goto(gpointer num)
{
    WINDOW_REC *window;
    GList *winlist;
    gint page;

    winlist = g_list_nth(windows, GPOINTER_TO_INT(num));
    if (winlist == NULL) return TRUE;

    window = winlist->data;

    if (WINDOW_GUI(window)->active != WINDOW_VIEW(window))
    {
	/* need to change the view */
	WINDOW_GUI(window)->active = WINDOW_VIEW(window);
	WINDOW_GUI(window)->parent->active = window;
	signal_emit("channel focused", 1, window->active);
	signal_emit("window focused", 1, window);
    }

    page = gtk_notebook_page_num(GTK_NOTEBOOK(WINDOW_GUI(window)->parent->notebook), WINDOW_GUI(window)->window);
    gtk_notebook_set_page(GTK_NOTEBOOK(WINDOW_GUI(window)->parent->notebook), GPOINTER_TO_INT(page));

    return TRUE;
}

#ifdef GTK_HEBREW

#include <fribidi/fribidi.h>
void hebrew_log2vis(guchar *log, guchar *vis)
{
   FriBidiChar *unilog, *univis;
   FriBidiCharType fbct;
   
   unilog = malloc(strlen(log) * 2 + 1);
   univis = malloc(strlen(log) * 2 + 1);
   fribidi_iso8859_8_to_unicode(log, unilog);
   fribidi_log2vis(unilog, strlen(log), &fbct, univis, NULL, NULL, NULL);
   fribidi_unicode_to_iso8859_8(univis, strlen(log), vis);
   free(unilog);
   free(univis);
}

static void log_insert(guchar *log, int pos, int ch)
{
   memmove(log + pos + 1, log + pos, strlen(log) - pos + 1);
   *(log + pos) = ch;
}

static void log_delete(guchar *log, int pos)
{
   memmove(log + pos, log + pos + 1, strlen(log) - pos);
}

#endif /* GTK_HEBREW */

void entry_set_specials(MAIN_WINDOW_REC *window, gchar *text)
{
    GList *tmp;
    gint pos;

    for (tmp = window->specials; tmp != NULL; tmp = tmp->next)
    {
	pos = GPOINTER_TO_INT(tmp->data);
	switch (text[pos])
	{
	    case SPECIAL_CHAR_BELL:
		text[pos] = 7;
		break;
	    case SPECIAL_CHAR_BOLD:
		text[pos] = 2;
		break;
	    case SPECIAL_CHAR_UNDERLINE:
		text[pos] = 31;
		break;
	    case SPECIAL_CHAR_REVERSE:
		text[pos] = 22;
		break;
	    case SPECIAL_CHAR_COLOR:
		text[pos] = 3;
		break;
	    case SPECIAL_CHAR_NORMAL:
		text[pos] = 15;
		break;
	}
    }
    g_list_free(window->specials);
    window->specials = NULL;
}

void entry_remove_specials(MAIN_WINDOW_REC *window, gchar *text)
{
    gchar *p, c;

    for (p = text; *p != '\0'; p++)
    {
	switch (*p)
	{
	    case 2:
		c = SPECIAL_CHAR_BOLD;
		break;
	    case 3:
		c = SPECIAL_CHAR_COLOR;
		break;
	    case 7:
		c = SPECIAL_CHAR_BELL;
		break;
	    case 22:
		c = SPECIAL_CHAR_REVERSE;
		break;
	    case 31:
		c = SPECIAL_CHAR_UNDERLINE;
		break;
	    case 15:
		c = SPECIAL_CHAR_NORMAL;
		break;
	    default:
		c = 0;
	}
	if (c != 0)
	{
	    *p = c;
	    window->specials = g_list_append(window->specials, GINT_TO_POINTER((gint) (p-text)));
	}
    }
}

/* signal: key pressed (in entry field) */
static gint sig_keypress(GtkWidget *widget, GdkEventKey *event, MAIN_WINDOW_REC *window)
{
    GUI_WINDOW_REC *guiwin;
    gboolean ret;
    gchar *key;

#ifdef GTK_HEBREW
    static gchar *log = NULL;
    static gchar *vis = NULL;

    if (log == NULL) {
       log = malloc(2048);
       log[0] = 0;
       vis = malloc(2048);
    }
#endif

    g_return_val_if_fail(event != NULL, 0);
    g_return_val_if_fail(window != NULL, 0);

    guiwin = window->active == NULL ? NULL : WINDOW_GUI(window->active);
    if (guiwin != NULL && guiwin->topicentry != NULL &&
	((gtk_object_get_data(GTK_OBJECT(guiwin->topicentry), "editable") &&
	  GTK_WIDGET_HAS_FOCUS(guiwin->topicentry)) ||
	 GTK_WIDGET_HAS_FOCUS(guiwin->limit_entry) ||
	 GTK_WIDGET_HAS_FOCUS(guiwin->key_entry)))
    {
        /* trying to write topic, don't disturb it.. */
        return 0;
    }

    /* send the key signal */
    key = gui_key2str(event);
    ret = key_pressed(key, window);
    g_free(key);

    if (ret && (event->keyval < 32 || event->keyval > 255) && event->keyval != GDK_Return && event->keyval != GDK_KP_Enter)
    {
	/* key was used, stop signal */
	gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
	return 1;
    }

    if ((event->state & (GDK_CONTROL_MASK|GDK_MOD1_MASK)) == 0 &&
	event->keyval != GDK_Control_L && event->keyval != GDK_Control_R &&
	event->keyval != GDK_Meta_L && event->keyval != GDK_Meta_R &&
	event->keyval != GDK_Alt_L && event->keyval != GDK_Alt_R)
    {
	/* normal key pressed, change focus to entry field */
	gtk_widget_grab_focus(window->entry);

#ifdef GTK_HEBREW
	{
	    int display = 0;
	    gint pos;

	    pos = gtk_editable_get_position(GTK_EDITABLE(window->entry));
	    if ((event->keyval & 0xFF00) == 0) {
		log_insert(log, pos, event->keyval);
		pos++;
		display = 1;
	    }
	    if (event->keyval == GDK_BackSpace) {
		pos--;
		log_delete(log, pos);
		display = 1;
	    }
	    if (event->keyval == GDK_Delete) {
		log_delete(log, pos);
		display = 1;
	    }
	    if (event->keyval == GDK_Return) {
		gtk_entry_set_text(GTK_ENTRY(window->entry), log);
		gtk_editable_set_position(GTK_EDITABLE(window->entry), 0);
		log[0] = 0;
		pos = 0;
	    }

	    if (display) {
		hebrew_log2vis(log, vis);
		gtk_entry_set_text(GTK_ENTRY(window->entry), vis);
		gtk_editable_set_position(GTK_EDITABLE(window->entry), pos);
		break;
	    }
	}
#endif /* GTK_HEBREW */
    }
    return 0;
}

/* entry signal: activate */
static void sig_entry_activated(GtkWidget *entry, MAIN_WINDOW_REC *window)
{
    gchar *str, *ptr;

    g_return_if_fail(entry != NULL);
    g_return_if_fail(window != NULL);

    /* replace the "special characters" with real characters.. */
    str = gtk_entry_get_text(GTK_ENTRY(entry));
    entry_set_specials(window, str);

    /* pasting text into entry can add some newlines in there, so we have to
       parse them here.. */
    while (str != NULL && *str != '\0')
    {
        ptr = strchr(str, '\n');
        if (ptr != NULL) *ptr++ = '\0';

        ui_history_add(window->active, str, FALSE);

        if (cur_channel != window->active->active)
        {
            /* we get here at least when using click-to-focus and keyboard
               focus being in different window than mouse. Is there some
               event we could use instead of enter_notify_event to find out
               what window really has the focus? .. probably is, I'm too lazy
               to find out what it is .. */
            cur_channel = window->active->active;
            signal_emit("channel changed", 1, cur_channel);
        }
        signal_emit("send command", 3, str, cur_channel->server, cur_channel);

        str = ptr;
    }

    gtk_entry_set_text(GTK_ENTRY(entry), "");
    ui_history_clear_pos(window->active);
}

static void sig_entry_inserted(GtkEntry *entry, gchar *text, gint length, gint *inspos, MAIN_WINDOW_REC *window)
{
    GList *tmp;
    gint pos;

    g_return_if_fail(entry != NULL);
    g_return_if_fail(window != NULL);

    if (*gtk_entry_get_text(entry) == '\0')
    {
	/* no text before, we could have ended up here by pressing key
	   up/down, don't try to update the specials */
        return;
    }

    /* new text, check if we need to move the position of the special chars */
    for (tmp = window->specials; tmp != NULL; tmp = tmp->next)
    {
	pos = GPOINTER_TO_INT(tmp->data);
	if (pos >= *inspos) tmp->data = GINT_TO_POINTER(pos+length);
    }
}

static void sig_entry_deleted(GtkWidget *entry, gint start_pos, gint end_pos, MAIN_WINDOW_REC *window)
{
    GList *tmp, *next;
    gint pos;

    g_return_if_fail(entry != NULL);
    g_return_if_fail(window != NULL);

    /* deleted text, check if we need to remove/move the position of the
       special chars */
    for (tmp = window->specials; tmp != NULL; tmp = next)
    {
	next = tmp->next;
	pos = GPOINTER_TO_INT(tmp->data);
	if (pos >= start_pos && pos < end_pos)
	{
	    /* character deleted */
	    window->specials = g_list_remove_link(window->specials, tmp);
            g_list_free_1(tmp);
	}
	else if (pos >= end_pos)
	{
	    tmp->data = GINT_TO_POINTER(pos-(end_pos-start_pos));
	}
    }
}

static void term_change_pos(MAIN_WINDOW_REC *window)
{
    /* window moved - redraw the transparent windows */
    GtkWidget *itext;
    GList *tmp;

    for (tmp = window->children; tmp != NULL; tmp = tmp->next)
    {
        WINDOW_REC *rec = tmp->data;

        if (WINDOW_VIEW(rec)->itext)
        {
            itext = WINDOW_VIEW(rec)->text;

            if (itext->window != NULL && (GTK_ITEXT(itext)->bg_flags & ITEXT_BG_TRANSPARENT))
                gtk_widget_queue_draw(itext);
        }
    }
}

MAIN_WINDOW_REC *gui_mainwindow_create(WINDOW_REC *child)
{
    proplist_t wprop;
    MAIN_WINDOW_REC *window;
    GtkWidget *hbox, *label;
    gint default_xsize, default_ysize, default_xpos, default_ypos;
    gint xsize, ysize, xpos, ypos;
    gchar *name;

    window = g_new0(MAIN_WINDOW_REC, 1);
    mainwindows = g_list_append(mainwindows, window);

    window->window = gnome_app_new(PACKAGE, PACKAGE);

    /* for repainting transparent itext widget after moving window */
    gtk_signal_connect_object(GTK_OBJECT(window->window), "configure_event",
                              GTK_SIGNAL_FUNC(term_change_pos), (GtkObject *) window);
    gtk_signal_connect(GTK_OBJECT(window->window), "delete_event",
                       GTK_SIGNAL_FUNC(sig_window_delete), window);
    gtk_signal_connect(GTK_OBJECT(window->window), "enter_notify_event",
                       GTK_SIGNAL_FUNC(sig_window_enter), window);
    /* redirect all key presses in window to entry field */
    gtk_signal_connect(GTK_OBJECT(window->window), "key_press_event",
                       GTK_SIGNAL_FUNC(sig_keypress), window);

    gtk_window_set_policy(GTK_WINDOW(window->window), TRUE, TRUE, FALSE);

    gui_menus_create(window, main_menu);
    if (setup_get_bool("toggle_show_toolbar")) gnome_app_create_toolbar_with_data(GNOME_APP(window->window), toolbar, window);

    window->container = gtk_vbox_new(FALSE, 0);
    gnome_app_set_contents(GNOME_APP(window->window), window->container);

    /* create statusbar */
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_end(GTK_BOX(window->container), hbox, FALSE, FALSE, 0);

    label = gtk_label_new(_("Cancel"));
    gtk_misc_set_padding(GTK_MISC(label), 5, 0);

    window->cancel = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(window->cancel), label);
    gtk_box_pack_start(GTK_BOX(hbox), window->cancel, FALSE, TRUE, 0);

    window->statusbar = gtk_statusbar_new();
    gtk_box_pack_start(GTK_BOX(hbox), window->statusbar, TRUE, TRUE, 0);

    window->main_id = gui_statusbar_push(window, "");

    /* create notebook for subwindows */
    window->notebook = gtk_notebook_new();
    gtk_notebook_set_scrollable(GTK_NOTEBOOK(window->notebook), TRUE);
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(window->notebook), setup_get_int("tab_orientation"));
    gtk_signal_connect(GTK_OBJECT(window->notebook), "switch_page",
                       GTK_SIGNAL_FUNC(sig_switch_page), window);

    gtk_box_pack_start(GTK_BOX(window->container), window->notebook, TRUE, TRUE, 0);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_container_border_width(GTK_CONTAINER(hbox), 2);
    gtk_box_pack_end(GTK_BOX(window->container), hbox, FALSE, FALSE, 0);

    /* Create server selection menu */
    window->servermenu = gtk_option_menu_new();
    gtk_widget_set_usize(window->servermenu, 100, -1);
    gtk_box_pack_start(GTK_BOX(hbox), window->servermenu, FALSE, TRUE, 0);

    gui_servermenu_create(window->servermenu, window);

    /* Create entry */
    window->entry = gtk_entry_new();
    gtk_signal_connect(GTK_OBJECT(window->entry), "insert_text",
                       GTK_SIGNAL_FUNC(sig_entry_inserted), window);
    gtk_signal_connect(GTK_OBJECT(window->entry), "delete_text",
                       GTK_SIGNAL_FUNC(sig_entry_deleted), window);
    gtk_signal_connect(GTK_OBJECT(window->entry), "activate",
                       GTK_SIGNAL_FUNC(sig_entry_activated), window);
    gtk_box_pack_start(GTK_BOX(hbox), window->entry, TRUE, TRUE, 0);

    /* Set the window size/position */
    name = gui_channel_get_name(child->channels->data);

    /* first get default values */
    wprop = config_get_prop(cprop, "windows");

    default_xsize = config_get_int(wprop, "default_xsize", gdk_screen_width()/2);
    default_ysize = config_get_int(wprop, "default_ysize", gdk_screen_height()*2/6);

    default_xpos = config_get_int(wprop, "default_xpos", -1);
    default_ypos = config_get_int(wprop, "default_ypos", -1);

    if (name == NULL)
    {
	/* empty window without name, use defaults */
	xsize = ysize = xpos = ypos = -1;
    }
    else
    {
	GString *str;

	/* then check if there's channel specific values */
	str = g_string_new(NULL);
	g_string_sprintf(str, "%s_xsize", name);
	xsize = config_get_int(wprop, str->str, -1);
	g_string_sprintf(str, "%s_ysize", name);
	ysize = config_get_int(wprop, str->str, -1);
	g_string_sprintf(str, "%s_xpos", name);
	xpos = config_get_int(wprop, str->str, -1);
	g_string_sprintf(str, "%s_ypos", name);
	ypos = config_get_int(wprop, str->str, -1);
	g_string_free(str, TRUE);
    }

    if (xsize == -1) xsize = default_xsize;
    if (ysize == -1) ysize = default_ysize;
    if (xpos == -1) xpos = default_xpos;
    if (ypos == -1) ypos = default_ypos;

    gtk_widget_set_usize(window->window, xsize, ysize);
    if (xpos != -1)
    {
	/* For some reason this needs to be set before AND .. */
	gtk_widget_set_uposition(window->window, xpos, ypos);
    }
    gtk_widget_show_all(window->window);
    if (xpos != -1)
    {
	/* AND .. after showing the main window.. If set before showing, it
	   will display the window in wrong position, if not set before it
	   will all the time keep moving the window back to the saved position
	   no matter how hard you try to move it elsewhere.. */
	gtk_widget_set_uposition(window->window, xpos, ypos);
    }

    gtk_widget_hide(window->cancel);
    if (!setup_get_bool("toggle_show_statusbar")) gtk_widget_hide(window->statusbar);
    gtk_widget_hide(window->servermenu);

    signal_emit("gui mainwindow created", 1, window);

    return window;
}

void gui_mainwindow_destroy(MAIN_WINDOW_REC *window)
{
    g_return_if_fail(window != NULL);
    if (window->destroying) return;

    mainwindows = g_list_remove(mainwindows, window);

    signal_emit("gui mainwindow destroyed", 1, window);

    window->destroying = TRUE;
    while (window->children != NULL)
        ui_window_destroy(window->children->data);
    window->destroying = FALSE;

    gui_servermenu_destroy(window->servermenu);
    gtk_widget_destroy(window->window);

    while (window->statusbar_texts != NULL)
        gui_statusbar_destroy(window, window->statusbar_texts->data);
    if (window->extramenu != NULL) g_free(window->extramenu);
    g_free(window);

    if (mainwindows == NULL)
        signal_emit("command quit", 1, "");
}

static gboolean gui_mainwindows_redraw(void)
{
    GList *tmp;

    for (tmp = g_list_first(mainwindows); tmp != NULL; tmp = tmp->next)
    {
        MAIN_WINDOW_REC *rec = tmp->data;

        gtk_notebook_set_tab_pos(GTK_NOTEBOOK(rec->notebook), setup_get_int("tab_orientation"));
    }

    return TRUE;
}

static gboolean sig_completion(gpointer data, MAIN_WINDOW_REC *window)
{
    gchar *line;
    gint pos;

    pos = gtk_editable_get_position(GTK_EDITABLE(window->entry));
    line = ui_completion_line(window->active->active, gtk_entry_get_text(GTK_ENTRY(window->entry)), &pos);
    if (line != NULL)
    {
	gtk_entry_set_text(GTK_ENTRY(window->entry), line);
	gtk_entry_set_position(GTK_ENTRY(window->entry), pos);
	g_free(line);
    }

    return TRUE;
}

static gboolean sig_replace(gpointer data, MAIN_WINDOW_REC *window)
{
    gchar *line;
    gint pos;

    pos = gtk_editable_get_position(GTK_EDITABLE(window->entry));
    line = ui_auto_completion(gtk_entry_get_text(GTK_ENTRY(window->entry)), &pos);
    if (line != NULL)
    {
	gtk_entry_set_text(GTK_ENTRY(window->entry), line);
	gtk_entry_set_position(GTK_ENTRY(window->entry), pos);
	g_free(line);
    }

    return TRUE;
}

static gboolean sig_prev_window(void)
{
    signal_emit("command window prev", 3, "", cur_channel->server, cur_channel);
    return TRUE;
}

static gboolean sig_next_window(void)
{
    signal_emit("command window next", 3, "", cur_channel->server, cur_channel);
    return TRUE;
}

static gboolean sig_prev_channel(void)
{
    signal_emit("command channel prev", 3, "", cur_channel->server, cur_channel);
    return TRUE;
}

static gboolean sig_next_channel(void)
{
    signal_emit("command channel next", 3, "", cur_channel->server, cur_channel);
    return TRUE;
}

static gboolean sig_addchar(gchar *data, MAIN_WINDOW_REC *window)
{
    gchar c;
    gint pos;

    switch (*data)
    {
	case 2:
	    c = SPECIAL_CHAR_BOLD;
	    break;
	case 3:
	    c = SPECIAL_CHAR_COLOR;
	    break;
	case 7:
	    c = SPECIAL_CHAR_BELL;
	    break;
	case 22:
	    c = SPECIAL_CHAR_REVERSE;
	    break;
	case 31:
	    c = SPECIAL_CHAR_UNDERLINE;
	    break;
	case 15:
	    c = SPECIAL_CHAR_NORMAL;
	    break;
	default:
	    return TRUE;
    }

    pos = gtk_editable_get_position(GTK_EDITABLE(window->entry));
    gtk_editable_insert_text(GTK_EDITABLE(window->entry), &c, 1, &pos);
    window->specials = g_list_append(window->specials, GINT_TO_POINTER(pos-1));
    gtk_editable_set_position(GTK_EDITABLE(window->entry), pos);

    return TRUE;
}

static gboolean sig_change_window(gchar *data)
{
    signal_emit("command window goto", 3, data, cur_channel->server, cur_channel);
    return TRUE;
}

static gboolean sig_history(gchar *data, MAIN_WINDOW_REC *window)
{
    gchar *text;
#ifdef GTK_HEBREW
    static gchar *log = NULL;
    static gchar *vis = NULL;

    if (log == NULL) {
       log = malloc(2048);
       log[0] = 0;
       vis = malloc(2048);
    }
#endif

    text = gtk_entry_get_text(GTK_ENTRY(window->entry));

    entry_set_specials(window, text);
    if (toupper(*data) == 'P')
	text = ui_history_prev(window->active, text);
    else
	text = ui_history_next(window->active, text);

    /* replace special chars with normal looking chars.. */
    text = g_strdup(text);
    entry_remove_specials(window, text);
#ifdef GTK_HEBREW
    strcpy(log, text);
    hebrew_log2vis(log, vis);
    g_free(text);
    text = g_strdup(vis);
#endif
    gtk_entry_set_text(GTK_ENTRY(window->entry), text);
    g_free(text);

    return TRUE;
}

static gboolean sig_prevpage(gchar *data, MAIN_WINDOW_REC *window)
{
    GtkAdjustment *adj;
    WINDOW_VIEW_REC *view;
    gint val;

    g_return_val_if_fail(window != NULL, FALSE);

    view = WINDOW_VIEW(window->active);
    adj = view->itext ?
	GTK_ITEXT(view->text)->adjustment :
	GTK_TEXT(view->text)->vadj;
    val = adj->value - (adj->page_size/2);
    gtk_adjustment_set_value(adj, val > 0 ? val : 0);

    return TRUE;
}

static gboolean sig_nextpage(gchar *data, MAIN_WINDOW_REC *window)
{
    GtkAdjustment *adj;
    WINDOW_VIEW_REC *view;
    gint val, max;

    g_return_val_if_fail(window != NULL, FALSE);

    view = WINDOW_VIEW(window->active);
    adj = view->itext ?
	GTK_ITEXT(view->text)->adjustment :
	GTK_TEXT(view->text)->vadj;

    val = adj->value + (adj->page_size/2);
    max = adj->upper - adj->lower - adj->page_size;
    gtk_adjustment_set_value(adj, val <= max ? val : max);
    return TRUE;
}

void gui_mainwindows_init(void)
{
    static gchar changekeys[] = "1234567890";
    gchar *key, *data;
    gint n;

    mainwindows = NULL;

    signal_add("setup changed", (SIGNAL_FUNC) gui_mainwindows_redraw);
    signal_add("gui window goto", (SIGNAL_FUNC) gui_window_goto);

    key_bind("completion", NULL, _("Nick completion"), "Tab", (SIGNAL_FUNC) sig_completion);
    key_bind("check replaces", NULL, _("Check word replaces"), " ", (SIGNAL_FUNC) sig_replace);
    key_bind("check replaces", NULL, NULL, "Return", (SIGNAL_FUNC) sig_replace);
    key_bind("window prev", NULL, _("Previous window"), "CTRL-P", (SIGNAL_FUNC) sig_prev_window);
    key_bind("window next", NULL, _("Next window"), "CTRL-N", (SIGNAL_FUNC) sig_next_window);
    key_bind("channel next", NULL, "Next channel", "CTRL-X", (SIGNAL_FUNC) sig_next_channel);
    key_bind("channel prev", NULL, "Next channel", NULL, (SIGNAL_FUNC) sig_prev_channel);

    key_bind("history", "PREV", _("Command history"), "Up", (SIGNAL_FUNC) sig_history);
    key_bind("history", "NEXT", NULL, "Down", (SIGNAL_FUNC) sig_history);
    key_bind("window prevpage", NULL, _("Previous page in window"), "Prior", (SIGNAL_FUNC) sig_prevpage);
    key_bind("window nextpage", NULL, _("Next page in window"), "Next", (SIGNAL_FUNC) sig_nextpage);

    key_bind("special char", "\x02", _("Insert special character"), "CTRL-B", (SIGNAL_FUNC) sig_addchar);
    key_bind("special char", "\x1f", NULL, "CTRL--", (SIGNAL_FUNC) sig_addchar);
    key_bind("special char", "\x03", NULL, "CTRL-C", (SIGNAL_FUNC) sig_addchar);
    key_bind("special char", "\x16", NULL, "CTRL-V", (SIGNAL_FUNC) sig_addchar);
    key_bind("special char", "\x07", NULL, "CTRL-G", (SIGNAL_FUNC) sig_addchar);
    key_bind("special char", "\x0f", NULL, "CTRL-O", (SIGNAL_FUNC) sig_addchar);

    for (n = 0; changekeys[n] != '\0'; n++)
    {
        key = g_strdup_printf("ALT-%c", changekeys[n]);
        data = g_strdup_printf("%d", n+1);
	key_bind("change window", data, _("Change window"), key, (SIGNAL_FUNC) sig_change_window);
	g_free(data); g_free(key);
    }
}

void gui_mainwindows_deinit(void)
{
    key_unbind("completion", (SIGNAL_FUNC) sig_completion);
    key_unbind("check replaces", (SIGNAL_FUNC) sig_replace);
    key_unbind("window prev", (SIGNAL_FUNC) sig_prev_window);
    key_unbind("window next", (SIGNAL_FUNC) sig_next_window);
    key_unbind("channel next", (SIGNAL_FUNC) sig_next_channel);
    key_unbind("channel prev", (SIGNAL_FUNC) sig_prev_channel);

    key_unbind("history", (SIGNAL_FUNC) sig_history);
    key_unbind("window prevpage", (SIGNAL_FUNC) sig_prevpage);
    key_unbind("window nextpage", (SIGNAL_FUNC) sig_nextpage);

    key_unbind("special char", (SIGNAL_FUNC) sig_addchar);
    key_unbind("change window", (SIGNAL_FUNC) sig_change_window);

    signal_remove("setup changed", (SIGNAL_FUNC) gui_mainwindows_redraw);
    signal_remove("gui window goto", (SIGNAL_FUNC) gui_window_goto);

    while (mainwindows != NULL)
        gui_mainwindow_destroy(mainwindows->data);
}
