/*
 modebox.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

#define CHANMODE_INVITE       0x01
#define CHANMODE_SECRET       0x02
#define CHANMODE_PRIVATE      0x04
#define CHANMODE_MODERATE     0x08
#define CHANMODE_NOMSGS       0x10
#define CHANMODE_OP_TOPIC     0x20

static GtkWidget *dialog = NULL;
static GtkWidget *opwidgets[4];

static void append_list(GtkList *list, gchar *label, gpointer data)
{
    GtkWidget *li;

    g_return_if_fail(list != NULL);

    li = gtk_list_item_new_with_label(label);
    gtk_widget_show(li);

    gtk_object_set_data(GTK_OBJECT(li), "data", data);
    gtk_list_append_items(list, g_list_append(NULL, li));
}

static void remove_list(GtkList *list, gpointer data)
{
    GList *tmp;

    g_return_if_fail(list != NULL);

    for (tmp = g_list_first(list->children); tmp != NULL; tmp = tmp->next)
    {
        if (data == gtk_object_get_data(GTK_OBJECT(tmp->data), "data"))
        {
            GList *items;

            items = g_list_alloc(); items->data = tmp->data;
            gtk_list_remove_items(list, items);
            g_list_free(items);
            break;
        }
    }
}

static void create_list(GtkList *list, GList *glist, gboolean bans)
{
    GtkWidget *li;
    GList *tmp, *items;

    g_return_if_fail(list != NULL);

    items = NULL;
    gtk_list_clear_items(GTK_LIST(list), 0, -1);
    for (tmp = g_list_first(glist); tmp != NULL; tmp = tmp->next)
    {
        BAN_REC *ban = tmp->data;

        li = gtk_list_item_new_with_label(bans ? ban->ban : tmp->data);
        gtk_widget_show(li);
        gtk_object_set_data(GTK_OBJECT(li), "data", ban);
        items = g_list_append(items, li);
    }
    gtk_list_append_items(list, items);
}

/* signal: apply button pressed in mode box dialog */
static void modebox_apply(GtkWidget *dialog)
{
    GtkEntry *topicentry, *limitentry, *keyentry;
    gchar *topic, *limitstr, *key;
    gint limit, mode;
    CHANNEL_REC *channel;
    GString *str;

    g_return_if_fail(dialog != NULL);

    channel = gtk_object_get_data(GTK_OBJECT(dialog), "channel");
    topicentry = gtk_object_get_data(GTK_OBJECT(dialog), "topic");
    limitentry = gtk_object_get_data(GTK_OBJECT(dialog), "limit");
    keyentry = gtk_object_get_data(GTK_OBJECT(dialog), "key");
    mode = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(dialog), "mode"));

    topic = gtk_entry_get_text(topicentry);
    key = gtk_entry_get_text(keyentry);
    limitstr = gtk_entry_get_text(limitentry);
    if (*limitstr == '\0')
        limit = 0;
    else
    {
        if (sscanf(limitstr, "%d", &limit) != 1 || limit == 0) limit = -1;
    }

    str = g_string_new(NULL);
    if (strcmp(channel->topic != NULL ? channel->topic : "", topic) != 0)
    {
        /* topic changed */
        g_string_sprintf(str, "TOPIC %s :%s", channel->name, topic);
        irc_send_cmd(channel->server, str->str);
    }

    if (strcmp(channel->key != NULL ? channel->key : "", key) != 0)
    {
        /* key changed */
        if (channel->key != NULL)
        {
            /* remove old key */
            g_string_sprintf(str, "MODE %s -k %s", channel->name, channel->key);
            irc_send_cmd(channel->server, str->str);
        }

        if (*key != '\0')
        {
            /* set new key */
            g_string_sprintf(str, "MODE %s +k %s", channel->name, key);
            irc_send_cmd(channel->server, str->str);
        }
    }

    g_string_sprintf(str, "MODE %s ", channel->name);

    if ((mode & CHANMODE_INVITE) != channel->mode_invite)
        g_string_sprintfa(str, "%ci", (mode & CHANMODE_INVITE) ? '+' : '-');
    if ((mode & CHANMODE_SECRET) != channel->mode_secret)
        g_string_sprintfa(str, "%cs", (mode & CHANMODE_SECRET) ? '+' : '-');
    if ((mode & CHANMODE_PRIVATE) != channel->mode_private)
        g_string_sprintfa(str, "%cp", (mode & CHANMODE_PRIVATE) ? '+' : '-');
    if ((mode & CHANMODE_MODERATE) != channel->mode_moderate)
        g_string_sprintfa(str, "%cm", (mode & CHANMODE_MODERATE) ? '+' : '-');
    if ((mode & CHANMODE_NOMSGS) != channel->mode_nomsgs)
        g_string_sprintfa(str, "%cn", (mode & CHANMODE_NOMSGS) ? '+' : '-');
    if ((mode & CHANMODE_OP_TOPIC) != channel->mode_optopic)
        g_string_sprintfa(str, "%ct", (mode & CHANMODE_OP_TOPIC) ? '+' : '-');

    if (limit != -1 && limit != channel->limit)
    {
        /* limit changed */
        if (limit == 0)
            g_string_sprintfa(str, "-l");
        else
            g_string_sprintfa(str, "+l %d", limit);
    }

    if (str->len != strlen(channel->name)+6)
        irc_send_cmd(channel->server, str->str);
    g_string_free(str, TRUE);
}

/* signal: ok button pressed in mode box dialog */
static void modebox_ok(GtkWidget *dialog)
{
    modebox_apply(dialog);

    gtk_widget_destroy(dialog);
}

static void check_clicked(GtkWidget *button, GtkWidget *dialog)
{
    gint mask;
    gint mode;

    g_return_if_fail(button != NULL);
    g_return_if_fail(dialog != NULL);

    mask = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(button), "mask"));
    mode = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(dialog), "mode")) ^ mask;
    gtk_object_set_data(GTK_OBJECT(dialog), "mode", GINT_TO_POINTER(mode));

    if ((mode & CHANMODE_PRIVATE) && (mode & CHANMODE_SECRET))
    {
        GtkToggleButton *but;

        /* can't be both private and secret */
        if (mask == CHANMODE_PRIVATE)
            but = gtk_object_get_data(GTK_OBJECT(dialog), "secretbut");
        else
            but = gtk_object_get_data(GTK_OBJECT(dialog), "privatebut");
        gtk_toggle_button_set_active(but, FALSE);
    }
}

/* create new check button for mode dialog */
static GtkWidget *new_check_button(GtkWidget *dialog, GtkWidget *vbox, gchar *text, gint mode)
{
    CHANNEL_REC *channel;
    GtkWidget *button;
    gboolean value;

    g_return_val_if_fail(dialog != NULL, NULL);
    g_return_val_if_fail(vbox != NULL, NULL);
    g_return_val_if_fail(text != NULL, NULL);

    channel = gtk_object_get_data(GTK_OBJECT(dialog), "channel");

    button = gtk_check_button_new_with_label(text);
    gtk_object_set_data(GTK_OBJECT(button), "mask", GINT_TO_POINTER(mode));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC (check_clicked), dialog);
    gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

    value = FALSE;
    switch (mode)
    {
        case CHANMODE_INVITE:
            value = channel->mode_invite;
            break;
        case CHANMODE_SECRET:
            value = channel->mode_secret;
            break;
        case CHANMODE_PRIVATE:
            value = channel->mode_private;
            break;
        case CHANMODE_MODERATE:
            value = channel->mode_moderate;
            break;
        case CHANMODE_NOMSGS:
            value = channel->mode_nomsgs;
            break;
        case CHANMODE_OP_TOPIC:
            value = channel->mode_optopic;
            break;
    }
    if (value) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

    return button;
}

/* signal: unban button pressed */
static void remove_ban(GtkWidget *dialog)
{
    CHANNEL_REC *channel;
    GtkList *list;
    BAN_REC *ban;
    GString *str;

    g_return_if_fail(dialog != NULL);

    channel = gtk_object_get_data(GTK_OBJECT(dialog), "channel");
    list = gtk_object_get_data(GTK_OBJECT(dialog), "banlist");
    if (list->selection == NULL) return;

    ban = gtk_object_get_data(GTK_OBJECT(list->selection->data), "data");
    str = g_string_new(NULL);
    g_string_sprintf(str, "MODE %s -b %s", channel->name, ban->ban);
    irc_send_cmd(channel->server, str->str);
    g_string_free(str, TRUE);
}

/* signal: remove ban exception button pressed */
static void remove_eban(GtkWidget *dialog)
{
    CHANNEL_REC *channel;
    GtkList *list;
    BAN_REC *ban;
    GString *str;

    g_return_if_fail(dialog != NULL);

    channel = gtk_object_get_data(GTK_OBJECT(dialog), "channel");
    list = gtk_object_get_data(GTK_OBJECT(dialog), "ebanlist");
    if (list->selection == NULL) return;

    ban = gtk_object_get_data(GTK_OBJECT(list->selection->data), "data");

    str = g_string_new(NULL);
    g_string_sprintf(str, "MODE %s -e %s", channel->name, ban->ban);
    irc_send_cmd(channel->server, str->str);
    g_string_free(str, TRUE);
}

/* signal: remove invite button pressed */
static void remove_invite(GtkWidget *dialog)
{
    CHANNEL_REC *channel;
    GtkList *list;
    GString *str;

    g_return_if_fail(dialog != NULL);

    channel = gtk_object_get_data(GTK_OBJECT(dialog), "channel");
    list = gtk_object_get_data(GTK_OBJECT(dialog), "invitelist");
    if (list->selection == NULL) return;

    str = g_string_new(NULL);
    g_string_sprintf(str, "MODE %s -I %s", channel->name,
                     (gchar *) gtk_object_get_data(GTK_OBJECT(list->selection->data), "data"));
    irc_send_cmd(channel->server, str->str);
    g_string_free(str, TRUE);
}

void sig_remove_dialog(GtkWidget *widget, GtkWidget **dialog)
{
    g_return_if_fail(dialog != NULL);
    g_return_if_fail(*dialog != NULL);

    *dialog = NULL;
}

static gboolean channel_synced(CHANNEL_REC *channel)
{
    GtkWidget *topic;

    if (dialog != NULL && gtk_object_get_data(GTK_OBJECT(dialog), "channel") == channel)
    {
	topic = gtk_object_get_data(GTK_OBJECT(dialog), "topic");
	gtk_entry_set_editable(GTK_ENTRY(topic), channel->chanop || !channel->mode_optopic);
	gtk_widget_set_sensitive(opwidgets[0], channel->chanop);
	gtk_widget_set_sensitive(opwidgets[1], channel->chanop);
	if (opwidgets[2] != NULL)
	    gtk_widget_set_sensitive(opwidgets[2], channel->chanop);
	if (opwidgets[3] != NULL)
	    gtk_widget_set_sensitive(opwidgets[3], channel->chanop);
    }

    return TRUE;
}

static gboolean channel_destroyed(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    if (dialog != NULL && gtk_object_get_data(GTK_OBJECT(dialog), "channel") == channel)
        gtk_widget_destroy(dialog);

    return TRUE;
}

static gboolean nick_mode_changed(CHANNEL_REC *channel, NICK_REC *nick)
{
    if (g_strcasecmp(nick->nick, channel->server->nick) == 0)
    {
        /* my channel mode changed, maybe opped/deopped.. */
        channel_synced(channel);
    }

    return TRUE;
}

static gboolean banlist_new(BAN_REC *ban)
{
    CHANNEL_REC *channel;
    GtkWidget *banlist;

    if (dialog != NULL)
    {
        banlist = gtk_object_get_data(GTK_OBJECT(dialog), "banlist");
        channel = gtk_object_get_data(GTK_OBJECT(dialog), "channel");
        append_list(GTK_LIST(banlist), ban->ban, ban);
    }

    return TRUE;
}

static gboolean banlist_remove(BAN_REC *ban)
{
    CHANNEL_REC *channel;
    GtkWidget *banlist;

    if (dialog != NULL)
    {
        banlist = gtk_object_get_data(GTK_OBJECT(dialog), "banlist");
        channel = gtk_object_get_data(GTK_OBJECT(dialog), "channel");
        remove_list(GTK_LIST(banlist), ban);
    }

    return TRUE;
}

static gboolean ebanlist_new(BAN_REC *ban)
{
    CHANNEL_REC *channel;
    GtkWidget *ebanlist;

    if (dialog != NULL)
    {
        ebanlist = gtk_object_get_data(GTK_OBJECT(dialog), "ebanlist");
        channel = gtk_object_get_data(GTK_OBJECT(dialog), "channel");
        append_list(GTK_LIST(ebanlist), ban->ban, ban);
    }

    return TRUE;
}

static gboolean ebanlist_remove(BAN_REC *ban)
{
    CHANNEL_REC *channel;
    GtkWidget *ebanlist;

    if (dialog != NULL)
    {
        ebanlist = gtk_object_get_data(GTK_OBJECT(dialog), "ebanlist");
        channel = gtk_object_get_data(GTK_OBJECT(dialog), "channel");
        remove_list(GTK_LIST(ebanlist), ban);
    }

    return TRUE;
}

static gboolean invitelist_new(CHANNEL_REC *invitechannel, gchar *mask)
{
    CHANNEL_REC *channel;
    GtkWidget *invitelist;

    if (dialog != NULL)
    {
        channel = gtk_object_get_data(GTK_OBJECT(dialog), "channel");
        if (channel == invitechannel)
        {
            invitelist = gtk_object_get_data(GTK_OBJECT(dialog), "invitelist");
            append_list(GTK_LIST(invitelist), mask, mask);
        }
    }

    return TRUE;
}

static gboolean invitelist_remove(CHANNEL_REC *invitechannel, gchar *mask)
{
    CHANNEL_REC *channel;
    GtkWidget *invitelist;

    if (dialog != NULL)
    {
        channel = gtk_object_get_data(GTK_OBJECT(dialog), "channel");
        if (channel == invitechannel)
        {
            invitelist = gtk_object_get_data(GTK_OBJECT(dialog), "invitelist");
            remove_list(GTK_LIST(invitelist), mask);
        }
    }

    return TRUE;
}

/* Display channel modes dialog */
void dialog_modes(CHANNEL_REC *channel)
{
    GtkWidget *topicentry, *limitentry, *keyentry;
    GtkWidget *button, *secretbut, *privatebut;
    GtkWidget *banlist, *ebanlist, *invitelist;
    GtkWidget *modebox, *vbox, *hbox;
    GtkWidget *scroll, *frame, *notebook;

    g_return_if_fail(channel != NULL);
    if (channel->type != CHANNEL_TYPE_CHANNEL) return;

    if (dialog != NULL)
    {
        /* Mode box already opened for this channel. */
        gdk_window_raise(dialog->window);
        return;
    }

    dialog = gnome_dialog_new(_("Channel modes"), GNOME_STOCK_BUTTON_OK,
                              GNOME_STOCK_BUTTON_APPLY, GNOME_STOCK_BUTTON_CANCEL, NULL);
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(modebox_ok), GTK_OBJECT(dialog));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(modebox_apply), GTK_OBJECT(dialog));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 2, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
   
    gtk_window_set_policy(GTK_WINDOW(dialog), TRUE, TRUE, FALSE);
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
                       GTK_SIGNAL_FUNC(sig_remove_dialog), &dialog);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    gtk_object_set_data(GTK_OBJECT(dialog), "mode", GINT_TO_POINTER(0));
    gtk_object_set_data(GTK_OBJECT(dialog), "channel", channel);

    /* topic */
    topicentry = gui_create_labelentry(GNOME_DIALOG(dialog)->vbox, _("Topic: "), channel->topic, FALSE);

    hbox = gtk_hbox_new(TRUE, 7);
    gtk_container_border_width(GTK_CONTAINER(hbox), 3);

    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 0);

    /* modes */
    frame = gtk_frame_new(_("Modes"));
    gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, FALSE, 0);

    opwidgets[0] = modebox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(frame), modebox);
    gtk_container_border_width(GTK_CONTAINER(modebox), 3);

    button = new_check_button(dialog, modebox, _("Invite"), CHANMODE_INVITE);
    secretbut = new_check_button(dialog, modebox, _("Secret"), CHANMODE_SECRET);
    privatebut = new_check_button(dialog, modebox, _("Private"), CHANMODE_PRIVATE);
    button = new_check_button(dialog, modebox, _("Moderated"), CHANMODE_MODERATE);
    button = new_check_button(dialog, modebox, _("No external msgs"), CHANMODE_NOMSGS);
    button = new_check_button(dialog, modebox, _("Only ops change topic"), CHANMODE_OP_TOPIC);

    limitentry = gui_create_labelentry(modebox, _("User limit: "), NULL, FALSE);

    if (channel->limit != 0)
    {
        GString *str;

        str = g_string_new(NULL);
        g_string_sprintf(str, "%d", channel->limit);
        gtk_entry_set_text(GTK_ENTRY(limitentry), str->str);
        g_string_free(str, TRUE);
    }

    keyentry = gui_create_labelentry(modebox, _("Channel key: "), channel->key, FALSE);
    gtk_container_border_width(GTK_CONTAINER(topicentry->parent), 5);

    notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(hbox), notebook, TRUE, TRUE, 0);

    /* ban list */
    vbox = gtk_vbox_new(FALSE, 3);
    gtk_container_border_width(GTK_CONTAINER(vbox), 2);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, gtk_label_new(_("Banlist")));

    scroll = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(vbox), scroll, TRUE, TRUE, 0);

    banlist = gtk_list_new();
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), banlist);

    create_list(GTK_LIST(banlist), channel->banlist, TRUE);

    opwidgets[1] = gtk_button_new_with_label(_("Unban"));
    gtk_box_pack_start(GTK_BOX(vbox), opwidgets[1], FALSE, FALSE, 0);
    gtk_signal_connect_object(GTK_OBJECT (opwidgets[1]), "clicked",
                              GTK_SIGNAL_FUNC (remove_ban), GTK_OBJECT(dialog));

    if (channel->server->emode_not_known)
    {
        ebanlist = NULL;
        invitelist = NULL;
        opwidgets[2] = NULL;
        opwidgets[3] = NULL;
    }
    else
    {
        /* exception ban list */
        vbox = gtk_vbox_new(FALSE, 3);
        gtk_container_border_width(GTK_CONTAINER(vbox), 2);

        gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, gtk_label_new(_("Banlist exceptions")));

        scroll = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
                                       GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        gtk_box_pack_start(GTK_BOX(vbox), scroll, TRUE, TRUE, 0);

        ebanlist = gtk_list_new();
        gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), ebanlist);

        create_list(GTK_LIST(ebanlist), channel->ebanlist, TRUE);

        opwidgets[2] = gtk_button_new_with_label(_("Remove"));
        gtk_box_pack_start(GTK_BOX(vbox), opwidgets[2], FALSE, FALSE, 0);
        gtk_signal_connect_object(GTK_OBJECT(opwidgets[2]), "clicked",
                                  GTK_SIGNAL_FUNC(remove_eban), GTK_OBJECT(dialog));

        /* invite list */
        vbox = gtk_vbox_new(FALSE, 3);
        gtk_container_border_width(GTK_CONTAINER(vbox), 2);
        gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, gtk_label_new(_("Invite list")));

        scroll = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
                                       GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        gtk_box_pack_start(GTK_BOX(vbox), scroll, TRUE, TRUE, 0);

        invitelist = gtk_list_new();
        gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), invitelist);

        create_list(GTK_LIST(invitelist), channel->invitelist, FALSE);

        opwidgets[3] = gtk_button_new_with_label(_("Remove"));
        gtk_box_pack_start(GTK_BOX(vbox), opwidgets[3], FALSE, FALSE, 0);
        gtk_signal_connect_object(GTK_OBJECT(opwidgets[3]), "clicked",
                                  GTK_SIGNAL_FUNC(remove_invite), GTK_OBJECT(dialog));
    }

    gtk_object_set_data(GTK_OBJECT(dialog), "banlist", banlist);
    gtk_object_set_data(GTK_OBJECT(dialog), "ebanlist", ebanlist);
    gtk_object_set_data(GTK_OBJECT(dialog), "invitelist", invitelist);
    gtk_object_set_data(GTK_OBJECT(dialog), "topic", topicentry);
    gtk_object_set_data(GTK_OBJECT(dialog), "limit", limitentry);
    gtk_object_set_data(GTK_OBJECT(dialog), "key", keyentry);
    gtk_object_set_data(GTK_OBJECT(dialog), "secretbut", secretbut);
    gtk_object_set_data(GTK_OBJECT(dialog), "privatebut", privatebut);

    channel_synced(channel);
    gtk_widget_show_all(dialog);
}

static gboolean cmd_modes(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    if (*data != '\0')
        channel = channel_find(server, data);

    if (channel == NULL || channel->type != CHANNEL_TYPE_CHANNEL)
        cmd_return_error(CMDERR_CHAN_NOT_FOUND);

    dialog_modes(channel);
    return TRUE;
}

void dialog_modes_init(void)
{
    signal_add("channel sync", (SIGNAL_FUNC) channel_synced);
    signal_add("channel destroyed", (SIGNAL_FUNC) channel_destroyed);
    signal_add("channel mode changed", (SIGNAL_FUNC) channel_synced);
    signal_add("nick mode changed", (SIGNAL_FUNC) nick_mode_changed);
    signal_add("ban new", (SIGNAL_FUNC) banlist_new);
    signal_add("ban remove", (SIGNAL_FUNC) banlist_remove);
    signal_add("ban exception new", (SIGNAL_FUNC) ebanlist_new);
    signal_add("ban exception remove", (SIGNAL_FUNC) ebanlist_remove);
    signal_add("invitelist new", (SIGNAL_FUNC) invitelist_new);
    signal_add("invitelist remove", (SIGNAL_FUNC) invitelist_remove);
    command_bind("modes", "GUI commands", (SIGNAL_FUNC) cmd_modes);
}

void dialog_modes_deinit(void)
{
    signal_remove("channel sync", (SIGNAL_FUNC) channel_synced);
    signal_remove("channel destroyed", (SIGNAL_FUNC) channel_destroyed);
    signal_remove("channel mode changed", (SIGNAL_FUNC) channel_synced);
    signal_remove("nick mode changed", (SIGNAL_FUNC) nick_mode_changed);
    signal_remove("ban new", (SIGNAL_FUNC) banlist_new);
    signal_remove("ban remove", (SIGNAL_FUNC) banlist_remove);
    signal_remove("ban exception new", (SIGNAL_FUNC) ebanlist_new);
    signal_remove("ban exception remove", (SIGNAL_FUNC) ebanlist_remove);
    signal_remove("invitelist new", (SIGNAL_FUNC) invitelist_new);
    signal_remove("invitelist remove", (SIGNAL_FUNC) invitelist_remove);
    command_unbind("modes", (SIGNAL_FUNC) cmd_modes);
}
