/*
 mysql.c : mysql support

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <irssi-plugin.h>
#include <mysql/mysql.h>
#include "sql.h"

static PLUGIN_DATA *plugdata;

/* Opens the SQL-connection */
static gboolean plugin_mysql_open(gchar *server, gchar *user, gchar *pass)
{
    plugdata->sql = mysql_connect(NULL, server, user, pass);
    return plugdata->sql != NULL;
}

/* Closes the SQL-connection */
static void plugin_mysql_close(void)
{
    if (plugdata->db != NULL)
    {
	g_free(plugdata->db);
	plugdata->db = NULL;
    }

    if (plugdata->sql != NULL)
    {
        mysql_close(plugdata->sql);
        plugdata->sql = NULL;
    }
}

/* Select database to use */
gboolean plugin_sql_select_db(gchar *db)
{
    if (plugdata->db != NULL && strcmp(plugdata->db, db) == 0)
	return TRUE;

    if (plugdata->db != NULL)
	g_free(plugdata->db);
    plugdata->db = g_strdup(db);

    return mysql_select_db(plugdata->sql, db) != -1;
}

/* Makes an SQL-query. Returns the SQL-result set (NULL = Error). */
gpointer plugin_sql_query(gchar *query, ...)
{
    MYSQL_RES *res;
    va_list valist;
    gchar *buf;
    gint ret;

    g_return_val_if_fail(query != NULL, NULL);
    g_return_val_if_fail(plugdata->sql != NULL, NULL);

/* Construct the query */
    va_start(valist, query);
    buf = g_strdup_vprintf(query, valist);
    va_end(valist);

/* Make the query */
    ret = mysql_query(plugdata->sql, buf);
    g_free(buf);
    if (ret == -1) return NULL;

/* Get the result */
    res = mysql_store_result(plugdata->sql);
    return res;
}

/* Gets the number of fields queried. (-1 = Error) */
gint plugin_sql_numfields(gpointer res)
{
    g_return_val_if_fail(res != NULL, -1);

    return mysql_num_fields((MYSQL_RES *) res);
}

/* Gets the number of rows queried. (-1 = Error) */
gint plugin_sql_numrows(gpointer res)
{
    g_return_val_if_fail(res != NULL, -1);

    return mysql_num_rows((MYSQL_RES *) res);
}

/* Fetches the next row (NULL is error/end of list) */
SQL_ROW plugin_sql_fetchrow(gpointer res)
{
    g_return_val_if_fail(res != NULL, NULL);

    return mysql_fetch_row((MYSQL_RES *) res);
}

/* Frees the result set. */
void plugin_sql_free(gpointer res)
{
    if (res != NULL)
        mysql_free_result((MYSQL_RES *) res);
}

gboolean plugin_sql_mysql_init(PLUGIN_REC *plugin)
{
    PLUGIN_DATA *data = plugin->data;

    plugdata = data;
    return plugin_mysql_open(data->server, data->user, data->pass);
}

void plugin_sql_mysql_deinit(PLUGIN_REC *plugin)
{
    plugin_mysql_close();
}
