/*
 users.c : IRC bot plugin for irssi - user handling

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#define _XOPEN_SOURCE
#include "bot.h"

#define WRITE_USERS_INTERVAL (60*15)

/* Keep these in the same order as USER_xxx flags */
static gchar *user_flags = "oavm";
static PLUGIN_DATA *plugdata;

static gint writeusers_tag;

static gint user_flags2value(gchar *flags)
{
    gchar *pos;
    gint val;

    g_return_val_if_fail(flags != NULL, 0);

    val = 0;
    while (*flags != '\0')
    {
	pos = strchr(user_flags, *flags);
	if (pos != NULL)
	    val |= 1 << (gint) (pos-user_flags);
        flags++;
    }

    return val;
}

static gchar *user_value2flags(gint value)
{
    gchar *str, *p;
    gint n;

    p = str = g_malloc(USER_FLAGS+1);
    for (n = 0; n < USER_FLAGS; n++)
    {
	if (value & (1 << n))
	    *p++ = user_flags[n];
    }
    *p = '\0';

    return str;
}

USER_MASK_REC *botuser_add_mask(USER_REC *user, gchar *mask)
{
    USER_MASK_REC *rec;

    rec = g_new0(USER_MASK_REC, 1);
    rec->mask = g_strdup(mask);

    user->masks = g_list_append(user->masks, rec);
    return rec;
}

static void read_users(PLUGIN_DATA *data)
{
    proplist_t prop, pusers, pvalue, pmask, pmasks, pchans;
    gint num, num2, max, max2;
    gchar *fname, *value;
    USER_REC *user;
    USER_CHAN_REC *userchan;
    USER_MASK_REC *usermask;

    /* Read users from ~/.irssi/users */
    fname = g_strdup_printf("%s/.irssi/users", g_get_home_dir());
    prop = PLGetProplistWithPath(fname);
    g_free(fname);

    if (prop == NULL)
    {
	/* No users.. */
	return;
    }

    pusers = config_get_prop(prop, "users");
    max = pusers == NULL ? 0 : PLGetNumberOfElements(pusers);

    for (num = 0; num < max; num++)
    {
	/* Read record, key = nick */
	pvalue = PLGetArrayElement(pusers, num);
	if (pvalue == NULL) continue; /* hm?? */

	value = config_get_str(pvalue, "nick", NULL);
	if (value == NULL) continue; /* nick is required! */

	/* Add new user */
	user = g_new0(USER_REC, 1);
	user->nick = g_strdup(value);
	g_hash_table_insert(data->users, user->nick, user);

	/* password, flags */
	value = config_get_str(pvalue, "password", NULL);
	user->password = value == NULL ? NULL : g_strdup(value);
	value = config_get_str(pvalue, "flags", "");
	user->flags = user_flags2value(value);

	/* get masks */
        user->masks = NULL;
	pmasks = config_get_prop(pvalue, "masks");
	max2 = pmasks == NULL ? 0 : PLGetNumberOfElements(pmasks);
	for (num2 = 0; num2 < max2; num2++)
	{
	    pmask = PLGetArrayElement(pmasks, num2);
	    if (pmask == NULL) continue;

	    value = config_get_str(pmask, "mask", NULL);
	    if (value == NULL) continue; /* mask is required */

	    usermask = botuser_add_mask(user, value);
	    value = config_get_str(pmask, "not_flags", "");
	    usermask->not_flags = user_flags2value(value);
	}

	/* get channels - must be last, messes up pvalue */
	user->channels = g_hash_table_new((GHashFunc) g_istr_hash, (GCompareFunc) g_istr_equal);
	pchans = config_get_prop(pvalue, "channels");
	max2 = pchans == NULL ? 0 : PLGetNumberOfElements(pchans);
	for (num2 = 0; num2 < max2; num2++)
	{
	    pvalue = PLGetArrayElement(pchans, num2);
	    if (pvalue == NULL) continue;

	    value = config_get_str(pvalue, "channel", NULL);
	    if (value == NULL) continue; /* channel is required */

	    /* create user channel specific record */
	    userchan = g_new0(USER_CHAN_REC, 1);
	    userchan->channel = g_strdup(value);
	    g_hash_table_insert(user->channels, userchan->channel, userchan);

	    value = config_get_str(pvalue, "flags", "");
	    userchan->flags = user_flags2value(value);
	}
    }

    PLRelease(prop);
}

/* save channel specific user record */
static void prop_add_user_chan(gchar *key, USER_CHAN_REC *rec, proplist_t *pchans)
{
    proplist_t dict;
    gchar *str;

    dict = PLMakeDictionaryFromEntries(NULL, NULL);
    *pchans = PLAppendArrayElement(*pchans, dict);

    dict = config_set_str(dict, "channel", rec->channel);

    if (rec->flags != 0)
    {
        str = user_value2flags(rec->flags);
        dict = config_set_str(dict, "flags", str);
        g_free(str);
    }
}

/* save user record */
static void prop_add_user(gchar *key, USER_REC *user, proplist_t *pusers)
{
    proplist_t dict, subdict, pmasks, pchans, pkey;
    GList *tmp;
    gchar *str;

    dict = PLMakeDictionaryFromEntries(NULL, NULL);
    *pusers = PLAppendArrayElement(*pusers, dict);

    /* nick, flags, password */
    dict = config_set_str(dict, "nick", user->nick);
    if (user->flags != 0)
    {
	gchar *str;

	str = user_value2flags(user->flags);
	dict = config_set_str(dict, "flags", str);
	g_free(str);
    }
    if (user->password != NULL)
	dict = config_set_str(dict, "password", user->password);

    if (user->masks != NULL)
    {
	/* Save masks */
	pmasks = PLMakeArrayFromElements(NULL);
	pkey = PLMakeString("masks");
	subdict = PLInsertDictionaryEntry(dict, pkey, pmasks);
	PLRelease(pkey);

	for (tmp = user->masks; tmp != NULL; tmp = tmp->next)
	{
	    USER_MASK_REC *rec = tmp->data;

	    subdict = PLMakeDictionaryFromEntries(NULL, NULL);
	    pmasks = PLAppendArrayElement(pmasks, subdict);

	    subdict = config_set_str(subdict, "mask", rec->mask);
	    if (rec->not_flags != 0)
	    {
		str = user_value2flags(rec->not_flags);
		subdict = config_set_str(subdict, "not_flags", str);
		g_free(str);
	    }
	}
    }

    /* Save channels */
    if (g_hash_table_size(user->channels) > 0)
    {
	pchans = PLMakeArrayFromElements(NULL);
	pkey = PLMakeString("channels");
	subdict = PLInsertDictionaryEntry(dict, pkey, pchans);
	PLRelease(pkey);

	g_hash_table_foreach(user->channels, (GHFunc) prop_add_user_chan, &pchans);
    }
}

static void write_users(PLUGIN_DATA *data)
{
    proplist_t prop, pfname, pusers;
    gchar *fname;

    /* Write users to ~/.irssi/users */
    prop = PLMakeDictionaryFromEntries(NULL, NULL);

    fname = g_strdup_printf("%s/.irssi/users", g_get_home_dir());
    pfname = PLMakeString(fname);
    prop = PLSetFilename(prop, pfname);
    PLRelease(pfname);
    g_free(fname);

    if (g_hash_table_size(data->users) > 0)
    {
	pusers = config_list_section(&prop, "users");
	g_hash_table_foreach(data->users, (GHFunc) prop_add_user, &pusers);
    }

    PLSave(prop, TRUE);
    PLRelease(prop);
}

static gboolean botuser_find_mask(USER_REC *user, gchar *nick, gchar *host)
{
    GList *tmp;

    g_return_val_if_fail(user != NULL, FALSE);
    g_return_val_if_fail(nick != NULL, FALSE);
    g_return_val_if_fail(host != NULL, FALSE);

    /* Check that masks match */
    for (tmp = user->masks; tmp != NULL; tmp = tmp->next)
    {
	USER_MASK_REC *rec = tmp->data;

	if (irc_mask_match_address(rec->mask, nick, host))
	{
	    user->not_flags = rec->not_flags;
	    return TRUE;
	}
    }

    return FALSE;
}

static void botuser_getusers_hash(gpointer key, USER_REC *user, GList **list)
{
    *list = g_list_append(*list, user);
}

USER_REC *botuser_find(gchar *nick, gchar *host)
{
    USER_REC *user;
    gchar *tnick;
    GList *users, *tmp;

    g_return_val_if_fail(nick != NULL, NULL);

    /* First check for user with same nick */
    tnick = nick_strip(nick);
    user = g_hash_table_lookup(plugdata->users, tnick);
    g_free(tnick);

    if (user != NULL && host != NULL && !botuser_find_mask(user, nick, host))
    {
	/* mask didn't match, check for more.. */
	user = NULL;
    }

    if (user == NULL && host != NULL)
    {
	/* Check for different nicks.. */
	users = NULL;
	g_hash_table_foreach(plugdata->users, (GHFunc) botuser_getusers_hash, &users);
	for (tmp = users; tmp != NULL; tmp = tmp->next)
	{
	    if (botuser_find_mask(tmp->data, nick, host))
	    {
		user = tmp->data;
		break;
	    }
	}
	g_list_free(users);
    }

    return user;
}

void botuser_set_password(USER_REC *user, gchar *password)
{
    gchar *pass, salt[3];

    g_return_if_fail(user != NULL);
    g_return_if_fail(password != NULL);

    salt[0] = rand()%20 + 'A';
    salt[1] = rand()%20 + 'A';
    salt[2] = '\0';
    pass = crypt(password, salt);

    if (user->password != NULL) g_free(user->password);
    user->password = g_strdup(pass);
}

gboolean botuser_verify_password(USER_REC *user, gchar *password)
{
    gchar *pass, salt[3];

    g_return_val_if_fail(user != NULL, FALSE);
    g_return_val_if_fail(password != NULL, FALSE);

    if (user->password == NULL || strlen(user->password) < 3)
	return FALSE;

    salt[0] = user->password[0];
    salt[1] = user->password[1];
    salt[2] = '\0';
    pass = crypt(password, salt);
    return strcmp(user->password, pass) == 0;
}

static gboolean event_massjoin(CHANNEL_REC *channel, GList *nicks)
{
    USER_REC *user;
    GList *users;

    g_return_val_if_fail(channel != NULL, FALSE);
    g_return_val_if_fail(nicks != NULL, FALSE);

    users = NULL;
    for (; nicks != NULL; nicks = nicks->next)
    {
	NICK_REC *rec = nicks->data;

	user = botuser_find(rec->nick, rec->host);
	if (user != NULL)
	{
	    user->nickrec = rec;
	    users = g_list_append(users, user);
	}
    }

    if (users != NULL)
    {
	signal_emit("bot massjoin", 2, channel, users);
	g_list_free(users);
    }

    return TRUE;
}

/* channel synced - find everyone's NICK_REC's */
static gboolean sig_channel_sync(CHANNEL_REC *channel)
{
    USER_REC *user;
    GList *tmp, *nicks;

    g_return_val_if_fail(channel != NULL, FALSE);

    nicks = nicklist_getnicks(channel);
    for (tmp = nicks; tmp != NULL; tmp = tmp->next)
    {
	NICK_REC *rec = tmp->data;

	if (rec->send_massjoin)
	    continue; /* This will be checked in "massjoin" signal */

	user = botuser_find(rec->nick, rec->host);
	if (user != NULL)
	    user->nickrec = rec;
    }
    g_list_free(nicks);

    return TRUE;
}

/* user left channel - remove from users record */
static gboolean sig_nicklist_remove(CHANNEL_REC *channel, NICK_REC *nick)
{
    USER_REC *user;

    g_return_val_if_fail(channel != NULL, FALSE);

    user = botuser_find(nick->nick, nick->host);
    if (user != NULL)
	user->nickrec = NULL;
    return TRUE;
}

/* Free memory used by user channel record */
static void user_destroy_chan(gchar *key, USER_CHAN_REC *rec)
{
    g_free(rec->channel);
    g_free(rec);
}

static void usermask_destroy(USER_MASK_REC *rec)
{
    g_free(rec->mask);
    g_free(rec);
}

/* Free memory used by user record */
static void user_destroy(gchar *key, USER_REC *user)
{
    g_list_foreach(user->masks, (GFunc) usermask_destroy, NULL);
    g_list_free(user->masks);

    g_hash_table_foreach(user->channels, (GHFunc) user_destroy_chan, NULL);
    g_hash_table_destroy(user->channels);

    g_free(user->nick);
    if (user->password != NULL) g_free(user->password);
    g_free(user);
}

static gint sig_write_users(PLUGIN_DATA *data)
{
    if (data->last_write+WRITE_USERS_INTERVAL <= time(NULL))
    {
	data->last_write = time(NULL);
	write_users(data);
    }
    return 1;
}

void plugin_bot_users_init(PLUGIN_REC *plugin)
{
    PLUGIN_DATA *data = plugin->data;

    plugdata = data;
    data->last_write = time(NULL);
    data->users = g_hash_table_new((GHashFunc) g_istr_hash, (GCompareFunc) g_istr_equal);
    read_users(data);

    writeusers_tag = gui_timeout_add(10000, (GUITimeoutFunction) sig_write_users, data);

    plugin_bind(plugin, "massjoin", (SIGNAL_FUNC) event_massjoin, TRUE);
    plugin_bind(plugin, "channel sync", (SIGNAL_FUNC) sig_channel_sync, TRUE);
    plugin_bind(plugin, "nicklist remove", (SIGNAL_FUNC) sig_nicklist_remove, TRUE);
}

void plugin_bot_users_deinit(PLUGIN_REC *plugin)
{
    PLUGIN_DATA *data = plugin->data;

    gui_timeout_remove(writeusers_tag);
    write_users(data);

    g_hash_table_foreach(data->users, (GHFunc) user_destroy, NULL);
    g_hash_table_destroy(data->users);
}
