/**************************************************************************** 
** File: udp.c
**
** Author: Mike Borella
**
** Comments: Dump UDP header information
**
*****************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <netdb.h>
#include "config.h"
#include "udp.h"
#include "rsip.h"
#include "dns.h"
#include "l2tp.h"
#include "dhcp.h"
#include "rip.h"
#include "mgcp.h"
#include "sip.h"
#include "isakmp.h"
#include "payload.h"
#include "addrtoname.h"
#include "error.h"

extern u_char *packet_end;
extern struct arg_t *my_args;

/*----------------------------------------------------------------------------
**
** dump_udp()
**
** Parse UDP header and dump fields
**
**----------------------------------------------------------------------------
*/

void dump_udp(u_char *bp, int length)
{
  UDPHdr *up;
  u_char *ep = bp + length;
  u_short sport, dport, len;
  struct servent *se;
  
  /*
   * Make sure we don't run off the end of the packet
   */

  if (ep > packet_end) ep = packet_end;

  /*
   * Check for truncated packet
   */

  if (length < sizeof(UDPHdr))
    {
      printf("Truncated header, length = %d bytes\n", length);
      return;
    }

  up = (UDPHdr *) bp;
  sport = ntohs(up->uh_src);
  dport = ntohs(up->uh_dst);
  len = ntohs(up->uh_len);

  /*
   * Dump UDP header info
   */
  
  printf("-----------------------------------------------------------------\n");
  printf("                        UDP Header\n");
  printf("-----------------------------------------------------------------\n");

  if (!my_args->t)
    {  
      printf("Source port:            %d", sport);
      if (sport < 1024) 
	printf(" (%s)\n", udpport_string(sport));
      else 
	printf("\n");
      
      printf("Destination port:       %d", dport);
      if (dport < 1024) 
	printf(" (%s)\n", udpport_string(dport));
      else 
	printf("\n");
      
      if (!my_args->m)
	{
	  printf("Length:                 %d\n", len);
	  printf("Checksum:               %d\n", ntohs(up->uh_chk));
	}
    }


  /*
   * Take care of DNS packets
   */

  se = getservbyname("domain", "udp");
  if (se == NULL) 
    GWF_error_system("can't get services entries");
  if (sport == ntohs(se->s_port) || dport == ntohs(se->s_port))
    dump_dns((u_char *) bp + sizeof(UDPHdr), len);

  /*
   * Take care of L2TP packets
   */

  if (sport == L2TP_PORT || dport == L2TP_PORT)
    dump_l2tp((u_char *) bp + sizeof(UDPHdr), len);

  /*
   * Take care of DHCP packets
   */

  if (sport == DHCP_CLIENT_PORT || dport == DHCP_CLIENT_PORT)
    dump_dhcp((u_char *) bp + sizeof(UDPHdr), len);

  /*
   * Take care of RIP packets
   */

  if (sport == RIP_PORT || dport == RIP_PORT)
    dump_rip((u_char *) bp + sizeof(UDPHdr), len);


  /*
   * Take care of SIP packets
   */

  if (sport == SIP_PORT || dport == SIP_PORT)
    dump_sip((u_char *) bp + sizeof(UDPHdr), len);

  /*
   * Take care of IKE packets
   */

  if (sport == ISAKMP_PORT || dport == ISAKMP_PORT)
    dump_isakmp((u_char *) bp + sizeof(UDPHdr), len);

  /*
   * Take care of RSIP packets
   */

  if (sport == RSIP_PORT || dport == RSIP_PORT)
    dump_rsip((u_char *) bp + sizeof(UDPHdr), len);

  /*
   * Take care of MGCP packets
   */

  if ( is_mgcp((u_char *) bp + sizeof(UDPHdr), len) )
     dump_mgcp((u_char *) bp + sizeof(UDPHdr), len);

  /*
   * print payload if there is one
   */

  if (my_args->p && len > 0) 
    dump_payload((u_char *) bp + sizeof(UDPHdr), len);

  return;
}
