#include "hx_types.h"
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <termcap.h>
#include <unistd.h>
#include <signal.h>
#include <sys/ioctl.h>
#include "term.h"
#include "screen.h"

int LI = 24, CO = 80;
char term_name[256] = "vt100", *HO, *ME, *MR, *US, *CL, *CE, *CM, *CS, *SF, *SR, *DC, *IC;
static char termcap[2048], tbuf[2048];

#define ttyfd 0

static struct termios old_tty, new_tty;

#define term_putc ((int(*)(int))__term_putc)

static void
__term_putc (int c)
{
	char ch = (char)c;

	write(1, &ch, 1);
}

int
term_init (void)
{
	char *ep, *bp = tbuf;

	if ((ep = getenv("TERM"))) {
		strncpy(term_name, ep, sizeof term_name - 1);
		term_name[sizeof term_name - 1] = 0;
	}
	switch (tgetent(termcap, term_name)) {
		case -1:
			term_printf("tgetent had trouble accessing the termcap database\n");
			return -1;
		case 0:
			term_printf("tgetent did not find an entry for '%s'\n", term_name);
			return -1;
	}
	if (!(ep = getenv("COLUMNS")) || !(CO = atoi(ep))) {
		if ((CO = tgetnum("co")) == -1)
			CO = 80;
	}
	if (!(ep = getenv("LINES")) || !(LI = atoi(ep))) {
		if ((LI = tgetnum("li")) == -1)
			LI = 24;
	}

	HO = tgetstr("ho", &bp);
	ME = tgetstr("me", &bp);
	MR = tgetstr("mr", &bp);
	US = tgetstr("us", &bp);
	CL = tgetstr("cl", &bp);
	CM = tgetstr("cm", &bp);
	CE = tgetstr("ce", &bp);
	CS = tgetstr("cs", &bp);
	DC = tgetstr("dc", &bp);
	IC = tgetstr("ic", &bp);
	SF = tgetstr("sf", &bp);
	SR = tgetstr("sr", &bp);

#ifndef __hpux__
	ep = tgetstr("pc", &bp);
	PC = ep ? *ep : 0;
	BC = tgetstr("le", &bp);
	UP = tgetstr("up", &bp);
#endif

	tcgetattr(ttyfd, &old_tty);
	new_tty = old_tty;
	new_tty.c_cc[VMIN] = 1;
	new_tty.c_cc[VTIME] = 0;
	new_tty.c_lflag &= ~(ICANON|ECHO);
	new_tty.c_iflag &= ~IXON;
	tcsetattr(ttyfd, TCSADRAIN, &new_tty);

	term_install_signal_handlers();

	return ttyfd;
}

void
term_puts (const char *buf, unsigned int len)
{
	write(1, buf, len);
}

void
term_printf (const char *fmt, ...)
{
	va_list ap;
	char buf[4096];
	int len;

	va_start(ap, fmt);
	if ((len = vsnprintf(buf, sizeof buf, fmt, ap)) == -1)
		len = sizeof buf;
	va_end(ap);

	write(1, buf, (unsigned)len);
}

#ifndef RETSIGTYPE
#define RETSIGTYPE void
#endif

void
term_install_signal_handlers (void)
{
	struct sigaction act;

	memset(&act, 0, sizeof act);

	act.sa_handler = SIG_IGN;
	sigaction(SIGPIPE, &act, 0);
#ifdef IRIX
	sigaction(SIGFPE, &act, 0);
#endif

	act.sa_handler = (RETSIGTYPE(*)(int))term_sigcont;
	sigaction(SIGINT, &act, 0);
	sigaction(SIGCONT, &act, 0);

	act.sa_handler = (RETSIGTYPE(*)(int))term_sigtstp;
	sigaction(SIGTSTP, &act, 0);

	act.sa_handler = (RETSIGTYPE(*)(int))term_sigwinch;
	sigaction(SIGWINCH, &act, 0);
}

void
term_set_scroll_region (int first, int last)
{
	tputs(tgoto(CS, last, first), 0, term_putc);
}

void
term_scroll_up (void)
{
	tputs(SR, 0, term_putc);
}

void
term_scroll_down (void)
{
	tputs(SF, 0, term_putc);
}

void
term_ce (void)
{
	tputs(CE, 0, term_putc);
}

void
term_clear (void)
{
	tputs(CL, 0, term_putc);
}

void
term_home (void)
{
	tputs(HO, 0, term_putc);
}

void
term_goto_line (int line)
{
	tputs(tgoto(CM, 0, line), 0, term_putc);
}

void
term_move_cursor (int line, int column)
{
	tputs(tgoto(CM, column, line), 0, term_putc);
}

void
term_char_delete (void)
{
	tputs(DC, 0, term_putc);
}

void
term_char_insert (void)
{
	tputs(IC, 0, term_putc);
}

void
term_mode_clear (void)
{
	tputs(ME, 0, term_putc);
}

void
term_mode_underline (void)
{
	tputs(US, 0, term_putc);
}

void
term_mode_reverse (void)
{
	tputs(MR, 0, term_putc);
}

void
term_reset (void)
{
	tcsetattr(ttyfd, TCSADRAIN, &old_tty);
	term_set_scroll_region(0, -1);
	term_move_cursor(LI, 0);
}

RETSIGTYPE
term_sigwinch (void)
{
	struct winsize win_size;
	char LINES[24], COLUMNS[24];

	if (ioctl(1, TIOCGWINSZ, &win_size))
		goto ret;

	LI = win_size.ws_row;
	CO = win_size.ws_col;
#ifdef USE_SETENV
	sprintf(LINES, "%d", win_size.ws_row);
	setenv("LINES", LINES, 1);
	sprintf(COLUMNS, "%d", win_size.ws_col);
	setenv("COLUMNS", COLUMNS, 1);
#else
	sprintf(LINES, "LINES=%d", win_size.ws_row);
	putenv(LINES);
	sprintf(COLUMNS, "COLUMNS=%d", win_size.ws_col);
	putenv(COLUMNS);
#endif
	scr_resize_all(LI - 4, CO);
	term_set_scroll_region(0, (signed)curscr->li);
	term_move_cursor(LI, curscr->history.cur->pos % CO);
ret:
	scr_draw(curscr);
	{}
}

RETSIGTYPE
term_sigtstp (void)
{
	tcgetattr(ttyfd, &new_tty);
	term_reset();
	write(1, "\r", 1);
	kill(getpid(), SIGSTOP);
}

RETSIGTYPE
term_sigcont (void)
{
	tcsetattr(ttyfd, TCSADRAIN, &new_tty);
	term_set_scroll_region(0, (signed)curscr->li);
	scr_draw(curscr);
}

/* from BitchX/source/misc.c */

#ifdef REVERSE_WHITE_BLACK
char const *colour_str[] = {
"[0m","[0;34m","[0;32m","[0;36m","[0;31m","[0;35m","[0;33m","[0;30m",
"[1;37m","[1;34m","[1;32m","[1;36m","[1;31m","[1;35m","[1;33m","[1;30m", "[0m",
"[0;47m", "[0;41m", "[0;42m","[0;43m", "[0;44m","[0;45m","[0;46m", "[0;40m",
"[1;47m", "[1;41m", "[1;42m","[1;43m", "[1;44m","[1;45m","[1;46m", "[1;40m",
"[7m", "[1m", "[5m", "[4m"};
#else
char const *colour_str[] = {
"[0;30m","[0;34m","[0;32m","[0;36m","[0;31m","[0;35m","[0;33m","[0m",
"[1;30m","[1;34m","[1;32m","[1;36m","[1;31m","[1;35m","[1;33m","[1;37m", "[0m",
"[0;40m", "[0;41m", "[0;42m","[0;43m", "[0;44m","[0;45m","[0;46m", "[0;47m",
"[1;40m", "[1;41m", "[1;42m","[1;43m", "[1;44m","[1;45m","[1;46m", "[1;47m",
"[7m", "[1m", "[5m", "[4m"};
#endif
