// Main file for the Darxite library, used by client programs
// to communicate with and control the daemon process.

#include <includes.h>

#include "global.h"
#include "files.h"
#include "sockets.h"

void DX_SayHello(void)
{
    dl_error(E_INFO, "Hello from the Darxite library");
}

void DX_ReadConfigFiles(void)
{
    dl_ReadConfigFiles();
}

// Per-client connection to daemon
int DX_ConnectClient(const char *name)
{
    return dl_ConnectClient(name);
}

int DX_ConnectRemoteClient(const char *host, int port, const char *password,
                           const char *name)
{
    return dl_ConnectRemoteClient(host, port, password, name);
}

// Close a client's connection to the daemon
int DX_DisconnectClient(int fd)
{
    if (dprintf(fd, "Disconnect\n") < 0)
    {
        DX_errno = errno;
        close(fd);
        return DX_LIB_ERROR;
    }
    
    close(fd);
    
    return DX_LIB_OK;
}

// Get a response from the daemon with a timeout (select is your friend)
int DX_GetResponse(int fd, char *response, int response_max, int timeout)
{
    fd_set read_fd_set;
    BOOL got_response = FALSE;
    struct timeval time_struct = { timeout, 0 };
    int i, rc, nbytes, total_bytes = 0;
    char *resp_ptr = response;

    memset(response, 0, response_max);
    while(1) {
	FD_ZERO(&read_fd_set);
	FD_SET(fd, &read_fd_set);
	
	rc = select(FD_SETSIZE, &read_fd_set, NULL, NULL, &time_struct);

	if (rc < 0)
	{
	    // select() returned an error
	    DX_errno = errno;
	    return DX_LIB_ERROR;
	}
	if(rc == 0) break;

	// now we have some data, let's read it into the buffer
        nbytes = read(fd, response + total_bytes, response_max - total_bytes);
        
	if (nbytes < 0)
        {
            DX_errno = errno;
            return DX_LIB_ERROR;
        }

        // our socket's been closed if we read 0 bytes.  if we got a
	// response, we'll just break out--we still want to give the
	// response; we'll let the next call to this function give the
	// disconnected error.
	if (nbytes == 0)
        {
	    if(got_response) break;
            return DX_LIB_DISCONNECTED;
        }
        
	if (nbytes > 0)
        {
            total_bytes += nbytes;
            // a null signifies the end of response - but we need to check
            // whether there's another response to append
            if (response[total_bytes - 1] == '\0')
            {
                got_response = TRUE;
            }
            else if (total_bytes >= response_max)
            {
                // remove NULLs here as well as below
                for (i = 0; i < response_max; i++)
                {
                    if (response[i] == '\0') continue;
                    
                    *(resp_ptr++) = response[i];
                }
                return DX_LIB_MORE_DATA;
            }
        }
        
	// if we're just checking for another response, we don't want to
        // wait for ages
        if (got_response)
            time_struct.tv_sec = 0;
        else
            time_struct.tv_sec = timeout;
        time_struct.tv_usec = 0;
    }

    // timed out, or stopped waiting for some other reason.
    
    /* rc == 0 */
    if (!got_response)
	return DX_LIB_TIMEOUT;
    /* we got a response */
    {
	char *p = response;
	// get rid of all embedded NULLs because if there are multiple
	// responses, we want to squidge them all together
	//printf("resp then: \"%s\", buffer = \"%s\", count = %d\n",
	//response, buffer, count);
	for (i = 0; i < response_max; i++)
	{
	    if (response[i] == '\0') continue;

	    *(p++) = response[i];
	}

	*(p++) = 0;
	//printf("resp now: \"%s\"", response);
	return DX_LIB_OK;
    }
}

// Read client information from database files
int DX_ReadClientDB(DX_ClientInfo **client_list)
{
    int total;
    char buffer[256];

    *client_list = NULL;
    // Read "~/.darxite/clients" first because we want its clients to
    // override the ones in /etc
    sprintf(buffer, "%s/clients", DX_ProgDir);
    total = dl_ReadClientsFile(buffer, client_list);
    sprintf(buffer, "%s/darxite-clients", DX_GlobalConfigPath());
    total += dl_ReadClientsFile(buffer, client_list);
    return total;
}

// Figures out where clients are - returns count that match
int DX_FindClients(const char *name, const char *type,
                   DX_ClientInfo *search_clients,
                   DX_ClientInfo **found_clients)
{
    DX_ClientInfo *client = search_clients, *found_client = NULL;
    int count = 0;

    *found_clients = NULL;
    while (client)
    {
        if (((name == NULL) || !strcasecmp(name, client->Name)) &&
            ((type == NULL) || !strcasecmp(type, client->Type)))
        {
            if (found_client == NULL)
            {
                *found_clients = malloc(sizeof(DX_ClientInfo));
                found_client = *found_clients;
                found_client->Prev = NULL;
            }
            else
            {
                found_client->Next = malloc(sizeof(DX_ClientInfo));
                found_client->Next->Prev = found_client;
                found_client = found_client->Next;
            }
            found_client->Next = NULL;
            found_client->Name = strdup(client->Name);
            found_client->Path = strdup(client->Path);
            found_client->Type = strdup(client->Type);
            found_client->Description = strdup(client->Description);
            //printf("Found name \"%s\", path \"%s\", type \"%s\", comment "
            //       "\"%s\"\n", client->Name, client->Path, client->Type,
            //       client->Description);
            count++;
        }
        client = client->Next;
    }
    return count;
}

// Frees the memory allocated for a client list
void DX_FreeClientList(DX_ClientInfo *client_list)
{
    DX_ClientInfo *client = client_list, *next_client;

    while (client)
    {
        next_client = client->Next;
        free(client->Name);
        free(client->Path);
        free(client->Type);
        free(client->Description);
        free(client);
        client = next_client;
    }
}

const char *DX_GlobalConfigPath(void)
{
    return ETCPATH;
}

const char *DX_BinPath(void)
{
    return BINPATH;
}

// Read a (possibly multi-line) "line" from a config file
char *DX_ReadFileLine(char *s, int size, FILE *stream)
{
    return dl_ReadFileLine(s, size, stream);
}

// Strip comments and whitespace from a line
char *DX_StripComments(char *line)
{
    return dl_StripComments(line);
}

// Strip a | separated line into fields
char *DX_GetNextField(char *line)
{
    return dl_GetNextField(line);
}

