// File containing file-related functions that are used
// by the library itself and NOT the user

#include <darxite.h>
#include "files.h"

// Read a "line" (which may be multi-line...) from a file.
// Note that the parameters etc. are as much like fgets
// as possible.
char *dl_ReadFileLine(char *s, int size, FILE *stream)
{
    char c;
    char *buffer;
    int count = 0;
    BOOL ignore_space = FALSE;

    buffer = malloc(size);
    while (((c = fgetc(stream)) != EOF) && (count < size))
    {
        if (ignore_space && isspace(c))
            continue;
	    
        ignore_space = FALSE;
        
        /* !ignore_space */
        if (c == '\\')
        {
            /* skip following whitespace */
            ignore_space = TRUE;
            continue;
        }
        buffer[count] = c;
        count++;
        if (c == '\n')
            break;
    }
    if (count == 0)
    {
        free(buffer);
        return NULL;
    }
    buffer[count] = 0;
    dl_error(E_TRACE, "Read line \"%s\"", buffer);
    strncpy(s, buffer, size);
    free(buffer);
    return s;
}

char *dl_StripComments(char *line)
{
    char *line_ptr = line;
    
    while (*line_ptr && isspace(*line_ptr))
        line_ptr++;
    if (!*line_ptr || (*line_ptr == '#'))
        return NULL;
    else
        return line_ptr;
}

// A function to get the next field, where the separator
// is a |. Note that, if the field is empty or it is the
// end of the line, an empty string will be returned. It
// changes the string you pass it (by inserting nulls into
// it) and a limitation is that the line should be terminated
// by lots of null characters. This is a bit lame, but allows
// it to be thread-safe. Also, it handles quotes correctly,
// but does not remove them for you.
char *dl_GetNextField(char *line)
{
    char *field_ptr = line, *end_ptr;
    BOOL quoted = FALSE;

    //dl_error(E_TRACE, "Line I received: \"%s\"", line);
    // Strip leading whitespace & field separator
    while (*field_ptr && isspace(*field_ptr))
        field_ptr++;
    if (*field_ptr == '|')
        field_ptr++;
    while (*field_ptr && isspace(*field_ptr))
        field_ptr++;

    if (*field_ptr == '\0')
    {
        dl_error(E_TRACE, "Read end of line");
        return field_ptr;
    }
    // if the field is empty then we have to backtrack
    if (*field_ptr == '|')
    {
        dl_error(E_TRACE, "Read empty field");
        field_ptr--;
        *field_ptr = '\0';
        return field_ptr;
    }
    end_ptr = field_ptr;
    while (*end_ptr && (quoted || (!isspace(*end_ptr) && (*end_ptr != '|'))))
    {
        if (*end_ptr == '"')
        {
            if (quoted)
                quoted = FALSE;
            else
                quoted = TRUE;
        }
        end_ptr++;
    }
    *end_ptr = '\0';
    dl_error(E_TRACE, "Read field \"%s\"", field_ptr);    
    return field_ptr;
}

// Returns the number of clients read
int dl_ReadClientsFile(const char *filename, DX_ClientInfo **client_list)
{
    FILE *file;
    char buffer[256], *buf_ptr, *field_ptr;
    int client_count = 0;
    DX_ClientInfo *client = *client_list;

    file = fopen(filename, "r");
    if (file == NULL)
        return 0;
    
    // find the last client in the list
    while (client && client->Next)
        client = client->Next;
    
    while (DX_ReadFileLine(buffer, sizeof(buffer), file))
    {
	strip(buffer);
            
        if (!(buf_ptr = DX_StripComments(buffer)))
            continue;
        
        //printf("Clients file: Read line \"%s\"\n", buf_ptr);
        if (*client_list == NULL)
        {
            *client_list = malloc(sizeof(DX_ClientInfo));
            client = *client_list;
            client->Prev = NULL;
        }
        else
        {
            client->Next = malloc(sizeof(DX_ClientInfo));
            client->Next->Prev = client;
            client = client->Next;
        }
        client->Next = NULL;
        
        field_ptr = DX_GetNextField(buf_ptr);
        if (*field_ptr)
        {
            if (*field_ptr == '"')
                field_ptr++;
            client->Name = strdup(field_ptr);
            if (lastchr(client->Name) == '"')
                lastchr(client->Name) = '\0';
        }
        field_ptr += strlen(field_ptr) + 1;
        
        field_ptr = DX_GetNextField(field_ptr);
        if (*field_ptr)
        {
            if (*field_ptr == '"')
                field_ptr++;
            client->Path = strdup(field_ptr);
            if (lastchr(client->Path) == '"')
                lastchr(client->Path) = '\0';
        }
        field_ptr += strlen(field_ptr) + 1;
        
        field_ptr = DX_GetNextField(field_ptr);
        if (*field_ptr)
        {
            if (*field_ptr == '"')
                field_ptr++;
            client->Type = strdup(field_ptr);
            if (lastchr(client->Type) == '"')
                lastchr(client->Type) = '\0';
        }
        field_ptr += strlen(field_ptr) + 1;
        
        field_ptr = DX_GetNextField(field_ptr);
        if (*field_ptr)
        {
            if (*field_ptr == '"')
                field_ptr++;
            client->Description = strdup(field_ptr);
            if (lastchr(client->Description) == '"')
                lastchr(client->Description) = '\0';
        }
        
//        error(E_TRACE, "Client name \"%s\", path \"%s\", type \"%s\", "
//                 "description \"%s\"", client->Name, client->Path,
//                 client->Type, client->Description);
        client_count++;
    }
    fclose(file);
    return client_count;
}

void dl_ReadConfigFiles(void)
{
    FILE *file;
    int i;
    char buffer[256];

    if (!strcmp(DX_HomeDir, ""))
        strcpy(DX_HomeDir, getenv("HOME"));
    if (!strcmp(DX_OutputDir, ""))
        strcpy(DX_OutputDir, DX_HomeDir);
    sprintf(DX_EMailAddress, "%s@localhost", getenv("USER"));
    // these variables *have* to be set to the defaults here, in case the
    // user reloads the config files
    strcpy(DX_SpoolDir, "");
    strcpy(DX_OutputToFile, "");
    strcpy(DX_OutputToProgram, "");
    strcpy(DX_FileLogName, "");
    strcpy(DX_ProxyHostName, "");
    strcpy(DX_DisconnectCommand, "");
    DX_EnableMirrors = DX_EnableSpooling = TRUE;
    DX_EnableReadingServerDB = DX_EnableWritingServerDB = TRUE;
    DX_EnableFileLogging = DX_ClearFileLogOnExit = TRUE;
    DX_LeaveCompleteFiles = DX_EnableMassiveDownload = FALSE;
    DX_EnableRenaming = DX_EnableOpenLocalFile = FALSE;
    DX_EnableErrorLogging = DX_LeaveCompleteFiles = FALSE;
    DX_DisconnectOnCompletion = DX_EnableMassiveAutoMirror = FALSE;
    DX_DetectPrematureCompletion = FALSE;
    DX_MassiveLimit = 100; // start massive download after 100K...
    DX_MassiveMaxConnections = 9; // maximum number of extra connections...
    DX_MassiveConnectAgainLimit = 100; // starting a new one for every 100K
    DX_MaxSimDownloads = 0; // no maximum no. of simultaneous downloads
    DX_MinRxBeforeSwitching = 100; // switch when rx rate < 100 bytes/s
    DX_TimeBeforeSwitching = 30;   // switch when no input for 30 seconds
    DX_MaxSimDownloads = -1; // no maximum number of downloads
    DX_SleepInterval = 5; // check for files every 5 seconds
    DX_ProxyPort = 80; // default proxy port
    DX_ListenPort = -1; // don't listen for connections
    DX_BufferSize = (16 * 1024); // default buffer size is 16K
    for (i = 0; i < DX_FileTypeCount; i++)
    {
        free(DX_FileType[0][i]);
        free(DX_FileType[1][i]);
    }
    DX_FileTypeCount = 0;
    
    // Read the settings.
    // We read from the old names first, so any settings in the new names
    // override the old ones.
    sprintf(buffer, "%s/darxiterc", DX_GlobalConfigPath());
    if ((file = fopen(buffer, "r")))
    {
        dl_ReadSettings(file);
        fclose(file);
    }
    sprintf(buffer, "%s/darxite.conf", DX_GlobalConfigPath());
    if ((file = fopen(buffer, "r")))
    {
        dl_ReadSettings(file);
        fclose(file);
    }
    sprintf(buffer, "%s/.darxiterc", DX_HomeDir);
    if ((file = fopen(buffer, "r")))
    {
        dl_ReadSettings(file);
        fclose(file);
    }
    if (!strcmp(DX_ProgDir, ""))
        sprintf(DX_ProgDir, "%s/.darxite", DX_HomeDir);
    MAKEPATH(DX_ProgDir);
    if (lastchr(DX_ProgDir) == '/')
        lastchr(DX_ProgDir) = '\0';
    sprintf(buffer, "%s/conf", DX_ProgDir);
    if ((file = fopen(buffer, "r")))
    {
        dl_ReadSettings(file);
        fclose(file);
    }
    
    if (!strcmp(DX_SpoolDir, ""))
        sprintf(DX_SpoolDir, "%s/spool", DX_ProgDir);
    
    // remove trailing /s
    if (lastchr(DX_HomeDir) == '/')
        lastchr(DX_HomeDir) = '\0';
    if (lastchr(DX_SpoolDir) == '/')
        lastchr(DX_SpoolDir) = '\0';
    if (lastchr(DX_OutputDir) == '/')
        lastchr(DX_OutputDir) = '\0';
    
    if (!strcmp(DX_FileLogName, ""))
        sprintf(DX_FileLogName, "%s/file-log.txt", DX_ProgDir);
}

void dl_ReadSettings(FILE *file)
{
    char buffer[256], *buf_ptr, *value_ptr;
    
    while (DX_ReadFileLine(buffer, sizeof(buffer), file))
    {
	strip(buffer);
            
        if (!(buf_ptr = DX_StripComments(buffer)))
            continue;
            
        dl_error(E_TRACE, "Settings file: Read line \"%s\"", buf_ptr);
            
        value_ptr = buf_ptr;
        while (*value_ptr && !isspace(*value_ptr++)) ;
        if (isspace(*(value_ptr - 1)))
            *(value_ptr - 1) = '\0';
        while (*value_ptr && isspace(*value_ptr++)) ;
        if (value_ptr > (buf_ptr + strlen(buf_ptr)))
            value_ptr--;
        if (*value_ptr == '"')
            value_ptr++;
        if (value_ptr[strlen(value_ptr) - 1] == '"')
            value_ptr[strlen(value_ptr) - 1] = '\0';
        dl_error(E_TRACE, "Attr: \"%s\" Value: \"%s\"", buf_ptr, value_ptr);

        if (!strcasecmp(buf_ptr, "DarxiteDir") && *value_ptr)
        {
            strcpy(DX_ProgDir, value_ptr);
            MAKEPATH(DX_ProgDir);
        }
        else if (!strcasecmp(buf_ptr, "OutputDir") && *value_ptr)
        {
            strcpy(DX_OutputDir, value_ptr);
            MAKEPATH(DX_OutputDir);
        }
        else if (!strcasecmp(buf_ptr, "SpoolDir") && *value_ptr)
        {
            strcpy(DX_SpoolDir, value_ptr);
            MAKEPATH(DX_SpoolDir);
        }
        else if (!strcasecmp(buf_ptr, "EMailAddress") && *value_ptr)
            strcpy(DX_EMailAddress, value_ptr);
        else if (!strcasecmp(buf_ptr, "OutputToFile") && *value_ptr)
            strcpy(DX_OutputToFile, value_ptr);
        else if (!strcasecmp(buf_ptr, "OutputToProgram") && *value_ptr)
            strcpy(DX_OutputToProgram, value_ptr);
        else if (!strcasecmp(buf_ptr, "FileLogName") && *value_ptr)
            strcpy(DX_FileLogName, value_ptr);
        else if (!strcasecmp(buf_ptr, "ProxyHostName") && *value_ptr)
            strcpy(DX_ProxyHostName, value_ptr);
		else if (!strcasecmp(buf_ptr, "DisconnectCommand") && *value_ptr)
			strcpy(DX_DisconnectCommand, value_ptr);
        else if (!strcasecmp(buf_ptr, "ProxyPort") && *value_ptr)
            DX_ProxyPort = atoi(value_ptr);
        else if (!strcasecmp(buf_ptr, "MinRxBeforeSwitching") && *value_ptr)
            DX_MinRxBeforeSwitching = atoi(value_ptr);
        else if (!strcasecmp(buf_ptr, "TimeBeforeSwitching") && *value_ptr)
            DX_TimeBeforeSwitching = atoi(value_ptr);
        else if (!strcasecmp(buf_ptr, "SleepInterval") && *value_ptr)
            DX_SleepInterval = atoi(value_ptr);
        else if (!strcasecmp(buf_ptr, "MaxSimDownloads") && *value_ptr)
            DX_MaxSimDownloads = atoi(value_ptr);
        else if (!strcasecmp(buf_ptr, "RemoveCompletedTime") && *value_ptr)
            DX_RemoveCompletedTime = atoi(value_ptr);
        else if (!strcasecmp(buf_ptr, "BufferSize") && *value_ptr)
            DX_BufferSize = atoi(value_ptr);
        else if (!strcasecmp(buf_ptr, "MassiveLimit") && *value_ptr)
            DX_MassiveLimit = atoi(value_ptr);
        else if (!strcasecmp(buf_ptr, "MassiveMaxConnections") && *value_ptr)
            DX_MassiveMaxConnections = atoi(value_ptr);
        else if (!strcasecmp(buf_ptr,"MassiveConnectAgainLimit") && *value_ptr)
            DX_MassiveConnectAgainLimit = atoi(value_ptr);
        else if (!strcasecmp(buf_ptr, "ListenOnPort") && *value_ptr &&
            (DX_ListenPort <= 0))
            DX_ListenPort = atoi(value_ptr);
        else if (!strcasecmp(buf_ptr, "EnableReadingServerDB"))
            DX_EnableReadingServerDB = TRUE;
        else if (!strcasecmp(buf_ptr, "DisableReadingServerDB"))
            DX_EnableReadingServerDB = FALSE;
        else if (!strcasecmp(buf_ptr, "EnableWritingServerDB") ||
                 !strcasecmp(buf_ptr, "AutoBookmark"))
            DX_EnableWritingServerDB = TRUE;
        else if (!strcasecmp(buf_ptr, "DisableWritingServerDB") ||
                 !strcasecmp(buf_ptr, "NoAutoBookmark"))
            DX_EnableWritingServerDB = FALSE;
        else if (!strcasecmp(buf_ptr, "ClearFileLogOnExit"))
            DX_ClearFileLogOnExit = TRUE;
        else if (!strcasecmp(buf_ptr, "LeaveFileLogOnExit"))
            DX_ClearFileLogOnExit = FALSE;
        else if (!strcasecmp(buf_ptr, "LeaveCompleteFiles"))
            DX_LeaveCompleteFiles = TRUE;
        else if (!strcasecmp(buf_ptr, "RemoveCompleteFiles"))
            DX_LeaveCompleteFiles = FALSE;
        else if (!strcasecmp(buf_ptr, "EnableMirrors"))
            DX_EnableMirrors = TRUE;
        else if (!strcasecmp(buf_ptr, "DisableMirrors"))
            DX_EnableMirrors = FALSE;
        else if (!strcasecmp(buf_ptr, "EnableRenaming"))
            DX_EnableRenaming = TRUE;
        else if (!strcasecmp(buf_ptr, "DisableRenaming"))
            DX_EnableRenaming = FALSE;        
        else if (!strcasecmp(buf_ptr, "EnableSpooling"))
            DX_EnableSpooling = TRUE;
        else if (!strcasecmp(buf_ptr, "DisableSpooling"))
            DX_EnableSpooling = FALSE;
        else if (!strcasecmp(buf_ptr, "EnableOpenLocalFile") ||
                 !strcasecmp(buf_ptr, "EnableOpenOutputFile"))
            DX_EnableOpenLocalFile = TRUE;
        else if (!strcasecmp(buf_ptr, "DisableOpenLocalFile") ||
                 !strcasecmp(buf_ptr, "DisableOpenOutputFile"))
            DX_EnableOpenLocalFile = FALSE;
		else if (!strcasecmp(buf_ptr, "DisconnectOnCompletion"))
			DX_DisconnectOnCompletion = TRUE;
		else if (!strcasecmp(buf_ptr, "NoDisconnectOnCompletion"))
			DX_DisconnectOnCompletion = FALSE;
        else if (!strcasecmp(buf_ptr, "EnableMassiveDownload"))
            DX_EnableMassiveDownload = TRUE;
        else if (!strcasecmp(buf_ptr, "DisableMassiveDownload"))
            DX_EnableMassiveDownload = FALSE;
        else if (!strcasecmp(buf_ptr, "EnableMassiveAutoMirror"))
            DX_EnableMassiveAutoMirror = TRUE;
        else if (!strcasecmp(buf_ptr, "DisableMassiveAutoMirror"))
            DX_EnableMassiveAutoMirror = FALSE;
        else if (!strcasecmp(buf_ptr, "EnableErrorLogging"))
            DX_EnableErrorLogging = TRUE;
        else if (!strcasecmp(buf_ptr, "DisableErrorLogging"))
            DX_EnableErrorLogging = FALSE;
        else if (!strcasecmp(buf_ptr, "EnableFileLogging"))
            DX_EnableFileLogging = TRUE;
        else if (!strcasecmp(buf_ptr, "DisableFileLogging"))
            DX_EnableFileLogging = FALSE;
        else if (!strcasecmp(buf_ptr, "DetectPrematureCompletion"))
            DX_DetectPrematureCompletion = TRUE;
        else if (!strcasecmp(buf_ptr, "NoDetectPrematureCompletion"))
            DX_DetectPrematureCompletion = FALSE;
        else if ((*buf_ptr == '.') && (DX_FileTypeCount < 100))
        {
            DX_FileType[0][DX_FileTypeCount] = strdup(buf_ptr);
            if (value_ptr)
                DX_FileType[1][DX_FileTypeCount] = strdup(value_ptr);
            else
                DX_FileType[1][DX_FileTypeCount] = strdup("");
            DX_FileTypeCount++;
        }
    }
}
