// File dealing with higher-level abstractions of the file transfer and so on

#include <signal.h>
#include <sys/stat.h>

#include "network.h"
#include "files.h"
#include "sockets.h"
#include "ftp.h"
#include "http.h"

void StartTransferCleanup(void *dummy)
{
    error(E_TRACE, "Transfer start thread exiting");
    if (ConnectMutexLocked)
    {
        error(E_TRACE, "Forcibly unlocking the connect mutex");
        UNLOCK_MUTEX(ConnectMutex);
    }
}

// Returns whether it has started or not
BOOL StartTransferringFile(FileInfo *file)
{
    int errn;
    struct stat stat_buf;
    char buffer[256];

    // ensure that a cancellation request is actioned immediately
    SET_CANCEL_TYPE(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
    
    error(E_TRACE, "Trying to start to transfer file %s", file->Path);
    file->Started = FALSE;
    file->Complete = FALSE;
    file->LocalFile = NULL;
    
    // is the file already on disk?
    if (!strcasecmp(file->Protocol, "file"))
    {
        error(E_TRACE, "Local file %s is already complete", file->Path);
        ExpandPath(file->Path, sizeof(file->Path));
        if (file->Upload)
        {
            if (stat(file->LocalPath, &stat_buf) != -1)
                file->CurrentSize = file->TotalSize = stat_buf.st_size;
            else
                file->CurrentSize = file->TotalSize = 0;
            CopyFile(file->Path, file->LocalPath, FALSE);
        }
        else
        {
            if (stat(file->Path, &stat_buf) != -1)
                file->CurrentSize = file->TotalSize = stat_buf.st_size;
            else
                file->CurrentSize = file->TotalSize = 0;
            CopyFile(file->LocalPath, file->Path, FALSE);
        }
        file->Complete = TRUE;
        FileComplete(file, "File complete");
        return FALSE;
    }
	
    // if necessary, stick a ".darxite" onto the file's name
    if (((DX_EnableRenaming && !strchr(file->Flags, 'R')) ||
        strchr(file->Flags, 'r')) && !file->Upload)
    {
        snprintf(buffer, sizeof(buffer), "%s.darxite", file->SpooledPath);
    }
    else
    {
        strcpy(buffer, file->SpooledPath);
    }
    
    // if we've been told to get the file from the beginning
    if (!file->Upload)
    {
        if (strchr(file->Flags, 'b'))
        {
            remove(buffer);
            file->CurrentSize = 0;
            RemoveFlag(file->Flags, 'b');
        }
        else
        {
            // If the file already exists on disk, get its size
            // for when we try to resume it - but massive download overrides
            if (!strchr(file->Flags, 'v'))
            {
                if (stat(buffer, &stat_buf) != -1)
                    file->CurrentSize = stat_buf.st_size;
                else
                    file->CurrentSize = 0;
            }
        }
        error(E_TRACE, "%s's current size is %d", file->SpooledPath,
              file->CurrentSize);
    }
    
    // Find the right mirror
    if (DX_EnableMirrors && !file->Upload && !strchr(file->Flags, 'M'))
    {
        if (file->NextMirror > -1)
            file->ActualServer = file->Server->Mirrors[file->NextMirror];
        else
            file->ActualServer = file->Server;
        file->NextMirror++;
        if (file->NextMirror >= file->Server->MirrorCount)
            file->NextMirror = -1;
    }
    else
    {
        file->ActualServer = file->Server;
    }
    
    // If the server has a default path, use it - unless we already have it
    // (ie. the start of the path is the same as it). This can cause problems,
    // eg. if you have a path /home/ashley/home/ashley, but is probably the
    // best solution.
    if (strncmp(file->ActualServer->DefaultPath, file->Path,
                strlen(file->ActualServer->DefaultPath)))
    {
        if (lastchr(file->ActualServer->DefaultPath) == '/')
        {
            strcpy(buffer, file->ActualServer->DefaultPath);
            lastchr(buffer) = '\0';
            strcat(buffer, file->Path);
            strcpy(file->Path, buffer);
        }
        else
        {
            sprintf(buffer, "%s%s", file->ActualServer->DefaultPath,
                    file->Path);
            strcpy(file->Path, buffer);
        }
    }
    else
    {
        error(E_TRACE, "File's path duplicates default path");
    }
    error(E_TRACE, "File's path is now \"%s\"", file->Path);
    error(E_TRACE, "File's protocol is \"%s\"", file->Protocol);
    
    // install a cleanup handler via pthread_cleanup_push() - for some reason
    // some versions of glibc segfault if this is done at the start of the
    // function
    SET_CLEANUP(StartTransferCleanup, NULL);
    
    // Connect and log in to the server
    if (!strcasecmp(file->Protocol, "ftp"))
    {
        errn = Connect(file->ActualServer->Name, DefaultFtpPort,
                       file->Activity, &(file->ControlSocket));
    }
    else
    {
        // Do proxy stuff if we have to
        if (strcmp(DX_ProxyHostName, "") && (DX_ProxyPort > 0))
        {
            errn = Connect(DX_ProxyHostName, htons(DX_ProxyPort),
                           file->Activity, &(file->ControlSocket));
            if (errn != DX_OK)
            {
                error(E_WARN, "Couldn't connect to proxy server %s",
                      DX_ProxyHostName);
                file->Starting = FALSE;
                strcpy(file->Activity, "Sleeping");
                return FALSE;
            }
            error(E_TRACE, "Connected to proxy OK, now to send stuff");
        }
        else
        {
            errn = Connect(file->ActualServer->Name, DefaultHttpPort,
                           file->Activity, &(file->ControlSocket));
        }
    }
    // we don't really need this, but pthread_cleanup_push() is implemented
    // as a macro which requires a closing } in pthread_cleanup_pop()
    UNSET_CLEANUP(0);
    
    if (errn != DX_OK)
    {
        //error(E_WARN, "Couldn't connect to server %s",
        //file->ActualServer->Name);
        file->Starting = FALSE;
        strcpy(file->Activity, "Sleeping");
        //sigprocmask(SIG_UNBLOCK, &signals, NULL);
        return FALSE;
    }
    
    if (!strcasecmp(file->Protocol, "http"))
    {
        file->DataSocket = file->ControlSocket;
        return HttpStartGettingFile(file);
    }
    else
    {
        if (file->Upload)
            return FtpStartPuttingFile(file);
        else
            return FtpStartGettingFile(file);
    }
    return FALSE;
}

void StartMassiveDownload(FileInfo *file)
{
    char buffer[256];
    struct stat stat_buf;
    FileInfo *massive_file;
    MassiveFileList *file_ptr;
    int i, extra_connections = 0;
    
    // if we want to open a new connection per xK - note that, when a massive
    // download is forced ('e'), we can't do this, so we have to use the
    // maximum number of connections instead
    if ((DX_MassiveConnectAgainLimit > 0) && !strchr(file->Flags, 'e'))
    {
        extra_connections = (file->TotalSize /
                             (DX_MassiveConnectAgainLimit * 1024)) - 1;
        if (extra_connections < 0)
        {
            return;
        }
        else if ((extra_connections > DX_MassiveMaxConnections) &&
                 (DX_MassiveMaxConnections > 0))
        {
            extra_connections = DX_MassiveMaxConnections;
        }
    }
    else if (DX_MassiveMaxConnections > 0)
    {
        extra_connections = DX_MassiveMaxConnections;
    }
    // we don't want to do a massive download after all
    else
    {
        return;
    }
    
    // we can't do a massive download when the user's already downloaded
    // more than the size of the first chunk
    if (file->CurrentSize > (file->TotalSize / (extra_connections + 1)))
    {
        error(E_WARN, "Can't start massive download: file is already %d bytes"
              " long", file->CurrentSize);
        return;
    }
    error(E_TRACE, "Starting massive download for \"%s://%s%s\": %d "
          "extra connections, size %d", file->Protocol,
          file->ActualServer->Name, file->Path, extra_connections,
          file->TotalSize);
    file->MassiveMaster = file;
    file->MassiveFiles = dxmalloc(sizeof(MassiveFileList));
    file->MassiveFiles->LocalPath = strdup(file->SpooledPath);
    file->MassiveFiles->Complete = FALSE;
    file->MassiveFiles->Next = NULL;
    file_ptr = file->MassiveFiles;
    for (i = 0; i < extra_connections; i++)
    {
        sprintf(buffer, "\"%s://%s%s\" | \"%s-part%d\" | %s | %s | %sv "
                "| %d", file->Protocol, file->Server->Name, file->Path,
                file->LocalPath, i + 2, file->LogIn, file->Password,
                file->Flags, ((i+2) * (file->TotalSize /
                                    (extra_connections + 1))));
        massive_file = AddFileToBatch(FALSE, buffer, NULL);
        if (massive_file == NULL)
        {
            // FIXME: should cleanup!
            error(E_WARN, "Massive download failed!");
            return;
        }
        // we don't want the child to be a forced massive d/l, because it
        // would spawn its own children ad infinitum
        RemoveFlag(massive_file->Flags, 'e');
        
        // sort out which mirror to use if necessary
        if (DX_EnableMassiveAutoMirror && !strchr(file->Flags, 'M') &&
            (massive_file->Server->MirrorCount > 0))
        {
            massive_file->NextMirror = i % massive_file->Server->MirrorCount;
            error(E_TRACE, "Next mirror: %s",
               massive_file->Server->Mirrors[massive_file->NextMirror]->Name);
        }
        massive_file->CurrentSize = ((i+1) * (file->TotalSize /
                                              (extra_connections + 1)));
        massive_file->StartOffset = massive_file->CurrentSize;
        
        // check whether this file already exists - we need to manually add
        // a "-partx" to the spooled file name because it is calculated from
        // the remote path, not the local one
        sprintf(buffer, "%s-part%d", massive_file->SpooledPath, i + 2);
        strcpy(massive_file->SpooledPath, buffer);
        // if necessary, stick a ".darxite" onto the file's name
        if (((DX_EnableRenaming && !strchr(file->Flags, 'R')) ||
             strchr(file->Flags, 'r')) && !file->Upload)
        {
            snprintf(buffer, sizeof(buffer), "%s.darxite",
                     massive_file->SpooledPath);
        }
        else
        {
            strcpy(buffer, massive_file->SpooledPath);
        }
        if (stat(buffer, &stat_buf) != -1)
            massive_file->CurrentSize += stat_buf.st_size;
        error(E_TRACE, "Size of %s: %d", buffer, stat_buf.st_size);
        
        error(E_TRACE, "Set file's current size to %d (local %d)",
              massive_file->CurrentSize, stat_buf.st_size);
        massive_file->MassiveMaster = file;
        // add the file's spooled path to the list of paths: this will later
        // be used to concatenate all the files
        file_ptr->Next = dxmalloc(sizeof(MassiveFileList));
        file_ptr->Next->LocalPath = strdup(massive_file->SpooledPath);
        file_ptr->Next->Complete = FALSE;
        file_ptr = file_ptr->Next;
        file_ptr->Next = NULL;
        // make sure the last connection gets all remaining data
        if (i == extra_connections - 1)
            massive_file->TotalSize = file->TotalSize;
    }
    file->TotalSize = file->TotalSize / (extra_connections + 1);
    file_ptr = file->MassiveFiles;
    while (file_ptr)
    {
        error(E_TRACE, "Path: \"%s\"", file_ptr->LocalPath);
        file_ptr = file_ptr->Next;
    }
}

void CancelStartGettingFile(FileInfo *file)
{
    //sigset_t signals;
    
    Disconnect(file->ControlSocket, FALSE);
    FreeSocket(&file->ControlSocket);
    
    file->Starting = FALSE;
    strcpy(file->Activity, "Sleeping");
    // unblock SIGIO
    //sigemptyset(&signals);
    //sigaddset(&signals, SIGIO);
    //sigprocmask(SIG_UNBLOCK, &signals, NULL);
}

// See if a file's going too slowly and should be disconnected. Returns TRUE
// if it's been disconnected, FALSE if it's still OK.
BOOL CheckFileSpeed(FileInfo *file)
{
    int time_now = time(NULL);

    // don't do it if the server doesn't support resume
    if (file->Complete || !file->Started ||
       !file->ActualServer->SupportsFileResume)
    {
        return FALSE;
    }
    /* always give 60 secs for it to get going */
    if (time_now - file->StartTime <= 60)
    {
        return FALSE;
    }
    
    // if there's been no data read for x seconds or the rx rate is low
    // then we should disconnect; this also applies to when the user
    // hangs up (note that we always give 60 secs for it to get going)
    if (time_now - file->TimeOfLastRxTx <= DX_TimeBeforeSwitching &&
        file->RxTxOverall / (time_now - file->StartTime + 1) >=
		DX_MinRxBeforeSwitching)
    {
        return FALSE;
    }
    error(E_TRACE, "Disconnecting file %s", file->Path);
    error(E_TRACE, "Time: %d, Rate: %d", time_now - file->TimeOfLastRxTx,
	  file->RxTxOverall / (time(NULL) - file->StartTime));
    CancelTransfer(file->ControlSocket, file->Protocol);
    DisconnectFile(file, TRUE);
    strcpy(file->Activity, "Sleeping");
    file->Started = FALSE;
    return TRUE;
}

/* BACKUP */
/*
BOOL CheckFileSpeed(FileInfo *file)
{
    int time_now = time(NULL);
    
    // if there's been no data read for x seconds or the rx rate is low
    // then we should disconnect; this also applies to when the user
    // hangs up (note that we always give 60 secs for it to get going)
    if (!file->Complete && file->Started &&
        (time_now - file->StartTime > 60) &&
        (((time_now - file->TimeOfLastRxTx) >
          DX_TimeBeforeSwitching) || ((file->RxTxOverall /
          (time_now - file->StartTime + 1)) < DX_MinRxBeforeSwitching)))
    {
        error(E_TRACE, "server's resume? %d",
              file->ActualServer->SupportsFileResume);
        error(E_TRACE, "Disconnecting file %s", file->Path);
        error(E_TRACE, "Time: %d, Rate: %d",
              time_now - file->TimeOfLastRxTx,
              file->RxTxOverall /
              (time(NULL) - file->StartTime));
        CancelTransfer(file->ControlSocket, file->Protocol);
        DisconnectFile(file, TRUE);
        strcpy(file->Activity, "Sleeping");
        file->Started = FALSE;
        return TRUE;
    }
    else
    {
        return FALSE;
    }
}
*/

// Checks whether we can read any data
int CheckReadSockets(void)
{
    char buffer[256];
    MassiveFileList *massive_file;
    int time_now, written = 0, rc;
    int bytes_to_write = 0, error_code = GOT_INPUT;
    FILE *local_file;
    FileInfo *file;
    BOOL massive_complete;
    THREAD_ID(thread_id)
    
    time_now = time(NULL);
    file = Batch.FirstFile;
    while (file)
    {
        // we're only interested in files which could have input
        if (file->Complete || !file->Started || file->Paused ||
            file->Starting || file->Upload ||
            !file->DataSocket || !file->DataSocket->Connected ||
            !file->ControlSocket || !file->ControlSocket->Connected)
        {
            file = file->Next;
            continue;
        }
        written = 0;
        if (!file->DataSocket->Eof && (file->DataSocket->DataLength > 0))
        {
            error(E_TRACE, "%s has some input", file->LocalPath);
            // if it's an HTTP file and we haven't got the header
            // yet, parse it, don't append it
            if (!strcasecmp(file->Protocol, "http") && !file->GotHeader)
            {
                HttpParseHeader(file);
            }
            // if necessary, stick a ".darxite" onto the file's name
            if ((DX_EnableRenaming && !strchr(file->Flags, 'R')) ||
                strchr(file->Flags, 'r'))
            {
                snprintf(buffer, sizeof(buffer), "%s.darxite",
                         file->SpooledPath);
            }
            else
            {
                strcpy(buffer, file->SpooledPath);
            }
            CreatePath(buffer);
            if (DX_EnableOpenLocalFile && (file->LocalFile != NULL))
            {
                local_file = file->LocalFile;
            }
            else
            {
                local_file = fopen(buffer, "a");
                file->LocalFile = local_file;
            }
            if (!local_file)
            {
                error(E_WARN, "Couldn't open output file \"%s\": %s",
                      buffer, strerror(errno));
                file->Complete = TRUE;
                continue;
            }
            
            if (file->TotalSize > 0)
            {
                bytes_to_write = min(file->TotalSize,
                                     file->DataSocket->DataLength +
                                     file->CurrentSize) - file->CurrentSize;
            }
            else
            {
                bytes_to_write = file->DataSocket->DataLength;
            }
            error(E_TRACE, "Writing %d bytes", bytes_to_write);
            
            rc = buf_repeat_fwrite(file->DataSocket->DataBuf,
                                   bytes_to_write, local_file);
            if (rc < 0)
            {
                error(E_WARN, "File \"%s\" write error: %s", buffer,
                      strerror(errno));
                CancelTransfer(file->ControlSocket, file->Protocol);
                DisconnectFile(file, FALSE);
                FileComplete(file, "Write error");
                error_code = INPUT_ERROR;
                continue;
            }
            file->CurrentSize += rc;
            file->RxTxOverall += rc;
            file->TimeOfLastRxTx = time(NULL);
            file->DataSocket->DataLength = 0;
            
            //if ((file->TotalSize != SIZE_UNKNOWN) &&
            //   (file->CurrentSize > file->TotalSize))
            //{
            //    error(E_WARN,
            //          "File \"%s\" is bigger than it should be - "
            //          "maybe the server doesn't support resume",
            //          buffer);
            //}
            
            if (!DX_EnableOpenLocalFile)
                fclose(local_file);
        }
        if (file->DataSocket->Eof && (file->CurrentSize < file->TotalSize))
        {
            error(E_TRACE, "%s completed prematurely", file->Path);
            if (DX_DetectPrematureCompletion &&
                file->ActualServer->SupportsFileResume)
            {
                error(E_TRACE, "Automatically resuming file");
                file->Started = FALSE;
                file->GotHeader = FALSE;
                WakeUpNow = TRUE;
                file = file->Next;
                continue;
            }
        }
        if (file->DataSocket->Eof ||
            ((file->CurrentSize >= file->TotalSize) && (file->TotalSize > -1)))
        {
            error(E_TRACE, "It seems that file %s is complete", file->Path);
            error(E_TRACE, "Eof is %d, Time %d",
                  file->DataSocket->Eof,
                  time(NULL) - file->StartTime);
            error(E_TRACE, "Cur size: %d, Written: %d, Total: %d",
                  file->CurrentSize, written, file->TotalSize);
            // if we haven't had an EOF, we've just overrun
            if (!file->DataSocket->Eof)
            {
                CancelTransfer(file->ControlSocket, file->Protocol);
            }
            // Get the "file complete" response... we hope!
            // Data disconnect beforehand - better?
            if (!strcasecmp(file->Protocol, "ftp"))
            {
                DataDisconnect(file->DataSocket);
                if (file->DataSocket->Eof)
                    GetResponse(file->ControlSocket, NULL, 0);
            }
            DisconnectFile(file, FALSE);
            file->Complete = TRUE;
            // if the file was part of a massive download, then notify its
            // master that it's complete
            if (file->MassiveMaster != NULL)
            {
                massive_file = file->MassiveMaster->MassiveFiles;
                while (massive_file)
                {
                    if (!strcmp(massive_file->LocalPath, file->SpooledPath))
                    {
                        error(E_TRACE, "Massive file is complete");
                        massive_file->Complete = TRUE;
                        break;
                    }
                    massive_file = massive_file->Next;
                }
            }
            // despool it if spooling's enabled and we know the file's
            // spooled filename and it's not a massive download
            if (((DX_EnableSpooling && !strchr(file->Flags, 'S')) ||
                 strchr(file->Flags, 's')) &&
                (file->Path[strlen(file->Path) - 1] != '/') &&
                (file->MassiveMaster == NULL))
            {
                //error(E_INFO,
                //"Starting moving process for file %d (%s)", i,
                //file->SpooledPath);
                strcpy(file->Activity, "Despooling file...");
                SPAWN_THREAD(ThreadedMoveFile, thread_id, file);
                DETACH_THREAD(thread_id);
                Sleep(50);	// lame but a simple way of avoiding
                    			// weirdnesses with threads
            }
            else
            {
                strcpy(file->Activity, "Complete");
                // if we need to rename it now that the transfer's
                // complete, then do so - note that SpooledPath should
                // be the same as LocalPath
                if ((DX_EnableRenaming && !strchr(file->Flags, 'R')) ||
                    strchr(file->Flags, 'r'))
                {
                    snprintf(buffer, sizeof(buffer), "%s.darxite",
                             file->SpooledPath);
                    rename(buffer, file->SpooledPath);
                }
                // only do completion stuff if the file isn't part of a
                // massive download
                if (file->MassiveMaster == NULL)
                    FileComplete(file, "File complete");
            }
            // if all the files in a massive download are complete then
            // concatenate them
            if (file->MassiveMaster != NULL)
            {
                massive_file = file->MassiveMaster->MassiveFiles;
                massive_complete = TRUE;
                while (massive_file)
                {
                    error(E_TRACE, "Massive completion: %s: %d",
                          massive_file->LocalPath, massive_file->Complete);
                    if (!massive_file->Complete)
                    {
                        massive_complete = FALSE;
                        break;
                    }
                    massive_file = massive_file->Next;
                }
                if (massive_complete)
                {
                    SPAWN_THREAD(ThreadedMassiveComplete, thread_id, file);
                    DETACH_THREAD(thread_id);
                    Sleep(50);
                }
            }
        }
        CheckFileSpeed(file);
        file = file->Next;
    }
    return error_code;
}

// Checks whether we've output any data on any write sockets
void CheckWriteSockets(void)
{
    FileInfo *file;
    
    // check write sockets
    file = Batch.FirstFile;
    while (file)
    {
        // we're only interested in files which could have output
        if (file->Complete || !file->Started || file->Paused ||
            !file->DataSocket || !file->ControlSocket || !file->Upload)
        {
            file = file->Next;
            continue;
        }

        if (!file->DataSocket->Eof)
        {
            if (file->DataSocket->DataWritten > 0)
            {
                file->CurrentSize += file->DataSocket->DataWritten;
                file->RxTxOverall += file->DataSocket->DataWritten;
                file->TimeOfLastRxTx = time(NULL);
				file->DataSocket->DataLength = 0;
                file->DataSocket->DataWritten = 0;
            }
        }
        else
        {
            error(E_TRACE, "It seems that file %s is complete", file->Path);
            error(E_TRACE, "Eof is %d, DataWritten %d, Time %d",
                  file->DataSocket->Eof,
                  file->DataSocket->DataWritten,
                  time(NULL) - file->StartTime);
            // We need to disconnect the data socket before getting the
            // response, so the FTP server knows it's an EOF
            if (!strcasecmp(file->Protocol, "ftp"))
            {
                DataDisconnect(file->DataSocket);
                GetResponse(file->ControlSocket, NULL, 0);
            }
            DisconnectFile(file, FALSE);
            file->Complete = TRUE;
            strcpy(file->Activity, "Complete");
            FileComplete(file, "File complete");
        }
        CheckFileSpeed(file);
        file = file->Next;
    }
}

// Reads data from a local file to upload into the buffer
int ReadUploadData(FileInfo *file)
{
    int rc, error_code = GOT_INPUT;
    FILE *local_file;
    
    error(E_TRACE, "Adding %s to write list", file->Path);
    
    if (DX_EnableOpenLocalFile && (file->LocalFile != NULL))
    {
        local_file = file->LocalFile;
    }
    else
    {
        local_file = fopen(file->LocalPath, "r");
        file->LocalFile = local_file;
        if (local_file)
        {
            rc = fseek(local_file, file->CurrentSize, SEEK_SET);
            if (rc != 0)
                error(E_WARN, "Couldn't fseek: %s", strerror(errno));
        }
    }
    if (!local_file)
    {
        error(E_WARN, "Couldn't open input file \"%s\": %s",
              file->LocalPath, strerror(errno));
        file->Complete = TRUE;
        return INPUT_ERROR;
    }
    
    error(E_TRACE, "Reading %d bytes from infile %d, offset %d",
          DX_BufferSize, local_file, file->CurrentSize);
    
    rc = buf_fread(file->DataSocket->DataBuf, 1, DX_BufferSize, local_file);
    if (rc < 0)
    {
        error(E_WARN, "File \"%s\" read error: %s",
              file->LocalPath, strerror(errno));
        CancelTransfer(file->ControlSocket, file->Protocol);
        DisconnectFile(file, FALSE);
        FileComplete(file, "Read error");
        error_code = INPUT_ERROR;
    }
    else
    {
        error(E_TRACE, "Actually read %d bytes", rc);
        file->DataSocket->DataLength = rc;
    }
    if (!DX_EnableOpenLocalFile)
        fclose(local_file);
    
    return error_code;
}

int WaitForInput(int timeout)
{
    SockInfo **read_sockets = NULL, **write_sockets = NULL;
    FileInfo *file;
    int rc, read_socket_count = 0, write_socket_count = 0;
    int error_code = GOT_INPUT;
    BOOL some_unstarted = FALSE;

//    error(E_TRACE, "Waiting for input");
    read_sockets = dxmalloc(sizeof(SockInfo) * Batch.FileCount);
    write_sockets = dxmalloc(sizeof(SockInfo) * Batch.FileCount);
    for (file = Batch.FirstFile; file; file = file->Next)
    {
        if (file->Complete)
            continue;
        
        if (!file->Started)
            some_unstarted = TRUE;
        
        if (!file->ControlSocket || !file->DataSocket ||
            !file->ControlSocket->Connected ||
            !file->DataSocket->Connected || file->Starting ||
            file->Paused || !file->Started)
        {
            continue;
        }
        if (file->Upload)
        {
            // only read data into the buffer if there's none there already
            if (file->DataSocket->DataLength == 0)
                error_code = ReadUploadData(file);
            write_sockets[write_socket_count++] = file->DataSocket;
        }
        else
        {
            //error(E_TRACE, "Adding %s to read list", file->Path);
            read_sockets[read_socket_count++] = file->DataSocket;
            // file->DataSocket->DataLength = 0;
        }
    }
    
    if ((read_socket_count == 0) && (write_socket_count == 0))
    {
        dxfree(read_sockets);
        dxfree(write_sockets);
        if (some_unstarted)
            return STARTED_DONE;
        else
            return ALL_COMPLETE;
    }
    
    rc = TransferData(read_sockets, read_socket_count, write_sockets,
                      write_socket_count, timeout);
    dxfree(read_sockets);
    dxfree(write_sockets);
    // rc 0 we allow because we might need to escape when checking sockets
    if (rc == -1)
        return INPUT_ERROR;
    
    error_code = CheckReadSockets();
    CheckWriteSockets();
    
    return error_code;
}

void DisconnectFile(FileInfo *file, BOOL quick)
{
    strcpy(file->Activity, "Disconnecting...");
    if (!strcasecmp(file->Protocol, "ftp"))
    {
        DataDisconnect(file->DataSocket);
        FreeSocket(&file->DataSocket);
        Disconnect(file->ControlSocket, quick);
        FreeSocket(&file->ControlSocket);
    }
    else
    {
        HttpDisconnect(file->ControlSocket);
        FreeSocket(&file->ControlSocket);
    }
    // Close the local file if it's still open
    if (DX_EnableOpenLocalFile && file->LocalFile)
    {
        fclose(file->LocalFile);
        file->LocalFile = NULL;
    }
    strcpy(file->Activity, "Sleeping");
}
