// The Darxite daemon
// Release: 0.4 (Agrajag)

#include <includes.h>

#include "sockets.h"
#include "global.h"
#include "main.h"
#include "files.h"
#include "network.h"
#include "clients.h"

static BOOL Fork = TRUE;

void *thread_func(void *params)
{
    error(E_INFO, "Thread function!");
    return NULL;
}

void PrintVersion(void)
{
    printf("%s daemon v%s, release %s (%s) by %s\n", PROG_NAME, VERSION,
           RELEASE_VER, RELEASE_NAME, AUTHORS);
    exit(0);
}

void Usage(char *prog_name)
{
    printf("Usage: %s [OPTION]...\n", prog_name);
    printf("where options are:\n");
    printf("  -v, --version\t\t\tShow version and exit\n");
    printf("  -h, --help\t\t\tShow some usage information\n");
    printf("  -d<dir>, --directory=<dir>\tUse directory <dir> for "
           "program files\n");
    printf("  -k, --kill\t\t\tDon't run, kill a running daemon\n");
    printf("  -l<port>, --listen=<port>\tListen for remote clients on "
			"<port>\n");
    printf("  -o, --output=<dir>\t\tOutput files to directory <dir>\n");
    /* glenn: this option is bad: it puts our password in ps */
    /* ashley: no it doesn't, the daemon rewrites this info */
    printf("  -p, --password=<pass>\t\tUse <pass> as remote password\n");
    printf("  -q, --quiet\t\t\tExit quietly if another copy is running\n");
    printf("  -r, --reload\t\t\tTell a running daemon to reload its config\n");
    printf("  -f, --foreground\t\tRun in the foreground (don't fork)\n");
    exit(0);
}

#define OPTION_STRING	"vhd:kl:o:p:qrf"

static struct option LongOptions[] = {
    { "version", 	0, 0, 'v' },
    { "help",		0, 0, 'h' },
    { "directory", 	1, 0, 'd' },
    { "kill",		0, 0, 'k' },
    { "listen",		1, 0, 'l' },
    { "output",		1, 0, 'o' },
    { "password", 	1, 0, 'p' },
    { "quiet",		0, 0, 'q' },
    { "reload",		0, 0, 'r' },
    { "foreground",	0, 0, 'f' },
    { 0,		0, 0,  0  }
};

void ReadCmdLine(int argc, char **argv)
{
    char opt;
    int i, option_index;
    
    while (1)
    {
        opt = getopt_long(argc, argv, OPTION_STRING, LongOptions,
                          &option_index);
        if(opt == -1) break;
        switch (opt)
        {
        case 'v':
            DX_QuietExit = TRUE;
            PrintVersion();
            break;
            
        case 'h':
            DX_QuietExit = TRUE;
            Usage(argv[0]);
            break;
            
        case 'd':
            strcpy(DX_ProgDir, optarg);
            break;

        case 'k':
        {
            int fd;
            char buffer[256];

            setuid(getuid());
            fd = DX_ConnectClient("Darxite-killer");
            if (fd == -1)
            {
                fprintf(stderr,
                        "Couldn't find an instance of Darxite to kill\n");
                exit(1);
            }
            
            strcpy(buffer, "Kill\n");
            write(fd, buffer, strlen(buffer));
            // give the daemon time to close down before we exit - this is
            // to allow people to type "darxite -k ; darxite"
            Sleep(50);
            exit(0);
        }
        
        case 'l':
            DX_ListenPort = atoi(optarg);
            break;
            
        case 'o':
            strcpy(DX_OutputDir, optarg);
            break;
            
        case 'p':
            strcpy(RemotePassword, optarg);
            break;
            
        case 'q':
            DX_QuietExit = TRUE;
            break;
            
        case 'f':
            Fork = FALSE;
            break;
            
        case 'r':
        {
            int fd;
            char buffer[256];

            setuid(getuid());
            fd = DX_ConnectClient("Darxite-reloader");
            if (fd == -1)
            {
                fprintf(stderr,
                        "Couldn't find an instance of Darxite to control\n");
                exit(1);
            }
            
            strcpy(buffer, "ReloadConfig\n");
            write(fd, buffer, strlen(buffer));
            exit(0);
        }
        
        }
    }
    
    if (optind < argc)
        Usage(argv[0]);

    for (i = 0; i < argc; i++)
        memset(argv[i], 0, strlen(argv[i]));
    strcpy(argv[0], "darxite");
}

void signal_func(int signum)
{
    switch (signum)
    {
    case SIGIO:
        // NB: If several of these arrive very close together,
        // they will be squidged into one signal!
        error(E_TRACE, "Caught a SIGIO");
        CheckClientSockets();
        error(E_TRACE, "Finished doing client stuff");
        break;

    case SIGTERM:
        error(E_TRACE, "Caught a SIGTERM: closing down gracefully.");
        exit(0);
        break;
        
    case SIGINT:
        error(E_TRACE, "Caught a SIGINT: exiting gracefully.");
        exit(0);
        break;

        /*case SIGCHLD:
        child_reap();
        child_clean();
        break;*/
        
    case SIGSEGV:
    {
        char* stupidities[] = { "Doh!", "Ashley is a moronic moron.",
                                "How tedious.", "Kaboom!",
                                "Time to do some debugging.",
                                "Didn't we tell you this was beta code?",
                                "Time to go for that job at Microsoft.",
                                "Read the log files.", "Oops.",
                                "Lots of features mean lots of bugs.",
                                "Many apologies.", "Oh no!",
                                "Now reboot your computer." };
        srand(time(NULL));
        error(E_FATAL, "\x1B[1mSegment violation. %s\x1B[0m",
              stupidities[rand() % 13]);
        break;
    }
     
    default:
        error(E_FAIL, "Caught unknown signal %d", signum);
        break;
    }
}

void exit_func(void)
{
    ClientInfo *client = FirstClient;
    FileInfo *file = Batch.FirstFile, *next_file;
    char resp_buf[256];

    sprintf(resp_buf, "%d Daemon closing down\n", DX_EXIT_EVENT);
    while (client)
    {
        if (client->Connected)
        {
            if (client->EventMask & EXIT_EVENT)
                write(client->Socket, resp_buf, strlen(resp_buf) + 1);
            DisconnectClient(client);
        }
        client = client->Next;
    }
    
    while (file)
    {
        next_file = file->Next;
        if (!file->Complete && file->Started)
        {
            error(E_TRACE, "Trying to close file \"%s://%s%s\"...",
                  file->Protocol, file->Server->Name, file->Path);
            CancelTransfer(file->ControlSocket, file->Protocol);
            DisconnectFile(file, TRUE);
        }
        else if (file->Complete)
        {
            RemoveFromBatch(file);
        }
        file = next_file;
    }
    if (DX_ClearFileLogOnExit)
        remove(DX_FileLogName);
    sock_DestroyDefaultSocket();
    sock_DestroyNetSocket(NetClientSocket);
    if (!DX_QuietExit)
        error(E_INFO, "Shutting down");
    error(E_TRACE, "===End of error log===");
}

static void main_loop(void)
{
    time_t sleep_start = 0;
    FileInfo *file;
    THREAD_ID(thread_id)
    int rc;
    BOOL any_starting = FALSE;
    
    while (1)
    {
        // This jump-point is needed for when a file is cancelled/paused by
        // a client, because another function could have major problems due
        // to it suddenly becoming NULL. We use siglongjmp() because it
        // allows the SIGIO stuff to work. Note that behaviour with threads
        // is fairly undefined - but it works fine with Linux. There may be
        // portability problems.
        if (sigsetjmp(MainJump, 1))
            error(E_TRACE, "We've jumped to the main loop...");
	
        if (!WakeUpNow && Batch.Complete)
        {
            //error(E_TRACE, "Sleeping because the batch is complete");
            /*-Nothing to do; sleep until a signal is received--probably
             * SIGIO. */
            Sleep(FOREVER);
            continue;
        }
        
        if (WakeUpNow || ((time(NULL) - sleep_start) >= DX_SleepInterval))
        {
            WakeUpNow = FALSE;
            error(E_TRACE, "Checking for unstarted files...");
            file = Batch.FirstFile;
            while (file)
            {
                if (!file->Started && !file->Complete &&
                    !file->Starting && !file->Paused)
                {
                    file->Starting = TRUE;
                    SPAWN_THREAD(StartTransferringFile, thread_id, file);
                    SET_THREAD_ID(file->ThreadId, thread_id);
                    DETACH_THREAD(thread_id);
                    Sleep(50);
                    Sleeping = FALSE;
                }
                file = file->Next;
            }
            time(&sleep_start);
        }
        
        if (!Sleeping)
        {
            // the important line!
            rc = WaitForInput(10);
            if (rc == ALL_COMPLETE)
            {
                error(E_TRACE, "All files complete");
                Batch.Complete = TRUE;
                if (DX_DisconnectOnCompletion)
                {
                    error(E_TRACE, "Calling \"%s\" to disconnect",
                          DX_DisconnectCommand);
                    system(DX_DisconnectCommand);
                }
            }
            else if (rc == STARTED_DONE)
            {
                // we can't go to sleep if there are any files which have
                // begun transferring in the background, because we will
                // want to transfer them soon
                //error(E_TRACE, "All started files complete");
                any_starting = FALSE;
                file = Batch.FirstFile;
                while (file)
                {
                    if (file->Starting)
                    {
                        //error(E_TRACE, "Not sleeping because file %s is "
                        //      "starting", file->Path);
                        any_starting = TRUE;
                        Sleep(500);
                        // sleep for a short time to conserve CPU time
                        break;
                    }
                    file = file->Next;
                }
                if (!any_starting)
                {
                    error(E_TRACE, "Starting to sleep...");
                    Sleeping = TRUE;
                    time(&sleep_start);
                }
            }
        }
        else
        {
            //error(E_TRACE, "Continuing to sleep...");
            //error(E_TRACE, "WakeUpNow: %d, Batch.Complete: %d",
            //      WakeUpNow, Batch.Complete);
            Sleep(DX_SleepInterval * 1000);
        }
    }
}

void init_signals(void)
{
    int i;

    if (signal(SIGHUP, SIG_IGN) == SIG_ERR) // signal 1
        error(E_FAIL, "Can't ignore SIGHUP");
    for (i = 2; i < 6; i++)
    {
        if (signal(i, signal_func) == SIG_ERR)
            error(E_FAIL, "Can't register handler for signal %d", i);
    }
    // don't trap SIGABRT (=SIGIOT, signal 6)
    for (i = 7; i < 9; i++)
    {
        if (signal(i, signal_func) == SIG_ERR)
            error(E_FAIL, "Can't register handler for signal %d", i);
    }
    // don't trap SIGUSR1, it's used for threads
    
    // we only do stuff on SEGV if we have forked, because this feedback is
    // useful for the user, who otherwise would not get anything
    if (Fork)
    {
        if (signal(SIGSEGV, signal_func) == SIG_ERR) // signal 11
            error(E_FAIL, "Can't register signal handler for SIGSEGV");
    }
    
    // don't trap SIGUSR2, it's used for threads
    if (signal(SIGPIPE, SIG_IGN) == SIG_ERR) // signal 13
        error(E_FAIL, "Can't ignore SIGPIPE");
    for (i = 14; i < 17; i++)
    {
        if (signal(i, signal_func) == SIG_ERR)
            error(E_FAIL, "Can't register handler for signal %d", i);
    }
    if (signal(SIGIO, signal_func) == SIG_ERR) // signal 29
        error(E_FAIL, "Can't register signal handler for SIGIO");
}

int main(int argc, char *argv[])
{
    char buffer[256];
    //DX_ClientInfo *client, *client2;
    int readyfds[2] = { -1, -1 };
    
    DX_ReadConfigFiles();
    // This has to be done AFTER DX_ReadConfigFiles because we may need to
    // override some of the config files. But it must be done before
    // DX_InitNetSocket because we may need to find out what port to listen on.
    ReadCmdLine(argc, argv);
    init_logging();
    DefaultClientSocket = sock_CreateDefaultSocket();
    NetClientSocket = sock_CreateNetSocket(DX_ListenPort);
    
    /*rc = DX_ReadClientDB(&client);
    error(E_TRACE, "Read %d clients", rc);
    rc = DX_FindClients(NULL, "util", client, &client2);
    error(E_TRACE, "Found %d clients", rc);*/
   
    // create the directories if they don't exist
    strcpy(buffer, DX_OutputDir);
    if (buffer[strlen(buffer) - 1] != '/')
        strcat(buffer, "/");
    ExpandPath(buffer, sizeof(buffer));
    CreatePath(buffer);
    strcpy(buffer, DX_ProgDir);
    if (buffer[strlen(buffer) - 1] != '/')
        strcat(buffer, "/");
    ExpandPath(buffer, sizeof(buffer));
    CreatePath(buffer);
    
    InitNetworking();
    ReadDataFiles();
    
    //SPAWN_THREAD(thread_func, thread, NULL); 
    //DX_SayHello();
    //while (1)
    //     Sleep(FOREVER);
    
    // Stick ourselves in the background
    if (Fork)
    {
        pid_t pid;
        pipe(readyfds);
        pid = fork();
        if (pid < 0)
        {
            perror("fork");
            return 1;
        }
        else if (pid > 0)
        {
            /* HACK: make sure the SIGIO handler is set up before
             * we exit */
            char c;
            close(readyfds[1]);
            /* Wait for the child to close the pipe, to indicate that
             * it's ready. */
            read(readyfds[0], &c, 1);
            
            return 0;
        }
        close(readyfds[0]);
        
        if (setsid() < 0)
            error(E_FAIL, "Couldn't create new session: %s", strerror(errno));
    }
    
    /* Our PID changed; reown the sockets. */
    sock_SetSocketPid(DefaultClientSocket);
    sock_SetSocketPid(NetClientSocket);
    
    INIT_MUTEX(BatchMutex);
    INIT_MUTEX(ConnectMutex);
    
    init_signals();
    
    if (atexit(exit_func) == -1)
        error(E_FAIL, "Can't register atexit() function");
    
    if(readyfds[1] != -1)
    {
        /* close the pipe to show we're ready */
        close(readyfds[1]);
    }
    
    main_loop();
    return 0;
}

