#include <includes.h>
#include "global.h"

/* Be careful to block SIGCHLD when changing this table; we don't want
 * a child handler popping up while we're deleting entries. */
static struct child_t {
	/* pid of child */
	int pid;

	/* id */
	char *name;
	
	/* if not NULL, called when child exits with return
	 * value */
	void (*exited)(int pid, int ret);

	/* this child is gone; free it. we don't do this in the
	 * signal handler; efence doesn't like that. */
	int delete;
	int exitstatus;

	struct child_t *n, *p;
} *children = NULL;

/* number of children awaiting reap */
static int needreap = 0;

static MUTEX(ChildMutex);
ONCE_ID(ChildMutexInit);

void Init_ChildMutex()
{
    INIT_MUTEX(ChildMutex);
}

static int real_make_child(void (*exited)(int pid, int ret), char *namebuf);

/* Assumption: this is always called before child_reap, and always before
 * child_clean is called where it might actually do something. Not a great
 * assumption, but I don't want to write wrappers like this for each of the
 * other functions ... */
int make_child(void (*exited)(int pid, int ret), char *name, ...)
{
    int oldcanceltype, ret;
    char buf[512];
    va_list msg;

    SET_CANCEL_TYPE(PTHREAD_CANCEL_DEFERRED, &oldcanceltype);
    ONCE(ChildMutexInit, Init_ChildMutex);

    va_start(msg, name);
    vsnprintf(buf, sizeof(buf), name, msg);
    va_end(msg);

    LOCK_MUTEX(ChildMutex);
    ret = real_make_child(exited, buf);
    
    SET_CANCEL_TYPE(oldcanceltype, NULL);
    UNLOCK_MUTEX(ChildMutex);

    return ret;
}

static int real_make_child(void (*exited)(int pid, int ret), char *namebuf)
{
    int ret;
    struct child_t *c;
    sigset_t newsigset, oldsigset;

    /* block SIGCHLD; can't have that going off while we mung with
     * our process table */
    sigemptyset (&newsigset);
    sigaddset (&newsigset, SIGCHLD);
    sigprocmask (SIG_BLOCK, &newsigset, &oldsigset);

    ret = fork();

    if(ret == -1) {
	error(E_FATAL, "Couldn't fork: %s", strerror(errno));
	return -1;
    }

    if(ret == 0) {
	/* child; just return, but block most signals */
	sigset_t newsigset;
	sigemptyset (&newsigset);
	sigaddset (&newsigset, SIGUSR1);
	sigaddset (&newsigset, SIGINT);
	sigaddset (&newsigset, SIGQUIT);
	sigaddset (&newsigset, SIGHUP);
	sigaddset (&newsigset, SIGTRAP);
	sigprocmask (SIG_BLOCK, &newsigset, NULL);

	return ret;
    }

    /* update the child table */
    c = dxmalloc(sizeof(struct child_t));
    c->pid = ret;
    c->exited = exited;
    c->n = children;
    c->name = strdup(namebuf);
    if(c->n) c->n->p = c;
    children = c;

    /* restore SIGCHLD */
    sigprocmask (SIG_SETMASK, &oldsigset, NULL);

    return ret;
}

static void child_exited(int pid, int ret)
{
    struct child_t *c;

    for(c = children; c; c=c->n) {
	if(c->pid == pid) break;
    }

    if(!c) {
	error(E_WARN, "Unknown child (%i)?", pid);
	return;
    }

    c->delete = 1;
    c->exitstatus = ret;
    needreap++;
}

void child_clean(void)
{
    struct child_t *c, *n;
    sigset_t newsigset, oldsigset;

    LOCK_MUTEX(ChildMutex);
    sigemptyset (&newsigset);
    sigaddset (&newsigset, SIGCHLD);
    sigprocmask (SIG_BLOCK, &newsigset, &oldsigset);

    c = children;
    while(needreap && c) {
	if(!c->delete) {
	    c=c->n;
	    continue;
	}

	if(WIFEXITED(c->exitstatus)) {
	    /* FIXME: we probably shouldn't make these calls from
	     * a sighandler */
	    error(E_TRACE, "Child %i (%s) exited normally.", c->pid, c->name);
	} else if(WIFSIGNALED(c->exitstatus)) {
	    error(E_TRACE, "Child %i (%s) exited with signal %i.", c->pid, c->name, WTERMSIG(c->exitstatus));
	}


	if(c->exited) c->exited(c->pid, c->exitstatus);

	n = c->n;

	if(c->n) c->n->p = c->p;
	if(c->p) c->p->n = c->n;

	if(c == children) children = c->n;

	free(c->name);
	free(c);
	c = n;

	needreap--;
    }

    /* restore SIGCHLD */
    sigprocmask (SIG_SETMASK, &oldsigset, NULL);
    UNLOCK_MUTEX(ChildMutex);
}

void child_reap()
{
    sigset_t sigset, oldset;
    LOCK_MUTEX(ChildMutex);

    sigfillset(&sigset);
    sigprocmask(SIG_BLOCK, &sigset, &oldset);

    while(1) {
	int pid, ret;

	pid = waitpid(-1, &ret, WNOHANG);
	if(pid <= 0) {
	    if(errno == EAGAIN) continue;
	    break;
	}

	child_exited(pid, ret);
    }

    sigprocmask(SIG_BLOCK, &oldset, NULL);
    UNLOCK_MUTEX(ChildMutex);
}

