// File dealing with connect dialogue etc.

#include "connect.h"
#include "commands.h"
#include "dialogues.h"
#include "main.h"

static BOOL ChangingText = FALSE;

/* "Hook" functions called by GTK */
static void toggle_anon(void)
{
    BOOL anon = GTK_TOGGLE_BUTTON(ConnectAnon)->active;

    if (!ChangingText)
    {
        gtk_widget_set_sensitive(ConnectUser, !anon);
        gtk_widget_set_sensitive(ConnectPassword, !anon);
        if (ServerIndex > -1)
            ServAnon[ServerIndex] = anon;
    }
}

static void edit_mirrors(void)
{
    if (ServerIndex > -1)
        ShowMirrorsDialogue(ServerIndex);
}

static void save_servers(void)
{
    FILE *file;
    char buffer[256], field[7][256];
    int i, j;
    
    sprintf(buffer, "%s/servers", DX_ProgDir);
    file = fopen(buffer, "w");
    if (!file)
    {
        AddFtpString("Couldn't save servers file!\n");
        return;
    }
    fprintf(file, "# Darxite servers file: "
            "contains a list of visited servers\n");
    fprintf(file, "# File created automagically by NoctFTP version %s\n\n",
            VERSION);
    for (i = 0; i < ServerCount; i++)
    {
        if (!strcmp(ServHost[i], ""))
            continue;
        strcpy(field[0], ServHost[i]);
        strcpy(field[1], ServPath[i]);
        strcpy(field[2], ServLogin[i]);
        strcpy(field[3], ServPassword[i]);
        strcpy(field[4], ServAlias[i]);
        strcpy(field[5], ServComment[i]);
        // make sure the path's right
        if (strcmp(field[1], ""))
        {
            if (*field[1] != '/')
            {
                sprintf(buffer, "/%s", field[1]);
                strcpy(field[1], buffer);
            }
            if (field[1][strlen(field[1]) - 1] == '/')
                field[1][strlen(field[1]) - 1] = '\0';
                
        }
        // stick quotes round the server and path if necessary
        if (strchr(field[0], ' ') || strchr(field[1], ' '))
        {
            sprintf(buffer, "\"%s", field[0]);
            strcpy(field[0], buffer);
            strcat(field[1], "\"");
        }
        // now check the other fields
        for (j = 2; j < 6; j++)
        {
            if (strchr(field[j], ' '))
            {
                sprintf(buffer, "\"%s\"", field[j]);
                strcpy(field[j], buffer);
            }
            if (strcmp(field[j], ""))
            {
                sprintf(buffer, " %s", field[j]);
                strcpy(field[j], buffer);
            }
        }
        if (MirrorCount[i] == 0)
        {
            fprintf(file, "%s%s |%s |%s |%s | \\\n   %s | |\n",
                    field[0], field[1], field[2], field[3], field[4],
                    field[5]);
        }
        else
        {
            fprintf(file, "%s%s |%s |%s |%s | \\\n   %s | | ",
                    field[0], field[1], field[2], field[3], field[4],
                    field[5]);
            for (j = 0; j < MirrorCount[i]; j++)
            {
                fprintf(file, "\\\n    %s ", ServMirror[i][j]);
            }
            fprintf(file, "\n");
        }
    }
    fclose(file);
    strcpy(buffer, "ReloadBookmarks\n");
    write(DaemonFd, buffer, strlen(buffer));
}

static void new_server(void)
{
    GList *list = NULL;
    GtkWidget *item;
    
    ServAlias[ServerCount] = strdup("*** New server ***");
    ServHost[ServerCount] = strdup("");
    ServPath[ServerCount] = strdup("");
    ServLogin[ServerCount] = strdup("anonymous");
    ServPassword[ServerCount] = strdup(DX_EMailAddress);
    ServComment[ServerCount] = strdup("");
    ServAnon[ServerCount] = TRUE;
    item = gtk_list_item_new_with_label(ServAlias[ServerCount]);
    gtk_widget_show(item);
    list = g_list_append(list, item);
    gtk_list_append_items(GTK_LIST(GTK_COMBO(ConnectCombo)->list), list);
    gtk_list_select_child(GTK_LIST(GTK_COMBO(ConnectCombo)->list), item);
    ServerCount++;
    CheckEnabled();
}

static void delete_server(void)
{
    int i;
    
    if (ServerIndex < 0)
        return;
    for (i = ServerIndex; i < ServerCount - 1; i++)
    {
        free(ServHost[i]);
        ServHost[i] = strdup(ServHost[i + 1]);
        free(ServPath[i]);
        ServPath[i] = strdup(ServPath[i + 1]);
        free(ServLogin[i]);
        ServLogin[i] = strdup(ServLogin[i + 1]);
        free(ServPassword[i]);
        ServPassword[i] = strdup(ServPassword[i + 1]);
        free(ServAlias[i]);
        ServAlias[i] = strdup(ServAlias[i + 1]);
        free(ServComment[i]);
        ServComment[i] = strdup(ServComment[i + 1]);
        ServAnon[i] = ServAnon[i + 1];
    }
    gtk_list_clear_items(GTK_LIST(GTK_COMBO(ConnectCombo)->list),
                         ServerIndex, ServerIndex + 1);
    ServerCount--;
    CheckEnabled();
}

static void do_rename(const char *text)
{
    GList *list = NULL;
    GtkList *listbox = GTK_LIST(GTK_COMBO(ConnectCombo)->list);
    GtkWidget *item[256];
    int i, si;
    
    if (!text)
        return;
    si = ServerIndex;
    free(ServAlias[si]);
    ServAlias[si] = strdup(text);
    gtk_list_clear_items(listbox, 0, ServerCount);
    for (i = 0; i < ServerCount; i++)
    {
        item[i] = gtk_list_item_new_with_label(ServAlias[i]);
        gtk_widget_show(item[i]);
        list = g_list_append(list, item[i]);
    }
    gtk_list_append_items(listbox, list);
    gtk_list_select_child(listbox, item[si]);
}

static void rename_server(void)
{
    char buffer[256], name_buf[256];

    if (ServerIndex < 0)
        return;
    
    if (!strcmp(ServAlias[ServerIndex], ""))
        strcpy(name_buf, ServHost[ServerIndex]);
    else
        strcpy(name_buf, ServAlias[ServerIndex]);
    sprintf(buffer, "Rename \"%s\" to:", name_buf);
    GetInput(do_rename, buffer, name_buf);
}

static void entry_changed(GtkWidget *widget)
{
    char *text = gtk_entry_get_text(GTK_ENTRY(widget));
    
    // we need this to stop it changing back when the list is clicked
    if (ChangingText)
        return;
    if (widget == ConnectHost)
    {
        free(ServHost[ServerIndex]);
        ServHost[ServerIndex] = strdup(text);
    }
    else if (widget == ConnectPath)
    {
        free(ServPath[ServerIndex]);
        ServPath[ServerIndex] = strdup(text);
    }
    else if (widget == ConnectComment)
    {
        free(ServComment[ServerIndex]);
        ServComment[ServerIndex] = strdup(text);
    }
    else if (widget == ConnectUser)
    {
        free(ServLogin[ServerIndex]);
        ServLogin[ServerIndex] = strdup(text);
    }
    else if (widget == ConnectPassword)
    {
        free(ServPassword[ServerIndex]);
        ServPassword[ServerIndex] = strdup(text);
    }
}

static void mirrors_ok(void)
{
    gtk_widget_hide(MirrorsDialogue);
}

static void connect_ok(void)
{
    char *path, *host, url[256];

    gtk_widget_hide(ConnectDialogue);
    PROCESS_EVENTS;
    //save_servers();
    host = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(ConnectCombo)->entry));
    path = gtk_entry_get_text(GTK_ENTRY(ConnectPath));
    sprintf(url, "ftp://%s%s%s", host, (*path == '/') ? "" : "/", path);
    if (GTK_TOGGLE_BUTTON(ConnectAnon)->active)
        OpenUrl(url, "anonymous", DX_EMailAddress);
    else
        OpenUrl(url, gtk_entry_get_text(GTK_ENTRY(ConnectUser)),
                gtk_entry_get_text(GTK_ENTRY(ConnectPassword)));
}

static void quick_connect_ok(void)
{
    char *url = gtk_entry_get_text(GTK_ENTRY(ConnectEntry));

    gtk_widget_hide(ConnectDialogue);
    PROCESS_EVENTS;
    OpenUrl(url, "anonymous", DX_EMailAddress);
}

static void list_select(GtkWidget *widget, GtkWidget *child)
{
    int pos;
    
    if (!child)
        return;

    ChangingText = TRUE;
    pos = gtk_list_child_position(GTK_LIST(GTK_COMBO(ConnectCombo)->list),
                                  child);
    gtk_entry_set_text(GTK_ENTRY(ConnectHost), ServHost[pos]);
    gtk_entry_set_text(GTK_ENTRY(ConnectPath), ServPath[pos]);
    gtk_entry_set_text(GTK_ENTRY(ConnectUser), ServLogin[pos]);
    gtk_entry_set_text(GTK_ENTRY(ConnectPassword), ServPassword[pos]);
    gtk_entry_set_text(GTK_ENTRY(ConnectComment), ServComment[pos]);
    gtk_widget_set_sensitive(ConnectUser, !ServAnon[pos]);
    gtk_widget_set_sensitive(ConnectPassword, !ServAnon[pos]);
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(ConnectAnon), ServAnon[pos]);
    ChangingText = FALSE;
    ServerIndex = pos;
}

static void mirror_clicked(GtkWidget *widget, GdkEvent *event, gpointer data)
{
    BOOL checked;
    char *label;
    int i, j;

    if (widget == NULL)
        return;
    checked = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
    gtk_label_get(GTK_LABEL(GTK_BIN(widget)->child), &label);
    if (checked)
    {
        // check to see if the mirror's already there; if not, add it
        for (i = 0; i < MirrorCount[MirrorIndex]; i++)
        {
            if (!strcmp(ServMirror[MirrorIndex][i], label))
                return;
        }
        ServMirror[MirrorIndex][MirrorCount[MirrorIndex]] = strdup(label);
        MirrorCount[MirrorIndex]++;
    }
    else
    {
        // delete the mirror if possible
        for (i = 0; i < MirrorCount[MirrorIndex]; i++)
        {
            if (!strcmp(ServMirror[MirrorIndex][i], label))
            {
                free(ServMirror[MirrorIndex][i]);
                for (j = i; j < MirrorCount[MirrorIndex]; j++)
                {
                    ServMirror[MirrorIndex][j] = ServMirror[MirrorIndex][j+1];
                }
                MirrorCount[MirrorIndex]--;
                return;
            }
        }
    }
}

void CheckEnabled(void)
{
    switch (ServerCount)
    {
    case 0:
        ChangingText = TRUE;
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(ConnectCombo)->entry), "");
        gtk_widget_set_sensitive(ConnectCombo, FALSE);
        gtk_widget_set_sensitive(ConnectDelete, FALSE);
        gtk_widget_set_sensitive(ConnectRename, FALSE);
        gtk_entry_set_text(GTK_ENTRY(ConnectHost), "");
        gtk_widget_set_sensitive(ConnectHost, FALSE);
        gtk_entry_set_text(GTK_ENTRY(ConnectPath), "");
        gtk_widget_set_sensitive(ConnectPath, FALSE);
        gtk_entry_set_text(GTK_ENTRY(ConnectUser), "");
        gtk_widget_set_sensitive(ConnectUser, FALSE);
        gtk_entry_set_text(GTK_ENTRY(ConnectPassword), "");
        gtk_widget_set_sensitive(ConnectPassword, FALSE);
        gtk_entry_set_text(GTK_ENTRY(ConnectComment), "");
        gtk_widget_set_sensitive(ConnectComment, FALSE);
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(ConnectAnon), TRUE);
        gtk_widget_set_sensitive(ConnectAnon, FALSE);
        ChangingText = FALSE;
        break;

    case 1:
        gtk_widget_set_sensitive(ConnectCombo, TRUE);
        gtk_widget_set_sensitive(ConnectDelete, TRUE);
        gtk_widget_set_sensitive(ConnectRename, TRUE);
        gtk_widget_set_sensitive(ConnectHost, TRUE);
        gtk_widget_set_sensitive(ConnectPath, TRUE);
        gtk_widget_set_sensitive(ConnectUser, !ServAnon[0]);
        gtk_widget_set_sensitive(ConnectPassword, !ServAnon[0]);
        gtk_widget_set_sensitive(ConnectComment, TRUE);
        gtk_widget_set_sensitive(ConnectAnon, TRUE);
        break;

    default:
        break;
    }
}

void CreateConnectDialogue(void)
{
    GtkWidget *line;
    
    // Window and main frames
    ConnectDialogue = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_container_set_border_width(GTK_CONTAINER(ConnectDialogue), 5);
    gtk_window_set_title(GTK_WINDOW(ConnectDialogue), "Connect to server");
    gtk_window_position(GTK_WINDOW(ConnectDialogue), GTK_WIN_POS_CENTER);
    gtk_window_set_policy(GTK_WINDOW(ConnectDialogue), FALSE, FALSE, FALSE);

    ConnectFrame = gtk_vbox_new(FALSE, 5);
    gtk_container_add(GTK_CONTAINER(ConnectDialogue), ConnectFrame);
    gtk_widget_show(ConnectFrame);

    ConnectQuickFrame = gtk_frame_new("Quick Connect");
    gtk_box_pack_start(GTK_BOX(ConnectFrame), ConnectQuickFrame,
                       FALSE, FALSE, 0);
    gtk_widget_show(ConnectQuickFrame);

    ConnectManagerFrame = gtk_frame_new("Connection Manager");
    gtk_box_pack_start(GTK_BOX(ConnectFrame), ConnectManagerFrame,
                       FALSE, FALSE, 0);
    gtk_widget_show(ConnectManagerFrame);

    // Quick connect section
    ConnectQuickRow = gtk_hbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(ConnectQuickFrame), ConnectQuickRow);
    gtk_widget_show(ConnectQuickRow);
    
    ConnectLabel = gtk_label_new("URL:");
    gtk_box_pack_start(GTK_BOX(ConnectQuickRow), ConnectLabel,
                       FALSE, FALSE, 5);
    gtk_widget_show(ConnectLabel);
    
    ConnectEntry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(ConnectQuickRow), ConnectEntry, TRUE, TRUE, 5);
    gtk_widget_show(ConnectEntry);

    ConnectGo = gtk_button_new_with_label("Go");
    gtk_signal_connect(GTK_OBJECT(ConnectGo), "clicked",
                       GTK_SIGNAL_FUNC(quick_connect_ok), NULL);
    gtk_signal_connect_object(GTK_OBJECT(ConnectEntry), "activate",
                              GTK_SIGNAL_FUNC(gtk_button_clicked),
                              GTK_OBJECT(ConnectGo));
    gtk_box_pack_start(GTK_BOX(ConnectQuickRow), ConnectGo, FALSE, FALSE, 2);
    gtk_widget_show(ConnectGo);

    // Connection manager section
    ConnectTable = gtk_table_new(12, 2, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(ConnectTable), 2);
    gtk_container_add(GTK_CONTAINER(ConnectManagerFrame), ConnectTable);
    gtk_widget_show(ConnectTable);

    // Host stuff
    ConnectComboLabel = gtk_label_new("Site:");
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectComboLabel,
                              0, 1, 0, 1);
    gtk_widget_show(ConnectComboLabel);
    ConnectCombo = gtk_combo_new();
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(ConnectCombo)->entry), FALSE);
    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(ConnectCombo)->list),
                       "select_child", list_select, NULL);
    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(ConnectCombo)->entry),
                       "changed", GTK_SIGNAL_FUNC(entry_changed), NULL);
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectCombo,
                              1, 2, 0, 1);
    gtk_widget_show(ConnectCombo);

    ConnectEditBox = gtk_hbox_new(TRUE, 5);
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectEditBox,
                              1, 2, 1, 2);
    gtk_widget_show(ConnectEditBox);

    ConnectNew = gtk_button_new_with_label("New");
    gtk_signal_connect(GTK_OBJECT(ConnectNew), "clicked",
                       GTK_SIGNAL_FUNC(new_server), NULL);
    gtk_box_pack_start(GTK_BOX(ConnectEditBox), ConnectNew, TRUE, TRUE, 0);
    gtk_widget_show(ConnectNew);

    ConnectDelete = gtk_button_new_with_label("Delete");
    gtk_signal_connect(GTK_OBJECT(ConnectDelete), "clicked",
                       GTK_SIGNAL_FUNC(delete_server), NULL);
    gtk_box_pack_start(GTK_BOX(ConnectEditBox), ConnectDelete,
                       TRUE, TRUE, 0);
    gtk_widget_show(ConnectDelete);

    ConnectRename = gtk_button_new_with_label("Rename");
    gtk_signal_connect(GTK_OBJECT(ConnectRename), "clicked",
                       GTK_SIGNAL_FUNC(rename_server), NULL);
    gtk_box_pack_start(GTK_BOX(ConnectEditBox), ConnectRename,
                       TRUE, TRUE, 0);
    gtk_widget_show(ConnectRename);
    
    line = gtk_hseparator_new();
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), line, 0, 2, 2, 3);
    gtk_widget_show(line);
    
    // Host info
    ConnectHostLabel = gtk_label_new("Host:");
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectHostLabel,
                              0, 1, 3, 4);
    gtk_widget_show(ConnectHostLabel);
    ConnectHost = gtk_entry_new();
    gtk_signal_connect(GTK_OBJECT(ConnectHost), "changed",
                       GTK_SIGNAL_FUNC(entry_changed), NULL);
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectHost,
                              1, 2, 3, 4);
    gtk_widget_show(ConnectHost);
    
    ConnectPathLabel = gtk_label_new("Path:");
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectPathLabel,
                              0, 1, 4, 5);
    gtk_widget_show(ConnectPathLabel);
    ConnectPath = gtk_entry_new();
    gtk_signal_connect(GTK_OBJECT(ConnectPath), "changed",
                       GTK_SIGNAL_FUNC(entry_changed), NULL);
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectPath,
                              1, 2, 4, 5);
    gtk_widget_show(ConnectPath);
    
    ConnectCommentLabel = gtk_label_new("Comment:");
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectCommentLabel,
                              0, 1, 5, 6);
    gtk_widget_show(ConnectCommentLabel);
    ConnectComment = gtk_entry_new();
    gtk_signal_connect(GTK_OBJECT(ConnectComment), "changed",
                       GTK_SIGNAL_FUNC(entry_changed), NULL);
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectComment,
                              1, 2, 5, 6);
    gtk_widget_show(ConnectComment);

    line = gtk_hseparator_new();
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), line, 0, 2, 6, 7);
    gtk_widget_show(line);
    
    ConnectUserLabel = gtk_label_new("User:");
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectUserLabel,
                              0, 1, 7, 8);
    gtk_widget_show(ConnectUserLabel);
    ConnectUser = gtk_entry_new();
        gtk_signal_connect(GTK_OBJECT(ConnectUser), "changed",
                       GTK_SIGNAL_FUNC(entry_changed), NULL);
    gtk_widget_set_sensitive(ConnectUser, FALSE);
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectUser,
                              1, 2, 7, 8);
    gtk_widget_show(ConnectUser);

    ConnectPasswordLabel = gtk_label_new("Password:");
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectPasswordLabel,
                              0, 1, 8, 9);
    gtk_widget_show(ConnectPasswordLabel);
    ConnectPassword = gtk_entry_new();
    gtk_entry_set_visibility(GTK_ENTRY(ConnectPassword), FALSE);
    gtk_signal_connect(GTK_OBJECT(ConnectPassword), "changed",
                       GTK_SIGNAL_FUNC(entry_changed), NULL);
    gtk_widget_set_sensitive(ConnectPassword, FALSE);
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectPassword,
                              1, 2, 8, 9);
    gtk_widget_show(ConnectPassword);

    ConnectAnon = gtk_check_button_new_with_label("Anonymous login");
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectAnon,
                              1, 2, 9, 10);
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(ConnectAnon), TRUE);
    gtk_signal_connect(GTK_OBJECT(ConnectAnon), "toggled",
                       GTK_SIGNAL_FUNC(toggle_anon), NULL);
    gtk_widget_show(ConnectAnon);

    // "Mirrors" and "Save" buttons
    ConnectButtonRow = gtk_hbox_new(TRUE, 2);
    gtk_table_attach_defaults(GTK_TABLE(ConnectTable), ConnectButtonRow,
                              0, 2, 10, 11);
    gtk_widget_show(ConnectButtonRow);

    ConnectMirrors = gtk_button_new_with_label("Mirrors...");
    gtk_signal_connect(GTK_OBJECT(ConnectMirrors), "clicked", 
                       GTK_SIGNAL_FUNC(edit_mirrors), NULL);
    gtk_box_pack_start(GTK_BOX(ConnectButtonRow), ConnectMirrors,
                       TRUE, TRUE, 5);
    gtk_widget_show(ConnectMirrors);
    
    ConnectSave = gtk_button_new_with_label("Save");
    gtk_signal_connect(GTK_OBJECT(ConnectSave), "clicked", 
                       GTK_SIGNAL_FUNC(save_servers), NULL);
    gtk_box_pack_start(GTK_BOX(ConnectButtonRow), ConnectSave,
                       TRUE, TRUE, 5);
    gtk_widget_show(ConnectSave);
    
    // Buttons at bottom
    ConnectBottomRow = gtk_hbox_new(TRUE, 2);
    gtk_box_pack_start(GTK_BOX(ConnectFrame),
                       ConnectBottomRow, FALSE, FALSE, 5);
    gtk_widget_show(ConnectBottomRow);
    
    ConnectOk = gtk_button_new_with_label("OK");
    gtk_box_pack_start(GTK_BOX(ConnectBottomRow), ConnectOk, TRUE, TRUE, 5);
    gtk_widget_show(ConnectOk);
    gtk_signal_connect(GTK_OBJECT(ConnectOk), "clicked",
                       GTK_SIGNAL_FUNC(connect_ok), NULL);
    ConnectCancel = gtk_button_new_with_label("Cancel");
    gtk_box_pack_start(GTK_BOX(ConnectBottomRow),
                       ConnectCancel, TRUE, TRUE, 5);
    gtk_signal_connect_object(GTK_OBJECT(ConnectCancel), "clicked",
                              GTK_SIGNAL_FUNC(gtk_widget_hide),
                              GTK_OBJECT(ConnectDialogue));
    gtk_widget_show(ConnectCancel);
    
    ReadServersFile();
    CreateMirrorsDialogue();
}

void CreateMirrorsDialogue(void)
{
    MirrorsDialogue = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_container_set_border_width(GTK_CONTAINER(MirrorsDialogue), 5);
    gtk_window_set_title(GTK_WINDOW(MirrorsDialogue), "Edit mirrors");
    gtk_window_position(GTK_WINDOW(MirrorsDialogue), GTK_WIN_POS_CENTER);
    gtk_window_set_default_size(GTK_WINDOW(MirrorsDialogue), 200, 200);
    gtk_window_set_policy(GTK_WINDOW(MirrorsDialogue), TRUE, TRUE, TRUE);
    gtk_window_set_modal(GTK_WINDOW(MirrorsDialogue), TRUE);
    
    MirrorsFrame = gtk_vbox_new(FALSE, 5);
    gtk_container_add(GTK_CONTAINER(MirrorsDialogue), MirrorsFrame);
    gtk_widget_show(MirrorsFrame);
    
    MirrorsBox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(MirrorsBox),
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(MirrorsFrame), MirrorsBox, TRUE, TRUE, 5);
    gtk_widget_show(MirrorsBox);
    
    MirrorsList = gtk_vbox_new(FALSE, 5);
    //gtk_container_add(GTK_CONTAINER(MirrorsBox), MirrorsList);
    // this causes major imlib crashes - why?
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(MirrorsBox),
                                          MirrorsList);
    gtk_widget_show(MirrorsList);
    
    MirrorsOk = gtk_button_new_with_label("OK");
    gtk_box_pack_start(GTK_BOX(MirrorsFrame), MirrorsOk, FALSE, FALSE, 5);
    gtk_widget_show(MirrorsOk);
    gtk_signal_connect(GTK_OBJECT(MirrorsOk), "clicked",
                       GTK_SIGNAL_FUNC(mirrors_ok), NULL);
}

void ShowMirrorsDialogue(int index)
{
    char buffer[256], *label;
    int i, j;

    if (!strcmp(ServAlias[index], ""))
        sprintf(buffer, "Mirrors for \"%s\"", ServHost[index]);
    else
        sprintf(buffer, "Mirrors for \"%s\"", ServAlias[index]);
    gtk_window_set_title(GTK_WINDOW(MirrorsDialogue), buffer);
    
    for (i = 0; i < MirrorCheckCount; i++)
    {
        gtk_widget_destroy(MirrorCheckBox[i]);
    }
    MirrorCheckCount = 0;
    for (i = 0; i < ServerCount; i++)
    {
        // we don't want to allow mirrors for ourself
        if (i == index)
            continue;
        sprintf(buffer, "%s%s", ServHost[i],
                (!strcmp(ServPath[i], "/") ? "" : ServPath[i]));
        MirrorCheckBox[MirrorCheckCount] =
            gtk_check_button_new_with_label(buffer);
        gtk_signal_connect(GTK_OBJECT(MirrorCheckBox[MirrorCheckCount]),
                           "clicked", GTK_SIGNAL_FUNC(mirror_clicked),
                           (gpointer)MirrorCheckCount);
        gtk_box_pack_start(GTK_BOX(MirrorsList),
                           MirrorCheckBox[MirrorCheckCount], FALSE, FALSE, 0);
        gtk_widget_show(MirrorCheckBox[MirrorCheckCount]);
        MirrorCheckCount++;
    }
    MirrorIndex = index;
    for (i = 0; i < MirrorCount[index]; i++)
    {
        for (j = 0; j < MirrorCheckCount; j++)
        {
            gtk_label_get(GTK_LABEL(GTK_BIN(MirrorCheckBox[j])->child),
                          &label);
            if (!strcmp(ServMirror[index][i], label))
            {
                gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(
                    MirrorCheckBox[j]), TRUE);
                break;
            }
        }
    }
    gtk_widget_show(MirrorsDialogue);
}


void ReadServersFile(void)
{
    FILE *file;
    char buffer[256], *field_ptr, *buf_ptr;
    char comment[256], user_name[256], password[256];
    char alias[256], flags[256], url[256], mirror[256];
    GList *list = NULL;
    GtkWidget *item[256];

    ServerCount = 0;
    sprintf(buffer, "%s/servers", DX_ProgDir);
    file = fopen(buffer, "r");
    if (file)
    {
        memset(buffer, 0, sizeof(buffer));
        while (DX_ReadFileLine(buffer, sizeof(buffer), file))
        {
            if (buffer[strlen(buffer) - 1] == '\n')
                buffer[strlen(buffer) - 1] = '\0';
            if (buffer[strlen(buffer) - 1] == '\r')
                buffer[strlen(buffer) - 1] = '\0';
            
            if (!(buf_ptr = DX_StripComments(buffer)))
                continue;
            
            //error(E_TRACE, "Servers file: Read line %s", buf_ptr);
            
// URL | UserName | Password | Alias | Comment | Flags | Mirrors

            strcpy(url, "");
            strcpy(user_name, "");
            strcpy(password, "");
            strcpy(alias, "");
            strcpy(comment, "");
            strcpy(flags, "");
            
            field_ptr = DX_GetNextField(buffer);
            if (*field_ptr)
            {
                // because the GetNextField call does not remove
                // quotes, we have to do it ourselves
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(url, field_ptr);
                if (url[strlen(url) - 1] == '"')
                    url[strlen(url) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;
            
            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(user_name, field_ptr);
                if (user_name[strlen(user_name) - 1] == '"')
                    user_name[strlen(user_name) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;

            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(password, field_ptr);
                if (password[strlen(password) - 1] == '"')
                    password[strlen(password) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;

            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(alias, field_ptr);
                if (alias[strlen(alias) - 1] == '"')
                    alias[strlen(alias) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;

            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(comment, field_ptr);
                if (comment[strlen(comment) - 1] == '"')
                    comment[strlen(comment) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;

            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(flags, field_ptr);
                if (flags[strlen(flags) - 1] == '"')
                    flags[strlen(flags) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;
            while (*field_ptr && (isspace(*field_ptr) || (*field_ptr == '|')))
                field_ptr++;
            
            while (*field_ptr)
            {
                field_ptr = DX_GetNextField(field_ptr);
                if (!*field_ptr)
                    continue;

                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(mirror, field_ptr);
                if (mirror[strlen(mirror) - 1] == '"')
                    mirror[strlen(mirror) - 1] = '\0';
                ServMirror[ServerCount][MirrorCount[ServerCount]++] =
                    strdup(mirror);
                field_ptr += strlen(field_ptr) + 1;
            }

            if (strcmp(url, ""))
            {
                ServHost[ServerCount] = malloc(256);
                ServPath[ServerCount] = malloc(256);
                DX_ParseUrl(url, buffer, ServHost[ServerCount], 256,
                            ServPath[ServerCount], 256);
                ServLogin[ServerCount] = strdup(user_name);
                ServPassword[ServerCount] = strdup(password);
                ServAlias[ServerCount] = strdup(alias);
                ServComment[ServerCount] = strdup(comment);
                if (!strcasecmp(ServLogin[ServerCount], "anonymous") ||
                    (strlen(ServLogin[ServerCount]) == 0))
                {
                    ServAnon[ServerCount] = TRUE;
                }
                else
                {
                    ServAnon[ServerCount] = FALSE;
                }
                if (strcmp(ServAlias[ServerCount], ""))
                {
                    item[ServerCount] =
                        gtk_list_item_new_with_label(ServAlias[ServerCount]);
                }
                else
                {
                    item[ServerCount] =
                        gtk_list_item_new_with_label(ServHost[ServerCount]);
                }
                gtk_widget_show(item[ServerCount]);
                list = g_list_append(list, item[ServerCount]);
                ServerCount++;
            }
        }
        gtk_list_append_items(GTK_LIST(GTK_COMBO(ConnectCombo)->list), list);
        fclose(file);
    }
    if (ServerCount == 0)
        ServerIndex = -1;
    CheckEnabled();
}
