#ifndef lint
static char rcsId[]="$Header: /home/linas/cvsroot/xacc/lib/XmHTML-1.1.0/src/forms.c,v 1.1 1997/11/30 05:13:09 linas Exp $";
#endif
/*****
* forms.c : XmHTML form support
*
* This file Version	$Revision: 1.1 $
*
* Creation date:		Thu Apr  3 16:06:10 GMT+0100 1997
* Last modification: 	$Date: 1997/11/30 05:13:09 $
* By:					$Author: linas $
* Current State:		$State: Exp $
*
* Author:				newt
*
* Copyright (C) 1994-1997 by Ripley Software Development 
* All Rights Reserved
*
* This file is part of the XmHTML Widget Library.
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the Free
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/
/*****
* ChangeLog 
* $Log: forms.c,v $
* Revision 1.1  1997/11/30 05:13:09  linas
* import XmHTML source for the help widget
*
* Revision 1.5  1997/08/31 17:35:00  newt
* Fixed translation table memory leak, translation tables are now only
* parsed once and then reused.
*
* Revision 1.4  1997/08/30 01:00:46  newt
* *Lots* of changes. Every HTML form component is now supported.
* Traversal and form reset more-or-less works.
*
* Revision 1.3  1997/08/01 13:00:42  newt
* Much enhancements: all elements except <textarea> are now supported.
*
* Revision 1.2  1997/05/28 01:47:30  newt
* ?
*
* Revision 1.1  1997/04/29 14:19:34  newt
* Initial Revision
*
*****/ 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* motif includes */
#include <Xm/DrawnB.h>
#include <Xm/FileSB.h>
#include <Xm/Form.h>
#include <Xm/List.h>
#include <Xm/PushB.h>
#include <Xm/PushBG.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/ToggleB.h>

#include "XmHTMLP.h"
#include "XmHTMLfuncs.h"

/*** External Function Prototype Declarations ***/

/*** Public Variable Declarations ***/

/*** Private Datatype Declarations ****/

/*** Private Function Prototype Declarations ****/

/*** Private Variable Declarations ***/
#define MY_FONTLIST_DEFAULT_TAG "XmHTMLDefaultFontList"

/* scratch stuff */
static XmHTMLFormData *current_form;
static XmHTMLForm *current_entry;
static XmFontList my_fontList;
static Arg args[20];
static Dimension argc;

/*****
* We override the following translations, we handle navigation accross form
* components ourselves.
******/
static char trav_translations[] =
"~Shift ~Meta ~Ctrl <Key>Tab: traverse-next()\n\
Shift ~Meta ~Ctrl <Key>Tab: traverse-prev()\n\
Ctrl Shift <Key>Tab: traverse-next-or-prev(1)\n\
Ctrl <Key>Tab: traverse-next-or-prev(0)\n\
Shift <Key>Tab: traverse-prev()\n\
<Key>Tab: traverse-next()";

/*****
* Textfield translations
*****/
static char textF_translations[] =
"~Meta ~Alt <Key>Tab: traverse-next()\n\
<Btn1Down>: grab-focus() traverse-current()";

/*****
* PushButton and toggleButton translations. We want it them to take focus
* when pressed.
*****/
static char pushB_translations[] =
"<Btn1Down>: Arm() traverse-current()";

static XtTranslations textFTranslations = NULL;
static XtTranslations travTranslations = NULL;
static XtTranslations pushBTranslations = NULL;

/*****
* Name: 		getInputType
* Return Type: 	componentType
* Description: 	retrieves the type of an <input> HTML form member.
* In: 
*	attrib..:	attributes to check
* Returns:
*	componenttype if ``type'' is present in attributes. FORM_TEXT is
*	returned if type is not present or type is invalid/misspelled.
*****/
static componentType
getInputType(String attributes)
{
	String chPtr;
	componentType ret_val = FORM_TEXT;

	/* if type isn't specified we default to a textfield */
	if((chPtr = _XmHTMLTagGetValue(attributes, "type")) == NULL)
		return(ret_val);

	if(!(strcasecmp(chPtr, "text")))
		ret_val = FORM_TEXT;
	else if(!(strcasecmp(chPtr, "password")))
		ret_val = FORM_PASSWD;
	else if(!(strcasecmp(chPtr, "checkbox")))
		ret_val = FORM_CHECK;
	else if(!(strcasecmp(chPtr, "radio")))
		ret_val = FORM_RADIO;
	else if(!(strcasecmp(chPtr, "submit")))
		ret_val = FORM_SUBMIT;
	else if(!(strcasecmp(chPtr, "reset")))
		ret_val = FORM_RESET;
	else if(!(strcasecmp(chPtr, "file")))
		ret_val = FORM_FILE;
	else if(!(strcasecmp(chPtr, "hidden")))
		ret_val = FORM_HIDDEN;
	else if(!(strcasecmp(chPtr, "image")))
		ret_val = FORM_IMAGE;
	free(chPtr);
	return(ret_val);
}

/*****
* Name: 		freeForm
* Return Type: 	void
* Description: 	releases all memory occupied by the given form component.
* In: 
*	entry:		form component to be released;
* Returns:
*	nothing.
*****/
static void
freeForm(XmHTMLForm *entry)
{
	XmHTMLForm *tmp;

	while(entry != NULL)
	{
		tmp = entry->next;
		if(entry->w)
		{
			/* move of screen */
			XtMoveWidget(entry->w, -1000, -1000);
			/* destroy */
			XtDestroyWidget(entry->w);
		}

		if(entry->name)
			free(entry->name);
		if(entry->value)
			free(entry->value);

		/* call ourselves recursively to destroy all option members */
		if(entry->options)
			freeForm(entry->options);

		free(entry);
		entry = tmp;
	}
}

/********** 
* Various callback routines: text activation & password modification, button
* activation, etc...
**********/

/*****
* Name: 		textActivateCB
* Return Type: 	void
* Description: 	textfield activation callback. Moves the focus down
*				when <return> is hit.
* In: 
*	w:			textfield widget id;
*	client..:	unused;
*	call_d..:	unused;
* Returns:
*	nothing.
*****/
static void
textActivateCB(Widget w, XtPointer client_data, XtPointer call_data)
{
	_XmHTMLProcessTraversal(w, XmTRAVERSE_NEXT);
}

/*****
* Name: 		passwdCB
* Return Type: 	void
* Description: 	password hiding routine. Modifies the chars entered to only
*				display '*' and saves the really entered text in the content
*				field of the current form entry.
* In: 
*	w:			TextField widget id;
*	client..:	client data registered with this callback, in this case the
*				password form component data;
*	call_d..:	call data for this callback, in this case a
*				verifyCallbackStruct.
* Returns:
*	nothing.
*****/
static void
passwdCB(Widget w, XtPointer client_data, XtPointer call_data)
{
	XmHTMLForm *entry = (XmHTMLForm*)client_data;
	XmTextVerifyCallbackStruct *cbs = (XmTextVerifyCallbackStruct*)call_data;
	String passwd = NULL;
	int len = 0;

	if(cbs->text->ptr == NULL)
	{
		/* backspace */
		if(entry->content == NULL)
			return;
		cbs->endPos = strlen(entry->content);	/* delete from here to end */
		entry->content[cbs->startPos] = 0;		/* backspace end */
		return;
	}
	/*
	* Only one char at a time, no pasting allowed so user *types* in the
	* password.
	*/
	if(cbs->text->length > 1)
	{
		cbs->doit = False;
		/* should we actually be ringing the bell here? */
		XBell(XtDisplay(w), 100);
		return;
	}
	passwd = (String)malloc(cbs->endPos+2);	/* new char + \0 */
	if(entry->content)
	{
		strcpy(passwd, entry->content);
		free(entry->content);
	}
	else
		passwd[0] = '\0';
	
	entry->content = passwd;
	strncat(entry->content, cbs->text->ptr, cbs->text->length);
	entry->content[cbs->endPos + cbs->text->length] = '\0';
	/* insert starts */
	for(len = 0; len < cbs->text->length; len++)
		cbs->text->ptr[len] = '*';
}

static void
buttonActivateCB(Widget w, XtPointer client_data, XtPointer call_data)
{
	XmHTMLForm *entry = (XmHTMLForm*)client_data;
	XmAnyCallbackStruct *cbs = (XmAnyCallbackStruct*)call_data;
	XmHTMLWidget html;

	/* our html widget id is stored in the parent form of this entry */
	html = (XmHTMLWidget)entry->parent->html;

	/* a submit button was pressed */
	if(entry->type == FORM_SUBMIT)
	{
		_XmHTMLDebug(12, ("forms.c: buttonActivateCB for FORM_SUBMIT\n"));
		_XmHTMLFormActivate(html, cbs->event, entry);
	}
	else if(entry->type == FORM_RESET)	/* a reset button was pressed */
	{
		_XmHTMLDebug(12, ("forms.c: buttonActivateCB for FORM_RESET\n"));
		_XmHTMLFormReset(html, entry);
	}
}

static void
radioChangedCB(Widget w, XtPointer client_data, XtPointer call_data)
{
	XmHTMLForm *tmp, *entry = (XmHTMLForm*)client_data;
	XmToggleButtonCallbackStruct *cbs =
		(XmToggleButtonCallbackStruct*)call_data;

	/* toggle set, unset all other toggles */
	if(cbs->set)
	{
		/* get start of this radiobox */
		for(tmp = entry->parent->components; tmp != NULL; tmp = tmp->next)
			if(tmp->type == FORM_RADIO && !(strcasecmp(tmp->name, entry->name)))
				break;

		/* sanity */
		if(tmp == NULL)
			return;

		/* unset all other toggle buttons in this radiobox */
		for(; tmp != NULL; tmp = tmp->next)
		{
			if(tmp->type == FORM_RADIO && tmp != entry)
			{
				/* same group, unset it */
				if(!(strcasecmp(tmp->name, entry->name)))
					XtVaSetValues(tmp->w, XmNset, False, NULL);
				/*****
				* Not a member of this group, we processed all elements in
				* this radio box, break out.
				*****/
				else
					break;
			}
		}
	}
	else /* current toggle can't be unset */
		XtVaSetValues(entry->w, XmNset, True, NULL);
}

static void
fileActivateCB(Widget w, XtPointer client_data, XtPointer call_data)
{
	XmHTMLForm *entry = (XmHTMLForm*)client_data;

	if(entry->parent->fileSB == NULL)
	{
		entry->parent->fileSB =XmCreateFileSelectionDialog(entry->parent->html,
			"_fileDialog", NULL, 0);

		/* callbacks */
		XtAddCallback(entry->parent->fileSB, XmNcancelCallback,
			(XtCallbackProc)XtUnmanageChild, NULL);

		XtVaSetValues(XtParent(entry->parent->fileSB),
			entry->value ? entry->value : "Select A File");
	}
	XtManageChild(entry->parent->fileSB);
	XtPopup(XtParent(entry->parent->fileSB), XtGrabNone);
	XMapRaised(XtDisplay(entry->parent->html),
		XtWindow(XtParent(entry->parent->fileSB)));
}

static void
finalizeEntry(XmHTMLWidget html, XmHTMLForm *entry, Boolean insert) 
{
	if(entry->w)
	{
		Dimension w = 0, h = 0;

		/*
		* Set values. We place this thing completely off screen so
		* no nasty things happen when this widget is mapped to it's correct
		* position for the first time.
		*/
		XtVaSetValues(entry->w,
			XmNx, -1000,
			XmNy, -1000,
			NULL);

		/* get widget dimensions */
		XtVaGetValues(entry->w,
			XmNwidth, &w,
			XmNheight, &h,
			NULL);
		entry->width = w;
		entry->height = h;
	}
	else
	{
		entry->width = 0;
		entry->height = 0;
	}

	/* add to parent form when requested */
	if(insert)
	{
		if(current_entry)
		{
			entry->prev = current_entry;
			current_entry->next = entry;
			current_entry = entry;
		}
		else
		{
			current_form->components = current_entry = entry;
		}
		/* and keep up component counter */
		current_form->ncomponents++;
	}
	_XmHTMLDebug(12, ("forms.c: finalizeEntry, added form entry, "
		"type = %i, name = %s\n", entry->type, entry->name));
}

/********
****** Public Functions
********/

/*****
* Name:			_XmHTMLStartForm
* Return Type:	void
* Description:	creates and initializes a new form parent
* In:
*	html:		XmHTMLWidget id;
*	attr..:		form attributes.
* Returns:
*	nothing but attaches it to the widget's form_data list
*****/
void
_XmHTMLStartForm(XmHTMLWidget html, String attributes)
{
	static XmHTMLFormData *form;
	XmFontListEntry my_fontEntry;
	String my_fonttag = MY_FONTLIST_DEFAULT_TAG;

	/* empty form, no warning, just return */
	if(attributes == NULL)
		return;

	/* allocate a new entry */
	form = (XmHTMLFormData*)malloc(sizeof(XmHTMLFormData));
	/* initialise to zero */
	(void)memset(form, 0, sizeof(XmHTMLFormData));

	/* this form starts a new set of entries */
	current_entry = NULL;

	/* set form owner */
	form->html = (Widget)html;

	/* pick up action */
	if((form->action = _XmHTMLTagGetValue(attributes, "action")) == NULL)
	{
		/* the action tag is required, so destroy and return if not found */
		free(form);
		form = NULL;
#ifdef PEDANTIC
		_XmHTMLWarning(__WFUNC__(html, "_XmHTMLStartForm"),
			"Bad HTML form: no action tag found, form ignored.");
#endif
		return;
	}
	/* default method is GET (0) */
	form->method = _XmHTMLTagCheck(attributes, "post");
	/* form encoding */
	if((form->enctype = _XmHTMLTagGetValue(attributes, "enctype")) == NULL)
		form->enctype = strdup("application/x-www-form-urlencoded");

	if(html->html.form_data)
	{
		form->prev = current_form;
		current_form->next = form;
		current_form = form;
	}
	else
		html->html.form_data = current_form = form;
	_XmHTMLDebug(12, ("forms.c: _XmHTMLStartForm, created a new form "
		"entry, action = %s\n", form->action));

	/* create a valid fontList context for our default font */
	my_fontEntry = XmFontListEntryCreate(my_fonttag, XmFONT_IS_FONT,
					(XtPointer)html->html.default_font->xfont);
	my_fontList = XmFontListAppendEntry(NULL, my_fontEntry);
	XmFontListEntryFree(&my_fontEntry);

	/* translations overrides */
	if(textFTranslations == NULL)
		textFTranslations = XtParseTranslationTable(textF_translations);
	if(travTranslations == NULL)
		travTranslations = XtParseTranslationTable(trav_translations);
	if(pushBTranslations == NULL)
		pushBTranslations = XtParseTranslationTable(pushB_translations);
}

/*****
* Name:			_XmHTMLEndForm
* Return Type: 	void
* Description: 	invalidates the current parent form.
* In: 
*	html:		XmHTMLWidget id.
* Returns:
*	nothing.
*****/
void
_XmHTMLEndForm(XmHTMLWidget html)
{
	current_entry = NULL;
#ifdef DEBUG
	_XmHTMLDebug(12, ("forms.c: _XmHTMLEndForm, listing for form %s.\n",
		current_form->action));
	for(current_entry = current_form->components; current_entry != NULL;
		current_entry = current_entry->next)
	{
		_XmHTMLDebug(12, ("\tname = %s, type = %i\n", current_entry->name,
			current_entry->type));
	}
#endif

	/* free fontList context */
	XmFontListFree(my_fontList);
	my_fontList  = (XmFontList)NULL;
}

/*****
* Name: 		_XmHTMLFormAddInput
* Return Type: 	XmHTMLForm
* Description: 	creates a form input entry 
* In: 
*	html:		XmHTMLWidget id;
*	attributes:	input attributes;
* Returns:
*	a new XmHTMLForm entry.
*****/
XmHTMLForm*
_XmHTMLFormAddInput(XmHTMLWidget html, String attributes)
{
	static XmHTMLForm *entry;
	String chPtr;
	XmString label;
	Widget parent;

	/*****
	* HTML form child widgets are childs of the workarea.
	* Making them a direct child of the widget itself messes up scrolling.
	*****/
	parent = html->html.work_area;

	if(attributes == NULL)
		return(NULL);

	if(current_form == NULL)
	{
		_XmHTMLWarning(__WFUNC__(html, "_XmHTMLFormAddInput"),
			"Bad HTML form: <INPUT> not within form.");
	}

	/* Create and initialise a new entry */
	entry = (XmHTMLForm*)malloc(sizeof(XmHTMLForm));
	(void)memset(entry, 0, sizeof(XmHTMLForm));

	/* set parent form */
	entry->parent = current_form;

	entry->type = getInputType(attributes);

	/* get name */
	if((chPtr = _XmHTMLTagGetValue(attributes, "name")) == NULL)
	{
		switch(entry->type)
		{
			case FORM_TEXT:
				chPtr = "text";
				break;
			case FORM_PASSWD:
				chPtr = "Password";
				break;
			case FORM_CHECK:
				chPtr = "CheckBox";
				break;
			case FORM_RADIO:
				chPtr = "RadioBox";
				break;
			case FORM_RESET:
				chPtr = "Reset";
				break;
			case FORM_FILE:
				chPtr = "File";
				break;
			case FORM_IMAGE:
				chPtr = "Image";
				break;
			case FORM_HIDDEN:
				chPtr = "Hidden";
				break;
			case FORM_SUBMIT:
				chPtr = "Submit";
				break;
		}
		entry->name = strdup(chPtr);
	}
	else
	{
		entry->name = strdup(chPtr);
		free(chPtr);
	}

	entry->value = _XmHTMLTagGetValue(attributes, "value");
	entry->checked = _XmHTMLTagCheck(attributes, "checked");
	if(entry->type == FORM_TEXT || entry->type == FORM_PASSWD)
	{
		/* default to 25 columns if size hasn't been specified */
		entry->size = _XmHTMLTagGetNumber(attributes, "size", 25);

		/* unlimited amount of text input if not specified */
		entry->maxlength = _XmHTMLTagGetNumber(attributes, "maxlength", -1);

		/* passwd can't have a default value */
		if(entry->type == FORM_PASSWD && entry->value)
		{
			free(entry->value);
			entry->value = NULL;
		}
		/* empty value if none given */
		if(entry->value == NULL)
		{
			entry->value = (String)malloc(1);
			entry->value[0] = '\0';
		}
	}
	else if(entry->type == FORM_FILE)
	{
		/* default to 5 files if size hasn't been specified */
		entry->size = _XmHTMLTagGetNumber(attributes, "size", 5);

		/* unlimited amount of files if not specified */
		entry->maxlength = _XmHTMLTagGetNumber(attributes, "maxlength", -1);
	}
	entry->align = _XmHTMLGetImageAlignment(attributes);

	/*****
	* go create the actual widget
	* As image buttons are promoted to image words we don't deal with the
	* FORM_IMAGE case. For hidden form fields nothing needs to be done.
	*****/
	if(entry->type != FORM_IMAGE && entry->type != FORM_HIDDEN)
	{
		/* defaults for all widgets */
		argc = 0;
		XtSetArg(args[argc], XmNmappedWhenManaged, False); argc++;
		XtSetArg(args[argc], XmNborderWidth, 0); argc++;
		XtSetArg(args[argc], XmNbackground, html->html.body_bg); argc++;
		XtSetArg(args[argc], XmNforeground, html->html.body_fg); argc++;
		XtSetArg(args[argc], XmNfontList, my_fontList); argc++;

		switch(entry->type)
		{
			/* text field, set args and create it */
			case FORM_TEXT:
			case FORM_PASSWD:
				XtSetArg(args[argc], XmNcolumns, entry->size); argc++;
				XtSetArg(args[argc], XmNvalue, entry->value); argc++;
				XtSetArg(args[argc], XmNhighlightThickness, 0); argc++;

				if(entry->maxlength != -1)
				{
					XtSetArg(args[argc], XmNmaxLength, entry->maxlength);
					argc++;
				}
				entry->w = XmCreateTextField(parent, entry->name, args, argc);

				/* override some translations */
				XtOverrideTranslations(entry->w, textFTranslations);
				XtOverrideTranslations(entry->w, travTranslations);

				/* manage it */
				XtManageChild(entry->w);

				/* callbacks */
				XtAddCallback(entry->w, XmNactivateCallback,
					(XtCallbackProc)textActivateCB, (XtPointer)entry);
				if(entry->type == FORM_PASSWD)
					XtAddCallback(entry->w, XmNmodifyVerifyCallback,
						(XtCallbackProc)passwdCB, (XtPointer)entry);

				break;
			/* toggle buttons, set args and create */
			case FORM_CHECK:
			case FORM_RADIO:
				XtSetArg(args[argc], XmNindicatorType,
					(entry->type == FORM_CHECK ? XmN_OF_MANY : XmONE_OF_MANY));
				argc++;
				/* empty label, text following this input will contain it */
				label = XmStringCreate("", MY_FONTLIST_DEFAULT_TAG);
				XtSetArg(args[argc], XmNlabelString, label); argc++;
				XtSetArg(args[argc], XmNset, entry->checked); argc++;

				entry->w = XmCreateToggleButton(parent, entry->name, args,
					argc);

				XmStringFree(label);

				/* override some translations */
				XtOverrideTranslations(entry->w, travTranslations);
				XtOverrideTranslations(entry->w, pushBTranslations);

				/* manage it */
				XtManageChild(entry->w);

				/* only do callbacks for radio buttons */
				if(entry->type == FORM_RADIO)
				{
					XtAddCallback(entry->w, XmNvalueChangedCallback,
						(XtCallbackProc)radioChangedCB, (XtPointer)entry);
				}
				break;

			/*
			* special case: this type of input is a listbox with a ``browse''
			* button. Position of the button is governed by the align attribute.
			* size of the listbox defaults to 5. maxlength indicates max no of
			* allowable entries.
			*/
			case FORM_FILE:
				{
					static Widget list, button;

					/* first create the form container */
					entry->w = XmCreateForm(parent, entry->name, args, argc);

					/* list widget */
					argc = 0;
					XtSetArg(args[argc], XmNlistSizePolicy, XmCONSTANT);
					argc++;
					XtSetArg(args[argc], XmNvisibleItemCount, entry->size);
					argc++;
					XtSetArg(args[argc], XmNselectionPolicy, XmSINGLE_SELECT);
					argc++;
					XtSetArg(args[argc], XmNfontList, my_fontList);
					argc++;
					XtSetArg(args[argc], XmNborderWidth, 0);
					argc++;
					XtSetArg(args[argc], XmNbackground, html->html.body_bg);
					argc++;
					XtSetArg(args[argc], XmNforeground, html->html.body_fg);
					argc++;
					/* constraints */
					XtSetArg(args[argc], XmNtopAttachment, XmATTACH_FORM);
					argc++;
					XtSetArg(args[argc], XmNleftAttachment, XmATTACH_FORM);
					argc++;

					/* create */
					list = XmCreateList(entry->w, "_fileList", args, argc);

					/* browse button */
					argc = 0;
					label = XmStringCreate(entry->value ? entry->value :
						"Browse", MY_FONTLIST_DEFAULT_TAG);
					XtSetArg(args[argc], XmNlabelString, label);
					argc++;
					/* constraints */
					XtSetArg(args[argc], XmNtopAttachment, XmATTACH_FORM);
					argc++;
					XtSetArg(args[argc], XmNrightAttachment, XmATTACH_WIDGET);
					argc++;
					XtSetArg(args[argc], XmNrightWidget, list);
					argc++;
					XtSetArg(args[argc], XmNborderWidth, 0);
					argc++;
					XtSetArg(args[argc], XmNfontList, my_fontList);
					argc++;
					XtSetArg(args[argc], XmNbackground, html->html.body_bg);
					argc++;
					XtSetArg(args[argc], XmNforeground, html->html.body_fg);
					argc++;

					/* create button */
					button = XmCreatePushButton(entry->w, "_fileButton", args,
						argc);

					XmStringFree(label);

					/* callbacks */
					XtAddCallback(button, XmNactivateCallback,
						(XtCallbackProc)fileActivateCB, (XtPointer)entry);
				}
				/* manage it */
				XtManageChild(entry->w);
				break;

			/* push buttons, set args and create */
			case FORM_RESET:
			case FORM_SUBMIT:

				label = XmStringCreate(entry->value ? entry->value :
					entry->name, MY_FONTLIST_DEFAULT_TAG);
				XtSetArg(args[argc], XmNlabelString, label); argc++;
				entry->w = XmCreatePushButton(parent, entry->name, args, argc);
				XmStringFree(label);

				/* override some translations */
				XtOverrideTranslations(entry->w, travTranslations);
				XtOverrideTranslations(entry->w, pushBTranslations);

				/* manage it */
				XtManageChild(entry->w);

				/* callbacks */
				XtAddCallback(entry->w, XmNactivateCallback,
					(XtCallbackProc)buttonActivateCB, (XtPointer)entry);
				break;

			default:
				break;
		}
	}

	/* do final stuff for this entry */
	finalizeEntry(html, entry, True);

	/* all done */
	return(entry);
}

/*****
* Name: 		_XmHTMLFormAddSelect
* Return Type: 	XmHTMLForm
* Description: 	creates a form select entry 
* In: 
*	html:		XmHTMLWidget id;
*	attributes:	select attributes;
* Returns:
*	a new XmHTMLForm entry.
*****/
XmHTMLForm*
_XmHTMLFormAddSelect(XmHTMLWidget html, String attributes)
{
	static XmHTMLForm *entry;
	String chPtr;
	Widget parent;

	/*****
	* HTML form child widgets are childs of the workarea.
	* Making them a direct child of the widget itself messes up scrolling.
	*****/
	parent = html->html.work_area;

	if(attributes == NULL)
		return(NULL);

	if(current_form == NULL)
	{
		_XmHTMLWarning(__WFUNC__(html, "_XmHTMLFormAddSelect"),
			"Bad HTML form: <SELECT> not within form.");
	}

	/* Create and initialise a new entry */
	entry = (XmHTMLForm*)malloc(sizeof(XmHTMLForm));
	(void)memset(entry, 0, sizeof(XmHTMLForm));

	/* set parent form */
	entry->parent = current_form;

	/* form type */
	entry->type = FORM_SELECT;

	/* get name */
	if((chPtr = _XmHTMLTagGetValue(attributes, "name")) == NULL)
		entry->name = strdup("Select");
	else
	{
		entry->name = strdup(chPtr);
		free(chPtr);
	}

	/* no of visible items in list */
	entry->size = _XmHTMLTagGetNumber(attributes, "size", 1);

	/* multiple select? */
	entry->multiple = _XmHTMLTagCheck(attributes, "multiple");

	/* default args for both multiple and single select */
	argc = 0;
	XtSetArg(args[argc], XmNborderWidth, 0); argc++;
	XtSetArg(args[argc], XmNbackground, html->html.body_bg); argc++;
	XtSetArg(args[argc], XmNforeground, html->html.body_fg); argc++;
	XtSetArg(args[argc], XmNfontList, my_fontList); argc++;

	/* multiple select or more than one item visible: it's a listbox */
	if(entry->multiple || entry->size > 1)
	{
		parent = html->html.work_area;
		XtSetArg(args[argc], XmNlistSizePolicy, XmCONSTANT); argc++;
		XtSetArg(args[argc], XmNscrollBarDisplayPolicy, XmAS_NEEDED); argc++;
		XtSetArg(args[argc], XmNmarginWidth, 0); argc++;
		XtSetArg(args[argc], XmNmarginHeight, 0); argc++;

		/* at least two items required for list boxes */
		XtSetArg(args[argc], XmNvisibleItemCount,
				(entry->size == 1 ? 2 : entry->size)); argc++;

		/* multiple selection possible */
		if(entry->multiple)
		{
			XtSetArg(args[argc], XmNselectionPolicy, XmMULTIPLE_SELECT);
			argc++;
		}

		/*
		* Create it. This returns the widget id of the listbox, *not*
		* the scrolledWindow parent!
		*/
		entry->w = XmCreateScrolledList(parent, entry->name, args, argc);

		/* override some translations */
		XtOverrideTranslations(entry->w, travTranslations);

		XtSetMappedWhenManaged(XtParent(entry->w), False);
		XtSetMappedWhenManaged(entry->w, False);

		/* manage it */
		XtManageChild(entry->w);
	}
	else	/* an option menu */
	{
		/* the menu that will contain the menu items */
		Widget menu;
		menu = XmCreatePulldownMenu(parent, entry->name, args, argc);
		entry->w = menu;
		/* override some translations */
		XtOverrideTranslations(entry->w, travTranslations);

	}
	/* set fg, bg and font to use, don't insert (yet) in parent form list */
	finalizeEntry(html, entry, False);

	/* this will be used to keep track of inserted menu items */
	entry->next = NULL;

	return(entry);
}

/*****
* Name:			_XmHTMLFormSelectAddOption
* Return Type: 	void
* Description: 	adds an option button to a form <select>.
* In: 
*	html:		XmHTMLWidget id;
*	entry:		parent for this option;
*	attrib..:	attributes for this option;
*	label:		label for the option button.
* Returns:
*	nothing, but the new button is added to the list of options of the
*	parent entry.
*****/
void
_XmHTMLFormSelectAddOption(XmHTMLWidget html, XmHTMLForm *entry,
	String attributes, String label)
{
	XmHTMLForm *item;
	XmString xms;

	/* Create and initialise a new entry */
	item = (XmHTMLForm*)malloc(sizeof(XmHTMLForm));
	(void)memset(item, 0, sizeof(XmHTMLForm));

	/* form type */
	item->type = FORM_OPTION;

	/* value. Use select name if none given */
	if((item->value = _XmHTMLTagGetValue(attributes, "value")) == NULL)
		item->value = strdup(entry->name);

	/* initial state */
	item->selected = _XmHTMLTagCheck(attributes, "selected");
	item->checked  = item->selected;

	/* list box selection */
	if(entry->multiple || entry->size > 1)
	{
		/* append item to bottom of list */
		xms = XmStringCreate(label, MY_FONTLIST_DEFAULT_TAG);
		XmListAddItem(entry->w, xms, 0);
		XmStringFree(xms);

		/* add this item to the list of selected items */
		if(item->checked)
		{
			/* single selection always takes the last inserted item */
			entry->selected = entry->maxlength;

			/*
			* Since we are always inserting items at the end of the list
			* we can simple select it by using 0 as the position arg
			*/
			XmListSelectPos(entry->w, 0, False);
		}
	}
	else
	{
		Widget w;

		/* add a new menu item */
		xms = XmStringCreate(label, MY_FONTLIST_DEFAULT_TAG);

		/* option menu button */
		argc = 0;
		XtSetArg(args[argc], XmNbackground, html->html.body_bg); argc++;
		XtSetArg(args[argc], XmNforeground, html->html.body_fg); argc++;
		XtSetArg(args[argc], XmNlabelString, xms); argc++;
		XtSetArg(args[argc], XmNfontList, my_fontList); argc++;
		w = XmCreatePushButton(entry->w, label, args, argc);
		XmStringFree(xms);

		XtManageChild(w);

		/* save as default menu item if initially selected */
		if(item->checked)
			entry->selected = entry->maxlength;
	}

	/* insert item, entry->next contains ptr to last inserted option */
	if(entry->next)
	{
		entry->next->next = item;
		entry->next = item;
	}
	else
	{
		entry->options = entry->next = item;
	}

	/* no of options inserted so far */
	entry->maxlength++;
}

/*****
* Name:			_XmHTMLFormSelectClose
* Return Type: 	void
* Description: 	closes a form <select> tag. Performs required wrapup
*				operations such as actual optionmenu creation and translation
*				stuff.
* In: 
*	html:		XmHTMLWidget id;
*	entry:		entry to be closed.
* Returns:
*	nothing.
*****/
void
_XmHTMLFormSelectClose(XmHTMLWidget html, XmHTMLForm *entry)
{
	/* option menu */
	if(!entry->multiple && entry->size == 1)
	{
		Widget menu = entry->w;		/* rowColumn containing all menubuttons */
		WidgetList children;
		int num_children;
		XmString xms;

		argc = 0;
		XtSetArg(args[argc], XmNx, 0); argc++;
		XtSetArg(args[argc], XmNy, 0); argc++;
		XtSetArg(args[argc], XmNmarginWidth, 0); argc++;
		XtSetArg(args[argc], XmNmarginHeight, 0); argc++;
		XtSetArg(args[argc], XmNsubMenuId, menu); argc++;
		XtSetArg(args[argc], XmNhighlightThickness, 0); argc++;
		XtSetArg(args[argc], XmNbackground, html->html.body_bg); argc++;
		XtSetArg(args[argc], XmNforeground, html->html.body_fg); argc++;
		XtSetArg(args[argc], XmNfontList, my_fontList); argc++;

		entry->w = XmCreateOptionMenu(html->html.work_area, "optionMenu",
			args, argc);

		/* override some translations */
		XtOverrideTranslations(entry->w, travTranslations);

		/*****
		* Remove the label gadget. We first *MUST* set an empty label
		* because Motif always reserves space for the label, EVEN when the
		* label is unmanaged. Sigh.
		*****/
		argc = 0;
		xms = XmStringCreate("", MY_FONTLIST_DEFAULT_TAG);
		XtSetArg(args[argc], XmNlabelString, xms); argc++;
		XtSetValues(XmOptionLabelGadget(entry->w), args, argc);
		XmStringFree(xms);
		XtUnmanageChild(XmOptionLabelGadget(entry->w));

		/* manage the menu */
		XtSetMappedWhenManaged(entry->w, False);
		XtManageChild(entry->w);

		/* get current list of menu buttons. This returns a rowColumn */
		menu = NULL;
		XtVaGetValues(entry->w, XmNsubMenuId, &menu, NULL);

		XtVaGetValues(menu,
			XmNnumChildren, &num_children,
			XmNchildren, &children,
			NULL);

		/* and select selected button (or first if no button is selected) */
		XtVaSetValues(entry->w,
			XmNmenuHistory, children[entry->selected], NULL);

		/* store menupane id as child id */
		entry->child = menu;

		/* safety */
		entry->next = NULL;

		/* do final stuff for this entry */
		finalizeEntry(html, entry, True);
	}
	else
	{
		/* safety */
		entry->next = NULL;

		/* list is child of ScrolledWindow */
		entry->child = entry->w;

		/* actual widget we'll be moving */
		entry->w = XtParent(entry->child);

		/* do final stuff for this entry */
		finalizeEntry(html, entry, True);

		/* make sure child is visible when parent is managed */
		XtSetMappedWhenManaged(entry->child, True);
	}
}

/*****
* Name:			_XmHTMLFormAddTextArea
* Return Type: 	XmHTMLForm*
* Description: 	creates a form <textarea> entry.
* In: 
*	html:		XmHTMLWidget id;
*	attrib..:	attributes for this textarea;
*	text:		default text for this entry.
* Returns:
*	a newly created entry.
*****/
XmHTMLForm*
_XmHTMLFormAddTextArea(XmHTMLWidget html, String attributes, String text)
{
	static XmHTMLForm *entry;
	String name = NULL;
	int rows = 0, cols = 0;
	Widget parent;

	/*****
	* HTML form child widgets are childs of the workarea.
	* Making them a direct child of the widget itself messes up scrolling.
	*****/
	parent = html->html.work_area;

	/* these are *required* */
	if(attributes == NULL)
		return(NULL);

	/* sanity, we must have a parent form */
	if(current_form == NULL)
	{
		_XmHTMLWarning(__WFUNC__(html, "_XmHTMLFormAddTextArea"),
			"Bad HTML form: <TEXTAREA> not within form.");
	}

	/* get form name. Mandatory so spit out an error if not found */
	if((name = _XmHTMLTagGetValue(attributes, "name")) == NULL)
	{
		_XmHTMLWarning(__WFUNC__(html, "_XmHTMLFormAddTextArea"),
			"Bad <TEXTAREA>: missing name attribute.");
		return(NULL);
	}

	/* get form dimensions. Mandatory so spit out an error if not found. */
	rows = _XmHTMLTagGetNumber(attributes, "rows", 0);
	cols = _XmHTMLTagGetNumber(attributes, "cols", 0);
	if(rows <= 0 || cols <= 0)
	{
		_XmHTMLWarning(__WFUNC__(html, "_XmHTMLFormAddTextArea"),
			"Bad <TEXTAREA>: invalid or missing ROWS and/or COLS attribute.");
	}

	/* Create and initialise a new entry */
	entry = (XmHTMLForm*)malloc(sizeof(XmHTMLForm));
	(void)memset(entry, 0, sizeof(XmHTMLForm));

	/* fill in appropriate fields */
	entry->name      = strdup(name);
	entry->parent    = current_form;
	entry->type      = FORM_TEXTAREA;
	entry->size      = cols;
	entry->maxlength = rows;
	free(name);

	/* default text. Empty value if not given */
	if((entry->value = text) == NULL)
	{
		entry->value = (String)malloc(1);
		entry->value[0] = '\0';
	}

	/* defaults for all widgets */
	argc = 0;
	XtSetArg(args[argc], XmNmappedWhenManaged, False); argc++;
#if 0
	XtSetArg(args[argc], XmNborderWidth, 0); argc++;
#endif
	XtSetArg(args[argc], XmNbackground, html->html.body_bg); argc++;
	XtSetArg(args[argc], XmNforeground, html->html.body_fg); argc++;
	XtSetArg(args[argc], XmNfontList, my_fontList); argc++;
	XtSetArg(args[argc], XmNvalue, entry->value); argc++;
	XtSetArg(args[argc], XmNcolumns, cols); argc++;
	XtSetArg(args[argc], XmNrows, rows); argc++;
	XtSetArg(args[argc], XmNeditMode, XmMULTI_LINE_EDIT); argc++;
	XtSetArg(args[argc], XmNscrollingPolicy, XmAUTOMATIC); argc++;
	XtSetArg(args[argc], XmNscrollBarDisplayPolicy, XmAS_NEEDED); argc++;
	XtSetArg(args[argc], XmNscrollBarPlacement, html->html.sb_placement);argc++;
	XtSetArg(args[argc], XmNhighlightThickness, 0); argc++;
	XtSetArg(args[argc], XmNborderWidth, 0); argc++;

	/* now create it */
	entry->child = XmCreateScrolledText(parent, entry->name, args, argc);
	entry->w = XtParent(entry->child);	/* actual widget id we'll be moving */

	/* don't let if popup */
	XtSetMappedWhenManaged(entry->w, False);
	XtSetMappedWhenManaged(entry->child, True);

	/* manage textArea child */
	XtManageChild(entry->child);

	/* safety */
	entry->next = NULL;

	/* do final stuff for this entry */
	finalizeEntry(html, entry, True);

	/* all done! */
	return(entry);
}

void
_XmHTMLFreeForm(XmHTMLWidget html, XmHTMLFormData *form)
{
	XmHTMLFormData *tmp;

	while(form != NULL)
	{
		tmp = form->next;
		freeForm(form->components);
		if(form->action)
			free(form->action);
		if(form->enctype)
			free(form->enctype);
		free(form);
		form = tmp;
	}
}

/*****
* Name:			_XmHTMLFormActivate
* Return Type: 	void
* Description: 	form activator. Will collect all data of the current form
*				and call the form callback.
* In: 
*	html:		XmHTMLWidget id;
*	event:		event structure triggering this callback;
*	entry:		form entry that triggered this routine.
* Returns:
*	nothing.
*****/
void
_XmHTMLFormActivate(XmHTMLWidget html, XEvent *event, XmHTMLForm *entry)
{
	XmHTMLFormCallbackStruct cbs;
	XmHTMLFormDataPtr components;

	_XmHTMLDebug(12, ("forms.c: _XmHTMLFormActivate, activated by component "
		"%s\n", entry->name));

	/* only do something when a form callback has been installed */
	if(html->html.form_callback == NULL)
		return;

	/*
	* INSERT CODE
	* to collect form data and call the form callback.
	* fields to be filled:
	*
	* int reason;			the reason the callback was called
	* XEvent *event;		event structure that triggered callback
	* String action;		URL or cgi-bin destination
	* String enctype;		form encoding
	* int method;			Form Method, GET (0) or POST (1)
	* int ncomponents;		no of components in this form
	* XmHTMLFormDataPtr components;
	*/
	(void)memset(&cbs, 0, sizeof(XmHTMLFormCallbackStruct));

	cbs.reason      = XmCR_HTML_FORM;
	cbs.event       = event;
	cbs.action      = strdup(entry->parent->action);
	cbs.method      = entry->parent->method;
	cbs.enctype     = strdup(entry->parent->enctype);
	cbs.ncomponents = 0;
	cbs.components  = (XmHTMLFormDataPtr)NULL;

	XtCallCallbackList((Widget)html, html->html.form_callback, &cbs);

	free(cbs.action);
	free(cbs.enctype);
}

/*****
* Name: 		_XmHTMLFormReset
* Return Type: 	void
* Description: 	resets all entries of the given form to their default values.
* In: 
*	html:		XmHTMLWidget id;
*	entry:		form entry that triggered this routine.
* Returns:
*	nothing.
*****/
void _XmHTMLFormReset(XmHTMLWidget html, XmHTMLForm *entry)
{
	XmHTMLFormData *form = entry->parent;
	XmHTMLForm *tmp, *option;
	int i;

	_XmHTMLDebug(12, ("forms.c: _XmHTMLFormReset start\n"));
	for(tmp = form->components; tmp != NULL; tmp = tmp->next)
	{
		_XmHTMLDebug(12, ("\tchecking %s\n", tmp->name));

		switch(tmp->type)
		{
			case FORM_TEXT:
			case FORM_PASSWD:
				_XmHTMLDebug(12, ("\t\tsetting XmNvalue to: %s\n", tmp->value));
				XtVaSetValues(tmp->w, XmNvalue, tmp->value, NULL);
				break;
			case FORM_TEXTAREA:
				_XmHTMLDebug(12, ("\t\tsetting XmNvalue to: %s\n", tmp->value));
				XtVaSetValues(tmp->child, XmNvalue, tmp->value, NULL);
				break;
			case FORM_CHECK:
			case FORM_RADIO:
				/* checkbuttons, set default state */
				_XmHTMLDebug(12, ("\t\tsetting state to %s\n", 
					tmp->checked ? "on" : "off"));
				XtVaSetValues(tmp->w, XmNset, tmp->checked, NULL);
				break;
			case FORM_SELECT:
				if(tmp->multiple || tmp->size > 1)
				{
					/* scrolled list. First deselect all items */
					XmListDeselectAllItems(tmp->child);

					/* now see what options should be selected */
					for(i = 0, option = tmp->options; option != NULL;
						option = option->next, i++)
					{
						if(option->checked)
							XmListSelectPos(tmp->child, i+1, False);
					}
					
				}
				else
				{
					Widget menu = NULL;
					WidgetList children;
					int num_children;
					XtVaGetValues(tmp->w, XmNsubMenuId, &menu, NULL);

					XtVaGetValues(menu,
						XmNnumChildren, &num_children,
						XmNchildren, &children,
						NULL);

					/* and set default button  */
					XtVaSetValues(tmp->w,
						XmNmenuHistory, children[tmp->selected], NULL);
				}
				break;
			default:
				break;
		}
	}
	_XmHTMLDebug(12, ("forms.c: _XmHTMLFormReset end.\n"));
}

/*****
* form widget traversal handling.
*****/
static Widget
getNextLeader(XmHTMLFormData *curr_group, int *y_pos)
{
	XmHTMLFormData *form;
	XmHTMLForm *entry = NULL;

	my_assert(curr_group != NULL);

	for(form = curr_group->next; form != NULL && entry == NULL;
		form = form->next)
	{
		for(entry = form->components; entry != NULL && entry->w == NULL;
			entry = entry->next);
	}
	_XmHTMLDebug(12, ("forms.c:getNextLeader, returning widget %s\n",
		entry ? XtName(entry->w) : "<NULL>"));

	if(entry)
	{
		*y_pos = entry->y;
		return(entry->w);
	}
	*y_pos = 0;
	return(NULL);
}

static Widget
getPrevLeader(XmHTMLFormData *curr_group, int *y_pos)
{
	XmHTMLFormData *form;
	XmHTMLForm *entry = NULL;

	my_assert(curr_group != NULL);

	/* need to walk all form groups */
	for(form = curr_group->prev; form != NULL && entry == NULL;
		form = form->prev)
	{
		for(entry = form->components; entry != NULL && entry->w == NULL;
			entry = entry->next);
	}
	_XmHTMLDebug(12, ("forms.c:getPrevLeader, returning widget %s\n",
		entry ? XtName(entry->w) : "<NULL>"));

	*y_pos = 0;
	if(entry)
	{
		*y_pos = entry->y;
		return(entry->w);
	}
	return(NULL);
}

static Widget
getNextTab(XmHTMLForm *curr_tab, Boolean start_at_current, int *y_pos)
{
	XmHTMLForm *entry = NULL;

	my_assert(curr_tab != NULL);

#ifdef DEBUG
	_XmHTMLDebug(12, ("forms.c: getNextTab, form listing:\n"));
	for(entry = curr_tab->parent->components; entry != NULL;
		entry = entry->next)
	{
		_XmHTMLDebug(12, ("\t%s", entry->name));
		if(entry == curr_tab)
			_XmHTMLDebug(12, ("\t(--> selected)"));
		_XmHTMLDebug(12, ("\n"));
	}
#endif

	/*****
	* If start_at_current is set, we need to start at the widhet with the
	* focus. This can happen when we are traversing down to the widget tree
	* for the *first* time, e.i., the current widget is the HTML widget itself.
	*****/
	if(start_at_current)
		entry = curr_tab;
	else
		entry = curr_tab->next;

	/* get to next visible widget */
	for(; entry != NULL && entry->w == NULL; entry = entry->next);

	_XmHTMLDebug(12, ("forms.c:getNextTab , returning widget %s\n",
		entry ? XtName(entry->w) : "<NULL>"));

	*y_pos = 0;
	if(entry)
	{
		*y_pos = entry->y;
		return(entry->w);
	}
	/* if we didn't find a next entry, get leader of next form */
	return(getNextLeader(curr_tab->parent, y_pos));
}

static Widget
getPrevTab(XmHTMLForm *curr_tab, int *y_pos)
{
	XmHTMLForm *entry = NULL;

	my_assert(curr_tab != NULL);

#ifdef DEBUG
	_XmHTMLDebug(12, ("forms.c: getPrevTab, form listing:\n"));
	for(entry = curr_tab->parent->components; entry != NULL;
		entry = entry->next)
	{
		_XmHTMLDebug(12, ("\t%s", entry->name));
		if(entry == curr_tab)
			_XmHTMLDebug(12, ("\t(--> selected)"));
		_XmHTMLDebug(12, ("\n"));
	}
#endif

	/* walk to previously visible widget */
	for(entry = curr_tab->prev; entry != NULL && entry->w == NULL;
			entry = entry->prev);

	_XmHTMLDebug(12, ("forms.c:getPrevTab , returning widget %s\n",
		entry ? XtName(entry->w) : "<NULL>"));

	*y_pos = 0;
	if(entry)
	{
		*y_pos = entry->y;
		return(entry->w);
	}
	/* if we didn't find a previous entry, get leader of previous form */
	return(getPrevLeader(curr_tab->parent, y_pos));
}

/*****
* Name: 		_XmHTMLProcessTraversal
* Return Type: 	void
* Description: 	XmHTML's version of XmProcessTraversal, required for proper
*				traversal amongst form widgets.
* In: 
*	w:			widget requiring traversal;
*	direction:	which way we should go;
* Returns:
*	nothing, but keyboard focus is changed. As a side effect this routine
*	*can* cause vertical and/or horizontal scrolling to happen (to place a
*	certain widget into focus).
*****/
void
_XmHTMLProcessTraversal(Widget w, int direction)
{
	XmHTMLWidget html;
	XmHTMLFormData *form, *curr_group;		/* current tabgroup */
	XmHTMLForm *entry = NULL, *curr_tab;	/* item that has the focus */
	Widget parent;							/* XmHTML parent */
	Widget current;							/* widget that has the focus */
	Widget next;							/* widget receiving focus */
	int y = 0;

	_XmHTMLDebug(12, ("forms.c: _XmHTMLProcessTraversal, direction = %i\n",
		direction));

	/* first get the XmHTML parent */
	current = parent = w;
	while(parent && !XmIsHTML(parent))
		parent = XtParent(parent);

	/* not found, nothing to do */
	if(!parent || !XmIsHTML(parent))
	{
		_XmHTMLDebug(12, ("forms.c: _XmHTMLProcessTraversal, end, no parent "
			"found\n"));
		return;
	}

	html = (XmHTMLWidget)parent;

	/*****
	* Check if we have any forms in here. If we don't we move traversal
	* on to Motif.
	*****/
	if(html->html.form_data == NULL)
	{
		_XmHTMLDebug(12, ("forms.c: _XmHTMLProcessTraversal, end, no form "
			"present\n"));
		/* pass down to Motif */
		XmProcessTraversal(w, direction);
		return;
	}

	/* now get the XmHTML parent */
	while(!XtIsShell(parent) && !XtIsTopLevelShell(parent))
		parent = XtParent(parent);

	/*****
	* Now get the widget that has the focus. If the current widget equals
	* the widget ID of the workArea, we are about to move into the form
	* hierarchy and thus everything should be initialized to the first form
	* widget. Otherwise we need to compare widget id's.
	*****/
	if(w != html->html.work_area)
	{
		form = html->html.form_data;
		while(form != NULL)
		{
			for(entry = form->components; entry != NULL; entry = entry->next)
			{
				_XmHTMLDebug(12, ("forms.c: _XmHTMLProcessTraversal, "
					"comparing %s with %s\n", entry->name, XtName(w)));
				if(entry->w == w)
					break;
			}
			if(entry)
				break;
			form = form->next;
		}
		/* no widget has got the focus (yet) */
		if(!entry)
		{
			_XmHTMLDebug(12, ("forms.c: _XmHTMLProcessTraversal, end, no "
				"valid entry found!\n"));
			return;
		}
		curr_group = form;
		curr_tab = entry;
		current = entry->w;
	}
	else
	{
		current = w;
		curr_group = html->html.form_data;
		curr_tab = curr_group->components;
	}

	_XmHTMLDebug(12, ("forms.c: _XmHTMLProcessTraversal, traversing %s\n",
		XtName(current)));

#ifdef DEBUG
	_XmHTMLDebug(12, ("forms.c: _XmHTMLProcessTraversal, using traversal "));

	switch(direction)
	{
		case XmTRAVERSE_CURRENT:
			_XmHTMLDebug(12, ("XmTRAVERSE_CURRENT\n"));
			break;
		case XmTRAVERSE_DOWN:
			_XmHTMLDebug(12, ("XmTRAVERSE_DOWN\n"));
			break;
		case XmTRAVERSE_RIGHT:
			_XmHTMLDebug(12, ("XmTRAVERSE_RIGHT\n"));
			break;
		case XmTRAVERSE_NEXT:
			_XmHTMLDebug(12, ("XmTRAVERSE_NEXT\n"));
			break;
		case XmTRAVERSE_UP:
			_XmHTMLDebug(12, ("XmTRAVERSE_UP\n"));
			break;
		case XmTRAVERSE_LEFT:
			_XmHTMLDebug(12, ("XmTRAVERSE_LEFT\n"));
			break;
		case XmTRAVERSE_PREV:
			_XmHTMLDebug(12, ("XmTRAVERSE_PREV\n"));
			break;
		case XmTRAVERSE_HOME:
			_XmHTMLDebug(12, ("XmTRAVERSE_HOME\n"));
			break;
		case XmTRAVERSE_NEXT_TAB_GROUP:
			_XmHTMLDebug(12, ("XmTRAVERSE_NEXT_TAB_GROUP\n"));
			break;
		case XmTRAVERSE_PREV_TAB_GROUP:
			_XmHTMLDebug(12, ("XmTRAVERSE_PREV_TAB_GROUP\n"));
			break;
		default:
			_XmHTMLDebug(12, ("unknown traversal method!!\n"));
			break;
	}
#endif

	switch(direction)
	{
		case XmTRAVERSE_CURRENT:
			/* focus stays at current */
			next = current;
			break;

		case XmTRAVERSE_NEXT:
			next = getNextTab(curr_tab, current == html->html.work_area, &y);
			break;

		case XmTRAVERSE_PREV:
			next = getPrevTab(curr_tab, &y);
			break;

		case XmTRAVERSE_HOME:
			next = (Widget)html;
			break;

		case XmTRAVERSE_NEXT_TAB_GROUP:
			next = getNextLeader(curr_group, &y);
			break;

		case XmTRAVERSE_PREV_TAB_GROUP:
			next = getPrevLeader(curr_group, &y);
			break;

		default:
			next = current;
			break;
	}
	/*
	* Set focus if we've found a valid widget. If we didn't find one, pass
	* down to Motif using the set action.
	*/
	if(next)
	{
		_XmHTMLDebug(12, ("forms.c: _XmHTMLProcessTraversal, setting focus "
			"to %s\n", XtName(next)));
		/*****
		* check if this widget is fully visible. If not we scroll so it
		* becomes visible.
		*****/
#if 0
		/*****
		* FIXME
		* must use bounding box to do this properly.
		*****/
		if(y > (html->html.scroll_y + html->html.work_height) ||
			y < html->html.scroll_y)
		{
			int max = 0, size = 0, value = y;
			XtVaGetValues(html->html.vsb,
				XmNmaximum, &max,
				XmNsliderSize, &size,
				NULL);
			if(value > (max - size))
				value = (max - size);
			_XmHTMLMoveToPos(html->html.vsb, html, value);
		}
#endif
		XtSetKeyboardFocus(parent, next);
	}
	else
	{
		_XmHTMLDebug(12, ("forms.c: _XmHTMLProcessTraversal, passing down to "
			"Motif (name of widget passed is %s)\n", XtName(w)));
		XmProcessTraversal(w, direction);
	}
}
