/** \file lispobject.h
 *  Implementation of basic LispObject, which is the base class for
 *  anything that can be put in a linked list.
 *
 * class LispObject. This class implements one lisp object, which is a
 * abstract class containing reference counting and a next pointer.
 * derive from this to implement specific types of list elements.
 * The local class LispPtr implements automatic garbage collection
 * through reference counting.
 *
 */


#ifndef __lispobject_h__
#define __lispobject_h__

#include "lisptype.h"
#include "lispstring.h"
#include "genericobject.h"
#include "evalfunc.h"

class LispObject;

/** class LispPtr. This class is a smart pointer type class to Lisp
 *  objects that can be inserted into linked lists. They do the actual
 *  reference counting, and consequent destruction of the object if
 *  nothing points to it.
 */
class LispPtr
{
public:
    LispPtr();
    LispPtr(LispPtr& aOther);
    LispPtr(LispObject& aOther);
    ~LispPtr();
    LispPtr& operator=(LispPtr& aOther);
    LispPtr& operator=(LispObject& aOther);
    void Set(LispObject* aNext);
    inline LispObject* Get() const;
    // For use as an iterator
    inline LispObject* operator()();
    inline void GoNext();
private:
    LispObject* iNext;
};



/** class LispObject is the base object class that can be put in
 *  linked lists.
 */
class LispObject
{
public:
    inline LispPtr& Next();
    inline LispShort RefCount() const;
    inline void IncreaseRefCount();
    inline void DecreaseRefCount();
    inline LispInt ReferenceCount() const;

public: //Derivables
    virtual ~LispObject();

    // String and SubList can return NULL, for a specific type of atom.
    virtual LispStringPtr String() const;
    virtual LispPtr* SubList();
    virtual GenericClass* Generic();
    virtual EvalFuncBase* EvalFunc();
    virtual void SetEvalFunc(EvalFuncBase* aEvalFunc);
    virtual LispObject* Copy(LispInt aRecursed)=0;
public:
    LispInt Equal(LispObject& aOther);
    inline LispInt operator==(LispObject& aOther);
    inline LispInt operator!=(LispObject& aOther);
    inline void SetFlag(LispInt aFlag);
    inline void ResetFlag(LispInt aFlag);
protected:
    LispObject();
    
private:
    ReferenceType iReferenceCount;
    LispPtr   iNext;
};


/**
 * class LispIterator works almost like LispPtr, but doesn't enforce
 * reference counting, so it should be slightly faster. This one
 * should be used in stead of LispPtr if you are going to traverse
 * a lisp expression in a non-destructive way.
 */
class LispIterator
{
public:
    LispIterator(LispPtr& aPtr);
    LispIterator& operator=(LispPtr& aPtr);
    inline LispObject* operator()();
    inline LispPtr* Ptr();
    inline void GoNext();
    inline void GoSub();
private:
    LispPtr* iPtr;
};


void IncNrObjects();
void DecNrObjects();


#include "lispobject.inl"

#endif



