/** \file lisphash.h
 *  hashing of strings. Each string will exist only once in the
 * hash table, and have an unique id.
 */


#ifndef __lisphash_h__
#define __lisphash_h__

#include "grower.h"
#include "lisptype.h"
#include "lispstring.h"

//
// This is the symbol table, implemented as a hash table for fast
// lookup. It is meant to store any string just once, have fast
// searching for strings and return a reference to the string.
//

const LispInt KSymTableSize = 211;
LispInt  LispHash( char *s );
LispInt  LispHashCounted( char *s,LispInt length );
LispInt  LispHashStringify( char *s );
LispInt  LispHashUnStringify( char *s );
LispInt LispHashPtr(LispStringPtr aString);


class LispHashTable
{
public:
    ~LispHashTable();
    // If string not yet in table, insert. Afterwards return the string.
    LispStringPtr LookUp(LispCharPtr aString,
                         LispBoolean aStringOwnedExternally=LispFalse);
    LispStringPtr LookUpCounted(LispCharPtr aString,LispInt aLength);
    LispStringPtr LookUpStringify(LispCharPtr aString,
                         LispBoolean aStringOwnedExternally=LispFalse);
    LispStringPtr LookUpUnStringify(LispCharPtr aString,
                         LispBoolean aStringOwnedExternally=LispFalse);

    // GarbageCollect
    void GarbageCollect();
private:
    CArrayGrower<LispStringPtr> iHashTable[KSymTableSize];
};



// LispAssociatedHash allows you to associate arbitrary
// information with a string in the above hash table.
//


template<class T>
class LAssoc
{
public:
    LAssoc(LispStringPtr aString,const T& aData)
        : iString(aString), iData(aData) {}
    LispStringPtr iString;
    T iData;
};

class VoidGrow : public CArrayGrower<void*>
{
};

template<class T>
class LispAssociatedHash
{
public:
    inline ~LispAssociatedHash();
    inline T* LookUp(LispStringPtr aString);
    inline void SetAssociation(const T& aData, LispStringPtr aString);
    inline void Release(LispStringPtr aString);

private:
    // The next array is in fact an array of arrays of type LAssoc<T>
    VoidGrow iHashTable[KSymTableSize];
    
};



#include "lisphash.inl"


#endif
