/** \file lispenvironment.h
 *  General environment access.
 *
 */

#include "lispobject.h"
#include "lisphash.h"
#include "lispevalhash.h"
#include "lispcleanupstack.h"
#include "deffile.h"
#include "ramdisk.h"
#include "lispio.h"

#ifndef __lispenvironment_h__
#define __lispenvironment_h__



class LispDefFiles;
class InputDirectories : public CArrayGrower<LispStringPtr>
{
public:
    inline ~InputDirectories()
    {
        LispInt nritems = NrItems();
        LispInt j;
        for (j=0;j<nritems;j++)
            delete Item(j);
    }
};

class LispInput;
class LispOutput;
class LispPrinter;
class LispOperators;
class LispUserFunctions;
class LispUserFunction;
class LispMultiUserFunction;
class LispEvaluatorBase;
class BasicEvaluator;
class LispEnvironment
{
public:
    LispEnvironment();
    virtual ~LispEnvironment();
public:
    virtual void SetVariable(LispStringPtr aString, LispPtr& aValue)=0;
    virtual LispPtr* GetVariable(LispStringPtr aString)=0;
    virtual void UnsetVariable(LispStringPtr aString)=0;
    virtual void PushLocalFrame(LispBoolean aFenced) = 0;
    virtual void PopLocalFrame() = 0;
    virtual void NewLocal(LispStringPtr aVariable,LispObject* aValue) = 0;
public:
    virtual LispCommands& Commands() = 0;
    virtual LispHashTable& HashTable() = 0;
    virtual LispUserFunction* UserFunction(LispPtr& aArguments) = 0;
    virtual LispUserFunction* UserFunction(LispStringPtr aName,LispInt aArity) = 0;
    virtual LispMultiUserFunction* MultiUserFunction(LispStringPtr aArguments)=0;
    virtual LispDefFiles& DefFiles() = 0;
    virtual void DeclareRuleBase(LispStringPtr aOperator, LispPtr& aParameters) = 0;
    virtual void DefineRule(LispStringPtr aOperator,LispInt aArity,
                            LispInt aPrecedence, LispPtr& aPredicate,
                            LispPtr& aBody) = 0;
    virtual void UnFenceRule(LispStringPtr aOperator,LispInt aArity) = 0;
    virtual void TryRetract(LispStringPtr aOperator,LispInt aArity) = 0;
    virtual void HoldArgument(LispStringPtr  aOperator,LispStringPtr aVariable) = 0;

public:
    inline void SetPrecision(LispInt aPrecision);
    inline LispInt Precision(void);
public:
    virtual LispPrinter& CurrentPrinter() = 0;
public:
    virtual LispOperators& PreFix() = 0;
    virtual LispOperators& InFix() = 0;
    virtual LispOperators& PostFix() = 0;
    virtual LispOperators& Bodied() = 0;
public:
    virtual LispInput* CurrentInput() = 0;
    virtual void SetCurrentInput(LispInput* aInput) = 0;
public:
    virtual LispOutput* CurrentOutput() = 0;
    virtual void SetCurrentOutput(LispOutput* aOutput) = 0;
public:
    virtual void SetUserError(LispCharPtr aErrorString)=0;
    virtual LispCharPtr ErrorString(LispInt aError)=0;
    
protected:
    LispInt iPrecision;
public:
    InputDirectories iInputDirectories;
    DeletingLispCleanup iCleanup;
    LispInt iEvalDepth;
    LispInt iMaxEvalDepth;
    LispRamDisk iRamDisk;
    LispEvaluatorBase* iEvaluator;
    
public: // Error information when some error occurs.
    InputStatus iInputStatus;
    LispInt iSecure;
public: // pre-found
    LispStringPtr iTrue;
    LispStringPtr iFalse;
    LispPtr iTrueAtom;
    LispPtr iFalseAtom;
};



inline void LispEnvironment::SetPrecision(LispInt aPrecision)
{
    iPrecision = aPrecision;
}

inline LispInt LispEnvironment::Precision(void)
{
    return iPrecision;
}




// Local lisp stack, unwindable by the exception handler
class LispLocalFrame : public LispBase
{
public:
    LispLocalFrame(LispEnvironment& aEnvironment, LispBoolean aFenced)
        : iEnvironment(aEnvironment)
    {
        iEnvironment.PushLocalFrame(aFenced);
        SAFEPUSH(iEnvironment,*this);
    };
    virtual ~LispLocalFrame()
    {
        SAFEPOP(iEnvironment);
        Delete();
    };
    virtual void Delete();
private:
    LispEnvironment& iEnvironment;
};



class LispSecureFrame : public LispBase
{
public:
    LispSecureFrame(LispEnvironment& aEnvironment)
        : iEnvironment(aEnvironment)
    {
        iPreviousSecure = iEnvironment.iSecure;
        iEnvironment.iSecure = 1;
        SAFEPUSH(iEnvironment,*this);
    };
    virtual ~LispSecureFrame()
    {
        SAFEPOP(iEnvironment);
        Delete();
    };
    virtual void Delete();
private:
    LispEnvironment& iEnvironment;
    LispInt iPreviousSecure;
};


// LispLocalInput takes ownership over the LispInput class
class LispLocalInput : public LispBase
{
public:
    LispLocalInput(LispEnvironment& aEnvironment, LispInput* aInput)
        : iEnvironment(aEnvironment)
    {
        iPreviousInput = iEnvironment.CurrentInput();
        iEnvironment.SetCurrentInput(aInput);
        SAFEPUSH(iEnvironment,*this);
    };
    virtual ~LispLocalInput()
    {
        SAFEPOP(iEnvironment);
        Delete();
    };
    virtual void Delete();
private:
    LispEnvironment& iEnvironment;
    LispInput* iPreviousInput;
};


// LispLocalInput takes ownership over the LispInput class
class LispLocalOutput : public LispBase
{
public:
    LispLocalOutput(LispEnvironment& aEnvironment, LispOutput* aOutput)
        : iEnvironment(aEnvironment)
    {
        iPreviousOutput = iEnvironment.CurrentOutput();
        iEnvironment.SetCurrentOutput(aOutput);
        SAFEPUSH(iEnvironment,*this);
    };
    virtual ~LispLocalOutput()
    {
        SAFEPOP(iEnvironment);
        Delete();
    };
    virtual void Delete();
private:
    LispEnvironment& iEnvironment;
    LispOutput* iPreviousOutput;
};


class LispLocalEvaluator
{
public:
    LispLocalEvaluator(LispEnvironment& aEnvironment,LispEvaluatorBase* aNewEvaluator);
    ~LispLocalEvaluator();
    
private:
    LispEvaluatorBase* iPreviousEvaluator;
    LispEnvironment& iEnvironment;
};

class LispLocalTrace
{
public:
    LispLocalTrace(LispUserFunction* aUserFunc);
    ~LispLocalTrace();
private:
    LispUserFunction* iUserFunc;
};

class LocalArgs
{
public:
    LocalArgs(LispPtr* aPtrs)
        : iPtrs(aPtrs)
    {};
    ~LocalArgs()
    {
        if (iPtrs)
            delete[] iPtrs;
    }
private:
    LispPtr* iPtrs;
};



#endif

