/*
   Egon Animator
   Copyright (C) 1997-1999  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

/* ---
   Module name:    window.c

   This module handles windows: creating, destroying, printing on windows.
--- */

#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Shell.h>
#include <X11/Xmu/Atoms.h>
#include <X11/Xmu/StdSel.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/List.h>
#include <X11/xpm.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Shell.h>

#include <X11/Xaw/Form.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Scrollbar.h>
#include "../xcommon/SiagMenuButton.h"
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Paned.h>
#include <X11/xpm.h>

#include "../siod/siod.h"
#include "../egon/types.h"
#include "../xcommon/Animator.h"
#include "../egon/egon.h"
#include "xegon.h"

#include "../common/common.h"
#include "../xcommon/xfonts.h"
#include "../xcommon/embed.h"
#include "../common/cmalloc.h"
#include "../xcommon/xcommon.h"
#include "../xcommon/Combo.h"
#include "../xcommon/Frame.h"
#include "../xcommon/plugin.h"
#include "../common/bitmaps/rbm.xbm"
#include "../common/bitmaps/egon.xpm"
#include "../xcommon/Animator.h"
#include "../xcommon/icon.h"
#include "../xcommon/embed.h"
#include "../xcommon/dialogs.h"
#include "../xcommon/xcommon.h"
#include "../xcommon/filesel.h"
#include "../xcommon/Rudegrid.h"
#include "../xcommon/Tooltip.h"
#include "../xcommon/Handle.h"

#include "../xcommon/DragAndDrop.h"

#define MENUBAR (1)
#define TOOLBAR (2)
#define FORMATBAR (4)

#define APPNAME "Egon"

String fallback_resources[] = {
#include "../xcommon/xcommon-ad.h"
#include "../xcommon/dialogs-ad.h"
#include "../xcommon/filesel-ad.h"
#include "app-defaults.h"
	NULL
};

static XtAppContext app_context;

Widget topLevel, topbox, frame1, frame2, frame3, menubox, toolbox, formatbox;
Widget textbox, statusbox, form, gridpane;
Widget text1, label1, label2, label3;

/* The toolbar */
Widget tbOpen, tbSave, tbView, tbPrint, tbHelp;

Widget btnFont, mnuFont, btnSize, mnuSize, btnStyle, mnuStyle, btnColor, mnuColor;
Widget cmdBold, cmdItalic, cmdHLeft, cmdHCenter, cmdHRight;
Widget shortcuts, tooltip;
Widget edit_shell, stage;

static int bars = 0;

AppData app_data;

#define XtNplugin "plugin"
#define XtCPlugin "Plugin"
#define XtNplay "play"
#define XtCPlay "Play"

static XtResource resources[] = {
	{
		XtNplugin,
		XtCPlugin,
		XtRBoolean,
		sizeof(Boolean),
		XtOffsetOf(AppData, plugin),
		XtRImmediate,
		(XtPointer)False,
	}, {
		XtNplay,
		XtCPlay,
		XtRBoolean,
		sizeof(Boolean),
		XtOffsetOf(AppData, play),
		XtRImmediate,
		(XtPointer)False,
	}
};

static sfmt fmt0 = {"Helvetica", 120, 0, 0, 0, "black", "white", 0, 0, 0, 0};

static XrmOptionDescRec options[] = {
	{"-plugin", "*plugin", XrmoptionNoArg, "True"},
	{"-play", "*play", XrmoptionNoArg, "True"}
};

window *w_list;
int editor_popped = 0;

Atom wm_delete_window;	/* Atom sent to destroy a window */
extern Atom target_atom;	/* used for selection */

/* ---
*/
int ask_for_str_comp(char *prompt, char *buffr, int (*comp) (char *))
{
	return dialog_input(topLevel, prompt, buffr, comp);
}

/* ---
   static int nocomp(char *b)
   This particular completion function doesn't complete at all, it just
   returns TRUE, making TAB equivalent to RET and LFD.
*/

static int nocomp(char *b)
{
	return TRUE;
}

/* ---
   int ask_for_str(char *prompt, char *buffr)
   Calls ask_for_str_comp with nocomp as completion function.
   X
95-06-29: changed "buffer" to "buffr" to please gcc
*/

int ask_for_str(char *prompt, char *buffr)
{
	return ask_for_str_comp(prompt, buffr, nocomp);
}

static void pr_grid(Widget w, XEvent * event,
		    String * params, Cardinal * num_params)
{
	if (ok2print) {
		draw_buffer(find_window_by_widget(w));
		show_cur(w_list);
	}
}

static LISP ltooltip_mode(LISP newmode)
{
	int mode = get_c_long(newmode);
	XtVaSetValues(tooltip,
		XtNtooltipMode, mode,
		XtNtooltipLabel, label2,
		(char *)0);
	return NIL;
}

static void siaghelp_action(Widget w, XEvent *event,
	String *params, Cardinal *num_params)
{
	char b[256];

	sprintf(b, "file://localhost%s/egon/%s", docdir, params[0]);
	if (!fork()) {
		execlp(siaghelp, "Siaghelp", b, NULL);
		exit(0);
	}
}

static void place_shortcuts(Widget w, XEvent *event,
                String *p, Cardinal *n)
{
        XButtonEvent *bev = (XButtonEvent *)event;
        int x, y;

        x = event->xbutton.x;
        y = event->xbutton.y;
        XtVaSetValues(shortcuts,
                XtNx, bev->x_root,
                XtNy, bev->y_root,
                (char *)0);
}

static void
execute_callback(Widget w, XtPointer client_data, XtPointer call_data)
{
	char *b = (char *) client_data;

	execute(b);
}

static struct {
	char *label;
	Widget button, menu;
} *menubar;

static int menucount = 0;

static LISP add_menu(LISP lisp_label)
{
	char *label;
	char button_name[80];

	if (topbox == None) return NIL;

	label = get_c_string(lisp_label);

	sprintf(button_name, "btn%s", label);
	menubar = crealloc(menubar, (menucount+1)*(sizeof *menubar));
	menubar[menucount].label = cstrdup(label);
	menubar[menucount].button = XtVaCreateManagedWidget(button_name,
		siagMenuButtonWidgetClass, menubox,
		(char *)0);
	label_set(menubar[menucount].button, label);
	menubar[menucount].menu = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, menubar[menucount].button, NULL);

	menucount++;

	return NIL;
}

static void remake_ylayout(void)
{
	char b[100];
	sprintf(b, "%s %s %s 30 100%% 30",
		(bars&MENUBAR) ? "30" : "0",
		(bars&TOOLBAR) ? "30" : "0",
		(bars&FORMATBAR) ? "30" : "0");
	XtVaSetValues(topbox,
		XtNyLayout, b,
		(char *)0);
}

static void attach_bar(Widget w)
{
	if (w == frame1) bars |= MENUBAR;
	if (w == frame2) bars |= TOOLBAR;
	if (w == frame3) bars |= FORMATBAR;
	remake_ylayout();
}

static void detach_bar(Widget w)
{
	if (w == frame1) bars &= ~MENUBAR;
	if (w == frame2) bars &= ~TOOLBAR;
	if (w == frame3) bars &= ~FORMATBAR;
	remake_ylayout();
}

static Widget make_handle(Widget pw, Widget victim)
{
	Pixmap pm;
	Pixel color;
	XtVaGetValues(pw,
		      XtNbackground, &color,
		      (char *)0);
	pm = load_pixmap(pw, "handlebg.xpm");
	return XtVaCreateManagedWidget("handle",
		handleWidgetClass, pw,
		XtNforeground, color,
		XtNbackgroundPixmap, pm,
		XtNshadowWidth, 1,
		XtNhandleDetach, detach_bar,
		XtNhandleAttach, attach_bar,
		XtNhandleVictim, victim,
		(char *)0);
}

static void init_menu(void)
{
	frame1 = XtVaCreateManagedWidget("menubar",
		frameWidgetClass, topbox,
		(char *)0);
	bars |= MENUBAR;
	menubox = XtVaCreateManagedWidget("menubox",
		boxWidgetClass, frame1,
		(char *)0);
	make_handle(menubox, frame1);
}

static Widget find_menu_by_name(char *label)
{
	int i;

	if (!cstrcasecmp("Shortcuts", label))
		return shortcuts;

	for (i = 0; i < menucount; i++) {
		if (!cstrcasecmp(menubar[i].label, label))
			return menubar[i].menu;
	}
	return NULL;
}

static LISP add_menu_entry(LISP menu, LISP label, LISP function)
{
	Widget entry, menuw;

	menuw = find_menu_by_name(get_c_string(menu));
	if (!menuw) {
#if 0
		fprintf(stderr, "No menu! Bailing out.\n");
#endif
		return NIL;
	}

	if (!strcmp(get_c_string(label), "-"))	/* line pane */
		entry = XtCreateManagedWidget("-",
				smeLineObjectClass,
				menuw,
				NULL, 0);
	else {
		entry = XtCreateManagedWidget(get_c_string(function),
				smeBSBObjectClass, menuw,
				NULL, 0);
		XtAddCallback(entry,
			XtNcallback, execute_callback,	
				cstrdup(get_c_string(function)));
		label_set(entry, get_c_string(label));
	}
	return NIL;
}

static struct {
        char *label, *sublabel;
        Widget entry, menu;
} *submenus;

static int submenucount = 0;

static Widget find_submenu_by_name(char *label, char *sublabel)
{
        int i;

        for (i = 0; i < submenucount; i++) {
                if (!cstrcasecmp(submenus[i].label, label) &&
                                !cstrcasecmp(submenus[i].sublabel, sublabel))
                        return submenus[i].menu;
        }
        return None;
}

static Widget find_submenu_by_entry(Widget entry)
{
        int i;

        for (i = 0; i < submenucount; i++) {
                if (entry == submenus[i].entry)
                        return submenus[i].menu;
        }
        return None;
}

static LISP add_submenu(LISP label, LISP sublabel)
{
        Widget menuw;
        static Pixmap rbm = None;

        menuw = find_menu_by_name(get_c_string(label));

        if (!menuw) {
#if 0
                fprintf(stderr, "add_submenu: No menu! Bailing out.\n");
#endif
                return NIL;
        }
        if (rbm == None) {
                rbm = XCreatePixmapFromBitmapData(XtDisplay(menuw),
                        XDefaultRootWindow(XtDisplay(menuw)),
                        rbm_bits, rbm_width, rbm_height, 1, 0, 1);
        }
        submenus = crealloc(submenus, (submenucount+1)*(sizeof *submenus));
        submenus[submenucount].label = cstrdup(get_c_string(label));
        submenus[submenucount].sublabel = cstrdup(get_c_string(sublabel));
        submenus[submenucount].entry =
                XtVaCreateManagedWidget(get_c_string(sublabel),
                        smeBSBObjectClass, menuw,
                        XtNlabel, translate(get_c_string(sublabel)),
                        XtNrightBitmap, rbm,
                        (char *)0);
        submenus[submenucount].menu = XtVaCreatePopupShell("submenu",
                simpleMenuWidgetClass, menuw, (char *)0);
        submenucount++;
        return NIL;
}

static LISP add_submenu_entry(LISP menu, LISP submenu,
                                LISP label, LISP function)
{
        Widget entry, menuw;

        menuw = find_submenu_by_name(get_c_string(menu),
                                get_c_string(submenu));
        if (!menuw) {
#if 0
                fprintf(stderr, "add_submenu_entry: No menu! Bailing out.\n");
#endif
                return NIL;
        }

        if (!strcmp(get_c_string(label), "-")) {        /* line pane */
                entry = XtCreateManagedWidget("-",
                        smeLineObjectClass,
                        menuw,
                        NULL, 0);
        } else {
                entry = XtVaCreateManagedWidget(get_c_string(function),
                        smeBSBObjectClass, menuw,
                        XtNlabel, translate(get_c_string(label)),
                        (char *)NULL);
                XtAddCallback(entry,
                        XtNcallback, execute_callback,
                        cstrdup(get_c_string(function)));
        }
        return NIL;
}

static void menu_motion(Widget w, XEvent *event,
                String *p, Cardinal *n)
{
        int fromright;
        Dimension width;
        XMotionEvent *mev = (XMotionEvent *)event;
        Widget menu, submenu;

        menu = XawSimpleMenuGetActiveEntry(w);
        if (menu == None) return;
        submenu = find_submenu_by_entry(menu);
        if (submenu == None) return;
        XtVaGetValues(w, XtNwidth, &width, (char *)0);
        fromright = width-mev->x;
        if (fromright > 0 && fromright < 20) {
                Position x, y, x_root, y_root;
                XtVaGetValues(menu,
                        XtNx, &x,
                        XtNy, &y,
                        (char *)0);
                XtTranslateCoords(w,
                        x+width-30, y, &x_root, &y_root);
                XtVaSetValues(submenu,
                        XtNx, x_root,
                        XtNy, y_root,
                        (char *)0);
                XtPopup(submenu, XtGrabNonexclusive);
        }
}

static void popdown_submenu(Widget w, XEvent *event,
                String *p, Cardinal *n)
{
        XtPopdown(w);
        XtPopdown(XtParent(w));
}


static XtActionsRec actions[] =
{
	{"pr-grid", pr_grid},
	{"siaghelp", siaghelp_action},
	{"place-shortcuts", place_shortcuts},
	{"menu-motion", menu_motion},
        {"popdown-submenu", popdown_submenu}
};


static Widget make_toggle(char *cmd, Widget pw, char *pm, char *tip)
{
        Widget w;
        Pixmap pm_return;
        Pixel color;

        XtVaGetValues(pw, XtNbackground, &color, (char *)0);

        w = XtVaCreateManagedWidget("toolbar_toggle",
                toggleWidgetClass, pw, (char *)NULL);
	pm_return = load_pixmap(pw, pm);

        XtVaSetValues(w,
                XtNbitmap, pm_return,
		XtNforeground, color,
                (char *)0);
        XtAddCallback(w,
                XtNcallback, execute_callback, (XtPointer)cmd);
	TooltipAdd(tooltip, w, translate(tip));
        return w;
}

static Widget make_command(char *cmd, Widget pw, char *pm, char *tip)
{
        Widget w;
        Pixmap pm_return;
        Pixel color;

        XtVaGetValues(pw, XtNbackground, &color, (char *)0);

        w = XtVaCreateManagedWidget("toolbar_command",
                commandWidgetClass, pw, (char *)NULL);
	pm_return = load_pixmap(pw, pm);

        XtVaSetValues(w,
                XtNbitmap, pm_return,
		XtNforeground, color,
                (char *)0);

        XtAddCallback(w,
                XtNcallback, execute_callback, (XtPointer)cmd);
	TooltipAdd(tooltip, w, translate(tip));
        return w;
}

static void make_vsep(Widget pw)
{
	XtCreateManagedWidget("vsep", labelWidgetClass, pw, NULL, 0);
}

/* ---
The toolbar
*/

static void init_toolbar(void)
{
	frame2 = XtVaCreateManagedWidget("toolbar",
		frameWidgetClass, topbox,
		(char *)0);
	bars |= TOOLBAR;
	toolbox = XtVaCreateManagedWidget("toolbox",
		boxWidgetClass, frame2,
		(char *)0);
	make_handle(toolbox, frame2);
	frame3 = XtVaCreateManagedWidget("formatbar",
		frameWidgetClass, topbox,
		(char *)0);
	bars |= FORMATBAR;
	formatbox = XtVaCreateManagedWidget("formatbox",
		boxWidgetClass, frame3,
		(char *)0);
	make_handle(formatbox, frame3);
	make_command("(new-buffer)", toolbox, "new.xpm",
		     "Start another instance of Egon Animator");
	make_command("(load-buffer)", toolbox, "fld_open.xpm",
		     "Open an Egon Animator document");
	make_command("(save-buffer-as)", toolbox, "save.xpm",
		     "Save the contents of the current buffer");
	make_command("(preview)", toolbox, "preview.xpm",
		     "Preview the contents of the current buffer");
	make_command("(print)", toolbox, "printer.xpm",
		     "Print the contents of the current buffer");
	make_vsep(toolbox);
	make_command("(ani-ctl ANI_STOP 0)", toolbox, "stop.xpm",
		     "Pause the animation");
	make_command("(ani-ctl ANI_PREVIOUS 0)", toolbox, "previous.xpm",
		     "Previous animation frame");
	make_command("(ani-ctl ANI_CONTINUE 0)", toolbox, "play.xpm",
		     "Play the animation");
	make_command("(ani-ctl ANI_NEXT 0)", toolbox, "next.xpm",
		     "Next animation frame");
	make_vsep(toolbox);
	make_command("(help-contents)", toolbox, "info.xpm",
		     "Display the Egon online documentation");
	make_command("(help-copyright)", toolbox, "copyright.xpm",
		     "Display the Gnu general public license");
}

static void init_toggle(void)
{
	cmdBold = make_toggle("(toggle-format \"bold\")",
		formatbox, "bold.xpm", "Bold text");
	cmdItalic = make_toggle("(toggle-format \"italic\")",
		formatbox, "italic.xpm", "Italic text");
}

static char *combo_sizes[] = {
	"8", "9", "10", "11", "12", "14", "16", "18",
	"20", "22", "24", "26", "28", "36", "48", "72"
};

static char *combo_objects[] = {
	"Line", "Rectangle", "Arc", "Ellipse",
	"Pixmap", "String", "Point", "Filled Rectangle",
	"Filled Arc", "Filled Ellipse"
};

static char **combo_fonts, **combo_colors;
static int ncombo_fonts, ncombo_colors;

static void cb_font(char *p)
{
	va_execute("(new-format \"family\" \"%s\")", p);
	activate_window(w_list);
}

static void cb_size(char *p)
{
	va_execute("(new-format \"size\" (* 10 %s))", p);
	activate_window(w_list);
}

static void cb_object(char *p)
{
	va_execute("(new-format \"style\" \"%s\")", p);
	activate_window(w_list);
}

static void cb_color(char *p)
{
	va_execute("(new-format \"fg\" \"%s\")", p);
	activate_window(w_list);
}

/* ---
*/
void setup_buttons(void)
{
	Widget formatframe;

	if (topbox == None) return;

	formatframe = XtVaCreateManagedWidget("formatframe",
		frameWidgetClass, formatbox,
		(char *)0);
	combo_fonts = font_list(&ncombo_fonts);
	btnFont = XtVaCreateManagedWidget("toolbar_command",
		comboWidgetClass, formatframe,
		XtNcomboList, cb_font,
		XtNcomboText, cb_font,
		XtNcomboTop, edit_shell,
		XtNcomboData, combo_fonts,
		XtNcomboNData, ncombo_fonts,
		XtNwidth, 160,
		(char *)0);
	TooltipAdd(tooltip, btnFont, translate("Change the font family"));

	formatframe = XtVaCreateManagedWidget("formatframe",
		frameWidgetClass, formatbox,
		(char *)0);
	btnSize = XtVaCreateManagedWidget("toolbar_command",
		comboWidgetClass, formatframe,
		XtNcomboList, cb_size,
		XtNcomboText, cb_size,
		XtNcomboTop, edit_shell,
		XtNcomboData, combo_sizes,
		XtNcomboNData, XtNumber(combo_sizes),
		XtNwidth, 40,
		(char *)0);
	TooltipAdd(tooltip, btnSize, translate("Change the font size"));

	formatframe = XtVaCreateManagedWidget("formatframe",
		frameWidgetClass, formatbox,
		XtNwidth, 100,
		(char *)0);
	btnStyle = XtVaCreateManagedWidget("toolbar_command",
		comboWidgetClass, formatframe,
		XtNcomboList, cb_object,
		XtNcomboText, cb_object,
		XtNcomboTop, edit_shell,
		XtNcomboData, combo_objects,
		XtNcomboNData, XtNumber(combo_objects),
		XtNwidth, 100,
		(char *)0);
	TooltipAdd(tooltip, btnStyle, translate("Change the display style"));

	formatframe = XtVaCreateManagedWidget("formatframe",
		frameWidgetClass, formatbox,
		(char *)0);
	combo_colors = color_list(&ncombo_colors);
	btnColor = XtVaCreateManagedWidget("toolbar_command",
		comboWidgetClass, formatframe,
		XtNcomboList, cb_color,
		XtNcomboText, cb_color,
		XtNcomboTop, edit_shell,
		XtNcomboData, combo_colors,
		XtNcomboNData, ncombo_colors,
		XtNwidth, 80,
		(char *)0);
	TooltipAdd(tooltip, btnColor, translate("Change the color"));
}

static LISP popup_editor(void)
{
	XtVaSetValues(btnFont, XtNwidth, 160, (char *)0);
	XtVaSetValues(btnSize, XtNwidth, 40, (char *)0);
	XtVaSetValues(btnStyle, XtNwidth, 100, (char *)0);
	XtVaSetValues(btnColor, XtNwidth, 80, (char *)0);
	XtPopup(edit_shell, XtGrabNone);
	XSetWMProtocols(XtDisplay(edit_shell), XtWindow(edit_shell),
			&wm_delete_window, 1);
	editor_popped = 1;
	return NIL;
}

static LISP popdown_editor(void)
{
	editor_popped = 0;
	XtPopdown(edit_shell);
	return NIL;
}

static void editor_init(void)
{
	edit_shell = XtVaCreatePopupShell("edit_shell",
		topLevelShellWidgetClass, topLevel,
		XtNtitle, translate("Egon Editor"),
		(char *)0);
	topbox = XtCreateManagedWidget("topbox",
		rudegridWidgetClass, edit_shell, NULL, 0);
	XtOverrideTranslations(edit_shell,
		XtParseTranslationTable(
			"<Message>WM_PROTOCOLS: execute(quit-program)"));
}

/* ---
*/
void drop_handler(Widget w, XtPointer data,
		XEvent *event, Boolean *b)
{
	unsigned char *Data, *filename;
	unsigned long DataSize;
	int DataType = DndDataType(event);

	if (DataType == DndNotDnd) return;
	DndGetData(&Data, &DataSize);
	switch (DataType) {
	case DndFile:
		filename = Data;
		if (!fork()) {
			execlp("egon", "Egon", filename, NULL);
			exit(1);
		}
	default:
		;
	}
}

/* ---
*/
void activate_window(window *w)
{
	char b[256];

	w_list = w;
	if (w) {
		strcpy(b, "Egon Animator: ");
		strncat(b, w->buf->name, 200);
		XtVaSetValues(topLevel, XtNtitle, b, NULL);
	}
}

static void object_select(Widget w, XtPointer client_data, XtPointer call_data)
{
	XawListReturnStruct *list_struct = (XawListReturnStruct *)call_data;
	String string = list_struct->string;
	char cmd[256];

	sprintf(cmd, "(select-object \"%s\")", string);
	execute(cmd);
	draw_buffer(w_list);
}

static void tick_select(Widget w, XtPointer client_data, XtPointer call_data)
{
	XawListReturnStruct *list_struct = (XawListReturnStruct *)call_data;
	String string = list_struct->string;
	char cmd[256];

	sprintf(cmd, "(select-tick %s)", string);
	execute(cmd);
	draw_buffer(w_list);
}

/* ---
*/
void draw_input(char *text)
{
	char b[256];
	if (!w_list->object)
		sprintf(b, "No object selected");
	else
		sprintf(b,
			translate("[%s at %d] [size = %dx%d] [duration=%d delta=%d]"),
			w_list->object->name, w_list->script->time,
			w_list->buf->width, w_list->buf->height,
			w_list->buf->duration, w_list->buf->delta);
	label_set(label1, b);
	label_set(label3, "Egon");
}

/* ---
*/
void draw_status(char *text)
{
	label_set(label2, text);
	XFlush(XtDisplay(topLevel));
}

/* ---
   void llpr(char *p)
   Prints the string p on the bottom line of the screen.  If p is empty and
   the last string printed was also empty, the string isn't printed.
*/

void llpr(char *p)
{
	static int isclear = FALSE;

	if (isclear && p[0] == '\0')
		return;
	isclear = (p[0] == '\0');

	draw_status(p);
}

#define FREE_LIST(l,n)\
	if (l) {\
		for (i = 0; i < (n); i++) cfree((l)[i]);\
		cfree(l);\
	}

#undef FREE_LIST

#define FREE_LIST(l,n)

/* ---
*/
void draw_buffer(window *w)
{
	buffer *b = w->buf;
	ani_object *o;
	ani_script *s;
	char p[256];
	static String fallback[1] = {"<none>"};
	static int on = 0, sn = 0, pn = 0;
	static char **ol = NULL, **sl = NULL, **pl = NULL;

	FREE_LIST(ol, on)
	FREE_LIST(sl, sn)
	FREE_LIST(pl, pn)

	/* select the first object if none is selected */
	if (!w->object) {
		w->object = w->buf->cast;
		if (w->object) w->script = w->object->script;
	}

	on = 0;
	for (o = b->cast; o; o = o->next) on++;

	if (!on) {
		XawListChange(w->ui->objl, fallback, 1, 0, True);
		XawListChange(w->ui->tickl, fallback, 1, 0, True);
		XawListChange(w->ui->propl, fallback, 1, 0, True);
		sn = pn = 0;
		goto The_end;
	}

	ol = (char **)cmalloc(on*sizeof(char *));
	on = 0;
	for (o = b->cast; o; o = o->next) {
		ol[on] = cstrdup(o->name);
		on++;
	}
	XawListChange(w->ui->objl, ol, on, 0, True);

	o = w_list->object;
	sn = 0;
	for (s = o->script; s; s = s->next) sn++;
	sl = (char **)cmalloc(sn*sizeof(char *));
	sn = 0;
	for (s = o->script; s; s = s->next) {
		sprintf(p, "%d", s->time);
		sl[sn] = cstrdup(p);
		sn++;
	}
	XawListChange(w->ui->tickl, sl, sn, 0, True);

	s = w_list->script;
	pn = 0;
	pl = (char **)cmalloc(9*sizeof(char *));
	sprintf(p, "Position = (%d, %d)", s->x, s->y);
	pl[pn++] = cstrdup(p);
	sprintf(p, "Size = %dx%d", s->width, s->height);
	pl[pn++] = cstrdup(p);
	sprintf(p, "Visible = %s", s->visible?"True":"False");
	pl[pn++] = cstrdup(p);
	sprintf(p, "Format = %d", o->fmt);
	pl[pn++] = cstrdup(p);
	sprintf(p, "String = %s", o->string?o->string:"EMPTY");
	pl[pn++] = cstrdup(p);
	XawListChange(w->ui->propl,
		pl, pn, 0, True);

The_end:

	XtVaSetValues(topLevel,
		XtNwidth, b->width,
		XtNheight, b->height,
		(char *)0);

	XtVaSetValues(stage,
		XtNwidth, b->width,
		XtNheight, b->height,
		XtNanimatorCast, b->cast,
		XtNanimatorDelta, b->delta,
		XtNanimatorDuration, b->duration,
		XtNanimatorBgPixmap, b->bg,
		XtNanimatorNow, b->now,
		XtNanimatorMode, b->state,
		(char *)0);
	XtVaSetValues(topLevel,
		XtNwidth, b->width,
		XtNheight, b->height,
		(char *)0);
/*	XtMapWidget(topLevel);
*/
	draw_input(NULL);
}

/* ---
*/
window *find_window_by_widget(Widget wdg)
{
	window *w = w_list;
	do {
		if (w->ui->viewport == wdg ||
			w->ui->objv == wdg || w->ui->objl == wdg ||
			w->ui->tickv == wdg || w->ui->tickl == wdg ||
			w->ui->propv == wdg || w->ui->propl == wdg)
			return w;
		w = w->next;
	} while (w != w_list);
	return NULL;
}

/* ---
*/
void free_window(window *w)
{
	window *pw;

	for (pw = w_list; pw->next != w && pw->next != pw; pw = pw->next);
	pw->next = w->next;

	if (w_list == w) w_list = w_list->next;
	if (w_list == w) w_list = NULL;
	if (w->ui->viewport != None)
		XtDestroyWidget(w->ui->viewport);
	cfree(w);
}

/* ---
When running as player, return a window structure without any widgets
Slightly modified for Egon: only one window at a time
*/

window *new_window(buffer *b, window *prev)
{
	window *w;

	if (w_list) w = w_list;
	else w = (window *)cmalloc(sizeof(window));

	if (w == NULL) return NULL;

	w->ui = (egon_ui *)cmalloc(sizeof(egon_ui));
	if (w->ui == NULL) {
		cfree(w);
		return NULL;
	}
	w->buf = b;

	if (prev == NULL) prev = w;
	else w->next = prev->next;
	prev->next = w;

	w->object = NULL;	/* none selected yet */
	w->script = NULL;

	w->ui->viewport = XtVaCreateManagedWidget("viewport",
		rudegridWidgetClass, gridpane,
		(char *)0);

	w->ui->objf = XtVaCreateManagedWidget("objf",
		frameWidgetClass, w->ui->viewport,
		(char *)0);
	w->ui->objt = XtVaCreateManagedWidget("objt",
		labelWidgetClass, w->ui->objf,
		(char *)0);
	label_set(w->ui->objt, "Objects");
	XtVaSetValues(w->ui->objf,
		XtNtitle, w->ui->objt,
		(char *)0);
	w->ui->objv = XtVaCreateManagedWidget("objv",
		viewportWidgetClass, w->ui->objf,
		(char *)0);
	w->ui->objl = XtVaCreateManagedWidget("objl",
		listWidgetClass, w->ui->objv, NULL);
	XtAddCallback(w->ui->objl, XtNcallback, object_select, NULL);

	w->ui->tickf = XtVaCreateManagedWidget("tickf",
		frameWidgetClass, w->ui->viewport,
		(char *)0);
	w->ui->tickt = XtVaCreateManagedWidget("tickt",
		labelWidgetClass, w->ui->tickf,
		(char *)0);
	label_set(w->ui->tickt, "Ticks");
	XtVaSetValues(w->ui->tickf,
		XtNtitle, w->ui->tickt,
		(char *)0);
	w->ui->tickv = XtVaCreateManagedWidget("tickv",
		viewportWidgetClass, w->ui->tickf,
		(char *)0);
	w->ui->tickl = XtVaCreateManagedWidget("tickl",
		listWidgetClass, w->ui->tickv, NULL);
	XtAddCallback(w->ui->tickl, XtNcallback, tick_select, NULL);

	w->ui->propf = XtVaCreateManagedWidget("propf",
		frameWidgetClass, w->ui->viewport,
		(char *)0);
	w->ui->propt = XtVaCreateManagedWidget("propt",
		labelWidgetClass, w->ui->propf,
		(char *)0);
	label_set(w->ui->propt, "Properties");
	XtVaSetValues(w->ui->propf,
		XtNtitle, w->ui->propt,
		(char *)0);
	w->ui->propv = XtVaCreateManagedWidget("propv",
		viewportWidgetClass, w->ui->propf,
		(char *)0);
	w->ui->propl = XtVaCreateManagedWidget("propl",
		listWidgetClass, w->ui->propv, (char *)0);

	XtVaSetValues(stage,
		XtNanimatorCast, b->cast,
		XtNanimatorNow, 0,
		XtNanimatorDelta, b->delta,
		XtNanimatorDuration, b->duration,
		XtNanimatorMode, ANI_STOP,
		XtNanimatorBgPixmap, b->bg,
		XtNwidth, b->width,
		XtNheight, b->height,
		(char *)0);

	return w;
}

/* ---
*/
int ani_ctl(int mode, unsigned int now)
{
	XtVaSetValues(stage,
		XtNanimatorMode, mode, (char *)0);
        return ANI_OK;
}

/* ---
*/
int remove_window(window *w)
{
	
	if (w == w->next) return FALSE;
	free_window(w);
	return TRUE;
}

/* ---
*/
int split_window(window *w)
{
	window *w2 = new_window(w->buf, w);

	if (w2 == NULL) return FALSE;
	return TRUE;
}

/* ---
this does probably not belong here in window.c,
   because it doesn't depend on X
*/

static void save_plugin(char *p)
{
        if (*p++ != ' ' || *p == '\0') printf("501 File name missing\n");
        else {
                if (savematrix(p, w_list->buf, NULL)) {
                        printf("501 Can't save %s\n", p);
                } else {
                        printf("250 Saved %s\n", p);
                }
        }
}

static void load_plugin(char *p)
{
        if (*p++ != ' ' || *p == '\0') printf("501 File name missing\n");
        else {
                if (loadmatrix(p, w_list->buf, NULL)) {
                        printf("501 Can't load %s\n", p);
                } else {
                        printf("250 Loaded %s\n", p);
                }
        }
}

static void exec_plugin(char *p)
{
        if (*p++ != ' ' || *p == '\0') printf("501 Command missing\n");
        else {
                execute(p);
                printf("250 OK\n");
        }
}

static void help_plugin(char *p)
{
        printf("214 SAVE LOAD EXEC HELP NOOP QUIT PRNT\n");
}

static void noop_plugin(char *p)
{
        printf("250 OK\n");
}

static void win_plugin(char *p)
{
        printf("250 %lx\n", (unsigned long)XtWindow(topLevel));
}

static void quit_plugin(char *p)
{
        printf("221 Over and out\n");
        execute("(quit-program)");
}

static void prnt_plugin(char *p)
{
        printf("502 Can't print yet\n");
}

static struct {
        char *verb;
        void (*cb)(char *);
} plugin_cmds[] = {
        {"SAVE", save_plugin},
        {"LOAD", load_plugin},
        {"EXEC", exec_plugin},
        {"HELP", help_plugin},
        {"NOOP", noop_plugin},
        {"WIN", win_plugin},
        {"QUIT", quit_plugin},
        {"PRNT", prnt_plugin},
        {NULL, NULL}
};

static void read_plugin_cmd(XtPointer client_data, int *fid, XtInputId *id)
{
        char b[1024], *p;
        int i, n;

        if ((n = read(*fid, b, 1020)) == -1) return;

        b[n] = '\0';
        if ((p = strchr(b, '\n')) == NULL) {
                printf("501 Incomplete command\n");
                fflush(stdout);
                return;
        }

        *p = '\0';
        for (i = 0; plugin_cmds[i].verb; i++) {
                if (!strncmp(b, plugin_cmds[i].verb,
                                strlen(plugin_cmds[i].verb)))
                        break;
        }
        if (plugin_cmds[i].verb)
                (*plugin_cmds[i].cb)(b+strlen(plugin_cmds[i].verb));
        else
                printf("500 What are you talking about\n");
        fflush(stdout);
}

/* ---
handle spontaneous exit of plugins
*/

static void handle_plugin_exit(int ph)
{
        buffer *b = b_list;

        do {
                int n = buffer_plugin2index(b, ph);
                if (n != -1) {
                        cfree(b->plugin[n].name);
                        b->nplugin--;
                        for (; n < b->nplugin; n++)
                                b->plugin[n] = b->plugin[n+1];
                        b->change = pr_scr_flag = TRUE;
                }
                b = b->next;
        } while (b != b_list);
}

static void init_windows1(int *argc, char **argv)
{
        unsigned long highlight_color, unhighlight_color;
	XtSetLanguageProc(NULL, (XtLanguageProc) NULL, NULL);

	topLevel = XtVaAppInitialize(
		    &app_context,	/* application context */
		    "Egon",		/* application class */
		    options,		/* command line options list */
		    XtNumber(options),
		    argc, argv,	/* command line args */
		    fallback_resources,	/* for missing app-defaults file */
		    NULL);		/* terminate varargs list */

	t1_init(topLevel);

        XtGetApplicationResources(topLevel, &app_data, resources,
                        XtNumber(resources), NULL, 0);

	XtAppAddActions(app_context, actions, XtNumber(actions));

	shortcuts = XtVaCreatePopupShell("shortcuts",
                simpleMenuWidgetClass, topLevel, (char *)0);
        XtVaCreateManagedWidget("-",
                smeLineObjectClass, shortcuts, (char *)0);

	stage = XtVaCreateManagedWidget("stage",
		animatorWidgetClass, topLevel,
		XtNanimatorCast, NULL,
		XtNanimatorMode, ANI_STOP,
		XtNanimatorNow, 0,
		(char *)0);
	editor_init();
        highlight_color = WhitePixel(XtDisplay(topLevel),
                                DefaultScreen(XtDisplay(topLevel)));

	XtVaGetValues(edit_shell,   /* or any widget with grey bg */
	        XtNbackground, &unhighlight_color,
	        (char *)0);
	tooltip_init(edit_shell, highlight_color, unhighlight_color);
	tooltip = XtVaCreatePopupShell("tooltip",
		tooltipWidgetClass, topLevel,
		(char *)0);

	init_menu();
	init_toolbar();
	setup_buttons();
	init_toggle();

	textbox = XtVaCreateManagedWidget("textbox",
		rudegridWidgetClass, topbox,
		(char *)0);
	label1 = XtVaCreateManagedWidget("label1",
		labelWidgetClass, textbox,
		(char *)0);

	gridpane = XtVaCreateManagedWidget("gridpane",
		panedWidgetClass, topbox,
		XtNallowResize, True,
		(char *)0);

	statusbox = XtVaCreateManagedWidget("statusbox",
		rudegridWidgetClass, topbox,
		(char *)0);
	label2 = XtVaCreateManagedWidget("label2",
		labelWidgetClass, statusbox,
		(char *)0);
	label3 = XtVaCreateManagedWidget("label3",
		labelWidgetClass, statusbox,
		(char *)0);
}

static void handle_plugin_cmd(char *p)
{
	execute(p);
}


/* ---
   void init_windows(buffer *b)
   Sets up the whole initial window structure and initializes scrupd.
   The window list w_list is set to a list with a single window with the
   buffer b.
*/

void init_windows(buffer * b, int *argc, char **argv)
{
	Display *dpy;
	unsigned long highlight_color, unhighlight_color;

	start_splash();

	init_windows1(argc, argv);
	interp_startup();

	dpy = XtDisplay(topLevel);

	init_format(dpy);
	encode_format(~0, &fmt0);

	XtRealizeWidget(topLevel);

        plugin_init(topLevel, handle_plugin_exit, handle_plugin_cmd);

	activate_window(new_window(b, NULL));

	wm_delete_window = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
	XtOverrideTranslations(topLevel,
		XtParseTranslationTable(
			"<Message>WM_PROTOCOLS: execute(quit-program)"));

	XSetWMProtocols(dpy, XtWindow(topLevel), &wm_delete_window, 1);

	draw_status("");

	/* Set up selection */
	target_atom = XInternAtom(dpy, "EGON_BLOCK", False);

	embed_init(topLevel);

	ok2print = 1;

	init_subr_0("popup-editor", popup_editor);
	init_subr_0("popdown-editor", popdown_editor);
	init_subr_1("add-menu", add_menu);
	init_subr_3("add-menu-entry", add_menu_entry);
	init_subr_2("add-submenu", add_submenu);
        init_subr_4("add-submenu-entry", add_submenu_entry);
	init_subr_1("tooltip-mode", ltooltip_mode);

	init_form(topLevel);

	icon_set(topLevel, egon_xpm);
	DndInitialize(topLevel);
	DndRegisterOtherDrop(drop_handler);

        highlight_color = WhitePixel(XtDisplay(topLevel),
				DefaultScreen(XtDisplay(topLevel)));
        XtVaGetValues(label1,   /* or any widget with grey bg */
                XtNbackground, &unhighlight_color,
                NULL);

	stop_splash();
}

/* ---
   void exit_windows()
   Cleans up after Calc before exit.  All buffers and windows are freed.
*/

void exit_windows(void)
{
	/* free all buffers */
	while (b_list != NULL)
		free_buffer(b_list);
	while (w_list != NULL)
		free_window(w_list);
}

/* ---
   static void pr_scr()
   Prints and refreshes all the windows.
   Sets pr_scr_flag to FALSE.
*/

static void pr_scr(void)
{
	window *w;
	int i;

	draw_status("");
	w = w_list;
	do {
		draw_buffer(w);
		for (i = 0; i < w->buf->nplugin; i++) {
                        if (!w->buf->plugin[i].displayed) {
                                plugin_show(w->buf->plugin[i].ph,
                                                stage);
                                w->buf->plugin[i].displayed = 1;
                        }
                }
		w = w->next;
	} while (w != w_list);
	pr_scr_flag = FALSE;
}	/* pr_scr */




/* ---
*/
void show_format(void)
{
	static int last_fmt = -1;
	int f = ret_format(w_list->buf,
				w_list->object,
				w_list->script);
	sfmt fmt;
	int hadj = HADJ_LEFT;
	int hadj_left = (hadj == HADJ_LEFT);
	int hadj_center = (hadj == HADJ_CENTER);
	int hadj_right = (hadj == HADJ_RIGHT);
	char b[100];

	decode_format(f, ~0, &fmt);
	/* menus */
	if (editor_popped) {
	   	int type;
	   	if (!w_list->buf || !w_list->object || !w_list->script)
	     		type = 0;
		else type = w_list->object->type;
		combo_text_change(btnFont, fmt.family);
		sprintf(b, "%d", fmt.size/10);
		combo_text_change(btnSize, b);
		combo_text_change(btnStyle, type2name(type));
		combo_text_change(btnColor, fmt.fg);
	}

	/* toggle buttons */
	state_set(cmdBold, (fmt.bold?1:0), 1, 0);
	state_set(cmdItalic, (fmt.italic?1:0), 1, 0);
	state_set(cmdHLeft, (hadj_left?1:0), 1, 0);
	state_set(cmdHCenter, (hadj_center?1:0), 1, 0);
	state_set(cmdHRight, (hadj_right?1:0), 1, 0);
	last_fmt = f;
}

int cursor_visible = FALSE;

/* ---
   void show_cur(window *w)
   Moves the cursor to reflect the position of point in w.
   If point is not visible, the window is moved so that point is in
   the middle of the screen.
*/

void show_cur(window *w)
{
	if (w) {
		if (pr_scr_flag) pr_scr();
		show_format();
	}
}	/* show_cur */

/* ---
*/
void hide_cur(window *w)
{
}

/* ---
*/
void mainloop(void)
{
        if (app_data.plugin) {
                /* control plugin from stdin */
                XtAppAddInput(XtWidgetToApplicationContext(topLevel),
                        fileno(stdin), (XtPointer)XtInputReadMask,
                        read_plugin_cmd, NULL);
                printf("220 %s\n", version);
                fflush(stdout);
        }

        XtAppMainLoop(XtWidgetToApplicationContext(topLevel));

        exit(0);
}

