/*
    Copyright (C) 1996-1998  Ulric Eriksson <ulric@edu.stockholm.se>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston,
    MA 02111-1307, USA.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "common.h"
#include "oldfonts.h"
#include "fonts.h"

siag_format format_table[256];
int nformat;

static struct {
	char *name;
	int mask;
} attrnames[] = {
	{"family", SFMT_FAMILY},
	{"size", SFMT_SIZE},
	{"fg", SFMT_FG},
	{"bg", SFMT_BG},
	{"bold", SFMT_BOLD},
	{"italic", SFMT_ITALIC},
	{"uline", SFMT_ULINE},
	{"hadj", SFMT_HADJ},
	{"vadj", SFMT_VADJ},
	{"borders", SFMT_BORDERS},
	{"style", SFMT_STYLE},
	{NULL, 0}
};

/* ---
Return an old-style attribute mask from its name.
*/

int fmt_attr2mask(char *attr)
{
	int i;

	for (i = 0; attrnames[i].name; i++) {
		if (!cstrcasecmp(attr, attrnames[i].name))
			return attrnames[i].mask;
	}
	return 0;
}

/* ---
Return an attribute name from its old-style mask.
*/

char *fmt_mask2attr(int mask)
{
	int i;

	for (i = 0; attrnames[i].name; i++) {
		if (mask == attrnames[i].mask)
			return attrnames[i].name;
	}
	return NULL;
}


/* ---
returns the index of a format structure with all the requested
   properties, or -1 for failure. It is assumed that the font has been
   allocated by calling lookup_font and the colors by calling
   lookup_color. lookup_format doesn't allocate any X resources.

   981106: Added "style", which is a bit tricky because we have no
   idea in this code what a style is or does: it varies between
   applications. The application encodes the style in to an integer
   and we don't care what it really means.
*/

int lookup_format(int font, int uline, int fgcolor, int bgcolor,
		int style, int siagfmt)
{
	int i;

	for (i = 0; i < nformat; i++) {
		if (format_table[i].font == font &&
			format_table[i].uline == uline &&
			format_table[i].fg == fgcolor &&
			format_table[i].bg == bgcolor &&
			format_table[i].style == style &&
			format_table[i].siagfmt == siagfmt)
				return i;
	}
	nformat++;
	format_table[i].font = font;
	format_table[i].uline = uline;
	format_table[i].fg = fgcolor;
	format_table[i].bg = bgcolor;
	format_table[i].style = style;
	format_table[i].siagfmt = siagfmt;
	return i;
}

/* ---
Loading and saving formats.

This code is shared between all programs that use the new formats.
That way there will be less duplication of code.

This requires new tags to store the formats. The new tag is

.ft N

where N is a number. This is followed by additional lines of
the format

name value

where name is the name of an attribute and
value is the value of the attribute. The beauty of this is
that it can be extended with new attributes without breaking
compatibility. Old versions of the program will simply not
understand new attributes and will therefore ignore them.
A format specification is terminated by the word "end" alone
on a line.

Example:

.ft 0
family Times
size 10
uline n
fg red
end
.ft 1
family Courier
size 12
italic y
end

It is not necessary to specify all attributes. The ones that are
not specified will use defaults compiled into the program.

Compatibility: Old files contain no .ft tags. For those, a function
will be used that translates from the old bitmap fields to new
structures. Another function translates in the other direction,
making it possible to store in a format old versions of Siag
can open.
--- */

/* ---
Convert an old-style bitmapped format to the new style index.
*/

int fmt_old2new(long oldfmt)
{
	sfmt fmt;

	switch (oldfmt & SIZE_MASK) {
	case SIZE_8: fmt.size = 80; break;
	case SIZE_10: fmt.size = 100; break;
	case SIZE_12: fmt.size = 120; break;
	case SIZE_14: fmt.size = 140; break;
	case SIZE_18: fmt.size = 180; break;
	case SIZE_24: fmt.size = 240; break;
	case SIZE_20: fmt.size = 200; break;
	default: fmt.size = 300; break;
	}
	fmt.italic = ((oldfmt & ITALIC) != 0);
	fmt.bold = ((oldfmt & BOLD) != 0);
	fmt.uline = ((oldfmt & ULINE) != 0);
	switch (oldfmt & FONT_MASK) {
	case COURIER: fmt.family = "Courier"; break;
	case HELVETICA: fmt.family = "Helvetica"; break;
	case NEW_CENTURY: fmt.family = "New Century Schoolbook"; break;
	default: fmt.family = "Times"; break;
	}
	switch (oldfmt & COLOR_MASK) {
	case COLOR_0: fmt.fg = "black"; break;
	case COLOR_1: fmt.fg = "red"; break;
	case COLOR_2: fmt.fg = "green"; break;
	case COLOR_3: fmt.fg = "blue"; break;
	case COLOR_4: fmt.fg = "yellow"; break;
	case COLOR_5: fmt.fg = "magenta"; break;
	case COLOR_6: fmt.fg = "cyan"; break;
	default: fmt.fg = "white"; break;
	}
	fmt.bg = "white";
	fmt.borders = oldfmt & BORDER_MASK;
	fmt.vadj = oldfmt & VADJ_MASK;
	fmt.hadj = oldfmt & HADJ_MASK;
	/* this works because the old styles are registered first */
	fmt.style = (oldfmt & FMT_MASK) >> FMT_SHIFT;
	return encode_format(~0, &fmt);
}

/* ---
Convert a new-style format index to the old bitmapped format.
this function could try harder
*/

long fmt_new2old(int newfmt)
{
	return SIZE_10 | HELVETICA;
}

/* the default format */
static sfmt dfmt = { "Helvetica", 120, 0, 0, 0, "black", "white",
			0, VADJ_CENTER, HADJ_LEFT, 0};

/* ---
the formats are not split up into fonts, colors and rest, but saved
   in a single level. The restructuring is done when the formats are read.
   Attributes: font, size, bold, italic, uline, fg, bg, siagfmt
*/

void save_formats(FILE *fp, int i)
{
	sfmt fmt;

	decode_format(i, ~0, &fmt);
	fprintf(fp, ".ft %d\n", i);
	if (cstrcasecmp(fmt.family, dfmt.family))
		fprintf(fp, "font %s\n", fmt.family);
	if (fmt.size != dfmt.size)
		fprintf(fp, "size %d\n", fmt.size);
	if (fmt.bold != dfmt.bold)
		fprintf(fp, "bold %d\n", fmt.bold);
	if (fmt.italic != dfmt.italic)
		fprintf(fp, "italic %d\n", fmt.italic);
	if (fmt.uline != dfmt.uline)
		fprintf(fp, "uline %d\n", fmt.uline);
	if (cstrcasecmp(fmt.fg, dfmt.fg))
		fprintf(fp, "fg %s\n", fmt.fg);
	if (cstrcasecmp(fmt.bg, dfmt.bg))
		fprintf(fp, "bg %s\n", fmt.bg);
	if (fmt.borders != dfmt.borders)
		fprintf(fp, "borders %d\n", fmt.borders);
	if (fmt.vadj != dfmt.vadj)
		fprintf(fp, "vadj %d\n", fmt.vadj);
	if (fmt.hadj != dfmt.hadj)
		fprintf(fp, "hadj %d\n", fmt.hadj);
	if (fmt.style != dfmt.style)
		fprintf(fp, "style %d\n", fmt.style);
	fprintf(fp, "end\n");
}

/* ---
When this function gets to borrow the file pointer, we are about to
   read the first attribute of a font. We keep on reading until eof
   or "end". The value returned is the index into the real format table.
*/

int load_formats(FILE *fp)
{
	char b[1000];
	char font[256], fg[256], bg[256];
	sfmt fmt = dfmt;

	while (fgets(b, sizeof b, fp) && strncmp(b, "end", 3)) {
		chomp(b);
		if (!strncmp(b, "font ", 5)) {
			strcpy(font, b+5);
			fmt.family = font;
		} else if (!strncmp(b, "size ", 5)) {
			fmt.size = strtol(b+5, NULL, 10);
		} else if (!strncmp(b, "bold ", 5)) {
			fmt.bold = strtol(b+5, NULL, 10);
		} else if (!strncmp(b, "italic ", 7)) {
			fmt.italic = strtol(b+7, NULL, 10);
		} else if (!strncmp(b, "uline ", 6)) {
			fmt.uline = strtol(b+6, NULL, 10);
		} else if (!strncmp(b, "fg ", 3)) {
			strcpy(fg, b+3);
			fmt.fg = fg;
		} else if (!strncmp(b, "bg ", 3)) {
			strcpy(bg, b+3);
			fmt.bg = bg;
		} else if (!strncmp(b, "borders ", 8)) {
			fmt.borders = strtol(b+8, NULL, 10);
		} else if (!strncmp(b, "vadj ", 5)) {
			fmt.vadj = strtol(b+5, NULL, 10);
		} else if (!strncmp(b, "hadj ", 5)) {
			fmt.hadj = strtol(b+5, NULL, 10);
		} else if (!strncmp(b, "style ", 6)) {
			fmt.style = strtol(b+6, NULL, 10);
		}
		/* ignore anything we don't grok */
	}
	return encode_format(~0, &fmt);
}

