/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <assert.h>
#include <stdlib.h>
#include "common.h"
#include "error.h"
#include "misc.h"
#include "lexer.h"
#include "var.h"
#include "str.h"
#include "cases.h"

#undef DEBUGGING
/*#define DEBUGGING 1*/
#include "debug-print.h"

/* Parses the NUMERIC command. */
int
cmd_numeric (void)
{
  int i;

  /* Names of variables to create. */
  char **v;
  int nv;

  /* Format spec for variables to create.  f.type==-1 if default is to
     be used. */
  fmt_spec f;

  match_id (NUMERIC);
  do
    {
      if (!parse_DATA_LIST_vars (&v, &nv, PV_NONE))
	return -1;

      /* Get the optional format specification. */
      if (match_tok ('('))
	{
	  if (!parse_format_specifier (&f, 0))
	    goto fail;
	  if (formats[f.type].cat & FCAT_STRING)
	    {
	      msg (SE, _("Format type %s may not be used with a numeric "
		   "variable."), fmt_to_string (&f));
	      goto fail;
	    }

	  if (!match_tok (')'))
	    {
	      msg (SE, _("`)' expected after output format."));
	      goto fail;
	    }
	}
      else
	f.type = -1;

      /* Create each variable. */
      for (i = 0; i < nv; i++)
	{
	  variable *new_var = create_variable (&default_dict, v[i],
					       NUMERIC, 0);
	  if (!new_var)
	    msg (SE, _("There is already a variable named %s."), v[i]);
	  else
	    {
	      if (f.type != -1)
		new_var->print = new_var->write = f;
	      envector (new_var);
	    }
	}

      /* Clean up. */
      for (i = 0; i < nv; i++)
	free (v[i]);
      free (v);
    }
  while (match_tok ('/'));

  if (token != '.')
    return syntax_error (_("expecting end of command"));

  return 1;

  /* If we have an error at a point where cleanup is required,
     flow-of-control comes here. */
fail:
  for (i = 0; i < nv; i++)
    free (v[i]);
  free (v);
  return -1;
}

/* Parses the STRING command. */
int
cmd_string (void)
{
  int i;

  /* Names of variables to create. */
  char **v;
  int nv;

  /* Format spec for variables to create. */
  fmt_spec f;

  /* Width of variables to create. */
  int width;

  match_id (STRING);
  do
    {
      if (!parse_DATA_LIST_vars (&v, &nv, PV_NONE))
	return -1;

      force_match ('(');
      if (!parse_format_specifier (&f, 0))
	goto fail;
      if (!(formats[f.type].cat & FCAT_STRING))
	{
	  msg (SE, _("Format type %s may not be used with a string "
	       "variable."), fmt_to_string (&f));
	  goto fail;
	}

      if (!match_tok (')'))
	{
	  msg (SE, _("`)' expected after output format."));
	  goto fail;
	}

      switch (f.type)
	{
	case FMT_A:
	  width = f.w;
	  break;
	case FMT_AHEX:
	  width = f.w / 2;
	  break;
	default:
	  assert (0);
	}

      /* Create each variable. */
      for (i = 0; i < nv; i++)
	{
	  variable *new_var = create_variable (&default_dict, v[i],
					       ALPHA, width);
	  if (!new_var)
	    msg (SE, _("There is already a variable named %s."), v[i]);
	  else
	    {
	      new_var->print = new_var->write = f;
	      envector (new_var);
	    }
	}

      /* Clean up. */
      for (i = 0; i < nv; i++)
	free (v[i]);
      free (v);
    }
  while (match_tok ('/'));

  if (token != '.')
    return syntax_error (_("expecting end of command"));

  return 1;

  /* If we have an error at a point where cleanup is required,
     flow-of-control comes here. */
fail:
  for (i = 0; i < nv; i++)
    free (v[i]);
  free (v);
  return -1;
}

/* Parses the LEAVE command. */
int
cmd_leave (void)
{
  variable **v;
  int nv;

  int i;

  match_id (LEAVE);
  if (!parse_variables (NULL, &v, &nv, PV_NONE))
    return 0;
  for (i = 0; i < nv; i++)
    {
      if (v[i]->left)
	continue;
      devector (v[i]);
      v[i]->left = 1;
      envector (v[i]);
    }
  free (v);
  if (token != '.')
    return syntax_error (_("expecting end of command"));
  return 1;
}
