/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#if !expr_h
#define expr_h 1

/* All the expression operators. */
enum
  {
    OP_ERROR,

    /* Basic operators. */
    OP_PLUS,			/* + */
    OP_MUL,			/* * */
    OP_POW,			/* ** */
    OP_AND,
    OP_OR,
    OP_NOT,
    OP_EQ,
    OP_GE,
    OP_GT,
    OP_LE,
    OP_LT,
    OP_NE,

    /* String operators. */
    OP_STRING_EQ,
    OP_STRING_GE,
    OP_STRING_GT,
    OP_STRING_LE,
    OP_STRING_LT,
    OP_STRING_NE,

    /* Unary functions. */
    OP_NEG,			/* unary minus */
    OP_ABS,
    OP_ARCOS,
    OP_ARSIN,
    OP_ARTAN,
    OP_COS,
    OP_EXP,
    OP_LG10,
    OP_LN,
    OP_MOD10,
    OP_RND,
    OP_SIN,
    OP_SQRT,
    OP_TAN,
    OP_TRUNC,

    /* N-ary numeric functions. */
    OP_ANY,
    OP_ANY_STRING,		/* string version */
    OP_CFVAR,
    OP_MAX,
    OP_MEAN,
    OP_MIN,
    OP_NMISS,
    OP_NVALID,
    OP_RANGE,
    OP_RANGE_STRING,		/* string version */
    OP_SD,
    OP_SUM,
    OP_VARIANCE,

    /* Time construction & extraction functions. */
    OP_TIME_HMS,

    OP_CTIME_DAYS,		/* These 5 never appear in a tree or an expression. */
    OP_CTIME_HOURS,		/* They disappear in expr-prs:unary_func(). */
    OP_CTIME_MINUTES,
    OP_CTIME_SECONDS,
    OP_TIME_DAYS,

    /* Date construction functions. */
    OP_DATE_DMY,
    OP_DATE_MDY,
    OP_DATE_MOYR,
    OP_DATE_QYR,
    OP_DATE_WKYR,
    OP_DATE_YRDAY,
    OP_YRMODA,

    /* Date extraction functions. */
    OP_XDATE_DATE,
    OP_XDATE_HOUR,
    OP_XDATE_JDAY,
    OP_XDATE_MDAY,
    OP_XDATE_MINUTE,
    OP_XDATE_MONTH,
    OP_XDATE_QUARTER,
    OP_XDATE_SECOND,
    OP_XDATE_TDAY,
    OP_XDATE_TIME,
    OP_XDATE_WEEK,
    OP_XDATE_WKDAY,
    OP_XDATE_YEAR,

    /* String functions. */
    OP_CONCAT,
    OP_INDEX,
    OP_INDEX_OPT,
    OP_RINDEX,
    OP_RINDEX_OPT,
    OP_LENGTH,
    OP_LOWER,
    OP_UPPER,
    OP_LPAD,
    OP_LPAD_OPT,
    OP_RPAD,
    OP_RPAD_OPT,
    OP_LTRIM,
    OP_LTRIM_OPT,
    OP_RTRIM,
    OP_RTRIM_OPT,
    OP_NUMBER,
    OP_NUMBER_OPT,
    OP_STRING,
    OP_SUBSTR,
    OP_SUBSTR_OPT,

    /* Artificial. */
    OP_INV,			/* reciprocal */
    OP_SQUARE,			/* squares the argument */
    OP_NUM_TO_BOOL,		/* converts ~0=>0, ~1=>1, sysmis=>sysmis,
				   others=>0 but with a warning */

    /* Weirdness. */
    OP_MOD,			/* modula function */
    OP_NORMAL,			/* generates normally distributed random number */
    OP_UNIFORM,			/* generates uniformly distributed random number */
    OP_SYSMIS,			/* tests whether the argument has SYSMIS value */
    OP_VEC_ELEM_NUM,		/* element of a numeric vector */
    OP_VEC_ELEM_STR,		/* element of a string vector */

    /* Terminals. */
    OP_TERMINAL,		/* this is not a valid type; it marks the boundary
				   between terminals and nonterminals */
    OP_NUM_CON,			/* numeric constant */
    OP_STR_CON,			/* string literal */
    OP_NUM_VAR,			/* numeric variable reference */
    OP_STR_VAR,			/* string variable reference */
    OP_NUM_LAG,			/* numeric variable from an earlier case */
    OP_STR_LAG,			/* string variable from an earlier case */
    OP_NUM_SYS,			/* SYSMIS(numvar) */
    OP_NUM_VAL,			/* VALUE(numvar) */
    OP_STR_MIS,			/* MISSING(strvar) */
    OP_CASENUM,			/* $CASENUM */
    OP_SENTINEL			/* sentinel */
  };

#define OP_VAR_ARGS     00001	/* 1=variable number of args */
#define OP_MIN_ARGS	00002	/* 1=can specific min args with .X */
#define OP_FMT_SPEC	00004	/* 1=includes a format specifier */
#define OP_ABSORB_MISS	00010	/* 1=may return other than SYSMIS if
				   given a SYSMIS argument */

typedef struct
  {
#if GLOBAL_DEBUGGING
    const char *name;		/* operator name */
#endif
    unsigned flags:4;		/* flags */
    signed height:4;		/* effect on stack height after this operator is
				   executed */
    unsigned skip:4;		/* number of operator items to skip after this
				   operator is seen */
  }
op_desc;

extern op_desc ops[];

typedef short int operator;

/* Tree structured expressions. */ 

/* Numeric constant. */
typedef struct
  {
    int type;			/* always OP_NUM_CON */
    double value;		/* numeric value */
  }
num_con_node;

/* String literal. */
typedef struct
  {
    int type;			/* always OP_STR_CON */
    int len;			/* length of string */
    char s[1];			/* string value */
  }
str_con_node;

/* Variable or test for missing values or cancellation of
   user-missing. */
typedef struct
  {
    int type;			/* OP_NUM_VAR, OP_NUM_SYS, OP_NUM_VAL,
				   OP_STR_MIS, or OP_STR_VAR */
    struct variable *v;		/* variable */
  }
var_node, sys_node, val_node;

/* Variable from an earlier case. */
typedef struct
  {
    int type;			/* always OP_NUM_LAG */
    struct variable *v;		/* relevant variable */
    int lag;			/* number of cases to lag */
  }
lag_node;

/* $CASENUM. */
typedef struct
  {
    int type;			/* always OP_CASENUM */
  }
casenum_node;

/* Any nonterminal node. */
typedef union any_node_union any_node;
typedef struct
  {
    int type;			/* always > OP_TERMINAL */
    int n;			/* number of arguments */
    any_node *arg[1];		/* arguments */
  }
nonterm_node;

/* Any node. */
union any_node_union
  {
    int type;
    nonterm_node nnt;
    num_con_node nmc;
    str_con_node stc;
    var_node var;
    lag_node lag;
    sys_node sys;
    val_node val;
    casenum_node cas;
  };

/* Public info. */

typedef enum
  {
    EX_ERROR,
    EX_BOOLEAN,
    EX_NUMERIC,
    EX_STRING
  }
exprtype;

/* An expression. */
typedef struct expression
  {
    exprtype type;		/* type of expression result */
    operator *op;		/* operators */
    struct variable **vars;	/* variables */
    double *dbl;		/* numeric operands */
    unsigned char *str;		/* string operands */
    union value *stack;		/* evaluation stack */
    unsigned char *str_stk;	/* string evaluation stack */
#if !PAGED_STACK
    size_t str_size;		/* size of string eval stack */
#endif
  }
expression;

/* (P)arse e(XP)ression flags */
#define PXP_NONE 000		/* no flags */
#define PXP_DUMP 001		/* dump postfix representation to screen;
				   only for use by EVALUATE */
#define PXP_BOOLEAN 002		/* coerce return value to Boolean */
#define PXP_NUMERIC 004		/* must be numeric result type */

expression *parse_expression (int flags);
nonterm_node *optimize_expression (nonterm_node *);
void dump_expression (any_node *, expression *);
void free_node (any_node *);
void free_expression (expression *);

struct ccase;
double evaluate_expression (expression *, struct ccase *, union value *);

double yrmoda (double year, double month, double day);

#endif /* !expr_h */
