/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <assert.h>
#include <math.h>
#include <ctype.h>
#include <stdarg.h>
#include <stdlib.h>
#include "error.h"
#include "expr.h"
#include "getline.h"
#include "str.h"
#include "common.h"
#include "misc.h"
#include "settings.h"
#include "lexer.h"
#include "var.h"

#undef DEBUGGING
/*#define DEBUGGING 1 */
#include "debug-print.h"

/* Globals. */

/* Maximum length of a field. */
#define MAX_FIELD 255

/* Buffer to hold a field while it is being parsed. */
static unsigned char buf[MAX_FIELD + 1];

/* Columns the field was taken from. */
static int f1, f2;

/* Format spec. */
static fmt_spec f;

/* Specialized error routine. */

static int dls_error (const char *format,...)
     __attribute__ ((format (printf, 1, 2)));

/* FIXME: Figure out clever way to include the field type in the error
   message. */
static int
dls_error (const char *format,...)
{
  va_list args;

  if (fmt_parse_ignore_error)
    return 0;

  cust_fc = f1;
  cust_lc = f2;
  cust_field = f;

  va_start (args, format);
  if (am_reading_script)
    vmsg (DE, format, args, _("data-file error: "));
  else
    vmsg (DE, format, args, NULL);
  va_end (args);

  return 0;
}

/* Format parsers. */ 

char overflow[] = N_("Overflow in floating-point constant: %s.");
char trailing[] = N_("Trailing characters in field: \"%s\".");
char hex[] = N_("This field may contain only hex digits: \"%s\".");
char even[] = N_("Field must have even length: \"%s\".");

static int parse_numeric (value * v);	/* E, F, COMMA, DOLLAR formats. */

static inline int
hexit_value (int c)
{
  if (isdigit (c))
    return (c - '0');
  else if (islower (c))
    return (c - 'a' + 10);
  else
    return (c - 'A' + 10);
}

static inline int
parse_N (value * v)
{
  unsigned long n;
  char *cp;

  for (cp = buf, n = 0; *cp; cp++)
    {
      if (!isdigit (*cp))
	return dls_error (_("All characters in field must be digits.  Field "
			  "actually contained: \"%s\"."), buf);
      n = (n * 10) + (*cp - '0');
    }

  v->f = (double) n;
  if (f.d)
    v->f /= pow (10.0, f.d);
  return 1;
}

static inline int
parse_PIBHEX (value * v)
{
  double n;
  char *cp = (char *) buf;

  while (isspace ((unsigned char) *cp))
    cp++;
  for (n = 0.0; *cp; cp++)
    if (isxdigit ((unsigned char) *cp))
      {
	if (n > DBL_MAX / 16)
	  return dls_error (gettext (overflow), buf);
	n = n * 16.0 + hexit_value ((unsigned char) *cp);
      }
    else
      break;
  while (isspace ((unsigned char) *cp))
    cp++;
  if (*cp)
    return dls_error (gettext (trailing), buf);

  v->f = n;
  return 1;
}

static inline int
parse_RBHEX (value * v)
{
  union
  {
    double d;
    unsigned char c[sizeof (double)];
  }
  u;
  int i;

  if (f.w % 2)
    return dls_error (gettext (even), buf);

  memset (u.c, 0, sizeof (u.c));
  for (i = 0; i < f.w / 2; i++)
    {
      if (!isxdigit ((unsigned char) buf[i * 2])
	  || !isxdigit ((unsigned char) buf[i * 2 + 1]))
	return dls_error (gettext (hex), buf);
      if (i < (int) sizeof (double))
	  u.c[i] = (16 * hexit_value ((unsigned char) buf[i * 2])
		    + hexit_value ((unsigned char) buf[i * 2] + 1));
    }
  v->f = u.d;
  return 1;
}

static inline int
parse_Z (value * v)
{
  static int warned = 0;
  unsigned char *tail;
  double n;
  int i, sign;

  if (!warned)
    {
      msg (MW, _("Quality of zoned decimal (Z) input format code is suspect.  "
	   "Check your results three times, report bugs to author."));
      warned = 1;
    }

  for (i = f.w - 2; i >= 0; i--)
    if (isspace ((unsigned char) buf[i]) || buf[i] == '.')
      continue;
    else if ((buf[i] & 0xf0) == 0xf0)
      {
	buf[i] = (buf[i] & 0xf) + '0';
	if (!isdigit ((unsigned char) buf[i]))
	  return dls_error (_("Bad digit in zoned decimal number."));
      }
    else
      return dls_error (_("Bad upper nibble in zoned decimal number."));

  switch (buf[f.w - 1] & 0xf0)
    {
    case 0xf0:
    case 0xc0:
      sign = 0;
      break;
    case 0xe0:
    case 0xd0:
      sign = 1;
      break;
    default:
      return dls_error (_("Bad sign nibble in final digit of "
			"zoned decimal number."));
    }
  buf[f.w - 1] = (buf[f.w - 1] & 0xf) + '0';
  if (!isdigit ((unsigned char) buf[f.w - 1]))
    return dls_error (_("Bad final digit in zoned decimal number."));

  n = strtod ((char *) buf, (char **) &tail);
  if (tail != &buf[f.w])
    return dls_error (_("Bad formatting in translated zoned decimal number: "
		      "`%s' is not a valid number."), buf);
  v->f = sign ? -n : n;

  return 1;
}

static inline int
parse_IB (value * v)
{
  double n = 0.0;
  int i = 0;

  /* This routine ignores overflow. */
  /* If the value is negative, we need to NOT each value before adding
     it on, then add 1 and take the negative, because of the
     properties of two's-complement notation. */
  if (endian == LITTLE)
    memrev (buf, f.w);
  if (buf[0] & 0x80)
    {
      for (; i < f.w; i++)
	n = (n * 256.0) + ~buf[i];
      n = -(n + 1);
    }
  else
    {
      for (; i < f.w; i++)
	n = (n * 256.0) + buf[i];
    }

  if (f.d)
    n /= pow (10.0, f.d);
  v->f = n;
  return 1;
}

static inline int
parse_PIB (value * v)
{
  double n;
  int i;

  /* This routine ignores overflow. */
  if (endian == LITTLE)
    memrev (buf, f.w);
  for (i = 0, n = 0.0; i < f.w; i++)
    n = (n * 256.0) + ~buf[i];

  if (f.d)
    n /= pow (10.0, f.d);
  v->f = n;
  return 1;
}

static inline int
parse_P (value * v)
{
  double n;
  int i;

  /* This routine ignores overflow. */
  for (i = 0, n = 0.0; i < f.w - 1; i++)
    n = (n * 100.0) + ((buf[i] >> 4) & 0xf) * 10.0 + (double) (buf[i] & 0xf);
  n = (n * 10.0) + (double) ((buf[i] >> 4) & 0xf);
  if ((buf[f.w - 1] & 0xf) == 0xd)
    n = -n;

  if (f.d)
    n /= pow (10.0, f.d);
  v->f = n;
  return 1;
}

static inline int
parse_PK (value * v)
{
  double n;
  int i;

  /* This routine ignores overflow. */
  for (i = 0, n = 0.0; i < f.w; i++)
    n = (n * 100.0) + ((buf[i] >> 4) & 0xf) * 10.0 + (double) (buf[i] & 0xf);

  if (f.d)
    n /= pow (10.0, f.d);
  v->f = n;
  return 1;
}

static inline int
parse_RB (value * v)
{
  union
  {
    double d;
    unsigned char c[sizeof (double)];
  }
  u;

  memset (u.c, 0, sizeof (u.c));
  memcpy (u.c, buf, min ((int) sizeof (u.c), f.w));
  v->f = u.d;
  return 1;
}

static inline int
parse_A (value * v)
{
  memcpy (v->s, buf, f.w);
  return 1;
}

static inline int
parse_AHEX (value * v)
{
  int i;

  if (f.w % 2)
    return dls_error (gettext (even), buf);

  for (i = 0; i < f.w / 2; i++)
    {
      if (!isxdigit ((unsigned char) buf[i * 2])
	  || !isxdigit ((unsigned char) buf[i * 2 + 1]))
	return dls_error (gettext (hex), buf);
      v->s[i] = hexit_value (buf[i * 2]) * 16 + hexit_value (buf[i * 2 + 1]);
    }
  return 1;
}

/* Date & time format components. */

#define date_error(MSG) 			\
	do					\
	  {					\
	    dls_error MSG;			\
	    *cp = NULL;				\
	    return;				\
	  } 					\
	while(0)

static inline void
parse_leader (char **cp)
{
  if (!*cp)
    return;
  while (isspace ((unsigned char) **cp))
    (*cp)++;
}

static inline void
parse_day (char **cp, int *day)
{
  if (!*cp)
    return;
  *day = strtol (*cp, cp, 10);
  if (*day < 1 || *day > 31)
    date_error ((_("Day (%d) must be between 1 and 31."), *day));
}

static inline void
parse_day_count (char **cp, int *day_count)
{
  char *tail;

  if (!*cp)
    return;
  *day_count = strtol (*cp, &tail, 10);
  if (tail == *cp)
    date_error ((_("Day count has bad format.")));
  *cp = tail;
}

static inline void
parse_date_delimiter (char **cp)
{
  int delim = 0;

  if (!*cp)
    return;
  while (**cp == '-' || **cp == '/' || isspace ((unsigned char) **cp)
	 || **cp == '.' || **cp == ',')
    delim = 1, (*cp)++;
  if (!delim)
    date_error ((_("Delimiter expected between fields in date.")));
}

/* Returns a pointer to a static character buffer containing roman
   numerals equivalent to X, or arabic numerals if the roman expansion
   would be too long. */
static char *
to_roman (int x)
{
  int sx = x;

  typedef struct
    {
      int value;		/* Value corresponding to this digit. */
      char name;		/* Digit name. */
    }
  roman_digit;

  static roman_digit roman_tab[7] =
  {
    {1000, 'M'},
    {500, 'D'},
    {100, 'C'},
    {50, 'L'},
    {10, 'X'},
    {5, 'V'},
    {1, 'I'},
  };

  static char roman[32];
  char *cp = roman;

  int i, j;

  assert (32 >= INT_DIGITS + 1);
  if (x == 0)
    goto arabic;

  if (x < 0)
    *cp++ = '-', x = -x;

  for (i = 0; i < 7; i++)
    {
      int digit = roman_tab[i].value;
      while (x >= digit)
	{
	  x -= digit;
	  if (cp > &roman[30])
	    goto arabic;
	  *cp++ = roman_tab[i].name;
	}

      for (j = i + 1; j < 7; j++)
	{
	  if (i == 4 && j == 5)	/* VX is not a shortened form of V. */
	    break;

	  digit = roman_tab[i].value - roman_tab[j].value;
	  while (x >= digit)
	    {
	      x -= digit;
	      if (cp > &roman[29])
		goto arabic;
	      *cp++ = roman_tab[j].name;
	      *cp++ = roman_tab[i].name;
	    }
	}
    }
  *cp = 0;
  return roman;

arabic:
  sprintf (roman, "%d", sx);
  return roman;
}

/* Returns true if C is a (lowercase) roman numeral. */
#define isrnum(C) 					\
	((C) == 'x' || (C) == 'v' || (C) == 'i')

/* Returns the value of a single roman numeral. */
#define rnumval(C) 				\
	((C)=='x' ? 10 : ((C)=='v' ? 5 : 1))

static inline void
parse_month (char **cp, int *month)
{
  static const char *months[12] =
  {
    "january", "february", "march", "april", "may", "june",
    "july", "august", "september", "october", "november", "december",
  };

  int c;
  char month_buf[32], *mp;

  if (!*cp)
    return;
  if (isdigit ((unsigned char) **cp))
    {
      *month = strtol (*cp, cp, 10);
      if (*month < 1 || *month > 12)
	date_error ((_("Month (%d) must be between 1 and 12."), *month));
      return;
    }

  c = tolower ((unsigned char) (**cp));
  if (isrnum (c))
    {
      int last = rnumval (c);

      *month = 0;
      for (;;)
	{
	  int value;

	  (*cp)++;
	  c = tolower ((unsigned char) (**cp));
	  if (!isrnum (c))
	    {
	      if (last != INT_MAX)
		*month += last;
	      break;
	    }

	  value = rnumval (c);
	  if (last == INT_MAX)
	    *month += value;
	  else if (value > last)
	    *month += value - last, last = INT_MAX;
	  else
	    *month += last, last = value;
	}
      if (*month < 1 || *month > 12)
	date_error ((_("Month (%s) must be between I and XII."), to_roman (*month)));
      return;
    }

  for (mp = month_buf;
       isalpha ((unsigned char) **cp) && mp < &month_buf[31]; (*cp)++)
    *mp++ = tolower ((unsigned char) (**cp));
  *mp = 0;
  if (isalpha ((unsigned char) **cp))
    date_error ((_("Month name (%s...) is too long."), month_buf));

  for (c = 0; c < 12; c++)
    if (id_match (months[c], month_buf))
      {
	*month = c + 1;
	return;
      }
  date_error ((_("Bad month name (%s)."), month_buf));
}

static inline void
parse_year (char **cp, int *year)
{
  char *tail;

  if (!*cp)
    return;
  *year = strtol (*cp, &tail, 10);
  if (tail == *cp)
    date_error ((_("Year has bad format.")));
  if (*year >= 0 && *year <= 199)
    *year += 1900;
  if (*year < 1582 || *year > 19999)
    date_error ((_("Year (%d) must be between 1582 and 19999."), *year));
  *cp = tail;
}

static inline void
parse_trailer (char **cp)
{
  if (!*cp)
    return;
  while (isspace ((unsigned char) **cp))
    (*cp)++;
  if (**cp)
    date_error ((_("Trailing garbage \"%s\" following date."), *cp));
}

static inline void
parse_julian (char **cp, long *julian)
{
  int year, day;

  if (!*cp)
    return;
  *julian = strtol (*cp, cp, 10);

  day = *julian % 1000;
  if (day < 1 || day > 366)
    date_error ((_("Julian day (%d) must be between 1 and 366."), day));

  year = *julian / 1000;
  if (year >= 0 && year <= 199)
    *julian += 1900000L;
  else if (year < 1582 || year > 19999)
    date_error ((_("Year (%d) must be between 1582 and 19999."), year));
}

static inline void
parse_quarter (char **cp, int *quarter)
{
  if (!*cp)
    return;
  *quarter = strtol (*cp, cp, 10);
  if (*quarter < 1 || *quarter > 4)
    date_error ((_("Quarter (%d) must be between 1 and 4."), *quarter));
}

static inline void
parse_q_delimiter (char **cp)
{
  if (!*cp)
    return;
  while (isspace ((unsigned char) **cp))
    (*cp)++;
  if (tolower ((unsigned char ) (*((*cp)++))) != 'q')
    date_error ((_("`Q' expected between quarter and year.")));
  while (isspace ((unsigned char) **cp))
    (*cp)++;
}

static inline void
parse_week (char **cp, int *week)
{
  if (!*cp)
    return;
  *week = strtol (*cp, cp, 10);
  if (*week < 1 || *week > 53)
    date_error ((_("Week (%d) must be between 1 and 53."), *week));
}

static inline void
parse_wk_delimiter (char **cp)
{
  if (!*cp)
    return;
  while (isspace ((unsigned char) **cp))
    (*cp)++;
  if (tolower ((unsigned char) ((*cp)[0])) != 'w'
      || tolower ((unsigned char) ((*cp)[1])) != 'k')
    date_error ((_("`WK' expected between week and year.")));
  (*cp) += 2;
  while (isspace ((unsigned char) **cp))
    (*cp)++;
}

static inline void
parse_time_delimiter (char **cp)
{
  int delim = 0;

  if (!*cp)
    return;
  while (**cp == ':' || **cp == '.' || isspace ((unsigned char) **cp))
    delim = 1, (*cp)++;
  if (!delim)
    date_error ((_("Delimiter expected between fields in time.")));
}

static inline void
parse_hour (char **cp, int *hour)
{
  char *tail;

  if (!*cp)
    return;
  *hour = strtol (*cp, &tail, 10);
  if (*hour < 0)
    date_error ((_("Hour (%d) must be positive."), *hour));
  *cp = tail;
}

static inline void
parse_minute (char **cp, int *minute)
{
  char *tail;

  if (!*cp)
    return;
  *minute = strtol (*cp, &tail, 10);
  if (*minute < 0 || *minute > 59)
    date_error ((_("Minute (%d) must be between 0 and 59."), *minute));
  *cp = tail;
}

static inline void
parse_opt_second (char **cp, double *second)
{
  int delim = 0;
  char *tail;

  if (!*cp)
    return;
  while (**cp == ':' || **cp == '.' || isspace ((unsigned char) **cp))
    delim = 1, (*cp)++;
  if (!delim || !isdigit ((unsigned char) **cp))
    {
      *second = 0.0;
      return;
    }

  *second = strtod (*cp, &tail);
  if (tail == *cp)
    date_error ((_("Bad format for seconds field of time value.")));
  if (*second < 0)
    date_error ((_("Seconds (%f) must be nonnegative."), *second));
  *cp = tail;
}

static inline void
parse_hour24 (char **cp, int *hour24)
{
  char *tail;

  if (!*cp)
    return;
  *hour24 = strtol (*cp, &tail, 10);
  if (*hour24 < 0 || *hour24 > 23)
    date_error ((_("Hour (%d) must be between 0 and 23."), *hour24));
  *cp = tail;
}

#define tuple(A,B) 				\
	(((A) << 8) + (B))
     
static inline void
parse_weekday (char **cp, int *weekday)
{
  if (**cp == 0)
    date_error ((_("Day of the week expected in date value.")));
  switch (tuple (tolower ((unsigned char) ((*cp)[0])),
		 tolower ((unsigned char) ((*cp)[1]))))
    {
    case tuple ('s', 'u'):
      *weekday = 1;
      break;
    case tuple ('m', 'o'):
      *weekday = 2;
      break;
    case tuple ('t', 'u'):
      *weekday = 3;
      break;
    case tuple ('w', 'e'):
      *weekday = 4;
      break;
    case tuple ('t', 'h'):
      *weekday = 5;
      break;
    case tuple ('f', 'r'):
      *weekday = 6;
      break;
    case tuple ('s', 'a'):
      *weekday = 7;
      break;
    default:
      date_error ((_("Day of the week expected in date value.")));
    }
  while (isalpha ((unsigned char) **cp))
    (*cp)++;
}
#undef tuple

static inline void
parse_spaces (char **cp)
{
  if (!*cp)
    return;
  while (isspace ((unsigned char) **cp))
    (*cp)++;
}

static inline void
parse_sign (char **cp, int *sign)
{
  if (!*cp)
    return;
  switch (**cp)
    {
    case '+':
      *sign = 0, (*cp)++;
      break;
    case '-':
      *sign = 1, (*cp)++;
      break;
    default:
      *sign = 0;
      break;
    }
}

#undef date_error

/* Date & time formats. */

char inv_date_msg[] = N_("Date is not in valid range between 15 Oct 1582 and "
			 "31 Dec 19999.");

#define prune_invalidate_dates				\
	do						\
	  {						\
	    if (v->f == SYSMIS)				\
	      {						\
		dls_error (gettext (inv_date_msg));	\
		return 0;				\
	      }						\
	  }						\
	while (0)

static inline int
parse_DATE (value * v)
{
  char *cp = buf;
  int day, month, year;

  parse_leader (&cp);
  parse_day (&cp, &day);
  parse_date_delimiter (&cp);
  parse_month (&cp, &month);
  parse_date_delimiter (&cp);
  parse_year (&cp, &year);
  parse_trailer (&cp);

  if (cp == NULL)
    return 0;

  v->f = yrmoda (year, month, day);
  prune_invalidate_dates;
  v->f *= 60. * 60. * 24.;

  return 1;
}

static inline int
parse_ADATE (value * v)
{
  char *cp = buf;
  int month, day, year;

  parse_leader (&cp);
  parse_month (&cp, &month);
  parse_date_delimiter (&cp);
  parse_day (&cp, &day);
  parse_date_delimiter (&cp);
  parse_year (&cp, &year);
  parse_trailer (&cp);

  if (cp == NULL)
    return 0;

  v->f = yrmoda (year, month, day);
  prune_invalidate_dates;
  v->f *= 60. * 60. * 24.;

  return 1;
}

static inline int
parse_EDATE (value * v)
{
  char *cp = buf;
  int month, day, year;

  parse_leader (&cp);
  parse_day (&cp, &day);
  parse_date_delimiter (&cp);
  parse_month (&cp, &month);
  parse_date_delimiter (&cp);
  parse_year (&cp, &year);
  parse_trailer (&cp);

  if (cp == NULL)
    return 0;

  v->f = yrmoda (year, month, day);
  prune_invalidate_dates;
  v->f *= 60. * 60. * 24.;

  return 1;
}

static inline int
parse_SDATE (value * v)
{
  char *cp = buf;
  int month, day, year;

  parse_leader (&cp);
  parse_year (&cp, &year);
  parse_date_delimiter (&cp);
  parse_month (&cp, &month);
  parse_date_delimiter (&cp);
  parse_day (&cp, &day);
  parse_trailer (&cp);

  if (cp == NULL)
    return 0;

  v->f = yrmoda (year, month, day);
  prune_invalidate_dates;
  v->f *= 60. * 60. * 24.;

  return 1;
}

static inline int
parse_JDATE (value * v)
{
  char *cp = buf;
  long julian;

  parse_leader (&cp);
  parse_julian (&cp, &julian);
  parse_trailer (&cp);

  if (cp == NULL)
    return 0;

  if (julian / 1000 == 1582)
    v->f = yrmoda (1583, 1, 1) - 365;
  else
    v->f = yrmoda (julian / 1000, 1, 1);
  prune_invalidate_dates;
  v->f = (v->f + julian % 1000 - 1) * 60. * 60. * 24.;
  if (v->f < 0)
    {
      v->f = SYSMIS;
      dls_error (gettext (inv_date_msg));
      return 0;
    }

  return 1;
}

static inline int
parse_QYR (value * v)
{
  char *cp = buf;
  int quarter, year;

  parse_leader (&cp);
  parse_quarter (&cp, &quarter);
  parse_q_delimiter (&cp);
  parse_year (&cp, &year);
  parse_trailer (&cp);

  if (cp == NULL)
    return 0;

  v->f = yrmoda (year, (quarter - 1) * 3 + 1, 1);
  prune_invalidate_dates;
  v->f *= 60. * 60. * 24.;

  return 1;
}

static inline int
parse_MOYR (value * v)
{
  char *cp = buf;
  int month, year;

  parse_leader (&cp);
  parse_month (&cp, &month);
  parse_date_delimiter (&cp);
  parse_year (&cp, &year);
  parse_trailer (&cp);

  if (cp == NULL)
    return 0;

  v->f = yrmoda (year, month, 1);
  prune_invalidate_dates;
  v->f *= 60. * 60. * 24.;

  return 1;
}

static inline int
parse_WKYR (value * v)
{
  char *cp = buf;
  int week, year;

  parse_leader (&cp);
  parse_week (&cp, &week);
  parse_wk_delimiter (&cp);
  parse_year (&cp, &year);
  parse_trailer (&cp);

  if (cp == NULL)
    return 0;

  v->f = yrmoda (year, 1, 1);
  prune_invalidate_dates;
  v->f = (v->f + (week - 1) * 7) * 60. * 60. * 24.;

  return 1;
}

static inline int
parse_TIME (value * v)
{
  char *cp = buf;
  int sign;
  double second;
  int hour, minute;

  parse_leader (&cp);
  parse_sign (&cp, &sign);
  parse_spaces (&cp);
  parse_hour (&cp, &hour);
  parse_time_delimiter (&cp);
  parse_minute (&cp, &minute);
  parse_opt_second (&cp, &second);

  if (cp == NULL)
    return 0;

  v->f = hour * 60. * 60. + minute * 60. + second;
  if (sign)
    v->f = -v->f;
  return 1;
}

static inline int
parse_DTIME (value * v)
{
  char *cp = buf;
  int sign;
  int day_count, hour;
  double second;
  int minute;

  parse_leader (&cp);
  parse_sign (&cp, &sign);
  parse_spaces (&cp);
  parse_day_count (&cp, &day_count);
  parse_time_delimiter (&cp);
  parse_hour (&cp, &hour);
  parse_time_delimiter (&cp);
  parse_minute (&cp, &minute);
  parse_opt_second (&cp, &second);

  if (cp == NULL)
    return 0;

  v->f = (day_count * 60. * 60. * 24.
	  + hour * 60. * 60.
	  + minute * 60.
	  + second);
  if (sign)
    v->f = -v->f;
  return 1;
}

static inline int
parse_DATETIME (value * v)
{
  char *cp = buf;
  int day, month, year;
  int hour24;
  double second;
  int minute;

  parse_leader (&cp);
  parse_day (&cp, &day);
  parse_date_delimiter (&cp);
  parse_month (&cp, &month);
  parse_date_delimiter (&cp);
  parse_year (&cp, &year);
  parse_time_delimiter (&cp);
  parse_hour24 (&cp, &hour24);
  parse_time_delimiter (&cp);
  parse_minute (&cp, &minute);
  parse_opt_second (&cp, &second);

  if (cp == NULL)
    return 0;

  v->f = yrmoda (year, month, day);
  prune_invalidate_dates;
  v->f = v->f * 60. * 60. * 24. + hour24 * 60. * 60. + minute * 60. + second;

  return 1;
}

static inline int
parse_WKDAY (value * v)
{
  char *cp = buf;
  int weekday;

  parse_leader (&cp);
  parse_weekday (&cp, &weekday);
  parse_trailer (&cp);

  if (cp == NULL)
    return 0;

  v->f = weekday;
  return 1;
}

static inline int
parse_MONTH (value * v)
{
  char *cp = buf;
  int month;

  parse_leader (&cp);
  parse_month (&cp, &month);
  parse_trailer (&cp);

  if (cp == NULL)
    return 0;

  v->f = month;
  return 1;
}

/* Parses a string S according to format *FP into value *V.  String S
   has length LEN and is not necessarily null-terminated, in fact it
   may contain nulls.  FC is the 1-based column number of the
   beginning of the string S, used only for error messages.  Returns 1
   on success, 0 on failure. */
int
parse_string_as_format (const char *s, int len, const fmt_spec *fp,
			int fc, value *v)
{
  fmt_desc *fmt;

  f = *fp;

  /* Default to SYSMIS or blanks. */
  fmt = &formats[f.type];
  if (fmt->cat & FCAT_STRING)
    {
#if __CHECKER__
      memset (v->s, ' ', ROUND_UP (fp->w, MAX_SHORT_STRING));
#else
      memset (v->s, ' ', fp->w);
#endif
    }
  else
    v->f = set_blanks;

  /* Check that we've got a string to work with. */
  if (len <= 0 || f.w <= 0)
    return 1;

  /* Make sure that the string isn't too long. */
  if (f.w > fmt->Imax_w)
    {
      dls_error (_("Field too long (%d characters).  Truncated after "
		   "character %d."),
		 len, fmt->Imax_w);
      f.w = fmt->Imax_w;
    }

  /* Set the column values. */
  f1 = fc;
  f2 = fc + f.w - 1;

  /* Copy the field into buf with exactly f.w length. */
  if (len < f.w)
    {
      memcpy (buf, s, len);
      memset (&buf[len], ' ', f.w - len);
    }
  else				/* len>=f.w */
    memcpy (buf, s, f.w);
  buf[f.w] = 0;

  /* This is not exactly a proper assertion, as it is possible that an
     input file could contain a null byte.  But for debugging purposes
     it is useful. */
#if DEBUGGING
  assert ((int) strlen (buf) == f.w);
#endif

  if ((fmt->cat & FCAT_BLANKS_SYSMIS) && (int) strspn (buf, " \t\v\r") == f.w)
    {
      v->f = set_blanks;
      return 1;
    }

  switch (f.type)
    {
    case FMT_E:
    case FMT_F:
    case FMT_COMMA:
    case FMT_DOT:
    case FMT_DOLLAR:
    case FMT_PCT:
      return parse_numeric (v);
    case FMT_N:
      return parse_N (v);
    case FMT_PIBHEX:
      return parse_PIBHEX (v);
    case FMT_RBHEX:
      return parse_RBHEX (v);
    case FMT_Z:
      return parse_Z (v);
    case FMT_IB:
      return parse_IB (v);
    case FMT_PIB:
      return parse_PIB (v);
    case FMT_P:
      return parse_P (v);
    case FMT_PK:
      return parse_PK (v);
    case FMT_RB:
      return parse_RB (v);
    case FMT_CCA:
    case FMT_CCB:
    case FMT_CCC:
    case FMT_CCD:
    case FMT_CCE:
      assert (0);
    case FMT_A:
      return parse_A (v);
    case FMT_AHEX:
      return parse_AHEX (v);
    case FMT_DATE:
      return parse_DATE (v);
    case FMT_EDATE:
      return parse_EDATE (v);
    case FMT_SDATE:
      return parse_SDATE (v);
    case FMT_ADATE:
      return parse_ADATE (v);
    case FMT_JDATE:
      return parse_JDATE (v);
    case FMT_QYR:
      return parse_QYR (v);
    case FMT_MOYR:
      return parse_MOYR (v);
    case FMT_WKYR:
      return parse_WKYR (v);
    case FMT_TIME:
      return parse_TIME (v);
    case FMT_DTIME:
      return parse_DTIME (v);
    case FMT_DATETIME:
      return parse_DATETIME (v);
    case FMT_WKDAY:
      return parse_WKDAY (v);
    case FMT_MONTH:
      return parse_MONTH (v);
    default:
      assert (0);
    }
#if __GNUC__ || __BORLANDC__
  return 0;
#endif
}

/* Copyright (C) 1991, 1992 Free Software Foundation, Inc.
   This file is part of the GNU C Library.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If
   not, write to the Free Software Foundation, Inc., 675 Mass Ave,
   Cambridge, MA 02139, USA.  */

/* Modified by blp for PSPP. */

#include <errno.h>
#include <float.h>
#include <ctype.h>
#include <string.h>
#include <math.h>
#include "str.h"

/* Converts BUF to a double in V according to type f.type, which may
   be FMT_F, FMT_E, FMT_PCT, FMT_DOLLAR, FMT_DOT, or FMT_COMMA.
   Returns nonzero to indicate success. */
static int
parse_numeric (value *v)
{
  register const char *s;
  short int sign;

  /* The number so far.  */
  double num;

  int got_dot;			/* Found a decimal point.  */
  int got_digit;		/* Count of digits.  */

  /* The exponent of the number.  */
  long int exponent;

  /* f.type except sometimes we change it for convenience. */
  int type = f.type;

  s = buf;

  /* Eat whitespace.  */
  while (isspace ((unsigned char) *s))
    ++s;

  if (type == FMT_DOLLAR && *s == '$')
    {
      ++s;
      type = FMT_COMMA;
    }

  /* Get the sign.  */
  sign = *s == '-' ? -1 : 1;
  if (*s == '-' || *s == '+')
    ++s;

  if (type == FMT_DOT)
    {
      int t = set_decimal;
      set_decimal = set_grouping;
      set_grouping = t;
    }

  v->f = SYSMIS;
  num = 0.0;
  got_dot = 0;
  got_digit = 0;
  exponent = 0;
  for (;; ++s)
    {
      if (isdigit ((unsigned char) *s))
	{
	  got_digit++;

	  /* Make sure that multiplication by 10 will not overflow.  */
	  if (num > DBL_MAX * 0.1)
	    /* The value of the digit doesn't matter, since we have already
	       gotten as many digits as can be represented in a `double'.
	       This doesn't necessarily mean the result will overflow.
	       The exponent may reduce it to within range.

	       We just need to record that there was another
	       digit so that we can multiply by 10 later.  */
	    ++exponent;
	  else
	    num = (num * 10.0) + (*s - '0');

	  /* Keep track of the number of digits after the decimal point.
	     If we just divided by 10 here, we would lose precision.  */
	  if (got_dot)
	    --exponent;
	}
      else if (!got_dot && *s == set_decimal)
	/* Record that we have found the decimal point.  */
	got_dot = 1;
      else if ((type != FMT_COMMA && type != FMT_DOT) || *s != set_grouping)
	/* Any other character terminates the number.  */
	break;
    }

  if (type == FMT_DOT)
    {
      int t = set_decimal;
      set_decimal = set_grouping;
      set_grouping = t;
    }

  if (!got_digit)
    {
      if (got_dot)
	{
	  v->f = SYSMIS;
	  return 1;
	}
      goto noconv;
    }
  
  if (tolower ((unsigned char) (*s)) == 'e'
      || tolower ((unsigned char) (*s)) == 'd'
      || (type == FMT_E && (*s == '+' || *s == '-')))
    {
      /* Get the exponent specified after the `e' or `E'.  */
      int save = errno;
      char *end;
      long int exp;

      errno = 0;
      if (isalpha ((unsigned char) *s))
	++s;
      exp = strtol (s, &end, 10);
      if (errno == ERANGE)
	{
	  /* The exponent overflowed a `long int'.  It is probably a safe
	     assumption that an exponent that cannot be represented by
	     a `long int' exceeds the limits of a `double'.  */
	  if (exp < 0)
	    goto underflow;
	  else
	    goto overflow;
	}
      else if (end == s)
	goto noconv;
      errno = save;
      s = end;
      exponent += exp;
    }
  else if (!got_dot)
    exponent -= f.d;

  while (isspace ((unsigned char) *s))
    s++;
  if (type == FMT_PCT && *s == '%')
    while (isspace ((unsigned char) *++s))
      ;
  if (*s)
    return dls_error (gettext (trailing), buf);

  if (num == 0.0)
    {
      v->f = 0.0;
      return 1;
    }

  /* Multiply NUM by 10 to the EXPONENT power, checking for overflow
     and underflow.  */

  if (exponent < 0)
    {
      if (-exponent + got_digit > -(DBL_MIN_10_EXP) + 5
	  || num < DBL_MIN * pow (10.0, (double) -exponent))
	goto underflow;
      num *= pow (10.0, (double) exponent);
    }
  else if (exponent > 0)
    {
      if (num > DBL_MAX * pow (10.0, (double) -exponent))
	goto overflow;
      num *= pow (10.0, (double) exponent);
    }

  if (sign < 0)
    v->f = -num;
  else
    v->f = num;
  return 1;

overflow:
  /* Return an overflow error.  */
  dls_error (_("Overflow in floating-point constant: \"%s\"."), buf);
  return 0;

underflow:
  /* Return an underflow error.  */
  dls_error (_("Underflow in floating-point constant: \"%s\"."), buf);
  return 0;

noconv:
  /* There was no number.  */
  dls_error (_("Field does not form a valid floating-point constant: \"%s\"."),
	     buf);
  return 0;
}
