/**********************************************************************
** This program is part of the kinetics library and is
**           copyright (C) 1995 Upinder S. Bhalla.
** It is made available under the terms of the
**           GNU Library General Public License. 
** See the file COPYRIGHT for the full notice.
**********************************************************************/
static char rcsid[] = "$Id: pool.c,v 1.1 1997/07/24 17:49:40 dhb Exp $";
 
/* $Log: pool.c,v $
 * Revision 1.1  1997/07/24 17:49:40  dhb
 * Initial revision
 *
 * Revision 1.5  1994/10/26  20:46:09  bhalla
 * Fixed bug with conserve option, where the nTotal got set to twice nInit
 *
 * Revision 1.4  1994/09/29  18:20:21  bhalla
 * -- Split up the SUMTOTAL msg into separate SUMTOTAL and CONSERVE
 * msgs, so that these two cases become more distinct and can coexist.
 * This vastly cleans up the calculation for SUMTOTAL, which usually
 * needs some form of CONSERVE as well.
 *
 * -- Changed the format for slave_enable. It now handles 3 flags:
 * NSLAVE = 0x01, CONCSLAVE = 0x02 and BUFFER = 0x04. The BUFFER
 * option is new, and simplifies the frequent process of using a pool
 * as a buffer by bypassing the whole calculations and just setting
 * n to nInit and Co to CoInit.
 *
 * Revision 1.3  1994/08/15  18:31:55  bhalla
 * Added option to use SLAVE msg to control concen rather than n,
 * using the slave_enable flag set to 2.
 *
 * Revision 1.2  1994/08/05  19:23:48  bhalla
 * conversion to using n rather than conc.
 * Also added new option for using SUMTOTAL msg to handle conservation
 * relationships
 *
 * Revision 1.1  1994/06/13  22:55:39  bhalla
 * Initial revision
 * */

#include "kin_ext.h"

/*
** A pool of a reagent. This version does not carry out any
** reactions, but interfaces to the kin and enz elements which do.
** This way there are no limitations on the number of reactions a
** pool can be involved in
*/

#define POOL_SUMTOTAL	0x01
#define POOL_CONSERVE	0x02
#define POOL_NSLAVE		0x01
#define POOL_CONCSLAVE	0x02
#define POOL_BUFFER		0x04

PoolFunc(pool,action)
register struct pool_type *pool;
Action		*action;
{
MsgIn	*msg;
double	A,B;
double	dt;
int		was_slaved;
double	sumtotal;
double	conserve;

    if(debug > 1){
		ActionHeader("ReactionPool",pool,action);
    }

    SELECT_ACTION(action){
	/* the CoRemaining stuff should actually be calculated
	** in a separate phase, at INIT, but it would be a very
	** expensive undertaking in terms of CPU cycles.
	*/
    case PROCESS:
				/* slave_enable of 3 is the same as buffering,
				** so we bypass everything else */
				if (pool->slave_enable & POOL_BUFFER) {
					pool->n = pool->nInit;
					pool->Co = pool->CoInit;
					break;
				}
				A = B = 0;
				was_slaved = 0;
				sumtotal = 0;
				conserve = 0;
        		MSGLOOP(pool,msg) {
            		case 0:		/* TYPE = REAC */
   		     			A += MSGVALUE(msg,0);
   		     			B += MSGVALUE(msg,1);
        			break;
					case 1:		/* TYPE = MM_PRD. */
   		     			A += MSGVALUE(msg,0);
					break;
					case 2:		/* TYPE = SLAVE */
								/* n = value to follow */
								/* msg 0 = new value of n */
								if (pool->slave_enable) {
									was_slaved = 1;
									/* hack to use concen for SLAVE
									** rather than n */
									if (pool->slave_enable == 
										POOL_CONCSLAVE)
										pool->n =
											pool->vol * MSGVALUE(msg,0);
									else
										pool->n = MSGVALUE(msg,0);
								}
					break;
					case 3:		/* TYPE = REMAINING */
							pool->nRemaining -= MSGVALUE(msg,0);
					break;
					case 4:		/* TYPE = CONSERVE */
							conserve += MSGVALUE(msg,0);
					break;
					case 5:		/* TYPE = VOL */
							pool->vol = MSGVALUE(msg,0);
					break;
					case 6:		/* TYPE = SUMTOTAL */
							sumtotal += MSGVALUE(msg,0);
					break;
        		}
				if (!was_slaved) {
					if (pool->consv_flag & POOL_SUMTOTAL)  {
						pool->nTotal = sumtotal;
						if (pool->consv_flag & ~POOL_CONSERVE)  {
							pool->n = sumtotal;
						}
					}
					if (pool->consv_flag & POOL_CONSERVE)  {
						pool->n = pool->nTotal - conserve;
						if (pool->n < 0)
							pool->n = 0;
					}
					if (!pool->consv_flag) {
						dt = Clockrate(pool);
						if (pool->n > 1.0e-10) {
							B /= pool->n;
							pool->n = IntegrateMethod(
								(B < 1.0e-10) ? FEULER_INT: pool->object->method,
								pool,pool->n,A,B,dt,"n");
						} else {
							pool->n = IntegrateMethod(FEULER_INT,
								pool,pool->n,A,B,dt,"n");
						}
						if (pool->n < pool->nMin) 
							pool->n = pool->nMin;
					}
				}
				/* Convert the n's to Co's */
				pool->nRemaining = pool->nTotal - pool->n;
				if (pool->vol > 0) {
					pool->Co = pool->n / pool->vol;
					pool->CoRemaining = pool->nRemaining / pool->vol;
				}
		break;
    case RESET:
		if (pool->vol > 0) {
			if (pool->keepconc) {
				pool->n = pool->Co * pool->vol;
				pool->nRemaining = pool->CoRemaining* pool->vol;
				pool->nTotal = pool->CoTotal * pool->vol;
				pool->nInit = pool->CoInit * pool->vol;
				pool->nMin = pool->CoMin * pool->vol;
			} else {
				pool->Co = pool->n / pool->vol;
				pool->CoRemaining = pool->nRemaining/pool->vol;
				pool->CoTotal = pool->nTotal / pool->vol;
				pool->CoInit = pool->nInit / pool->vol;
				pool->CoMin = pool->nMin / pool->vol;
			}
		}
		pool->consv_flag = 0;
		sumtotal = 0;
		conserve = pool->nInit;
   		MSGLOOP(pool,msg) {
       		case 6:		/* TYPE = SUMTOTAL. The 2nd slot is for init */
				pool->consv_flag |= POOL_SUMTOTAL;
				sumtotal += MSGVALUE(msg,1);
			break;
			case 4:		/* TYPE = CONSERVE. The 2nd slot is for init */
				pool->consv_flag |= POOL_CONSERVE;
				conserve += MSGVALUE(msg,1);
			break;
		}
		if (pool->consv_flag & POOL_SUMTOTAL) { /*This controls nTotal*/
			pool->nTotal = sumtotal;
		}
		if (pool->consv_flag & POOL_CONSERVE) { /* This controls n. */
			/* add up all the relevant nInits. */
			if (pool->consv_flag & POOL_SUMTOTAL) {
				if (sumtotal < conserve) {
					printf("Warning: Initial values for sumtotal %g are less than for conserve=%g on %s\n",
					sumtotal,conserve, Pathname(pool));
					conserve = sumtotal;
				}
				pool->nInit = sumtotal - conserve;
			} else {
				pool->nTotal = conserve;
			}
		}

		if (pool->nTotal < pool->nInit)
			pool->nTotal = pool->nInit;
		pool->n = pool->nInit;
		/* redo the Co values that might have changed in the RESET */
		if (pool->vol > 0) {
			pool->Co = pool->n / pool->vol;
			pool->CoTotal = pool->nTotal / pool->vol;
		}
    break;
	case CREATE:
		pool->vol = 1; /* need to initialize this */
		pool->nMin = pool->CoMin = 0;
	break;
    case SET :
		if (action->argc == 2) {
			char *field = action->argv[0];
			double newval;
			if (strcmp(field,"vol") == 0 ||
				strcmp(field,"n") == 0 ||
				strcmp(field,"nInit") == 0 ||
				strcmp(field,"nRemaining") == 0 ||
				strcmp(field,"nTotal") == 0 ||
				strcmp(field,"nMin") == 0 ||
				strcmp(field,"Co") == 0 ||
				strcmp(field,"CoInit") == 0 ||
				strcmp(field,"CoRemaining") == 0 ||
				strcmp(field,"CoTotal") == 0 ||
				strcmp(field,"CoMin") == 0) {
					newval = Atof(action->argv[1]);
					if (newval < 0) {
						/* Error(); */
						printf("Cannot set field=%s of pool=%s \n to negative value, using 0.0\n",
						field,Pathname(pool));
						/* return(1); */
						newval = 0.0;
					}
				}
				if (strcmp(field,"vol") == 0 && newval > 0) {
					pool->vol = newval;
					if (pool->keepconc) {
						pool->n = pool->Co * pool->vol;
						pool->nRemaining = pool->CoRemaining* pool->vol;
						pool->nTotal = pool->CoTotal * pool->vol;
						pool->nInit = pool->CoInit * pool->vol;
						pool->nMin = pool->CoMin * pool->vol;
					} else {
						pool->Co = pool->n / pool->vol;
						pool->CoRemaining = pool->nRemaining/pool->vol;
						pool->CoTotal = pool->nTotal / pool->vol;
						pool->CoInit = pool->nInit / pool->vol;
						pool->CoMin = pool->nMin / pool->vol;
					}
					return(1);
				}
				if (pool->vol > 0) {
					if (strcmp(field,"n") == 0)
						pool->Co = newval / pool->vol;
					if (strcmp(action->argv[0],"nInit") == 0)
						pool->CoInit = newval / pool->vol;
					if (strcmp(action->argv[0],"nRemaining") == 0)
						pool->CoRemaining = newval / pool->vol;
					if (strcmp(action->argv[0],"nMin") == 0)
						pool->CoMin = newval / pool->vol;
					if (strcmp(action->argv[0],"nTotal") == 0)
						pool->CoTotal = newval / pool->vol;

					if (strcmp(field,"Co") == 0)
						pool->n = newval * pool->vol;
					if (strcmp(action->argv[0],"CoInit") == 0)
						pool->nInit = newval * pool->vol;
					if (strcmp(action->argv[0],"CoRemaining") == 0)
						pool->nRemaining = newval * pool->vol;
					if (strcmp(action->argv[0],"CoMin") == 0)
						pool->nMin = newval * pool->vol;
					if (strcmp(action->argv[0],"CoTotal") == 0)
						pool->nTotal = newval * pool->vol;
				}
				/* Do the normal set on all the fields */
				return(0);
		}
        return(0); /* do the normal set */
		break;
	}
}
#undef POOL_SUMTOTAL
#undef POOL_CONSERVE
#undef POOL_NSLAVE
#undef POOL_CONCSLAVE
#undef POOL_BUFFER

/* This function is declared as an extern void() */
void copyleft_kin()
{
	printf("The kinetics library is copylefted under the LGPL, see kinetics/COPYRIGHT.\n\n");
}
