;;;
;;; Wanderlust -- Yet Another Message Interface on Emacsen.
;;;
;;; Copyright (C) 1998 Yuuichi Teranishi <teranisi@gohome.org>
;;;
;;; Time-stamp: <99/06/10 00:19:23 teranisi>

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.
;;;

(require 'wl-vars)
(require 'wl-highlight)

(eval-when-compile
  (when wl-use-semi
    (require 'mime-view)
    (require 'mmelmo-imap4))
  (mapcar
   (function
    (lambda (symbol)
      (unless (boundp symbol)
	(make-local-variable symbol)
	(eval (list 'setq symbol nil)))))
   '(mime-view-ignored-field-list mmelmo-imap4-skipped-parts))
  (or (fboundp 'event-window)
      (defun event-window (a)))
  (or (fboundp 'posn-window)
      (defun posn-window (a)))
  (or (fboundp 'event-start)
      (defun event-start (a)))
  (or (fboundp 'mime-open-entity)
      (defun mime-open-entity (a b)))
  (or (fboundp 'mime-display-message)
      (defun mime-display-message (a b))))

(defvar wl-original-buf-name "*Message*")
(defvar wl-message-buf-name "Message")
(defvar wl-message-buffer-cur-summary-buffer nil)
(defvar wl-message-buffer-cur-folder nil)
(defvar wl-message-buffer-cur-number nil)

(mapcar 
 (function make-variable-buffer-local)
 (list 'wl-message-buffer-cur-folder
       'wl-message-buffer-cur-number
       ))

(provide 'wl-message)

(defun wl-select-buffer (buffer)
  (let ((gbw (get-buffer-window buffer))
	(sum (car wl-message-window-size))
	(mes (cdr wl-message-window-size))
	whi)
    (when (and gbw
	       (not (eq (save-excursion (set-buffer buffer)
					wl-message-buffer-cur-summary-buffer)
			(current-buffer))))
      (delete-window gbw)
      (run-hooks 'wl-message-window-deleted-hook)
      (setq gbw nil))
    (if gbw
	(select-window gbw)
;      (if (or (null mes)
;	      wl-stay-folder-window)
;	  (delete-other-windows))
      (setq whi (window-height))
      (if mes
	  (progn
	    (let ((total (+ sum mes)))
	      (setq sum (max window-min-height (/ (* whi sum) total)))
	      (setq mes (max window-min-height (/ (* whi mes) total))))
	    (if (< whi mes)
		(enlarge-window (- mes whi)))))
      (split-window (get-buffer-window (current-buffer)) sum)
      (other-window 1))
    (switch-to-buffer buffer)))

;;
;; called by wl-summary-mode buffer
;;
(defvar wl-message-func-called-hook nil)

(defun wl-message-scroll-down (amount)
  (let ((view-message-buffer (get-buffer-create wl-message-buf-name))
	(cur-buf (current-buffer)))
    (wl-select-buffer view-message-buffer)
    (if (bobp)
	()
      (scroll-down))
    (select-window (get-buffer-window cur-buf))    
    ))  

(defun wl-message-scroll-up (amount)
  (let ((view-message-buffer (get-buffer-create wl-message-buf-name))
	(cur-buf (current-buffer)))
    (wl-select-buffer view-message-buffer)
    (save-excursion
      (save-restriction
	(widen)
	(forward-page 1)
	(if (pos-visible-in-window-p (point))
	    (wl-message-narrow-to-page 1))))		;Go to next page.
    (if (eobp)
	()
      (scroll-up))
    (select-window (get-buffer-window cur-buf))    
    ))
  
(defun wl-message-call-func (func-name)
  (interactive)
  (let ((view-message-buffer (get-buffer-create wl-message-buf-name))
	(method (intern (format 
			 (if wl-use-semi
			     "mime-preview-%s"
			   "mime-viewer/%s"
			   )
			 func-name))))
    (wl-select-buffer view-message-buffer)
    (funcall method)
    (run-hooks 'wl-message-func-called-hook)
    ))

(defun wl-message-original-mode ()
  (setq major-mode 'wl-message-original-mode)
  (setq mode-name "Original")
  (setq buffer-read-only t)
  (if (fboundp 'set-buffer-file-coding-system)
      (set-buffer-file-coding-system wl-cs-noconv)))

(defun wl-message-mode ()
  (interactive)
  (setq major-mode 'wl-message-mode)
  (setq mode-name "Message")
  )

(defun wl-message-get-buffer-create ()
  (save-excursion
    (let (ret-val)
      (if (get-buffer wl-message-buf-name)
	  (set-buffer wl-message-buf-name)
	(setq ret-val
	      (set-buffer (get-buffer-create wl-message-buf-name)))
	(wl-message-mode)
	(run-hooks 'wl-message-buffer-created-hook)
	ret-val
	))))

(defun wl-message-original-get-buffer-create ()
  (let (ret-val)
    (if (get-buffer wl-original-buf-name)
	(set-buffer wl-original-buf-name)
      (save-excursion
	(setq ret-val
	      (set-buffer (get-buffer-create wl-original-buf-name)))
	(wl-message-original-mode)
	ret-val))))
  
(defun wl-message-exit ()
  (interactive)
  (let (summary-buf summary-win)
    (if (setq summary-buf wl-message-buffer-cur-summary-buffer)
	(if (setq summary-win (get-buffer-window summary-buf))
	    (select-window summary-win)))
    (run-hooks 'wl-message-exit-hook)))

(defun wl-message-decode (outbuf inbuf flag)
  (cond
   ((eq flag 'all-header)
    (and wl-mime-decode-with-all-header-func
	 (funcall wl-mime-decode-with-all-header-func outbuf inbuf)))
   ((eq flag 'no-mime)
    (save-excursion
      (set-buffer inbuf)
      (let ((buffer-read-only nil))
	(copy-to-buffer outbuf (point-min) (point-max))
	(set-buffer outbuf)
	(local-set-key "q" 'wl-message-exit)
	(local-set-key "p" 'wl-message-exit)
	(local-set-key "n" 'wl-message-exit)
	;(insert (decode-mime-charset-string contents wl-mime-charset))
	(decode-mime-charset-region (point-min) (point-max) wl-mime-charset)
	(wl-highlight-message (point-min)
			      (save-excursion
				(goto-char (point-min))
				(re-search-forward "^$" nil t)) nil)
	)))
   (t					; normal
    (save-excursion
      (set-buffer inbuf)
      (let ((buffer-read-only nil))
	(decode-mime-charset-region (point-min) 
				    (save-excursion
				      (goto-char (point-min))
				      (re-search-forward "^$" nil t)
				      (point))
				    wl-mime-charset)))
    (and wl-mime-decode-func
	 (funcall wl-mime-decode-func outbuf inbuf)))))

(defun wl-message-prev-page (&optional lines)
  "Scroll down this message. Returns non-nil if top of message"
  (interactive)
  (let ((cur-buf (current-buffer))
	(view-message-buffer (get-buffer-create wl-message-buf-name))
	ret-val)  
    (wl-select-buffer view-message-buffer)
    (move-to-window-line 0)
    (if (and wl-break-pages
	     (bobp)
	     (not (save-restriction (widen) (bobp))))
	(progn
	  (wl-message-narrow-to-page -1)
	  (goto-char (point-max))
	  (recenter -1))
      (if (not (bobp))
	  (scroll-down lines)
	(setq ret-val t)))
    (select-window (get-buffer-window cur-buf))
    ret-val))

;; for toolbar
(defun wl-message-read ()
  (funcall wl-message-read-func))

(defun wl-message-next-content ()
  (funcall wl-message-next-content-func))

(defun wl-message-prev-content ()
  (funcall wl-message-prev-content-func))

(defun wl-message-play-content ()
  (funcall wl-message-play-content-func))

(defun wl-message-extract-content ()
  (funcall wl-message-extract-content-func))

(defun wl-message-quit ()
  (funcall wl-message-quit-func))

(defun wl-message-next-page (&optional lines)
  "Scroll up this message. Returns non-nil if bottom of message"
  (interactive)
  (let ((cur-buf (current-buffer))
	(view-message-buffer (get-buffer-create wl-message-buf-name))
	ret-val)
    (wl-select-buffer view-message-buffer)
    (move-to-window-line -1)
    (if (save-excursion
	  (end-of-line)
	  (and (pos-visible-in-window-p)
	       (eobp)))
	(if (or (null wl-break-pages)
		(save-excursion
		  (save-restriction
		    (widen) (forward-line) (eobp))))
	    (setq ret-val t)
	  (wl-message-narrow-to-page 1)
	  (setq ret-val nil)
	  )
      (condition-case ()
	  (scroll-up lines)
	(end-of-buffer
	 (goto-char (point-max))))
      (setq ret-val nil)
      )
    (select-window (get-buffer-window cur-buf))
    ret-val
    ))

(defun wl-message-narrow-to-page (&optional arg)
  (interactive "P")
  (setq arg (if arg (prefix-numeric-value arg) 0))
  (save-excursion
    (forward-page -1) ; Beginning of current page.
    (forward-char 1)  ; for compatibility with emacs-19.28 and emacs-19.29
    (widen)
    (cond
     ((> arg 0)	(forward-page arg))
     ((< arg 0) (forward-page (1- arg)))
     )
    (forward-page)
    (if wl-break-pages
	(narrow-to-region (point)
			  (progn
			    (forward-page -1)
			    (if (and (eolp) (not (bobp)))
				(forward-line))
			    (point)))) ))

(defun wl-message-toggle-disp-summary ()
  (interactive)
  (let (summary-buf summary-win)
    (if (setq summary-buf (get-buffer wl-message-buffer-cur-summary-buffer))
	(if (setq summary-win (get-buffer-window summary-buf))
	    (delete-window summary-win)
	  (switch-to-buffer summary-buf)
	  (wl-select-buffer wl-message-buf-name))
      (wl-summary-goto-folder-subr wl-message-buffer-cur-folder 'no-sync
				   nil nil t)
  					; no summary-buf
      (let ((sum-buf (current-buffer)))
 	(wl-select-buffer wl-message-buf-name)
	(setq wl-message-buffer-cur-summary-buffer sum-buf))      
      )))

(defun wl-message-normal-get-original-buffer ()
  (let (ret-val)
    (if (setq ret-val (get-buffer wl-original-buf-name))
	ret-val
      (set-buffer (setq ret-val 
			(get-buffer-create wl-original-buf-name)))
      (wl-message-original-mode)
      ret-val)))

(if wl-use-semi
    (defalias 'wl-message-get-original-buffer 
      'mmelmo-get-original-buffer)
  (defalias 'wl-message-get-original-buffer 
    'wl-message-normal-get-original-buffer))

(defvar wl-message-redisplay-func 'wl-normal-message-redisplay)

(defun wl-message-redisplay (folder number flag msgdb &optional force-reload)
  (let ((default-mime-charset wl-mime-charset)
	(buffer-read-only nil))
    (if wl-message-redisplay-func
	(funcall wl-message-redisplay-func
		 folder number flag msgdb force-reload))))

;; nil means don't fetch all.
(defun wl-message-decide-backend (folder number message-id size)
  (let ((dont-do-that (and (integerp size)
			   (>= size wl-fetch-confirm-threshold)
			   (not 
			    (elmo-cache-exists-p message-id 
						 folder number))
			   (not (y-or-n-p 
				 (format "Fetch entire message? (%dbytes)" 
					 size))))))
    (message "")
    (cond ((and dont-do-that
		(eq (elmo-folder-number-get-type folder number) 'imap4)
		(not (and (elmo-use-cache-p folder number)
			  (elmo-cache-exists-p message-id folder number))))
	   'elmo-imap4)
	  (t (if (not dont-do-that) 'elmo)))))

;; Works on FLIM-1.9.0/SEMI-1.8.2 or later (maybe).
(defun wl-mmelmo-message-redisplay (folder number flag msgdb
					   &optional force-reload)
  (let* ((cur-buf (current-buffer))
	 (view-message-buffer (wl-message-get-buffer-create))
	 (inhibit-read-only t)
	 (message-id (cdr (assq number 
				(elmo-msgdb-get-number-alist msgdb))))
	 (size (elmo-msgdb-overview-entity-get-size
		(assoc message-id 
		       (elmo-msgdb-get-overview msgdb))))
	 (backend (wl-message-decide-backend folder number message-id size))
	 cur-entity ret-val header-end real-fld-num summary-win)
    (require 'mmelmo)
    (wl-select-buffer view-message-buffer)
    (unwind-protect
	(progn
	  (setq wl-message-buffer-cur-summary-buffer cur-buf)
	  (setq wl-message-buffer-cur-folder folder)
	  (setq wl-message-buffer-cur-number number)
	  (erase-buffer)
	  (save-excursion
	    (if backend
		(let ((mime-display-header-hook 'wl-highlight-headers)
		      (mime-view-ignored-field-list 
		       (if (eq flag 'all-header)
			   nil
			 mime-view-ignored-field-list))
		      (mmelmo-force-reload force-reload)
		      (mmelmo-imap4-threshold wl-fetch-confirm-threshold))
		  (setq real-fld-num (elmo-get-real-folder-number
				      folder number))
		  (setq cur-entity (mime-open-entity 
				    backend 
				    (if (eq backend 'elmo)
					(list folder
					      number
					      msgdb nil) ; location
				      (list (car real-fld-num)
					    (cdr real-fld-num)
					    msgdb nil))
				    ))
		  (setq mmelmo-imap4-skipped-parts nil)
		  (mime-display-message cur-entity view-message-buffer)
		  (if mmelmo-imap4-skipped-parts
		      (progn
			(message "Skipped fetching of %s."
				 (mapconcat 
				  (lambda (x)
				    (format "[%s]" x))
				  mmelmo-imap4-skipped-parts ","))))
		  (if (and (eq backend 'elmo-imap4)
			   (null mmelmo-imap4-skipped-parts))
		      (message "No part was skipped."))
		  (setq ret-val (not (eq backend 'elmo-imap4))))
	      (message "Skipped fetching.")
	      (setq ret-val nil))))
      (wl-message-overload-functions)
      (setq buffer-read-only t)
      ;; highlight body
      (when wl-highlight-body-too 
	(wl-highlight-body)
;	(wl-highlight-citation)
	)
      (condition-case ()
	  (wl-message-narrow-to-page)
	(error nil));; ignore errors.
      (set-buffer-modified-p nil)
      (setq mode-line-buffer-identification
	    (format "Wanderlust: << %s / %s >>"
		    (if wl-use-folder-petname-on-modeline
			(wl-folder-get-petname folder)
		      folder) number))
      (goto-char (point-min))
      (unwind-protect
	  (run-hooks 'wl-message-redisplay-hook)
	;; go back to summary mode
	(set-buffer cur-buf)
	(setq summary-win (get-buffer-window cur-buf))
	(if (window-live-p summary-win)
	    (select-window summary-win)))
      ret-val
      )))

(defun wl-normal-message-redisplay (folder number flag msgdb 
					   &optional force-reload)
  (interactive)
  (let* ((cur-buf (current-buffer))
	 (original-message-buffer (wl-message-get-original-buffer))
	 (view-message-buffer (wl-message-get-buffer-create))
	 (inhibit-read-only t)
	 (buffer-read-only nil)
	 (message-id (cdr (assq number 
				(elmo-msgdb-get-number-alist msgdb))))
	 (size (elmo-msgdb-overview-entity-get-size
		(assoc message-id 
		       (elmo-msgdb-get-overview msgdb))))
	 header-end ret-val summary-win
	 )
    (wl-select-buffer view-message-buffer)
    (unwind-protect
	(progn
	  (setq wl-message-buffer-cur-summary-buffer cur-buf)
	  (setq wl-message-buffer-cur-folder folder)
	  (setq wl-message-buffer-cur-number number)
	  (setq buffer-read-only nil)
	  (erase-buffer)
	  (if (or (eq (elmo-folder-number-get-type folder number) 'localdir)
		  (not (and (integerp size)
			    (>= size wl-fetch-confirm-threshold)
			    (not (elmo-cache-exists-p message-id 
						      folder number))
			    (not (y-or-n-p
				  (format "Fetch entire message? (%dbytes)" 
					  size))))))
	      (progn
		(save-excursion
		  (set-buffer original-message-buffer)
		  (setq buffer-read-only nil)
		  (elmo-read-msg folder number original-message-buffer 
				 msgdb force-reload)
		  ;;(decode-mime-charset-region (point-min) 
		  ;;		      (save-excursion
		  ;;		(goto-char (point-min))
		  ;;	(re-search-forward "^$" nil t)
		  ;;(point))
		  ;;			      wl-mime-charset)
		  (setq buffer-read-only t)
		  )
		;; decode MIME message.
		(wl-message-decode 
		 view-message-buffer 
		 original-message-buffer flag)
		(setq ret-val t))
	    (save-excursion
	      (set-buffer view-message-buffer)
	      (insert "\n\n"))))
      (wl-message-overload-functions)
      ;; highlight body
      (and wl-highlight-body-too (wl-highlight-body))
      (condition-case ()
	  (wl-message-narrow-to-page)
	(error nil)) ; ignore errors.
      (set-buffer-modified-p nil)      
      (setq mode-line-buffer-identification
	    (format "Wanderlust: << %s / %s >>" 
		    (if wl-use-folder-petname-on-modeline
			(wl-folder-get-petname folder)
		      folder)
		    number))
      (goto-char (point-min))
      (unwind-protect
	  (run-hooks 'wl-message-redisplay-hook)
	;; go back to summary mode
	(set-buffer cur-buf)
	(setq summary-win (get-buffer-window cur-buf))
	(if (window-live-p summary-win)
	    (select-window summary-win)))
      ret-val
      )))

(defun wl-message-refer-article-or-url (e)
  "Read article specified by message-id around point. If failed,
   attempt to execute button-dispatcher."
  (interactive "e")
  (mouse-set-point e)
  (save-restriction
    (let ((point (point))
          (beg (save-excursion (beginning-of-line) (point)))
          (end (save-excursion (end-of-line) (point)))
	  msg-id)
      (search-forward ">" end t)      ;Move point to end of "<....>".
      (if (and (re-search-backward "\\(<[^<> \t\n]+@[^<> \t\n]+>\\)" beg t)
	      (not (string-match "mailto:" (setq msg-id (wl-match-buffer 1)))))
	  (progn
	    (goto-char point)
	    (switch-to-buffer-other-window wl-message-buffer-cur-summary-buffer)
	    (wl-summary-jump-to-msg-by-message-id msg-id)
	    (wl-summary-redisplay))
	(wl-message-button-dispatcher e)))))

(defun wl-message-button-dispatcher (event)
  (funcall wl-message-button-dispatcher-func event))

(defun wl-message-uu-substring (buf outbuf &optional first last)
  (save-excursion
    (set-buffer buf)
    (search-forward "\n\n")
    (let ((sp (point))
	  ep filename case-fold-search)
      (if first
	  (progn
	    (re-search-forward "^begin[ \t]+[0-9]+[ \t]+\\([^ ].*\\)" nil t)
	    (setq filename (buffer-substring (match-beginning 1)(match-end 1))))
	(re-search-forward "^M.*$" nil t)) ; uuencoded string
      (beginning-of-line)
      (setq sp (point))
      (goto-char (point-max))
      (if last
	  (re-search-backward "^end" sp t)
        (re-search-backward "^M.*$" sp t)) ; uuencoded string
      (forward-line 1)
      (setq ep (point))
      (set-buffer outbuf)
      (goto-char (point-max))
      (insert-buffer-substring buf sp ep)
      (set-buffer buf)
      filename)))
