#! --PERL--
##
## Sympa - Programme principal

## RCS identification.
#my $id = '@(#)$Id: sympa.pl,v 1.4 1998/12/22 13:52:42 sympa Exp $';

## Load the modules and whatever we need.
use strict;

use lib '--DIR--/bin';
use Getopt::Std;

use Mail::Address;
use Mail::Internet;

use Commands;
use Conf;
use Language;
use Log;
use Version;
use smtp;
use MIME::QuotedPrint;
use List;

require 'mail.pl';
require 'tools.pl';
require 'msg.pl';
require 'parser.pl';


## Internal tuning
# delay between each read of the expirequeue
my $expiresleep = 50 ; 
my $is_signed = {}; 

# delay between each read of the digestqueue
my $digestsleep = 5; 

## Options :  d		-> debug
##            D         -> Debug with many logs
##            f		-> name of configuration file
##            m		-> log invocations to sendmail.
##            l		-> language
##            F		-> Foreground and log to stderr also.

Getopt::Std::getopts('DdFf:ml:');

my @parser_param = ($*, $/);
my %loop_info;

## Load configuration file
unless (Conf::load($Getopt::Std::opt_f ? $Getopt::Std::opt_f : '--CONFIG--')) {
   print Msg(1, 1, "Configuration file has errors.\n");
   exit(1);
}

## Open the syslog and say we're read out stuff.
do_openlog($Conf{'syslog'}, $Conf{'log_socket_type'});
do_log('info', 'Configuration file read'); 

## Probe Db if defined
if ($Conf{'db_name'} and $Conf{'db_type'}) {
    unless ($List::use_db = &List::probe_db()) {
	&fatal_err('Database %s defined in sympa.conf has not the right structure or is unreachable. If you don\'t use any database, comment db_xxx parameters in sympa.conf', $Conf{'db_name'});
    }
}

## Set locale configuration
$Getopt::Std::opt_l =~ s/\.cat$//; ## Compatibility with version < 2.3.3
$Language::sympa_lang = $Getopt::Std::opt_l || $Conf{'lang'};
Language::SetLang($Language::sympa_lang);

## Check locale version
if (Msg(1, 102, $Version) ne $Version){
    &do_log('info', 'NLS message file version %s different from src version %s', Msg(1, 102,""), $Version);
} 

## Main program
if (!chdir($Conf{'home'})) {
   fatal_err("Can't chdir to %s: %m", $Conf{'home'});
   ## Function never returns.
}

## Sets the UMASK
umask($Conf{'umask'});

## Check for several files.
unless (Conf::checkfiles()) {
   fatal_err("Missing files. Aborting.");
   ## No return.
}

## Put ourselves in background if we're not in debug mode. That method
## works on many systems, although, it seems that Unix conceptors have
## decided that there won't be a single and easy way to detach a process
## from its controlling tty.
unless ($Getopt::Std::opt_d || $Getopt::Std::opt_F) {
   if (open(TTY, "/dev/tty")) {
       ioctl(TTY, 0x20007471, 0);         # XXX s/b &TIOCNOTTY
#       ioctl(TTY, &TIOCNOTTY, 0);
       close(TTY);
   }
   open(STDERR, ">> /dev/null");
   open(STDOUT, ">> /dev/null");
   setpgrp(0, 0);
   if (($_ = fork) != 0) {
      do_log('debug', "Starting server, pid $_");
      exit(0);
   }
   do_openlog($Conf{'syslog'}, $Conf{'log_socket_type'});
}

## Create and write the pidfile
unless (open(LOCK, "+>> $Conf{'pidfile'}")) {
   fatal_err("Could not open %s, exiting", $Conf{'pidfile'});
   ## No return.
}
unless (flock(LOCK, 6)) {
   fatal_err("Could not lock %s: Sympa is probably already running.", $Conf{'pidfile'});
   ## No return.
}

unless (open(LCK, "> $Conf{'pidfile'}")) {
   fatal_err("Could not open %s, exiting", $Conf{'pidfile'});
   ## No return.
}
unless (truncate(LCK, 0)) {
   fatal_err("Could not truncate %s, exiting.", $Conf{'pidfile'});
   ## No return.
}
print LCK "$$\n";
close(LCK);

## Most initializations have now been done.
do_log('notice', "Sympa $Version Started");
printf "Sympa $Version Started\n";

## Catch SIGTERM, in order to exit cleanly, whenever possible.
$SIG{'TERM'} = 'sigterm';

my $end = 0;
my $index_queuedigest = 0; # verify the digest queue
my $index_queueexpire = 0; # verify the expire queue
my ($qdir, @qfile);

## This is the main loop : look after files in the directory, handles
## them, sleeps a while and continues the good job.
while (!$end) {
    $qdir = $Conf{'queue'};
    if (!opendir(DIR, $qdir)) {
	fatal_err("Can't open dir %s: %m", $qdir); ## No return.
    }
    @qfile = sort grep (!/^\./,readdir(DIR));
    closedir(DIR);
    
    ## Scan queuedigest
    if ($index_queuedigest++ >=$digestsleep){
	$index_queuedigest=0;
	&SendDigest();
    }
    ## Scan the queueexpire
    if ($index_queueexpire++ >=$expiresleep){
	$index_queueexpire=0;
	&ProcessExpire();
    }

    my $filename;
    my $listname;
    my $priority = 'z'; ## lowest priority
    
    ## Scans files in queue
    foreach my $t_filename (sort @qfile) {
	my $t_priority;
	my $type;
	my $list;

	# trying to fix a bug (perl bug ??) of solaris version
	($*, $/) = @parser_param;

	## test ever if it is an old bad file
	if ($t_filename =~ /^BAD\-/i){
	    if ((stat "$qdir/$t_filename")[9] < (time - $Conf{'clean_delay_queue'}*86400) ){
		unlink ("$qdir/$t_filename") ;
		do_log('notice', "Deleting bad message %s because too old", $t_filename);
	    };
	    next;
	}

	## z and Z are a null priority, so file stay in queue and are processed
	## only if renamed by administrator
	next unless ($t_filename =~ /^(\S+)\.\d+\.\d+$/);

	## Don't process temporary files created by queue (T.xxx)
	next if ($t_filename =~ /^T\./);

	## Extract listname from filename
	$listname = $1;
	$listname =~ s/\@.*$//;
	$listname =~ y/A-Z/a-z/;
	if ($listname =~ /^(\S+)-(request|owner|editor|subscribe|unsubscribe)$/) {
	    ($listname, $type) = ($1, $2);
	}

	unless ($listname =~ /^(sympa|listserv|majordomo)(\@(\S+))?$/i) {
	    unless ($list = new List ($listname)) {
		rename("$qdir/$t_filename", "$qdir/BAD-$t_filename");
		do_log('notice', "Renaming bad file %s to BAD-%s", $t_filename, $t_filename);
	    }
	    next unless $list;
	}
	
	if ($type eq 'request') {
	    $t_priority = $Conf{'request_priority'};
	}elsif ($type eq 'owner') {
	    $t_priority = $Conf{'owner_priority'};
	}elsif ($listname eq 'sympa') {	
	    $t_priority = $Conf{'sympa_priority'};
	}else {
	    $t_priority = $list->{'admin'}{'priority'};
	}
	
	if (ord($t_priority) < ord($priority)) {
	    $priority = $t_priority;
	    $filename = $t_filename;
	}
    } ## END of spool lookup

    &smtp::reaper;

    unless ($filename) {
	sleep($Conf{'sleep'});
	next;
    }

    do_log('debug', "Processing %s with priority %s", "$qdir/$filename", $priority) 
	if ($Getopt::Std::opt_d);

    my $status = &DoFile($listname, "$qdir/$filename");
    
    if (defined($status)) {
	do_log('debug', "Finished %s", "$qdir/$filename") if ($Getopt::Std::opt_d);
	unlink("$qdir/$filename");
    }else {
	rename("$qdir/$filename", "$qdir/BAD-$filename");
	do_log('notice', "Renaming bad file %s to BAD-%s", $filename, $filename);
    }
    
} ## END of infinite loop

## Dump of User files in DB
List::dump();

## Disconnect from Database
List::db_disconnect if ($List::dbh);

do_log('notice', 'Sympa exited normally due to signal');
exit(0);

## When we catch SIGTERM, just change the value of the loop
## variable.
sub sigterm {
    do_log('notice', 'signal TERM received, still processing current task');

    $end = 1;
}

## Handles a file received and files in the queue directory. This will
## read the file, separate the header and the body of the message and
## call the adequate function wether we have received a command or a
## message to be redistributed to a list.
sub DoFile {
    my ($listname, $file) = @_;
    &do_log('debug2', 'DoFile(%s)', $file);
    
    my $status;
    
    ## Open and parse the file   
    if (!open(IN, $file)) {
	&do_log('info', 'Can\'t open %s: %m', $file);
    }
    
    my $msg = new Mail::Internet [<IN>];
    my $hdr = $msg->head;
    
    ## Ignoring messages without From: field
    unless ($hdr->get('From')) {
	do_log('notice', 'No From found in message, skipping.');
	return undef;
    }
    
    my @sender_hdr = Mail::Address->parse($hdr->get('From'));
    
    if ($#sender_hdr == -1) {
	do_log('notice', 'No valid address in From: field, skipping');
	return undef;
    }

    my $sender = $sender_hdr[0]->address;

    ## Loop prevention
    if ($sender =~ /^(mailer-daemon|sympa|listserv|mailman|majordomo|smartlist)/mio) {
	do_log('notice','Ignoring message which would cause a loop, sent by %s', $sender);
	return undef;
    }

    ## Initialize command report
    undef @msg::report;  
    
    ## Q- and B-decode subject
    my $subject_field = &tools::decode_string($hdr->get('Subject'));
    chomp $subject_field;
    $hdr->replace('Subject', $subject_field);
    
    my $bytes = -s $file;
    
    my $list;   
    $list = new List ($listname);
    my $host = $list->{'admin'}{'host'};
    my $name = $list->{'name'};
    
    ## Search the X-Sympa-To header.
    my $rcpt = $hdr->get('X-Sympa-To');
    unless ($rcpt) {
	do_log('notice', 'no X-Sympa-To found, ignoring message file %s', $file);
	return undef;
    }
    
    ## Strip of the initial X-Sympa-To field
    $hdr->delete('X-Sympa-To');
    
    ## Loop prevention
    my $loop;
    foreach $loop ($hdr->get('X-Loop')) {
	foreach my $l (split(/[\s,]+/, lc($loop))) {
	    if ($l eq lc("$name\@$host")) {
		do_log('notice', "Ignoring message which would cause a loop (X-Loop: $loop)");
		return undef;
	    }
	}
    }
    
    ## Content-Identifier: Auto-replied is generated by some non standard X400 mailer
    if ($hdr->get('Content-Identifier') =~ /Auto-replied/i) {
	do_log('notice', "Ignoring message which would cause a loop (Content-Identifier: Auto-replied)");
	return undef;
    }
        
    if ($Conf{'openssl'} && $hdr->get('Content-Type') =~ /multipart\/signed/i) {
#       do_log ('debug2', "-=-=-==-=-=-=-=-==-=-=-=-  verifying signed message"); 
	$is_signed = &tools::smime_sign_check ($file,$sender);
    }else{
	do_log ('debug2', "-=-=-==-=-=-=-=-==-=-=-=-  message not signed or openssl not configured"); 
    }
    
    ## Mail adressed to the robot and mail 
    ## to <list>-subscribe or <list>-unsubscribe are commands
    if (($rcpt =~ /^(sympa|listserv|majordomo)(\@(\S+))?$/o) || ($rcpt =~ /^(\S+)-(subscribe|unsubscribe)(\@(\S+))?$/o)) {
	$status = &DoCommand($rcpt, $msg);
	
	## forward mails to <list>-request <list>-owner etc
    }elsif ($rcpt =~ /^(\S+)-(request|owner|editor)(\@(\S+))?$/o) {
	my ($name, $function) = ($1, $2);
	
	## Simulate Smartlist behaviour with command in subject
	if (($function eq 'request') and ($subject_field =~ /^\s*(subscribe|unsubscribe)(\s*$name)?\s*$/i) ) {
	    my $command = $1;
	    
	    $status = &DoCommand("$name-$command", $msg);
	}else {
	    $status = &DoForward($name, $function, $msg);
	}       
    }else {
	$status =  &DoMessage($rcpt, $msg, $bytes);
    }
    
    ## Mail back the result.
    if (@msg::report) {

	## Loop prevention

	## Count reports sent to $sender
	$loop_info{$sender}{'count'}++;
	
	## Sampling delay 
	if ((time - $loop_info{$sender}{'date_init'}) < $Conf{'loop_command_sampling_delay'}) {

	    ## Notify listmaster of first rejection
	    if ($loop_info{$sender}{'count'} == $Conf{'loop_command_max'}) {
		## Notify listmaster
		&List::send_notify_to_listmaster('loop_command', $file);
	    }
	    
	    ## Too many reports sent => message skipped !!
	    if ($loop_info{$sender}{'count'} >= $Conf{'loop_command_max'}) {
		&do_log('notice', 'Ignoring message which would cause a loop, %d messages sent to %s', $loop_info{$sender}{'count'}, $sender);
		
		return undef;
	    }
	}else {
	    ## Sampling delay is over, reinit
	    $loop_info{$sender}{'date_init'} = time;

	    ## We apply Decrease factor if a loop occured
	    $loop_info{$sender}{'count'} *= $Conf{'loop_command_decrease_factor'};
	}

	## Prepare the reply message
	my $reply_hdr = new Mail::Header;
	$reply_hdr->add('From', sprintf Msg(12, 4, 'SYMPA <%s>'), $Conf{'sympa'});
	$reply_hdr->add('To', $sender);
	$reply_hdr->add('Subject', Msg(4, 17, 'Output of your commands'));
	$reply_hdr->add('X-Loop', $Conf{'sympa'});
	$reply_hdr->add('MIME-Version', Msg(12, 1, '1.0'));
	$reply_hdr->add('Content-type', sprintf 'text/plain; charset=%s', 
			Msg(12, 2, 'us-ascii'));
	$reply_hdr->add('Content-Transfer-Encoding', Msg(12, 3, '7bit'));
	
	## Open the SMTP process for the response to the command.
	*FH = &smtp::smtpto($Conf{'request'}, \$sender);
	$reply_hdr->print(\*FH);
	
	foreach (@msg::report) {
	    print FH;
	}
	
	close(FH);
    }
    
    return $status;
}

## Handles a message sent to [list]-editor, [list]-owner or [list]-request
sub DoForward {
    my($name,$function,$msg) = @_;
    &do_log('debug2', 'DoForward(%s, %s)', $name, $function);

    my $hdr = $msg->head;
    my $messageid = $hdr->get('Message-Id');
    ##  Search for the list
    my $list = new List ($name);
    my $host = $list->{'admin'}{'host'};

    my @rcpt;
    my $recepient="$name-$function";
    
    do_log('info', "Processing message for %s with priority %s, %s",$recepient,$list->{'admin'}{'priority'}, $messageid );
    
    unless ($list) {
	do_log('notice', "Message for %s-%s ignored, unknown list %s",$name,$function,$name );
     return undef;
    }
    
    my $admin = $list->{'admin'};
    
    $hdr->add('X-Loop', "$name-$function\@$host");
    $hdr->delete('X-Sympa-To:');

   if ($function eq "request") {
      foreach my $i (@{$admin->{'owner'}}) {
          next if ($i->{'reception'} eq 'nomail');
          push(@rcpt, $i->{'email'}) if ($i->{'email'});
      }
      do_log('notice', 'Warning : no owner defined or all of them use nomail option in list %s', $name ) unless (@rcpt);
   }elsif ($function eq "editor") {
      foreach my $i (@{$admin->{'editor'}}) {
          next if ($i->{'reception'} eq 'nomail');
          push(@rcpt, $i->{'email'}) if ($i->{'email'});
      }
      do_log('notice', 'Warning : no editor defined or  all of them use nomail option in list %s', $name ) unless (@rcpt);
   }
		      
   if ($#rcpt < 0) {
     do_log('notice', "Message for %s-%s ignored, %s undefined in list %s",$name,$function,$function,$name);
     return undef;
   }
   *SIZ = smtp::smtpto($Conf{'request'}, \@rcpt);
   $msg->print(\*SIZ);
   close(SIZ);
   
   do_log('info',"Message for %s-%s forwarded",$name,$function);
   return 1;
}



## Handles a message sent to a list.
sub DoMessage {
    my($which, $msg, $bytes) = @_;
    &do_log('debug2', 'DoMessage(%s, %s)', $which, $bytes);
    
    ## List and host.
    my($listname, $host) = split(/[@\s]+/, $which);
#    $host = $Conf{'host'} unless ($host);

    ## Search for the list
    my $list = new List ($listname);
    return undef unless $list;
 
    my $name = $list->{'name'};
    my $host = $list->{'admin'}{'host'};

    my $start_time = time;
    
    ## Now check if the sender is an authorized address.
    my $hdr = $msg->head;
    
    my $from_field = $hdr->get('From');
    my $messageid = $hdr->get('Message-Id');

    do_log('info', "Processing message for %s with priority %s, %s", $name,$list->{'admin'}{'priority'}, $messageid );
    
    my @sender_hdr = Mail::Address->parse($from_field);

    my $sender = $sender_hdr[0]->address || '';
    if ($sender =~ /^(mailer-daemon|sympa|listserv|majordomo|smartlist|mailman)/mio) {
	do_log('notice', 'Ignoring message which would cause a loop');
	return undef;
    }
    
    ## Check the message for commands and catch them.
    return undef if (tools::checkcommand($msg, $sender));
       
    my $admin = $list->{'admin'};
    return undef unless $admin;
    
    my $customheader = $admin->{'custom_header'};
    $host = $admin->{'host'} if ($admin->{'host'});
    
    ## Check if the message is too large
    my $max_size = $list->get_max_size() || $Conf{'max_size'};
    if ($max_size && $bytes > $max_size) {
	do_log('notice', 'Message for %s from %s too large (%d > %d)', $name, $sender, $bytes, $max_size);
	*SIZ  = smtp::smtpto($Conf{'request'}, \$sender);
	print SIZ "From: " . sprintf (Msg(12, 4, 'SYMPA <%s>'), $Conf{'request'}) . "\n";
	printf SIZ "To: %s\n", $sender;
	printf SIZ "Subject: " . Msg(4, 11, "Your message for list %s has been rejected") . "\n\n", $name;
	print SIZ Msg(4, 12, $msg::msg_too_large);
	$msg->print(\*SIZ);
	close(SIZ);
	return undef;
    }
    
    ## Call scenarii assuming smtp auth method, this should be change when introducing S/MIME
    ## auth_method MD5 do not have any sense in send scenarii because auth is perfom by
    ## distribute or reject command.

    my $action ;
    if ($is_signed->{'body'}) {
	$action = &List::get_action ('send',$name,$sender,'smime',$hdr);
    }else{
	$action = &List::get_action ('send',$name,$sender,'smtp',$hdr);
    }
    if ($action =~ /^do_it/) {
	
	## Change the reply-to header if necessary. Never change it if already there.

	## Hide the sender if the list is anonymoused
	if ( $list->{'admin'}{'anonymous_sender'} ) {
	    foreach my $field ('Sender','X-Sender','Received','Message-id','From',
			       'Resent-From','Reply-To') {
		$hdr->delete($field);
	    }
	    $hdr->add('From',"$list->{'admin'}{'anonymous_sender'}");
	}


	my $reply = $list->get_reply_to();
	if ($reply && !$hdr->get('Reply-To') && $reply !~ /^sender$/io) {
	    if ($reply =~ /^list$/io) {
		$reply = "$name\@$host";
	    }
	    $hdr->add('Reply-To', $reply);
	}
	
	## Remove unwanted headers if present.
	$hdr->delete('Return-Receipt-To');
	$hdr->delete('Precedence');
	$hdr->delete('X-Sequence');
	$hdr->add('X-Loop', "$name\@$host");
	
	## Update the stats, and returns the new X-Sequence, if any.
	$hdr->add('X-Sequence', $list->update_stats($bytes));
	$list->savestats();
	## Add other useful headers
	$hdr->add('Precedence', 'list');
	foreach my $i (@{$customheader}) {
	    $hdr->add($1, $2) if ($i=~/^([\S\-\:]*)\s(.*)$/);
	}
	
	## Blindly send the message to all users.
	my $numsmtp;
	unless ($numsmtp = $list->send_msg($msg)) {
	    do_log('info','Unable to send message to list %s', $name);
	    return undef;
	}
	
	$list->archive_msg($msg);
	do_log('info', 'Message for %s from %s accepted (%d seconds, %d sessions), size=%d', $name, $sender, time - $start_time, $numsmtp, $bytes);
	
	## Does the list accept digest mode
	if ($list->is_digest()){
	   
	    ## Send this msg into the digestqueue
	    my $numsmtp2 = $list->archive_msg_digest($msg);
	}
	
	## Everything went fine, return TRUE in order to remove the file from
	## the queue.
	return 1;
    }elsif($action =~ /^request_auth/){
    	my $key = $list->send_auth($sender, $msg);
	do_log('notice', 'Message for %s from %s kept for authentication with key %s', $name, $sender, $key);
	return 1;
    }elsif($action =~ /^editorkey/){
	my $key = $list->send_to_editor('md5',$msg);
	do_log('info', 'Key %s for list %s from %s sent to editors', $key, $name, $sender);
	$list->notify_sender($sender);
	return 1;
    }elsif($action =~ /editor/){
	my $key = $list->send_to_editor('smtp',$msg);
	do_log('info', 'Message for %s from %s sent to editors', $name, $sender);
	$list->notify_sender($sender);
	return 1;
    }elsif($action =~ /^reject/) {
   
	do_log('notice', 'Message for %s from %s rejected because sender not allowed', $name, $sender);
	*SIZ  = smtp::smtpto($Conf{'request'}, \$sender);
	print SIZ "From: " . sprintf (Msg(12, 4, 'SYMPA <%s>'), $Conf{'request'}) . "\n";
	printf SIZ "To: %s\n", $sender;
	printf SIZ "Subject: " . Msg(4, 11, "Your message for list %s has been rejected") . "\n\n", $name;
	print SIZ Msg(4, 15, $msg::list_is_private);
	$msg->print(\*SIZ);
	close(SIZ);
	return undef;
    }
}

## Handles a message sent to a list.

## Handles a command sent to the list manager.
sub DoCommand {
    my($rcpt, $msg) = @_;
    &do_log('debug2', 'DoCommand(%s)', $rcpt);

    ## Now check if the sender is an authorized address.
    my $hdr = $msg->head;
    
    my $from_field = $hdr->get('From');
    my $messageid = $hdr->get('Message-Id');
    my $content_type = $hdr->get('Content-type');
    my $transfert_encoding = $hdr->get('Content-transfer-encoding');
    my ($success, $status);
    
    do_log('debug', "Processing command with priority %s, %s", $Conf{'sympa_priority'}, $messageid );
    
    my @sender_hdr = Mail::Address->parse($from_field);
    my $sender = $sender_hdr[0]->address;

    ## If X-Sympa-To = <listname>-<subscribe|unsubscribe> parse as a unique command
    if ($rcpt =~ /^(\S+)-(subscribe|unsubscribe)(\@(\S+))?$/o) {
	do_log('debug',"processing message for $1-$2");
	&Commands::parse($sender,"$2 $1");
	return 1; 
    }
    
    ## Process the Subject of the message
    ## Search and process a command in the Subject field
    my $subject_field = $hdr->get('Subject');
    chomp $subject_field;
    $subject_field =~ s/\n//mg; ## multiline subjects
    $subject_field =~ s/^\s*(Re:)?\s*(.*)\s*$/$2/i;

    $success ||= &Commands::parse($sender, $subject_field) ;
    
    ## check Content-type
    my $mime = $hdr->get('Mime-Version') ;
    
    unless (($content_type =~ /text/i and !$mime)
	    or !($content_type) 
	    or ($content_type =~ /text\/plain/i)) {
	do_log('notice', "Ignoring message body not in text/plain, Content-type: %s", $content_type);
	print Msg(4, 37, "Ignoring message body not in text/plain, please use text/plain only \n(or put your command in the subject).\n");
	
	return $success ;
    }
    
    
    my @msgexpire;
    my ($expire, $i);
    
    ## Process the body of the message
    unless ($success) { ## unless subject contained commands
	foreach $i (@{$msg->body}) {
	    if ($transfert_encoding =~ /quoted-printable/i) {
		$i = MIME::QuotedPrint::decode($i);
	    }
	    if ($expire){
		if ($i =~ /^(quit|end|stop)/io){
		    last;
		}
		# store the expire message in @msgexpire
		push(@msgexpire, $i);
		next;
	    }
	    $i =~ s/^\s*>?\s*(.*)\s*$/$1/g;
	    next if ($i =~ /^$/); ## skip empty lines
	    
	    # exception in the case of command expire
	    if ($i =~ /^exp(ire)?\s/i){
		$expire = $i;
		print "> $i\n\n";
		next;
	    }
	    
	    push @msg::report, "> $i\n\n";
	    my $size = $#msg::report;
	    

	    if ($i =~ /^(quit|end|stop|-)/io) {
		push @msg::report, sprintf Msg(4, 18, "Found `quit' command, exiting.\n");
		last;
	    }
	    
	    unless ($status = Commands::parse($sender, $i)) {
		push @msg::report, sprintf Msg(4, 19, "Command not understood: ignoring end of message.\n");
		last;
	    }

	    if ($#msg::report > $size) {
		## There is a command report
		push @msg::report, "\n";
	    }else {
		## No command report
		pop @msg::report;
	    }
	    
	    $success ||= $status;
	}
    }

    ## No command found
    unless (defined($success)) {
	do_log('info', "No command found in message");
    }
    
    # processing the expire function
    if ($expire){
	print STDERR "expire\n";
	unless (&Commands::parse($sender, $expire, @msgexpire)) {
	    print Msg(4, 19, "Command not understood: ignoring end of message.\n");
	}
    }

    return $success;
}

## Read the queue and send old digests to the subscibers with the digest option.
sub SendDigest{
    &do_log('debug2', 'SendDigest()');

    my $ddir = $Conf{'queuedigest'};
    if (!opendir(DIR, $ddir)) {
	fatal_err(Msg(3, 1, "Can't open dir %s: %m"), $ddir); ## No return.
    }
    my @dfile =( sort grep (!/^\./,readdir(DIR)));
    closedir(DIR);

    foreach my $digest (@dfile){
	## Load list info....if necessary reload config, subscribers, ....
	my $list = new List ($digest);
	unless ($list) {
	    &do_log('info', 'Unknown list, deleting digest file %s', $digest);
	    unlink "$ddir/$digest";
	    return undef;
	}
	
	my @timedigest= (stat "$ddir/$digest")[9];

	if ($list->get_nextdigest()){
	    ## Blindly send the message to all users.
	    do_log('info', "Sending digest to list %s", $digest);
	    my $start_time = time;
	    $list->send_msg_digest($digest);
	    do_log('info', 'Digest of the list %s sent (%d seconds)',
		   $digest, time - $start_time);

	    unlink("$ddir/$digest");
 	}
    }
}


## Read the EXPIRE queue and check if a process has ended
sub ProcessExpire{
    &do_log('debug2', 'ProcessExpire()');

    my $edir = $Conf{'queueexpire'};
    if (!opendir(DIR, $edir)) {
	fatal_err("Can't open dir %s: %m", $edir); ## No return.
    }
    my @dfile =( sort grep (!/^\./,readdir(DIR)));
    closedir(DIR);
    my ($d1, $d2, $proprio, $user);

    foreach my $expire (@dfile) {
#   while ($expire=<@dfile>){	
	## Parse the expire configuration file
	if (!open(IN, "$edir/$expire")) {
	    next;
	}
	if (<IN> =~ /^(\d+)\s+(\d+)$/) {
	    $d1=$1;
	    $d2=$2;
	}	

	if (<IN>=~/^(.*)$/){
	    $proprio=$1; 
	}
	close(IN);

	## Is the EXPIRE process finished ?
	if ($d2 <= time){
	    my $list = new List ($expire);
	    my $listname = $list->{'name'};
	    unless ($list){
		unlink("$edir/$expire");
		next;
	    };
	
	    ## Prepare the reply message
	    my $reply_hdr = new Mail::Header;
	    $reply_hdr->add('From', sprintf Msg(12, 4, 'SYMPA <%s>'), $Conf{'sympa'});
	    $reply_hdr->add('To', $proprio);
 	    $reply_hdr->add('Subject',sprintf( Msg(4, 24, 'End of your command EXPIRE on list %s'),$expire));

	    $reply_hdr->add('MIME-Version', Msg(12, 1, '1.0'));
	    my $content_type = 'text/plain; charset='.Msg(12, 2, 'us-ascii');
	    $reply_hdr->add('Content-type', $content_type);
	    $reply_hdr->add('Content-Transfer-Encoding', Msg(12, 3, '7bit'));

	    ## Open the SMTP process for the response to the command.
	    *FH = &smtp::smtpto($Conf{'request'}, \$proprio);
	    $reply_hdr->print(\*FH);
	    my $fh = select(FH);
	    my $limitday=$d1;
	    #converting dates.....
	    $d1= int((time-$d1)/86400);
	    #$d2= int(($d2-time)/86400);
	
	    my $cpt_badboys;
	    ## Amount of unconfirmed subscription

	    unless ($user = $list->get_first_user()) {
		return undef;
}

	    while ($user = $list->get_next_user()) {
		$cpt_badboys++ if ($user->{'date'} < $limitday);
	    }

	    ## Message to the owner who launched the expire command
	    printf Msg(4, 28, "Among the subscribers of list %s for %d days, %d did not confirm their subscription.\n"), $listname, $d1, $cpt_badboys;
	    print "\n";
	    printf Msg(4, 26, "Subscribers who do not have confirm their subscription:\n");	
	    print "\n";
	
	    my $temp=0;

	    unless ($user = $list->get_first_user()) {
		return undef;
	    }

	    while ($user = $list->get_next_user()) {
		next unless ($user->{'date'} < $limitday);
		print "," if ($temp == 1);
		print " $user->{'email'} ";
		$temp=1 if ($temp == 0);
	    }
	    print "\n\n";
	    printf Msg(4, 27, "You must delete these subscribers from this list with the following commands :\n");
	    print "\n";

	    unless ($user = $list->get_first_user()) {
		return undef;
	    }
	    while ($user = $list->get_next_user()) {
		next unless ($user->{'date'} < $limitday);
		print "DEL   $listname   $user->{'email'}\n";
	    }
	    ## Mail back the result.
	    select($fh);
	    close(FH);
	    unlink("$edir/$expire");
	    next;
	}
    }
}


1;
