#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <unistd.h>

#include <Y2/Y.h>
#include <Y2/Ylib.h>

#include "../include/string.h"

#ifndef DEF_CON_ARG
# define DEF_CON_ARG    "127.0.0.1:9433"
#endif            



/*
 *	Print help.
 */
void PrintUsage()
{
	printf("\
Usage: yrecinfo [options]\n\
\n\
    [options] can be any of the following:\n\
\n\
        --modes                      Print list of Audio modes.\n\
        --recorder <address:port>    Specify which YIFF server to connect to.\n\
\n"
	);


	return;
}



int main(int argc, char *argv[])
{
	int i, total;
	int print_audio_modes = 0;
	char *con_arg = NULL;

	YConnection *con = NULL;
	YEventAudioStats astats;
	YEventServerStats sstats;
	YAudioModeValuesStruct **audio_mode;


	/* Parse arguments. */
	for(i = 1; i < argc; i++)
	{
	    if(argv[i] == NULL)
		continue;

	    /* Help. */
	    if(strcasepfx(argv[i], "--h") ||
               strcasepfx(argv[i], "-h") ||
               !strcmp(argv[i], "?")
	    )
	    {
		PrintUsage();
		return(0);
	    }
	    /* Print Audio modes list. */
	    else if(strcasepfx(argv[i], "--m") ||
                    strcasepfx(argv[i], "-m")
            )
	    {
		print_audio_modes = 1;
	    }
	    /* Connect address. */
	    else if(strcasepfx(argv[i], "--rec") ||
                    strcasepfx(argv[i], "-rec")
	    )
	    {
		i++;
		if(i < argc)
		{
		    free(con_arg);
		    con_arg = StringCopyAlloc(argv[i]);
		}
		else
		{
		    fprintf(stderr,
			"%s: Requires argument.\n",
			argv[i - 1]
		    );
		}
	    }
	}


	/*
	 *	Connect to YIFF server.
	 */
	con = YOpenConnection(
	    NULL,		/* No start argument. */
	    con_arg
	);
	if(con == NULL)
	{
	    printf("%s: Cannot connect to YIFF server.\n",
		(con_arg == NULL) ? DEF_CON_ARG : con_arg
	    );

            free(con_arg);
            con_arg = NULL;

	    return(-1);
	}



	/*
	 *	Request and print stats.
	 */
        printf("recorder: %s\n",
            ((con_arg == NULL) ? DEF_CON_ARG : con_arg)
        );

	/* Server stats. */
	if(!YGetServerStats(con, &sstats))
	{
            printf("version: %i.%i\n",
                sstats.protocol_version_major,
		sstats.protocol_version_minor
            );

            printf("cycle load: %.1lf%%\n",
                sstats.cycle_load * 100
            );

            printf("maximum protocol statement: %i bytes\n",
                YNetRecvBufLen
            );
	}

	/* Audio mode values. */
	if(!YGetAudioStats(con, &astats))
	{
	    printf("cycle set: %i\n",
		astats.cycle_set
	    );

            printf("cycle: %i microseconds\n",
		astats.cycle_us
	    );
            printf("compensated cycle: %i microseconds\n",
                astats.compensated_cycle_us
            );

            printf("write ahead: %i microseconds\n",
                astats.write_ahead_us
            );
            printf("cumulative latency: %i microseconds\n",
                astats.cumulative_latency_us
            );

            printf("sample rate: %i Hz\n",
                astats.sample_rate
            );
            printf("channels: %i\n",
                astats.channels
            );
            printf("sample size: %i bits\n",
                astats.sample_size
            );
            printf("bytes per second: %i bytes\n",
                astats.bytes_per_sec
            );

            printf("allow sound buffer fragments: %s\n",  
                ((astats.allow_fragments) ? "yes" : "no")
            );
            printf("sound buffer fragments: %i\n",
		astats.num_fragments
	    );
	    printf("sound buffer size: %i bytes\n",
		astats.fragment_size
	    );

            printf("flip stereo: %s\n",
                ((astats.flip_stereo) ? "yes" : "no")
            );

            printf("direction: %i\n",
                astats.direction
            );
	}

	/* Print audio modes. */
	if(print_audio_modes)
	{
	 printf("\n");
	 audio_mode = YGetAudioModes(con, &total);
	 for(i = 0; i < total; i++)
	 {
	    if(audio_mode[i] == NULL)
		continue;

	    printf(
		"audio: %s\n",
		audio_mode[i]->name
	    );
	    printf("  sample rate: %i Hz\n",
		audio_mode[i]->sample_rate
            );
            printf("  channels: %i\n",
                audio_mode[i]->channels
            );
            printf("  sample size: %i bits\n",
                audio_mode[i]->sample_size
            );
	    if(audio_mode[i]->allow_fragmenting)
	    {
                printf("  sound buffer size: %i bytes\n",
                    audio_mode[i]->fragment_size_bytes
                );
                printf("  sound buffer fragments: %i\n",
                    audio_mode[i]->num_fragments
                );
	    }
	    printf("\n");
	 }
	 YFreeAudioModesList(audio_mode, total);
	}

	/*
	 *	Disconnect from YIFF server
	 */
	YCloseConnection(con, False);
	con = NULL;

	free(con_arg);
	con_arg = NULL;


	return(0);
}
