/*
                         Program Primary Routines

	Functions:

	int antishift(int in)
	void CRSanitize(char *s)

	void YiffSignalHandler(int sig)
	void YiffResetMixer(void)

	int YiffInit(int argc, char **argv)
	void YiffShutdown(void)

	int YiffCheckNewConnection(int socket)
	void YiffCloseConnection(YConnectionNumber con_num)
	void YiffManageConnections(void)

	int YiffCreatePlaystack(
		char *path,
		YConnectionNumber owner,
		YID yid,
		YDataPosition pos,
		Coefficient volume_left,
                Coefficient volume_right,
		int repeats
	)
	void YiffDestroyPlaystack(int n)
	void YiffManageSound(void)

	void YiffUpdateTimmers(void)
	void YiffResetTimmers(void)

	int main(int argc, char *argv[])

	---

 */

#include <stdio.h>
#include <db.h>
#include <sys/types.h> 
#include <malloc.h>
#include <string.h>
#include <errno.h>
extern int errno;
#include <unistd.h>
#include <signal.h>

#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>		/* For inet_ntoa. */
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/disk.h"
#include "ymode.h"
#include "yhost.h"
#include "ynet.h"
#include "ymixer.h"
#include "rcfile.h"
#include "yiff.h"
#include "options.h"
#include "config.h"



#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define MAX(a,b)	(((a) > (b)) ? (a) : (b))

#define MINSC(a,b)	((a < b) ? (char)127 : (char)(a))
#define MAXSC(a,b)	((a > b) ? (char)-128 : (char)(a))


PlayStack **playstack;
int total_playstacks;

SoundPath **soundpath;
int total_soundpaths;

YConnection **yconnection;
int total_yconnections;

YMode **ymode;
int total_ymodes;

YHost **yhost;   
int total_yhosts;

yiff_option_struct option;
yiff_stats_struct ystats;
yiff_fname_struct fname;

int runlevel;
YTime cur_time;

yiff_next_struct next;

Recorder *recorder;

int listen_socket;



/*
 *	Local timmings:
 */
time_t next_sound_delta_us;	/* In microseconds. */



/*
 *	Local functions:
 */
int antishift(int in);
void CRSanitize(char *s);


/*
 *	Anti shift macro:
 */
int antishift(int in)
{
        int s, i;

        for(s = 0, i = in; i; i >>= 1)
            s++;

        return(s);
}


/*
 *	Replaces occurances of \r with \n.
 */
void CRSanitize(char *s)
{
	char replace = '\r';	/* Replace with this char. */


	if(s == NULL)
	    return;

	s = strchr(s, replace);
	while(s != NULL)
	{
	    *s = '\n';
	    s = strchr(s, replace);
	}

	return;
}


/*
 *	Procedure to perform when a SIGHUP is recieved.
 */
void YiffHangupHandle(void)
{
	int i;


	/* Reload configuration. */
	RCLoadFromFile(fname.rcfile);


        /* Delete all playstacks and notify owner connections about it. */
        for(i = 0; i < total_playstacks; i++)
            YiffDestroyPlaystack(i);


	/* Close sound device. */
        YSoundShutdown(recorder);


	/*   Do not free recorder (other functions may be reffering to it
	 *   during threaded execution of this function.
	 */


	/* Initialize recorder. */
	if(YSoundInit(recorder, &option.audio))
	{
	    fprintf(stderr,
"Error: Unable to initialize recorder.\n\
To try again type: `kill -s HUP %i'\n\
To terminate type: `kill -s INT %i'\n",
		PROG_NAME, PROG_VERSION,
		ystats.pid,
                ystats.pid
	    );
	    return;
	}

        /* Load mixer settings from file. */
        if(MixerRCLoadFromFile(fname.mixerrcfile, recorder))
        {
            /* Could not load settings from file, so reset mixer
             * values.
             */
            YiffResetMixer();
        }

        /* Update timmers (to help sync audio device). */
        YiffUpdateTimmers(); 

        /* Need to sync audio device right after initialization. */
        YSoundSync(recorder, 0);  

        /* Reset global next sound delta. */
        next_sound_delta_us = 0;


        /* Schedual next refresh. */
        next.refresh.ms = cur_time.ms + option.refresh_int.ms;
        next.refresh.us = 0;


	return;
}

/*
 *	Signal handler.
 */
void YiffSignalHandler(int sig)
{
	switch(sig)
	{
	  case SIGHUP:
	    signal(SIGHUP, YiffSignalHandler);
	    YiffHangupHandle();
	    break;

	  case SIGINT:
            runlevel = 1;
            break;

          case SIGKILL:  
            runlevel = 1;
            break;

          case SIGPIPE:
            signal(SIGPIPE, YiffSignalHandler);
            break;

          case SIGSEGV:
            runlevel = 1;
            break;

	  case SIGTERM:
	    runlevel = 1;
	    break;

	  default:
	    break;
	}

	return;
}


/*
 *	Procedure to reset mixer values.
 */
void YiffResetMixer(void)
{
	if(recorder == NULL)
	    return;


	YMixerSet(recorder, YMixerCodeBass,    0.70, 0.70);
        YMixerSet(recorder, YMixerCodeCD,      0.70, 0.70);
        YMixerSet(recorder, YMixerCodeGainIn,  0.70, 0.70);
        YMixerSet(recorder, YMixerCodeGainOut, 0.70, 0.70);

        YMixerSet(recorder, YMixerCodeLine,    0.70, 0.70);
        YMixerSet(recorder, YMixerCodeLine1,   0.70, 0.70);
        YMixerSet(recorder, YMixerCodeLine2,   0.70, 0.70);
        YMixerSet(recorder, YMixerCodeLine3,   0.70, 0.70);

        YMixerSet(recorder, YMixerCodeMic,     0.20, 0.20);
        YMixerSet(recorder, YMixerCodeMix,     0.70, 0.70);
        YMixerSet(recorder, YMixerCodePCM,     0.70, 0.70);

        YMixerSet(recorder, YMixerCodePCM2,    0.70, 0.70);

        YMixerSet(recorder, YMixerCodeRec,     0.20, 0.20);
        YMixerSet(recorder, YMixerCodeSpeaker, 0.70, 0.70);
        YMixerSet(recorder, YMixerCodeSynth,   0.70, 0.70);
        YMixerSet(recorder, YMixerCodeTreble,  0.70, 0.70);
        YMixerSet(recorder, YMixerCodeVolume,  0.70, 0.70);


	return;
}



/*
 *	Initializes the program, should call this function
 *	at the very start.
 */
int YiffInit(int argc, char **argv)
{
	int i, n, status;
	char *strptr;
	struct sockaddr_in addr;
	char cwd[PATH_MAX];
	struct stat stat_buf;

	int override_device = 0;
	char device[PATH_MAX + NAME_MAX];
        int override_mixer = 0;
	char mixer[PATH_MAX + NAME_MAX];
	int override_port = 0;
	int port = -1;
	YIPUnion ip;


	getcwd(cwd, PATH_MAX);
	cwd[PATH_MAX - 1] = '\0';


	/* Reset global variables. */
	option.debug = False;
	option.port = DEF_PORT;

        option.audio.cycle_set = CYCLE_SET_USER;

        option.audio.cycle.ms = 30;
        option.audio.cycle.us = 0;

        option.audio.compensated_cycle.ms = option.audio.cycle.ms;
        option.audio.compensated_cycle.us = option.audio.cycle.us;

        option.audio.write_ahead.ms = 45;
        option.audio.write_ahead.us = 0;

        option.audio.cycle_ahead_left.ms = 0;
        option.audio.cycle_ahead_left.us = 0;

        option.audio.cumulative_latency.ms = 0;
        option.audio.cumulative_latency.us = 0;

        option.audio.sample_size = DEF_SAMPLE_SIZE;
        option.audio.channels = DEF_CHANNELS;
        option.audio.sample_rate = DEF_SAMPLE_RATE;

        option.audio.bytes_per_second = 0;	/* Recalculated later
						 * in YSoundInit().
						 */

#ifdef OSS_BUFFRAG
        option.audio.allow_fragments = True;
        option.audio.num_fragments = 2;
        option.audio.fragment_size = antishift(512 - 1);
#endif  /* OSS_BUFFRAG */

        option.audio.flip_stereo = False;
        option.audio.direction = AUDIO_DIRECTION_PLAY;

	option.refresh_int.ms = 60000;		/* 1 minute. */
	option.refresh_int.us = 0;

	option.midi_play_cmd = StringCopyAlloc(DEF_MIDI_CMD);

#ifdef ALSA
	option.midi_device_number = 0;
#endif	/* ALSA */


        strncpy(fname.rcfile, DEF_RCFILE_NAME, PATH_MAX + NAME_MAX);
        fname.rcfile[PATH_MAX + NAME_MAX - 1] = '\0';

	strptr = getenv("HOME");
	if(strptr == NULL)
	    strncpy(
		fname.mixerrcfile,
		PrefixPaths("/", DEF_RCMIXERFILE_NAME),
		PATH_MAX + NAME_MAX
	    );
	else
	    strncpy(
                fname.mixerrcfile,
                PrefixPaths(strptr, DEF_RCMIXERFILE_NAME),
                PATH_MAX + NAME_MAX
            );

	strncpy(fname.device, DEF_DEVICE, PATH_MAX + NAME_MAX);
	fname.device[PATH_MAX + NAME_MAX - 1] = '\0';

        strncpy(fname.mixer, DEF_MIXER, PATH_MAX + NAME_MAX);
        fname.mixer[PATH_MAX + NAME_MAX - 1] = '\0';


	ystats.pid = getpid();
	ystats.start_time = time(NULL);
	ystats.cycle_load = 0.00;


	/* Delete any existing sound paths. */
	SoundPathDeleteAll();

	/* Allocate current directory as first sound path. */
	i = SoundPathAllocate();
	if(SoundPathIsAllocated(i))
	{
	    free(soundpath[i]->path);
	    soundpath[i]->path = StringCopyAlloc(cwd);
	}


	/* Delete any existing yhosts. */
	YHostDeleteAll();

	/* Add localhost to list of available hosts (127.0.0.1). */
	ip.charaddr[0] = 127;
	ip.charaddr[1] = 0;
        ip.charaddr[2] = 0;
        ip.charaddr[3] = 1;
	YHostAllocate(&ip);


        /* ****************************************************** */
        /* Parse arguments. */
	for(i = 1; i < argc; i++)
	{
	    if(argv[i] == NULL)
		continue;

	    /* Print help. */
	    if(strcasepfx(argv[i], "--h") ||
               strcasepfx(argv[i], "-h") ||
               strcasepfx(argv[i], "-?")
	    )
	    {
		printf(PROG_HELP_MESG);
		return(-4);
	    }
            /* Print version. */
            else if(strcasepfx(argv[i], "--ver") ||
                    strcasepfx(argv[i], "-ver")
            )
            {
                printf("%s Version %s\n", PROG_NAME, PROG_VERSION);
		printf("%s\n", PROG_COPYRIGHT);
                return(-4);
            }
            /* Device. */
            else if(strcasepfx(argv[i], "--d") ||
                    strcasepfx(argv[i], "-d")
            )
            {
                i++;
                if(i < argc)
                {
		    strncpy(device, argv[i], PATH_MAX + NAME_MAX);
		    device[PATH_MAX + NAME_MAX - 1] = '\0';

		    override_device = 1;
		}
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
	    }
#ifdef ALSA
	    /* MIDI device port number. */
	    else if(strcasepfx(argv[i], "--midi_port"))
	    {
		i++;
                if(i < argc)
                {
		    option.midi_device_number = atoi(argv[i]);
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
	    }
#endif	/* ALSA */
            /* Mixer settings file. */
	    else if(strcasepfx(argv[i], "--mixer_rc") ||
                    strcasepfx(argv[i], "-mixer_rc")
	    )
	    {
                i++;
                if(i < argc)
                {
                    strncpy(fname.mixerrcfile, argv[i], PATH_MAX + NAME_MAX);
                    fname.mixerrcfile[PATH_MAX + NAME_MAX - 1] = '\0';
                }
                else
                {   
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
 	    }
            /* Mixer. */
            else if(strcasepfx(argv[i], "--m") ||
                    strcasepfx(argv[i], "-m")
            )
            {
                i++;
                if(i < argc)
                {
                    strncpy(mixer, argv[i], PATH_MAX + NAME_MAX);
                    mixer[PATH_MAX + NAME_MAX - 1] = '\0';

                    override_mixer = 1;
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
            }
            /* Port. */
            else if(strcasepfx(argv[i], "--po") ||
                    strcasepfx(argv[i], "-po")
            )
            {
                i++;
                if(i < argc)
                {
		    port = atoi(argv[i]);
                    override_port = 1;
                }   
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
            }
            /* Sound path. */
            else if(strcasepfx(argv[i], "--pa") ||
                    strcasepfx(argv[i], "-pa")
            )
            {
                i++;
                if(i < argc)
                {
		    n = SoundPathAllocate();
		    if(SoundPathIsAllocated(n))
		    {
			free(soundpath[n]->path);
			soundpath[n]->path = StringCopyAlloc(argv[i]);

			if(stat(soundpath[n]->path, &stat_buf))
			{
			    fprintf(stderr,
 "%s: Warning: No such directory.\n",
				soundpath[n]->path
			    );
			}
			else
			{
			    if(!S_ISDIR(stat_buf.st_mode))
                                fprintf(stderr,
 "%s: Warning: Not a directory.\n",
				    soundpath[n]->path
                                );
			}
		    }
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
            }
	    /* All else, it's the rcfile. */
	    else
	    {
		strncpy(fname.rcfile, argv[i], PATH_MAX + NAME_MAX);
		fname.rcfile[PATH_MAX + NAME_MAX - 1] = '\0';
	    }
	}

        /* ****************************************************** */
        /* Load configuration. */
	status = RCLoadFromFile(fname.rcfile);
	if(status)
	    return(-1);

	/*   Need to set current mode and options values to match
	 *   default Audio mode.
	 */
	n = 0;	/* Default audio mode is the first one. */
	if(YModeIsAllocated(n))
	{
            /* Set new Audio parameters to option's Audio from Audio
	     * mode's values.
	     */

	    /* Audio modes are always considered user set. */
            option.audio.cycle_set = CYCLE_SET_USER;

            option.audio.cycle.ms = ymode[n]->cycle.ms;
            option.audio.cycle.us = ymode[n]->cycle.us;

	    /* This will be recalculated by YSoundInit(). */
            option.audio.compensated_cycle.ms = option.audio.cycle.ms;
	    option.audio.compensated_cycle.us = option.audio.cycle.us;

            option.audio.write_ahead.ms = ymode[n]->write_ahead.ms;
            option.audio.write_ahead.us = ymode[n]->write_ahead.us;

	    /* This will be reset by YSoundInit(). */
            option.audio.cycle_ahead_left.ms = 0;
	    option.audio.cycle_ahead_left.us = 0;

	    /* This will be reset by YSoundInit(). */
            option.audio.cumulative_latency.ms = 0;
            option.audio.cumulative_latency.us = 0;

            option.audio.sample_size = ymode[n]->sample_size;
            option.audio.channels = ymode[n]->channels;
            option.audio.sample_rate = ymode[n]->sample_rate;

	    /* This will be recalculated by YSoundInit(). */
            option.audio.bytes_per_second = 0;

#ifdef OSS_BUFFRAG
            option.audio.allow_fragments = ymode[n]->allow_fragments;
            option.audio.num_fragments = ymode[n]->num_fragments;  
            option.audio.fragment_size = ymode[n]->fragment_size;  
#endif  /* OSS_BUFFRAG */

	    option.audio.flip_stereo = ymode[n]->flip_stereo;
            option.audio.direction = ymode[n]->direction;
	}
	else
	{
	    fprintf(
		stderr,
		"Warning: Audio mode `Default' not defined.\n"
	    );
	}

	/* Override device? */
	if(override_device)
	{
	    strncpy(fname.device, device, PATH_MAX + NAME_MAX);
	    fname.device[PATH_MAX + NAME_MAX - 1] = '\0';
	}
        /* Override mixer? */
        if(override_mixer)
        {
            strncpy(fname.mixer, mixer, PATH_MAX + NAME_MAX);
            fname.mixer[PATH_MAX + NAME_MAX - 1] = '\0';
        }
        /* Override port? */
        if(override_port)
        {
            option.port = port;
        }


        /* ****************************************************** */
        /* Set signals to watch. */
	signal(SIGHUP, YiffSignalHandler);
        signal(SIGINT, YiffSignalHandler);
        signal(SIGKILL, YiffSignalHandler);
        signal(SIGPIPE, YiffSignalHandler);
        signal(SIGSEGV, YiffSignalHandler);
	signal(SIGTERM, YiffSignalHandler);


        /* ****************************************************** */
	/* Reset timmers. */
	YiffResetTimmers();


        /* ****************************************************** */
	/* Allocate recorder structure and initialize sound. */
	recorder = (Recorder *)calloc(1, sizeof(Recorder));
	if(recorder == NULL)
	    return(-1);

	/* Initialize sound. */
	if(YSoundInit(recorder, &option.audio))
	    return(-1);


	/* Load mixer settings from file. */
	if(MixerRCLoadFromFile(fname.mixerrcfile, recorder))
	{
	    /* Could not load settings from file, so reset mixer
	     * values.
	     */
	    YiffResetMixer();
	}


	/* ****************************************************** */
	/* Set up listening socket. */
	listen_socket = socket(AF_INET, SOCK_STREAM, 0);
	if(listen_socket < 0)
            return(-1);

        addr.sin_family = AF_INET;
        addr.sin_port = htons(option.port);
        addr.sin_addr.s_addr = INADDR_ANY;
        memset(&addr.sin_zero, 0, 8);

        /* Bind the socket. */
        status = bind(
	    listen_socket,
            (struct sockaddr *)&addr,
            sizeof(struct sockaddr)
	);
	if(status)
            return(-1);

	status = listen(listen_socket, LISTEN_SOCKET_BACKLOG);
	if(status)
            return(-1);

        /* Set listening port socket to nonblocking. */
        fcntl(listen_socket, F_SETFL, O_NONBLOCK);


	/* ********************************************************** */
	/* Update timmers (to help sync audio device). */
	YiffUpdateTimmers();

	/* Need to sync audio device right after initialization. */
	YSoundSync(recorder, 0);

	/* Reset global next sound delta. */
	next_sound_delta_us = 0;


	/* schedual next refresh. */
        next.refresh.ms = cur_time.ms + option.refresh_int.ms;
	next.refresh.us = 0;


	return(0);
}



void YiffShutdown(void)
{
	int i;
	YConnection **con_ptr;


	/* Save mixer settings. */
	MixerRCSaveToFile(fname.mixerrcfile, recorder);


	/* Notify all connections about shutdown. */
	for(i = 0, con_ptr = yconnection;
            i < total_yconnections;
            i++, con_ptr++
	)
	{
	    if(*con_ptr == NULL)
		continue;

	    if((*con_ptr)->socket < 0)
		continue;

            YNetSendShutdown(i, 0);
	}


	/* Free all sound paths. */
	SoundPathDeleteAll();

	/* Free all play stacks. */
	PlayStackDeleteAll();

        /* Deallocate recorder structure and shutdown sound. */
	YSoundShutdown(recorder);
	free(recorder);
	recorder = NULL;


	/* Close listening socket. */
	if(listen_socket > -1)
	{
#ifndef SHUT_RDWR
# define SHUT_RDWR	2
#endif
	    shutdown(listen_socket, SHUT_RDWR);
	    listen_socket = -1;
	}

	/* Disconnect all connections. */
	ConnectionDeleteAll();

	/* Free all yhosts. */
	YHostDeleteAll();

	/* Free all ymodes. */
	YModeDeleteAll();


	/* Free midi play command. */
	free(option.midi_play_cmd);
	option.midi_play_cmd = NULL;


	return;
}


/*
 *	Check for and handle any incoming connections.
 */
int YiffCheckNewConnection(int socket)
{
	int i;
	YConnection *con_ptr;

        struct timeval timeout;
        fd_set readfds;
        int new_socket;

        int sin_size;
        struct sockaddr_in foreign_addr;
	char sndbuf[256];


        timeout.tv_sec = 0;
        timeout.tv_usec = 0;
        FD_ZERO(&readfds);
        FD_SET(socket, &readfds);
        select(socket + 1, &readfds, NULL, NULL, &timeout);
  
        if(!FD_ISSET(socket, &readfds))
            return(0);

        /* Handle new connection. */
        sin_size = sizeof(struct sockaddr_in);
        new_socket = accept(
            socket,
            (struct sockaddr *)&foreign_addr,
            &sin_size
        );
        if(new_socket == -1)
            return(-1);
	/* Set new socket non-blocking. */
        fcntl(new_socket, F_SETFL, O_NONBLOCK);


	/* Allocate a new connection structure. */
	i = ConnectionAllocate();
	con_ptr = ConnectionGetPtr(i);
	if(con_ptr == NULL)
	{
	    close(new_socket);
	    new_socket = -1;

	    return(-1);
	}	


	/* Set up connection. */
	con_ptr->socket = new_socket;
	con_ptr->ip.whole = foreign_addr.sin_addr.s_addr;

/*
printf("Server: Got connection from %i.%i.%i.%i.\n",
 con_ptr->ip.charaddr[0],
 con_ptr->ip.charaddr[1],
 con_ptr->ip.charaddr[2],
 con_ptr->ip.charaddr[3]
);
*/

	/* Is incoming connection's host allowed? */
	if(!YHostInList(&con_ptr->ip))
	{
/*
	    fprintf(
 "Y Server: Connection %i.%i.%i.%i denied, not in allowed YHosts list.\n",
		 con_ptr->ip.charaddr[0],
                 con_ptr->ip.charaddr[1],
                 con_ptr->ip.charaddr[2],
                 con_ptr->ip.charaddr[3]
	    );
 */

	    /* Send disconnect reason. */
	    YNetSendDisconnect(i, 0);

	    /* Close and delete connection. */
	    close(con_ptr->socket);
	    con_ptr->socket = -1;
	    ConnectionDelete(i);
	}


	return(0);
}


/*
 *	Closes connection con_num and deallocates its owned
 *	resources.
 */
void YiffCloseConnection(YConnectionNumber con_num)
{
	int i;
	YConnection *con_ptr;
	char sndbuf[512];
	PlayStack **ps_ptr;


	con_ptr = ConnectionGetPtr(con_num);
	if(con_ptr == NULL)
	    return;


	/* Delete all playstacks associated with this connection. */
	for(i = 0, ps_ptr = playstack;
            i < total_playstacks;
            i++, ps_ptr++
	)
	{
	    if(*ps_ptr == NULL)
		continue;

	    if((*ps_ptr)->owner == con_num)
		YiffDestroyPlaystack(i);
	}


        /* Notify connection about disconnect. */
        if(con_ptr->socket > -1)
            YNetSendDisconnect(con_num, 0);

	/* Close connection. */
	if(con_ptr->socket > -1)
	{
	    close(con_ptr->socket);
	    con_ptr->socket = -1;
	}


	/*   Delete this connection structure and all its allocated
         *   resources.
         */
	ConnectionDelete(con_num);


	return;
}

/*
 *	Check for incoming connections, handle them and
 *	manage each existing connection.
 */
void YiffManageConnections(void)
{
	int i;
	YConnection **ptr;


        /*   Check for and handle any new incoming
	 *   connections.
	 */
	YiffCheckNewConnection(listen_socket);


        /* Handle each connection for incoming data. */
        for(i = 0, ptr = yconnection;
            i < total_yconnections;
            i++, ptr++
        )
	{
	    if(*ptr == NULL)
		continue;

	    if((*ptr)->socket < 0)
		continue;

	    YNetRecv(i);
	}


	return;
}



/*
 *	Attempts to allocate a new playstack, if successful,
 *	then sets it up to the given values and loads it according
 *	to the file specified in path.
 *
 *	owner is the connection number that is to own this
 *	playstack.
 */
int YiffCreatePlaystack(
	char *path,
	YConnectionNumber owner,
	YID yid,
	YDataPosition pos,
	YVolumeStruct *volume,
	int repeats
)
{
	int i;
	PlayStack *ps_ptr, **ptr;
	int bytes_per_cycle;
	char sndbuf[256];


	if(!ConnectionIsConnected(owner))
	    return(-1);

	if((recorder == NULL) ||
           (path == NULL) ||
	   (yid == YIDNULL) ||
           (volume == NULL)
	)
	    return(-1);

	/* Allocate new playstack. */
	i = PlayStackAllocate();
	ps_ptr = PlayStackGetPtr(i);
	if(ps_ptr == NULL)
	{
            YNetSendSoundObjectPlay(
                owner,
                YIDNULL,	/* Implies play failure. */
                path,
                pos,
                volume,
                repeats
            );

	    return(-1);
	}

	ps_ptr->yid = yid;
	ps_ptr->owner = owner;
	ps_ptr->position = pos;
	ps_ptr->repeats = 0;
        ps_ptr->total_repeats = ((repeats <= 0) ? -1 : repeats);
	ps_ptr->data_length = 0;

	ps_ptr->volume_left = (Coefficient)volume->left /
	    (Coefficient)((u_int16_t)-1);
	ps_ptr->volume_right = (Coefficient)volume->right /
	    (Coefficient)((u_int16_t)-1);

	/* Get bytes per cycle. */
        bytes_per_cycle = recorder->sound.buffer_length;

	/* Open audio file. */
	if(
	    AFWOpen(
	        path,
	        &ps_ptr->afw_data
	    )
	)
	{
	    PlayStackDelete(i);

            YNetSendSoundObjectPlay(
                owner,
                YIDNULL,        /* Implies play failure. */
                path,
                pos,
                volume,
                repeats
            );

	    return(-1);
	}

	ps_ptr->data_length = ps_ptr->afw_data.entire_length;

	ps_ptr->block_align = ps_ptr->afw_data.block_align;
	ps_ptr->block_length = ps_ptr->afw_data.block_length;


        /* Check sound object's type. */
        if(ps_ptr->afw_data.sndobj_format == SndObjTypeDSP)
        {
            /* DSP Sound Object. */  

	    /* Load first segment. */
            AFWLoadSegment(
                &ps_ptr->afw_data,
                ps_ptr->position,
	        bytes_per_cycle,
		&recorder->audio
            );
	}
        else if(ps_ptr->afw_data.sndobj_format == SndObjTypeMIDI)
        {
            /* MIDI Sound Object. */

	    /* Check if another playstack is playing this sound. */
	    for(i = 0, ptr = playstack;
                i < total_playstacks;
                i++, ptr++
	    )
	    {
		if(*ptr == NULL)
		    continue;

		if((*ptr)->afw_data.sndobj_format != SndObjTypeMIDI)
		    continue;

                /* Skip this playstack. */
                if(*ptr == ps_ptr)
                    continue;

/* Let's let the AFW handle this and we respond properly to the AFW. */
	    }

            /* Begin playing MIDI. */
            AFWLoadSegment(
                &ps_ptr->afw_data,
                0,		/* Position n/a. */
                0,		/* Length n/a. */
		&recorder->audio
            );
        }


	/* Notify connection of successful create. */
        YNetSendSoundObjectPlay(
            owner,
            yid,
            path,
            pos,
            volume,
            repeats
        );


	return(0);
}

/*
 *	Destroys the playstack n, notifies the owner connection
 *	about it, and closes the reffered sound object on file.
 */
void YiffDestroyPlaystack(int n)
{
	PlayStack *ps_ptr;


	/* Check if playstack n is allocated. */
	if(PlayStackIsAllocated(n))
	    ps_ptr = playstack[n];
	else
	    return;

	/* Check if playstack owner connection is connected. */
	if(ConnectionIsConnected(ps_ptr->owner))
	{
	    /* Notify owner of this sound object being destroyed. */
            YNetSendSoundObjectKill(ps_ptr->owner, ps_ptr->yid);
	}

	/*   Delete this playstack, closing the audio file
	 *   and deallocating its resources.
	 */
	PlayStackDelete(n);


	return;
}

/*
 *	Manage the recorder and the sound related resources.
 */
void YiffManageSound(void)
{
	int i, status;
	PlayStack *ps_ptr;
	int bytes_per_cycle;

	SoundBuffer *tar_buf, *src_buf, *tmp_ptr;
	YDataLength len;

	char sndbuf[1024];


	if(recorder == NULL)
	    return;

	if(1)
        {
	    /* Record or play? */
	    if(recorder->audio.direction == AUDIO_DIRECTION_RECORD)
	    {
		/* Record. */

	    }
	    else	/* AUDIO_DIRECTION_PLAY */
	    {
		/* Play. */

	        /* Play and reset midway buffer. */
	        YSoundPlayBuffer(recorder);

	        bytes_per_cycle = recorder->sound.buffer_length;
	        tar_buf = recorder->sound.buffer;

	        /* Manage each playstack. */
	        for(i = 0; i < total_playstacks; i++)
	        {
		    ps_ptr = playstack[i];

		    if(ps_ptr == NULL)
		        continue;

		    /* Check sound object type. */
		    if(ps_ptr->afw_data.sndobj_format == SndObjTypeDSP)
		    {
			/* DSP Sound Object. */

		        /* Mix to buffer. */
                        src_buf = ps_ptr->afw_data.buffer;
		        if(src_buf != NULL)
		        {
                            len = MIN(bytes_per_cycle,
                                      ps_ptr->afw_data.length
                            );

		            YiffMixBuffers(
			        &recorder->audio,
		                tar_buf,
			        src_buf,
			        len,
			        ps_ptr->volume_left,
                                ps_ptr->volume_right,
                                ps_ptr->volume_up,
                                ps_ptr->volume_down,
                                ps_ptr->volume_away
		            );
		        }

		        /* Increment position. */
		        ps_ptr->position += bytes_per_cycle;

			/* Position matched or exceeded data length? */
		        if(ps_ptr->position >= ps_ptr->data_length)
		        {
		            ps_ptr->position = 0;

		            /* Increment repeats. */
		            ps_ptr->repeats++;

		            /* Repeats exceeded? */
		            if((ps_ptr->total_repeats >= 0) &&
                               (ps_ptr->repeats >= ps_ptr->total_repeats)
		            )
		            {
				YiffDestroyPlaystack(i);
			        continue;
		            }
		        }
		        /* Read next buffer segment. */
		        if(
		            AFWLoadSegment(
		                &ps_ptr->afw_data,
		                ps_ptr->position,
			        bytes_per_cycle,
				&recorder->audio
		            )
		        )
		        {
                            /* Error loading audio data segment. */
                            YiffDestroyPlaystack(i);
                            continue;
			}
		    }
		    else if(ps_ptr->afw_data.sndobj_format == SndObjTypeMIDI)
		    {
		        /* MIDI Sound Object. */

                        /*   Call AFWLoadSegment() for MIDI Sound Objects,
			 *   this will check if it is still being played.
			 *   Returns:
			 *	0  on success
			 *	-1 on error
			 *	-2 stopped because another MIDI play started
			 *	-3 stopped normally.
			 */
                        status = AFWLoadSegment(
                            &ps_ptr->afw_data,
                            0,			/* Position n/a. */
                            0,			/* Length n/a. */
			    &recorder->audio
                        );
                        if(status == -1)
			{
			    /* General error, destroy it. */
                            YiffDestroyPlaystack(i);
                            continue;
			}
			else if(status == -2)
			{
			    /* Another play started, thus this one must stop. */
                            YiffDestroyPlaystack(i);
                            continue;
                        }
			else if(status == -3)
			{
			    /* Playback stopped normally, repeat as needed. */
                            ps_ptr->repeats++;

                            /* Repeats exceeded? */
                            if((ps_ptr->total_repeats >= 0) &&
                               (ps_ptr->repeats >= ps_ptr->total_repeats)
                            )
                            {
                                YiffDestroyPlaystack(i);
                                continue;
                            }

			    /* Begin playing midi object again. */
			    if(AFWLoadSegment(
                                &ps_ptr->afw_data,
                                0,	/* Position n/a. */
                                0,	/* Length n/a. */
				&recorder->audio
                            ))
			    {
				/* Definate error in playing MIDI object. */
                                YiffDestroyPlaystack(i);
                                continue;
			    }
                        }
			else
			{
			    /* Still playing MIDI object, do nothing. */
			}

		    }
		    else
		    {
			/* Unsupported sound object type. */
		    }
		}

                /* Shift target buffer from being signed 8 to unsigned 8. */
	        for(i = 0, tmp_ptr = tar_buf;
                    i < bytes_per_cycle;
                    i++
                )
	            *tmp_ptr++ = (int)(*tmp_ptr) + (int)128;

	    }
        }


        return;
}


/*
 *	Procedure to update timmers.
 */
void YiffUpdateTimmers(void)
{
	YTime new_millitime;
	time_t audio_cycle_us, cycle_ahead_us;


	GetCurrentTime(&new_millitime);
	if(new_millitime.ms < cur_time.ms)
	{
	    /*   Timmers need to be reset. Note that cur_time will
	     *   be updated in YiffResetTimmers();
	     */
	    YiffResetTimmers();
	}
	else
	{
	    if(recorder != NULL)
	    {
	        /* Get audio cycle in microseconds. */
		audio_cycle_us = (recorder->audio.compensated_cycle.ms *
                    1000) + recorder->audio.compensated_cycle.us;

		cycle_ahead_us = (recorder->audio.cycle_ahead_left.ms *
		    1000) + recorder->audio.cycle_ahead_left.us;

                /*
                 *   Calculate next_sound_delta_us, this is the
		 *   delta time from now untill the next sound
		 *   sound be played.
		 *
		 *   audio_device_cycle - (current_time -
		 *   time_audio_was_last_played) - cycle_ahead
		 *   - 100 us
		 */
	        next_sound_delta_us = audio_cycle_us -
                    (((new_millitime.ms * 1000) + new_millitime.us) -
                    ((cur_time.ms * 1000) + cur_time.us)) - cycle_ahead_us
                    - 100;

		/* Sanitize. */
		if(next_sound_delta_us > audio_cycle_us)
		    next_sound_delta_us = audio_cycle_us;
		if(next_sound_delta_us < 100)
		    next_sound_delta_us = 100;

		/* Stats: calculate cycle_load. */
		if(audio_cycle_us != 0)
		    ystats.cycle_load = (double)(audio_cycle_us -
		        next_sound_delta_us) /
		        (double)audio_cycle_us;

/*
if(cycle_ahead_us > 0) 
        fprintf(stderr,
            "cycle_ahead_us: %ld us\n",
            cycle_ahead_us
        );
 */

                /* Decrease cycle_ahead_left as needed. */
                cycle_ahead_us = MAX(cycle_ahead_us -
                    ((recorder->audio.compensated_cycle.ms * 1000) +
			recorder->audio.compensated_cycle.us),
                    0
                );
		recorder->audio.cycle_ahead_left.ms = cycle_ahead_us / 1000;
		recorder->audio.cycle_ahead_left.us = cycle_ahead_us % 1000;


/*
fprintf(stderr, "===> %ld (%ld %ld) us\n",
 next_sound_delta_us,
 recorder->audio.compensated_cycle.ms,
(((new_millitime.ms * 1000) + new_millitime.us) -
                    ((cur_time.ms * 1000) + cur_time.us))
);
*/

	    }
	    else
	    {
		/*   No audio device connected, set a reasonable
		 *   value for next_sound_delta_us since it is
		 *   used for the main loop usleep()
		 */
		next_sound_delta_us = 8000;

                /* Status: calculate cycle_load. */
                ystats.cycle_load = 0.00;
	    }


	    /* Update cur_time as usual. */
	    cur_time.ms = new_millitime.ms;
	    cur_time.us = new_millitime.us;
	}


	return;
}


/*
 *	Resets all timmers to 0.
 */
void YiffResetTimmers(void)
{
	/* Directly get current time (for syncing audio device). */
	GetCurrentTime(&cur_time);

	/* Reset next schedual timmers. */
	next.refresh.ms = 0;
	next.refresh.us = 0;


        /* Sync audio device. */
        YSoundSync(recorder, 0);

	/* How long to sleep before playing sound again. */
	next_sound_delta_us = 10;	/* Use minimum. */


	return;
}



int main(int argc, char *argv[])
{
	int status;


	runlevel = 1;

	/* Initialize everything. */
	status = YiffInit(argc, argv);
	switch(status)
	{
	  case -4:
	    YiffShutdown();
	    return(0);
	    break;

	  case 0:
	    break;

	  default:
	    YiffShutdown();
	    return(1);
	    break;
	}

	/*   Main Y server loop:
	 *
	 *   |--+----------+-------------------------------|
	 *   A  B          C                               D
	 *
	 *   At point A the curent time is fetched (GetCurrentTime())
	 *   and then the Sound buffer (already mixed and ready to play)
	 *   is played (YiffManageSound()). During the call to
	 *   YiffManageSound(); first the Sound buffer is played, then
	 *   it is immediatly mixed again and thus ready for the next
	 *   play.
	 *
	 *   At point B, the YiffManageSound() call has finished and
	 *   the call to YiffManageConnections() takes place. This will
	 *   handle any client programs connected to us that have sent
	 *   data.
	 *
	 *   At point C, all the management and maintainance (as needed)
	 *   calls are performed. Also at point C the function
	 *   YiffUpdateTimmers() is called. This function updates all
	 *   timming and calculates how long it took to process from
	 *   point A to point C, subtracts that amount from the
	 *   `cycle' interval and sets the value into variable
	 *   next_sound_delta_us.  usleep() is then called to sleep
	 *   for next_sound_delta_us microseconds. The call to
	 *   YiffUpdateTimmers() needs to be as fast as possible to get
	 *   the most accurate next_sound_delta_us. Process sleeps
	 *   untill point D.
	 *
	 *   At point D, the execution loops back to point A.
	 */
	runlevel = 2;
	while(runlevel > 1)
	{
	    /* Get new time just after usleep(). */
	    GetCurrentTime(&cur_time);


	    /* Manage sound. */
            YiffManageSound();

	    /* Manage connections. */
            YiffManageConnections();


	    /* Time to refresh resources? */
	    if(next.refresh.ms < cur_time.ms)
	    {
		/* Refresh resources. */
		if(recorder != NULL)
		{
		    /* Sync audio device. */
		    YSoundSync(recorder, 0);
		}
		next.refresh.ms = cur_time.ms + option.refresh_int.ms;
	    }

	    /* Update timmings. */
            YiffUpdateTimmers();  

	    /* Sleep for a calculated amount of time, this time is
	     * calculated by the above call to YiffUpdateTimmers().
	     */
	    usleep(next_sound_delta_us);
	}

	/* Shutdown everything. */
	YiffShutdown();

	runlevel = 0;

	return(0);
}
