/*								-*- C++ -*-
 * $Id: WIN_canvas.cpp,v 1.3 1997-01-15 14:58:57+01 mho Exp $
 *
 * Purpose: canvas panel item
 *
 * Authors: Markus Holzem and Julian Smart
 *
 * Copyright: (C) 1995, AIAI, University of Edinburgh (Julian)
 * Copyright: (C) 1995, GNU (Markus)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Additionally everyone using this library has to announce it with:
 *
 *   This software uses the wxWindows-Xt GUI library
 *   (C) Markus Holzem, available via
 *       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
 */

#ifdef __GNUG__
#pragma implementation "WIN_canvas.h"
#endif

#define  Uses_XtIntrinsic
#define  Uses_wxCanvas
#include "wx.h"
#define  Uses_EnforcerWidget
#define  Uses_ScrollWinWidget
#define  Uses_CanvasWidget
#include <widgets.h>

//-----------------------------------------------------------------------------
// create and destroy canvas
//-----------------------------------------------------------------------------

IMPLEMENT_DYNAMIC_CLASS(wxCanvas, wxItem)

wxCanvas::wxCanvas(void) : wxItem()
{
    __type = wxTYPE_CANVAS;

    hsize   = vsize   = 0;	// virtual size of canvas
    vphsize = vpvsize = 0;	// viewport size of canvas (client size)
    hunits  = vunits  = 0;	// amount to scroll on arrows
    hpage   = vpage   = 0;	// amount to scroll on space below/above thumb
    hstart  = vstart  = 0;	// start of canvas inside viewport
    hscroll = vscroll = TRUE;	// do smooth scrolling

    quoted = FALSE;
}

wxCanvas::wxCanvas(wxWindow *parent, int x, int y, int width, int height,
		   int style, Constdata char *name) : wxItem()
{
    __type = wxTYPE_CANVAS;

    hsize   = vsize   = 0;	// virtual size of canvas
    vphsize = vpvsize = 0;	// viewport size of canvas (client size)
    hunits  = vunits  = 0;	// amount to scroll on arrows
    hpage   = vpage   = 0;	// amount to scroll on space below/above thumb
    hstart  = vstart  = 0;	// start of canvas inside viewport
    hscroll = vscroll = TRUE;	// do smooth scrolling

    quoted = FALSE;

    Create(parent, x, y, width, height, style, name);
}

Bool wxCanvas::Create(wxWindow *panel, int x, int y, int width, int height,
		      int style, Constdata char *name)
{
    if (!panel || !panel->IsKindOf(CLASSINFO(wxPanel)))
	wxFatalError("parent has to be a wxFrame, wxPanel, or any subtype",
		     "wxCanvas");

    ChainToPanel((wxPanel*)panel, style, name);

    // create frame
    FWidget() = XtVaCreateManagedWidget
	("Enforcer", xfwfEnforcerWidgetClass, GetParentWidget(parent),
	 XtNalignment,          XfwfTop,
	 XtNbackground,         bg.GetPixel(&cmap),
	 XtNforeground,         label_fg.GetPixel(&cmap),
	 XtNfont,               label_font.GetInternalFont(),
	 XtNhighlightThickness, 2,
	 XtNlabelOffset,        2,
	 XtNtraversalKeys,      XfwfTraverseKeyTab,
	 NULL);
    // create scrolled area
    PWidget() = XtVaCreateManagedWidget
	("ScrolledWindow", xfwfScrolledWindowWidgetClass, FWidget(),
	 XtNhighlightThickness, 0,
	 XtNbackground, bg.GetPixel(&cmap),
	 XtNhideHScrollbar, TRUE,
	 XtNhideVScrollbar, TRUE,
	 XtNscrollingPolicy, (style & wxBACKINGSTORE ?
			      XfwfUserScrolling : XfwfVirtualScrolling),
	 XtNtraversalKeys,  XfwfTraverseKeyTab,
	 NULL);
    // create canvas
    HWidget() = XtVaCreateManagedWidget
	(name, xfwfCanvasWidgetClass, PWidget(),
	 XtNbackingStore, ((style & wxREDUCED_BACKINGSTORE) ||
			   (style & wxBACKINGSTORE)) ? Always : NotUseful,
	 XtNborderWidth,  0,
	 NULL);
    // Initialize CanvasDC and draw white background
    CreateDC(); SetBackground(wxWHITE_BRUSH);
    // initial frame size
    hsize = (width  > -1 ? width  : wxCANVAS_WIDTH);
    vsize = (height > -1 ? height : wxCANVAS_HEIGHT);
    ((wxPanel*)panel)->PositionItem(this, x, y, hsize, vsize);
    // initial client size
    vphsize = wxMax(0, hsize - 12); // 12 = 2 * 6 is the usual border
    vpvsize = wxMax(0, vsize - 12); // width of a canvas
    // add event handlers
    AddEventHandlers();
    if ( !(style & wxBACKINGSTORE) )
	XtAddCallback(PWidget(), XtNresizeCallback,
		      (XtCallbackProc)wxCanvas::ResizeProc,
		      (XtPointer)this);
    // ready
    return TRUE;
}

//-----------------------------------------------------------------------------
// overriden parent functions
//-----------------------------------------------------------------------------

void wxCanvas::GetClientSize(int *width, int *height)
{
    if (width)  *width  = vphsize;
    if (height) *height = vpvsize;
}

void wxCanvas::SetClientSize(int WXUNUSED(x), int WXUNUSED(y),
			     int width, int height)
{
    // get size of frame to compute difference betwwen size and client size
    Position dummy; Dimension hInner, vInner; int hOuter, vOuter;
    XfwfCallComputeInside(PWidget(), &dummy, &dummy, &hInner, &vInner);
    GetSize(&hOuter, &vOuter);
    // adjust new size and client size
    if (width > 0) {
	vphsize  = width;
	width   += hOuter - hInner;
    }
    if (height > 0) {
	vpvsize  = height;
	height  += vOuter - vInner;
    }
    // adjust frame size
    SetSize(width, height);
}

void wxCanvas::SetSize(int x, int y, int width, int height, int flags)
{
    // keep size of viewport in vphsize/vpvsize
    Position dummy; Dimension hInner, vInner; int hOuter, vOuter;
    XfwfCallComputeInside(PWidget(), &dummy, &dummy, &hInner, &vInner);
    GetSize(&hOuter, &vOuter);
    // set widget size
    wxItem::SetSize(x, y, width, height, flags);
    // check for fit into frame
    if (flags & wxSIZE_FIT_ONE)
	GetSize(&width, &height);
    if (width  > 0) vphsize = width  - (hOuter - hInner);
    if (height > 0) vpvsize = height - (vOuter - vInner);
}

//-----------------------------------------------------------------------------
// handle scrollbars, pointer, and virtual size
//-----------------------------------------------------------------------------

void wxCanvas::EnableScrolling(Bool x, Bool y)
{
    hscroll = x; vscroll = y;
}

int wxCanvas::GetScrollPage(int orient)
{
    return ((orient == wxHORIZONTAL) ? hpage : vpage);
}


void wxCanvas::GetScrollPixelsPerUnit(int *x_units, int *y_units)
{
    if (x_units) *x_units = hunits;
    if (y_units) *y_units = vunits;
}

void wxCanvas::GetScrollUnitsPerPage(int *x, int *y)
{
    if (x) *x = hpage;
    if (y) *y = vpage;
}

void wxCanvas::GetVirtualSize(int *x, int *y)
{
    if (x) *x = hsize;
    if (y) *y = vsize;
}

Bool wxCanvas::IsRetained(void)
{
    return FALSE;
}

void wxCanvas::Scroll(int x, int y)
{
    if ( !((x == -1) || (x == hstart)) || !((y == -1) || (y == vstart)) ) {
	int  oldx = hstart;
	int  oldy = vstart;
	Bool smooth_scrolling = FALSE;

	if (hunits && x > -1) {
	    hstart           = x;
	    smooth_scrolling = (smooth_scrolling || hscroll);
	}
	if (vunits && y > -1) {
	    vstart           = y;
	    smooth_scrolling = (smooth_scrolling || vscroll);
	}
	// adjust hstart, vstart to fit into canvas
	if (hunits) {
	    hstart = wxMin(hstart, (hsize + hunits - vphsize - 1) / hunits);
	    hstart = wxMax(0, hstart);
	} else
	    hstart = 0;
	if (vunits) {
	    vstart = wxMin(vstart, (vsize + vunits - vpvsize - 1) / vunits);
	    vstart = wxMax(0, vstart);
	} else
	    vstart = 0;
	// move/redraw canvas if coordinates changed
	if ( (oldx != hstart) || (oldy != vstart) ) {
	    // when the canvas uses backing store is has to behave like wxItem
	    if (style & wxBACKINGSTORE) {
		wxItem::Scroll(hstart*hunits, vstart*vunits);
	    } else {
		SetDeviceOrigin(float(-hstart*hunits), float(-vstart*vunits));
		if (smooth_scrolling) {
		    // do a XCopyArea for the visible part
		    // expose is generated automatically by the widget
		    XfwfScrollCanvas(HWidget(),
				     (oldx-hstart)*hunits, (oldy-vstart)*vunits,
				     GetDC()->BgGC());
		} else {
		    // refresh complete window
		    wxRectangle rect(0, 0, vphsize, vpvsize);
		    Refresh(TRUE, &rect);
		}
	    }
	}
    } else if (!hscroll || !vscroll) {
	wxRectangle rect(0, 0, vphsize, vpvsize);
	Refresh(TRUE, &rect);
    }
    if ( !(style & wxBACKINGSTORE) ) {
	// set scrollbars
	//TOMSMOD was causing divide by zero if hsize, vsize = 0;
        float hs = (hsize==0?0:float(vphsize)/hsize);
        float vs = (vsize==0?0:float(vpvsize)/vsize);
	XfwfSetScrolledWindow(PWidget(),
			      hstart*hunits, vstart*vunits,
			      hsize, vsize,
			      hs, vs);
	//ENDMOD
    }
}

void wxCanvas::SetScrollbars(int _hunits,  int _vunits,  int _hlen, int _vlen,
			     int _hpage, int _vpage, int x, int y,
			     Bool setVirtualSize)
{
    if (_hunits > 0) {
	hunits = _hunits;
	hsize  = _hunits * _hlen;
	hpage  = _hpage;
    } else
	hsize = hunits = hpage = hstart = 0;
    if (_vunits > 0) {
	vunits = _vunits;
	vsize  = _vunits * _vlen;
	vpage  = _vpage;
    } else
	vsize = vunits = vpage = vstart = 0;

    // when the canvas uses backing store is has to bahave like wxItem
    if (style & wxBACKINGSTORE) {
	setVirtualSize = TRUE;
    }
    // set scrollbars
    Bool old_hscroll = hscroll; hscroll = FALSE;
    Bool old_vscroll = vscroll; vscroll = FALSE;
    wxItem::SetScrollbars(_hunits, _vunits, _hlen, _vlen, _hpage, _vpage, x, y,
			  setVirtualSize);
    hscroll = old_hscroll;
    vscroll = old_vscroll;
}

void wxCanvas::SetScrollPage(int orient, int range)
{
    if (orient == wxHORIZONTAL) {
	hpage = range;
	wxItem::SetScrollPage(orient, hpage * hunits);
    } else {
	vpage = range;
	wxItem::SetScrollPage(orient, vpage * vunits);
    }
}

void wxCanvas::SetVirtualSize(int _hsize, int _vsize)
{
    hsize = _hsize;
    vsize = _vsize;

    if (style & wxBACKINGSTORE) {
	wxItem::SetVirtualSize(_hsize, _vsize);
	return;
    } else {
	Scroll(-1, -1); // redisplay scrollbars
    }
}

void wxCanvas::ViewStart(int *x, int *y)
{
    if (x) *x = hstart;
    if (y) *y = vstart;
}

void wxCanvas::WarpPointer(int x, int y)
{
    if (HWidget() && XtWindow(HWidget()))
	if (style & wxBACKINGSTORE)
	    XWarpPointer(XtDisplay(HWidget()), None, XtWindow(HWidget()),
			 0, 0, 0, 0, x+hstart, y+vstart);
	else
	    XWarpPointer(XtDisplay(HWidget()), None, XtWindow(HWidget()),
			 0, 0, 0, 0, x, y);
}

//-----------------------------------------------------------------------------
// overriden event handlers
//-----------------------------------------------------------------------------

void wxCanvas::OnChar(wxKeyEvent &event)
{
    // quoted keys are processed immediately
    if (quoted) {
	quoted = FALSE;
	wxItem::OnChar(event);
	return;
    }
    // new and old position
    int x = -1, y = -1, vx, vy; ViewStart(&vx, &vy);
    // do quoting and scrolling
    if (event.ControlDown())
	switch (event.KeyCode()) {
	case 'q': case 'Q':  quoted = TRUE;            return;
	case 'l': case 'L':  Refresh();		       return;
	case WXK_LEFT:       x = wxMax(0, vx - hpage); break;
	case WXK_RIGHT:      x = vx + hpage;	       break;
	}
    else
	switch (event.KeyCode()) {
	case WXK_PRIOR:  y = wxMax(0, vy - vpage); break;
	case WXK_NEXT:   y = vy + vpage;	   break;
	case WXK_UP:     y = vy - 1;		   break;
	case WXK_DOWN:   y = vy + 1;		   break;
	case WXK_LEFT:   x = vx - 1;		   break;
	case WXK_RIGHT:  x = vx + 1;		   break;
	case WXK_HOME:   x = y = 0;		   break;
	case WXK_TAB:
	    XtCallActionProc(FWidget(), (event.ShiftDown() ?
					 "traversePrev" : "traverseNext"),
			     (XEvent*)event.eventHandle, NULL, 0);
	    return;
	}
    // scroll to x, y - may be -1, -1 if no scrolling is done
    Scroll(x, y);
}

void wxCanvas::OnScroll(wxCommandEvent& event)
{
    // scroll to new position. the unused scrollbar is -1
    if (WXSCROLLORIENT(event) == wxHORIZONTAL)
	Scroll(WXSCROLLPOS(event)/hunits, -1);
    else
	Scroll(-1, WXSCROLLPOS(event)/vunits);
}

//-----------------------------------------------------------------------------
// resize canvas widget when scrolled window is resized
//-----------------------------------------------------------------------------

void wxCanvas::ResizeProc(Widget WXUNUSED(w), wxCanvas *can, XRectangle* bounds)
{
    int newx = -1, newy = -1;

    // set size of viewport
    can->vphsize = bounds->width;
    can->vpvsize = bounds->height;
    // resize canvas widget to fit into scrolled window widget
    XtVaSetValues(can->HWidget(),
		  XtNwidth,  Dimension(can->vphsize),
		  XtNheight, Dimension(can->vpvsize),
		  NULL);
    // adjust hstart and hsize
    if (can->hunits) {
	if (can->hstart * can->hunits + can->vphsize > can->hsize)
	    newx = wxMax(0, (can->hsize - can->vphsize + can->hunits - 1)
			    / can->hunits);
    } else {
	can->hstart = 0;
	can->hsize  = can->vphsize;
    }
    // adjust vstart and vsize
    if (can->vunits) {
	if (can->vstart * can->vunits + can->vpvsize > can->vsize)
	    newy = wxMax(0, (can->vsize - can->vpvsize + can->vunits - 1)
		            / can->vunits);
    } else {
	can->vstart = 0;
	can->vsize  = can->vpvsize;
    }
    // check bounds
    can->hstart = wxMax(0, can->hstart);
    can->vstart = wxMax(0, can->vstart);
    can->hsize  = wxMax(1, can->hsize);
    can->vsize  = wxMax(1, can->vsize);
    // scroll and set scrollbars
    can->Scroll(newx, newy);
}
