/*
 * ===========================
 * VDK Visual Development Kit
 * Version 0.5
 * November 1998
 * ===========================
 *
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 */

#ifndef VDKBTREES_H
#define VDKBTREES_H
#ifdef NULL
#undef NULL
#define NULL 0x0000
#endif
// --------------------------
// Abstract  class template, 
// --------------------------
template<class T, class Node>
class AbstractBinaryNode {
public:
    AbstractBinaryNode() { }
    AbstractBinaryNode(const T& _object, 
               Node *_parent = NULL, 
               Node *_left = NULL, 
               Node *_right = NULL);
    virtual ~AbstractBinaryNode() { }

    // Subtree arrangement
    virtual Node *Clone(Node *_parent) const;
    virtual void RemoveSubtree();
    virtual Node *LeftRotate(Node *root);
    virtual Node *RightRotate(Node *root);
    virtual int CheckTreeProperties(const Node *);
    virtual int IsLeftChild() const;
    virtual int IsRightChild() const;
    // Adding a node and deleting
    virtual Node *add(const T& x);
    virtual Node *unlink(Node *z);

    // Find
    virtual Node *find(const T& x);
    virtual Node *findNearest(const T& x);

    // Tree trasverse
    virtual Node *Minimum();
    virtual Node *Maximum();
    virtual Node *Predecessor();
    virtual Node *Successor();

    // Miscellaneous
    virtual T *Object();

public:
    T object;
    Node *left, *right, *parent;

};

template<class T>
class BinaryNode : public AbstractBinaryNode<T, BinaryNode<T> > {
public:
    // Constructors and destructor
    BinaryNode() { }
    BinaryNode(const T& object, 
               BinaryNode<T> *parent = NULL, 
               BinaryNode<T> *left = NULL, 
               BinaryNode<T> *right = NULL):
        AbstractBinaryNode<T, BinaryNode<T> >(object, parent, left, right) { }
    virtual ~BinaryNode() { }
};

//////////////////////////////////////////////////////
// Different ways to iterate on the binary tree.

enum BtreeIteratorMode { BtMinKey, BtRootKey, BtMaxKey };

template<class T, class Node>
class AbstractBinaryTree {

public:

    AbstractBinaryTree();
    AbstractBinaryTree(const AbstractBinaryTree<T, Node>&);
    AbstractBinaryTree<T, Node>& operator=(const AbstractBinaryTree<T, Node>&);
    virtual ~AbstractBinaryTree();


    virtual void add(const T&);      // Add a node
    virtual void unlink(const T&);      // Remove a node
    virtual T *find(const T& q);
    virtual int IsEmpty() const { return root == NULL; }


    virtual Node *IteratorRoot() const;
    virtual Node *IteratorFind(const T& x) const;
    virtual int CheckTreeProperties();
    unsigned int size() { return count; }
    // Iterator class 
    class Iterator {

    public:
        Iterator(const AbstractBinaryTree<T, Node>& _tree, 
                 enum BtreeIteratorMode start = BtMinKey) 
        {
            tree = (AbstractBinaryTree<T, Node> *) (&_tree);
            StartAt(start);
        }

        // Start iterator over at the minimum, maximum or
        // root node of the binary tree.
        void StartAt(enum BtreeIteratorMode start) 
        {
            ptr = tree->IteratorRoot();
            if (start == BtMinKey)
                ptr = ptr->Minimum();
            else if (start == BtMaxKey)
                ptr = ptr->Maximum();
        }

        virtual ~Iterator() { }

        virtual void Previous() 
        {
            if (ptr)
                ptr = ptr->Predecessor();
        }

        virtual void Next() 
        {
            if (ptr)
                ptr = ptr->Successor();
        }

        virtual void Parent() 
        {
            if (ptr)
                ptr = ptr->parent;
        }

        virtual void find(const T& x) 
        {
            ptr = tree->IteratorFind(x);
        }

        virtual operator int() const 
        {
            return ptr != NULL;
        }

        // Dereferencing operator returns the object of the
        // node currently pointed to by the iterator.
        virtual T operator*() const {  return ptr->object; }
	virtual T current() const {  return ptr->object; }

	virtual Node* current_node() const
	  { return ptr; }
        // Object returns a pointer to the object of the
        // node currently pointed to (as opposed to returning
        // a copy of the node, as the dereferencing operator
        // above does).
        virtual const T *Object() const 
        {
            if (ptr)
                return &ptr->object;
            return (T*) NULL;
        }
        virtual T *Object() 
        {
            if (ptr)
                return &ptr->object;
            return NULL;
        }

        virtual void operator++() { Next(); }
        virtual void operator++(int) { Next(); }
        virtual void operator--() { Previous(); }
        virtual void operator--(int) { Previous(); }

    protected:
        Node *ptr;
        AbstractBinaryTree<T, Node> *tree;
    };

protected:
    Node *root;
    unsigned int count;
};

///////////////////////////////////////
template<class T>
class BinaryTree : public AbstractBinaryTree<T, BinaryNode<T> > {
public:
    BinaryTree() { }
};



// --------------------------------------------------------
// AbstractBinaryNode implementation.
// --------------------------------------------------------
template<class T, class Node>
AbstractBinaryNode<T, Node>::AbstractBinaryNode(const T& _object, 
                          Node *_parent,
                          Node *_left, 
                          Node *_right) 
{
    object = _object;
    parent = _parent;
    left = _left;
    right = _right;
}

template<class T, class Node>
Node *
AbstractBinaryNode<T, Node>::Clone(Node *_parent) const
{
    Node *ret = new Node( *((Node *) this));
    if (left)
        ret->left = left->Clone(ret);
    if (right)
        ret->right = right->Clone(ret);
    ret->parent = _parent;
    return ret;
}

template<class T, class Node> 
Node *
AbstractBinaryNode<T, Node>::LeftRotate(Node *root)
{
    Node *ret = root;
    Node *y = right;
    right = y->left;
    if (right)
        right->parent = (Node *) this;
    y->parent = parent;
    if (parent) {
        if (this == parent->left)
            parent->left = y;
        else
            parent->right = y;
    }
    else
        ret = y;
    y->left = (Node *) this;
    parent = y;
    return ret;
}

template<class T, class Node> 
Node *
AbstractBinaryNode<T, Node>::RightRotate(Node *root)
{
    Node *ret = root;
    Node *x = left;
    left = x->right;
    if (left)
        left->parent = (Node *) this;
    x->parent = parent;
    if (parent) {
        if (this == parent->left)
          parent->left = x;
        else
          parent->right = x;
    }
    else
        ret = x;
    x->right = (Node *) this;
    parent = x;
    return ret;
}

template<class T, class Node>
int
AbstractBinaryNode<T, Node>::IsLeftChild() const
{
    return (parent && parent->left == (Node *) this);
}

template<class T, class Node>
int
AbstractBinaryNode<T, Node>::IsRightChild() const
{
    return (parent && parent->right == (Node *) this);
}

template<class T, class Node>
Node *
AbstractBinaryNode<T, Node>::find(const T& x)
{
    Node *sc = (Node *) this;
    while (sc) {
        if (x == sc->object)
            return sc;
        if (x < sc->object)
            sc = sc->left;
        else
            sc = sc->right;
    }
    return NULL;
}

template<class T, class Node>
Node *
AbstractBinaryNode<T, Node>::findNearest(const T& x)
{
    Node *sc = (Node *) this;
    Node *prev = NULL;
    while (sc) {
        prev = sc;
        if (x < sc->object)
            sc = sc->left;
        else
            sc = sc->right;
    }
    return prev;
}

template<class T, class Node>
Node *
AbstractBinaryNode<T, Node>::add(const T& x) 
{
    Node *nearest = findNearest(x);
    if (x < nearest->object)
        nearest->left = new Node(x, nearest);
    else
        nearest->right = new Node(x, nearest);
    return (Node *) this;
}

template<class T, class Node>
Node *
AbstractBinaryNode<T, Node>::unlink(Node *z)
{
    Node *root = (Node *) this;
    Node *x, *y;
    if (! z)
        return root;
    if (! z->left || ! z->right)
        y = z;
    else
        y = z->Successor();
    if (y->left)
        x = y->left;
    else
        x = y->right;
    if (x)
        x->parent = y->parent;
    if (y->parent) {
        if (y == y->parent->left)
            y->parent->left = x;
        else
            y->parent->right = x;
    }
    else
        root = x;
    if (y != z)
        z->object = y->object;
    delete y;
    return root;
}

template<class T, class Node>
Node *
AbstractBinaryNode<T, Node>::Minimum()
{
    Node *sc = (Node *) this;
    while (sc && sc->left)
        sc = sc->left;
    return sc;
}

template<class T, class Node>
Node *
AbstractBinaryNode<T, Node>::Maximum()
{
    Node *sc = (Node *) this;
    while (sc && sc->right)
        sc = sc->right;
    return sc;
}

template<class T, class Node>
Node *
AbstractBinaryNode<T, Node>::Predecessor()
{
    if (left)
        return left->Maximum();
    Node *x = (Node *) this;
    Node *y = parent;
    while (y && x == y->left) {
        x = y;
        y = y->parent;
    }
    return y;
}

template<class T, class Node>
Node *
AbstractBinaryNode<T, Node>::Successor() 
{
    if (right)
        return right->Minimum();
    Node *x = (Node *) this;
    Node *y = parent;
    while (y && x == y->right) {
        x = y;
        y = y->parent;
    }
    return y;
}

template<class T, class Node>
void
AbstractBinaryNode<T, Node>::RemoveSubtree()
{
    if (left) {
        left->RemoveSubtree();
        delete  left;
    }
    if (right) {
        right->RemoveSubtree();
        delete right;
    }
}

template<class T, class Node>
T *
AbstractBinaryNode<T, Node>::Object()
{
    return &object;
}

template<class T, class Node>
int
AbstractBinaryNode<T, Node>::CheckTreeProperties(const Node *_parent)
{
    if (parent != _parent)
        return NULL;
    if (left) {
        if (object < left->object)
            return NULL;
        if (! left->CheckTreeProperties((Node *) this))
            return NULL;
    }
    if (right) {
        if (right->object < object)
            return NULL;
        if (! right->CheckTreeProperties((Node *) this))
            return NULL;
    }
    return 1;
}

// --------------------------------------------------------
// AbstractBinaryTree class template implementation.
// --------------------------------------------------------

template<class T, class Node>
AbstractBinaryTree<T, Node>::AbstractBinaryTree()
{
    root = NULL;
    count = NULL;
}

template<class T, class Node>
AbstractBinaryTree<T, Node>::AbstractBinaryTree(const AbstractBinaryTree<T, Node>& x)
{
    if (x.root)
        root = x.root->Clone(NULL);
    else
        root = NULL;
    count = x.count;
}

template<class T, class Node>
AbstractBinaryTree<T, Node>&
AbstractBinaryTree<T, Node>::operator=(const AbstractBinaryTree<T, Node>& x)
{
    if (root) {
        root->RemoveSubtree();
        delete root;
    }
    if (x.root)
        root = x.root->Clone(NULL);
    else
        root = NULL;
    count = x.count;
    return *this;
}

template<class T, class Node>
AbstractBinaryTree<T, Node>::~AbstractBinaryTree()
{
    if (root) {
        root->RemoveSubtree();
        delete root;
    }
} 

template<class T, class Node>
void
AbstractBinaryTree<T, Node>::add(const T& x)
{
  count++;
  if (root == NULL)
    root = new Node(x);
  else
    root = root->add(x);
}

template<class T, class Node>
Node *
AbstractBinaryTree<T, Node>::IteratorRoot() const
{
    return root;
}

template<class T, class Node>
Node *
AbstractBinaryTree<T, Node>::IteratorFind(const T& x) const
{
    return (root) ? root->find(x) : NULL;
}

template<class T, class Node>
void
AbstractBinaryTree<T, Node>::unlink(const T& _x)
{
  count--;
  if (! root)
    return;
  root = root->unlink(root->find(_x));
}

template<class T, class Node>
T *
AbstractBinaryTree<T, Node>::find(const T& q)
{
    Node *p = (root) ? root->find(q) : NULL;
    return (p) ? &p->object : NULL;
}

template<class T, class Node>
int
AbstractBinaryTree<T, Node>::CheckTreeProperties()
{
  if (root->CheckTreeProperties(NULL) == NULL)
    return NULL;
  return 1;
}

/////////////////////////////
// balanced binary trees 
// (using red & black trees)
/////////////////////////////
typedef enum RedBlack { Black, Red };
template<class T, class Node>
class AbstractRedBlackNode : public AbstractBinaryNode<T, Node> {
public:

  RedBlack clr;
  // Constructors.  Node always starts out red.
  AbstractRedBlackNode() { clr = Red; }
  AbstractRedBlackNode(const T& X, 
		       Node *P = NULL,
		       Node *L = NULL,
		       Node *R = NULL):
    AbstractBinaryNode<T, Node>(X, P, L, R) { }
  AbstractRedBlackNode(const T& X, RedBlack Clr, Node *P = NULL,
		       Node *L = NULL, Node *R = NULL):
    AbstractBinaryNode<T, Node>(X, P, L, R), clr(Clr) { }
  virtual ~AbstractRedBlackNode() { }

    // Tree manipulations used during insertion and deletion
    virtual Node *RemoveFixup(Node *x, Node *p);
    
    // Operations defined on binary trees.  All run in O(lgN) time.
    virtual Node *add(const T& AddMe);
    virtual Node *unlink(Node *z);
    
    // Returns NULL if the red-black invariant holds.
    virtual int CheckTreeProperties(const Node *);
};

template<class T>
class RedBlackNode : public AbstractRedBlackNode<T, RedBlackNode<T> > {
public:

    // Constructors.  Node always starts out red.
    RedBlackNode() { }
    RedBlackNode(const T& X, 
                 RedBlackNode<T> *P = NULL,
                 RedBlackNode<T> *L = NULL,
                 RedBlackNode<T> *R = NULL):
        AbstractRedBlackNode<T, RedBlackNode<T> >(X, P, L, R) { }
    RedBlackNode(const T& X, RedBlack Clr, RedBlackNode<T> *P = NULL,
            RedBlackNode<T> *L = NULL, RedBlackNode<T> *R = NULL):
        AbstractRedBlackNode<T, RedBlackNode<T> >(X, Clr, P, L, R) { }
    virtual ~RedBlackNode() { }
};


// --------------------------------------------------------
// AbstractRedBlackTree class template.
// --------------------------------------------------------
template<class T, class Node> 
class AbstractRedBlackTree : public AbstractBinaryTree<T, Node> {
// The following is accessible only to classes that inherit
// from AbstractRedBlackTree, since they deal directly with RedBlackNodes.
protected:
    virtual Node *FindNode(T q) const
        { return (root) ? (Node *) root->find(q) : NULL; }
};

template <class T>
class VDKBtree : public AbstractRedBlackTree<T, RedBlackNode<T> > {
public:
    VDKBtree() { }
};



template<class T, class Node> 
Node *
AbstractRedBlackNode<T, Node>::add(const T& AddMe)
{
    Node *root = (Node *) this;
    Node *x = (Node *) this;
    Node *y = NULL;
    while (x) {
        y = x;
        x = (AddMe < x->object) ? x->left : x->right;
    }
    Node *addme = new Node(AddMe, y);
    if (! y)
        root = addme;
    else {
      if (AddMe < y->object)
          y->left = addme;
      else
          y->right = addme;
    }
    addme->clr = Red;
    while (addme != root && 
           addme->parent->parent && 
           addme->parent->clr == Red) {
        Node *y;

        if (addme->parent == addme->parent->parent->left) {
            y = addme->parent->parent->right;
            if (y && y->clr == Red) {
                // Case 1: x's uncle is red
                addme->parent->clr = Black;
                y->clr = Black;
                addme->parent->parent->clr = Red;
                addme = addme->parent->parent;
            }
            else {
                if (addme == addme->parent->right) {
                    // Case 2: x is a right child
                    // Rotate to transform to case 3
                    addme = addme->parent;
                    root = addme->LeftRotate(root);
                }
                // Case 3: x is a left child
                addme->parent->clr = Black;
                if (addme->parent->parent) {
                    addme->parent->parent->clr = Red;
                    root = addme->parent->parent->RightRotate(root);
                }
                // The while loop will terminate 
                // on the next iteration.
            }
        }
        else {
            y = addme->parent->parent->left;
            if (y && y->clr == Red) {
                addme->parent->clr = Black;
                y->clr = Black;
                addme->parent->parent->clr = Red;
                addme = addme->parent->parent;
            }
            else {
                if (addme == addme->parent->left) {
                  addme = addme->parent;
                  root = addme->RightRotate(root);
                }
                addme->parent->clr = Black;
                if (addme->parent->parent) {
                    addme->parent->parent->clr = Red;
                    root = addme->parent->parent->LeftRotate(root);
                }
            }
        }
    }
    root->clr = Black;
    return root;
}

template<class T, class Node> 
Node *
AbstractRedBlackNode<T, Node>::RemoveFixup(Node *x, Node *p)
{
    Node *root = (Node *) this;

    while (x != root && (! x || x->clr == Black)) {
        Node *w;
        if (x == p->left) {
            if (! p)
                return root;
            w = p->right;
            if (! w)
                return root;
            if (w->clr == Red) {
                w->clr = Black;
                p->clr = Red;
                root = p->LeftRotate(root);
                w = p->right;
                if (! p || ! w)
                    return root;
            }
            if ( ((! w->left) || w->left->clr == Black) &&
                 ((! w->right) || w->right->clr == Black)) {
                  w->clr = Red;
                  x = p;
                  p = p->parent;
                  continue;
            }
            else if ((! w->right) || w->right->clr == Black) {
                w->left->clr = Black;
                w->clr = Red;
                root = w->RightRotate(root);
                w = p->right;
                if (! p || ! w)
                    return root;
            }
            w->clr = p->clr;
            if (p)
                p->clr = Black;
            w->right->clr = Black;
            if (p)
                root = p->LeftRotate(root);
            x = root;
        }
        else {
            if (! p)
                return root;
            w = p->left;
            if (! p || ! w)
                return root;
            if (w->clr == Red) {
                w->clr = Black;
                p->clr = Red;
                root = p->RightRotate(root);
                w = p->left;
                if (! p || ! w)
                    return root;
            }
            if ( ((! w->right) || w->right->clr == Black) &&
                 ((! w->left) || w->left->clr == Black)) {
                w->clr = Red;
                x = p;
                p = p->parent;
                continue;
            }
            else if ((! w->left) || w->left->clr == Black) {
                w->right->clr = Black;
                w->clr = Red;
                root = w->LeftRotate(root);
                w = p->left;
                if (! p || ! w)
                    return root;
            }
            w->clr = p->clr;
            if (p)
                p->clr = Black;
            w->left->clr = Black;
            if (p)
                root = p->RightRotate(root);
            x = root;
        }
    }
    if (x)
        x->clr = Black;
    return root;
}

template<class T, class Node> 
Node *
AbstractRedBlackNode<T, Node>::unlink(Node *z)
{
    Node *root = (Node *) this;
    Node *x, *y;

    if (! z)
        return root;
    y = (! z->left || ! z->right) ? z : (Node *) z->Successor();
    x = (y->left) ? y->left : y->right;

    if (x)
        x->parent = y->parent;

    if (y->parent) {
        if (y == y->parent->left)
            y->parent->left = x;
        else
            y->parent->right = x;
    }
    else
        root = x;
    if (y != z)
        z->object = y->object;
    if (y->clr == Black) {
        if (root)
            root = root->RemoveFixup(x, y->parent);
    }
    delete y;
    return root;
}

template<class T, class Node> 
int
AbstractRedBlackNode<T, Node>::CheckTreeProperties(const Node *_parent)
{
    static int BlackHeight;

    if (_parent == NULL)
        BlackHeight = -1;

    // Check binary tree properties.
    if (parent != _parent)
        return NULL;
    if (left) {
        if (object < left->object)
            return NULL;
    }
    if (right) {
        if (right->object < object)
            return NULL;
    }

    // Now check red-black tree properties.

    // If a node is red, then both its children are black
    // (NULL nodes are black).
    if (clr == Red) {
        if ((left && left->clr != Black) ||
            (right && right->clr != Black))
            return NULL;
    }

    // The black-heights of all leaf nodes are equal.
    int bh = NULL;

    if ((! left) && (! right)) {
        // Compute black-height of node
        for (Node *sc = (Node *) this; sc; sc = sc->parent)
            if (sc->clr == Black)
                bh += 1;

        if (BlackHeight == -1) {
            BlackHeight = bh;
        }
        else {
            if (bh != BlackHeight)
                return NULL;
        }
    }
    if (left && (! left->CheckTreeProperties((Node *) this)))
        return NULL;
    if (right && (! right->CheckTreeProperties((Node *) this)))
        return NULL;
    return 1;
}
#endif





