/*
 *  haval.h:  specifies the interface to the HAVAL (V.1) hashing library.
 *
 *      HAVAL is a one-way hashing algorithm with the following
 *      collision-resistant property:
 *             It is computationally infeasible to find two or more
 *             messages that are hashed into the same fingerprint.
 *
 *  Reference:
 *       Y. Zheng, J. Pieprzyk and J. Seberry:
 *       ``HAVAL --- a one-way hashing algorithm with variable
 *       length of output'', Advances in Cryptology --- AUSCRYPT'92,
 *       Lecture Notes in Computer Science, Springer-Verlag, 1993.
 *
 *      This library provides routines to hash
 *        -  a string,
 *        -  a file,
 *        -  input from the standard input device,
 *        -  a 32-word block, and
 *        -  a string of specified length.
 *
 *  Author:     Yuliang Zheng
 *              Department of Computer Science
 *              University of Wollongong
 *              Wollongong, NSW 2522, Australia
 *              Email: yuliang@cs.uow.edu.au
 *              Voice: +61 42 21 4331 (office)
 *
 *  Date:       June 1993
 *
 *      Copyright (C) 1993 by C^3SR. All rights reserved. 
 *      This program may not be sold or used as inducement to
 *      buy a product without the written permission of C^3SR.
 */

#ifndef MHASH_HAVAL_H
#define MHASH_HAVAL_H

#include "libdefs.h"

#define HAVAL_FPTLEN 256		/*
								   length of fingerprint (128,160,192,224 or 256) 
								 */
#define PASS 5					/*
								   number of rounds (3, 4 or 5) 
								 */

typedef word32 haval_word;

typedef struct {
	haval_word count[2];		/*
								   number of bits in a message 
								 */
	haval_word fingerprint[8];	/*
								   current state of fingerprint 
								 */
	haval_word block[32];		/*
								   buffer for a 32-word block 
								 */
	unsigned char remainder[32 * 4];	/*
										   unhashed chars (No.<128) 
										 */
} haval_state;

void haval_string(char *, unsigned char *);		/*
												   hash a string 
												 */
int haval_file(char *, unsigned char *);	/*
											   hash a file 
											 */
void haval_stdin(void);			/*
								   filter -- hash input from stdin 
								 */
void haval_start(haval_state *);	/*
									   initialization 
									 */
void haval_hash(haval_state *, const unsigned char *,
				unsigned int);	/*
								   updating routine 
								 */
void haval_end(haval_state *, unsigned char *);		/*
													   finalization 
													 */
void haval_hash_block(haval_state *);	/*
										   hash a 32-word block 
										 */

#endif
