
/* Test out the window manager interaction functions */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "SDL.h"

/* Is the cursor visible? */
static int visible = 1;

SDL_Surface *LoadIconSurface(char *file, Uint8 **maskp)
{
	SDL_Surface *icon;
	Uint8       *pixels;
	Uint8       *mask;
	int          mlen, i;

	*maskp = NULL;

	/* Load the icon surface */
	icon = SDL_LoadBMP(file);
	if ( icon == NULL ) {
		fprintf(stderr, "Couldn't load %s: %s\n", file, SDL_GetError());
		return(NULL);
	}

	/* Check width and height */
	if ( (icon->w%8) != 0 ) {
		fprintf(stderr, "Icon width must be a multiple of 8!\n");
		SDL_FreeSurface(icon);
		return(NULL);
	}
	if ( icon->format->palette == NULL ) {
		fprintf(stderr, "Icon must have a palette!\n");
		SDL_FreeSurface(icon);
		return(NULL);
	}

	/* Set the colorkey */
	SDL_SetColorKey(icon, SDL_SRCCOLORKEY, *((Uint8 *)icon->pixels));

	/* Create the mask */
	pixels = (Uint8 *)icon->pixels;
	printf("Transparent pixel: (%d,%d,%d)\n",
				icon->format->palette->colors[*pixels].r,
				icon->format->palette->colors[*pixels].g,
				icon->format->palette->colors[*pixels].b);
	mlen = icon->w*icon->h;
	mask = (Uint8 *)malloc(mlen/8);
	if ( mask == NULL ) {
		fprintf(stderr, "Out of memory!\n");
		SDL_FreeSurface(icon);
		return(NULL);
	}
	memset(mask, 0, mlen/8);
	for ( i=0; i<mlen; ) {
		if ( pixels[i] != *pixels )
			mask[i/8] |= 0x01;
		++i;
		if ( (i%8) != 0 )
			mask[i/8] <<= 1;
	}
	*maskp = mask;
	return(icon);
}

int FilterEvents(const SDL_Event *event)
{
	static int reallyquit = 0;

	switch (event->type) {

		case SDL_ACTIVEEVENT:
			/* See what happened */
			printf("App %s ",
				event->active.gain ? "gained" : "lost");
			if ( event->active.state & SDL_APPACTIVE )
				printf("active ");
			if ( event->active.state & SDL_APPMOUSEFOCUS )
				printf("mouse ");
			if ( event->active.state & SDL_APPINPUTFOCUS )
				printf("input ");
			printf("focus\n");

			/* See if we are iconified or restored */
			if ( event->active.state & SDL_APPACTIVE ) {
				printf("App has been %s\n",
					event->active.gain ?
						 "restored" : "iconified");
			}
			return(0);

		/* This is important!  Queue it if we want to quit. */
		case SDL_QUIT:
			if ( ! reallyquit ) {
				reallyquit = 1;
				printf("Quit requested\n");
				return(0);
			}
			printf("Quit demanded\n");
			return(1);

		/* We want to toggle visibility on buttonpress */
		case SDL_MOUSEBUTTONDOWN:
		case SDL_MOUSEBUTTONUP:
			if ( event->button.state == SDL_PRESSED ) {
				visible = !visible;
				SDL_ShowCursor(visible);
			}
			printf("Mouse button %d has been %s\n",
				event->button.button,
				(event->button.state == SDL_PRESSED) ?
						"pressed" : "released");
			return(0);

		/* Show relative mouse motion */
		case SDL_MOUSEMOTION:
			printf("Mouse relative motion: {%d,%d}\n",
				event->motion.xrel, event->motion.yrel);
			return(0);
		
		/* Drop all other events */
		default:
			return(0);
	}
}

main(int argc, char *argv[])
{
	SDL_Surface *screen;
	SDL_Event event;
	char *title;
	SDL_Surface *icon;
	Uint8 *icon_mask;
	SDL_Cursor  *cursor;
	int i, parsed;
	Uint8 *buffer;
	SDL_Color palette[256];
	Uint8  video_bpp;
	Uint32 video_flags;

	if ( SDL_Init(SDL_INIT_VIDEO) < 0 ) {
		fprintf(stderr,
			"Couldn't initialize SDL: %s\n", SDL_GetError());
		exit(1);
	}
	atexit(SDL_Quit);

	/* Set the icon -- this must be done before the first mode set */
	icon = LoadIconSurface("icon.bmp", &icon_mask);
	if ( icon != NULL ) {
		SDL_WM_SetIcon(icon, icon_mask);
	}
	if ( icon_mask != NULL )
		free(icon_mask);

	/* Set the title bar */
	if ( argv[1] == NULL )
		title = "Testing  1.. 2.. 3...";
	else
		title = argv[1];
	SDL_WM_SetCaption(title, "testwm");

	/* See if it's really set */
	SDL_WM_GetCaption(&title, NULL);
	if ( title )
		printf("Title was set to: %s\n", title);
	else
		printf("No window title was set!\n");

	/* Initialize the display */
	video_bpp = 8;
	video_flags = SDL_SWSURFACE;
	parsed = 1;
	while ( parsed ) {
		if ( (argc >= 2) && (strcmp(argv[1], "-fullscreen") == 0) ) {
			video_flags |= SDL_FULLSCREEN;
			argc -= 1;
			argv += 1;
		} else
		if ( (argc >= 3) && (strcmp(argv[1], "-bpp") == 0) ) {
			video_bpp = atoi(argv[2]);
			argc -= 2;
			argv += 2;
		} else {
			parsed = 0;
		}
	}
	screen = SDL_SetVideoMode(640, 480, video_bpp, video_flags);
	if (  screen == NULL ) {
		fprintf(stderr, "Couldn't set 640x480x%d video mode: %s\n",
						video_bpp, SDL_GetError());
		exit(1);
	}
	printf("Running in %s mode\n", screen->flags & SDL_FULLSCREEN ?
						"fullscreen" : "windowed");

	/* Set an event filter that discards everything but QUIT */
	SDL_SetEventFilter(FilterEvents);

	/* Ignore keystrokes, they don't even get filtered */
	SDL_EventState(SDL_KEYDOWN, SDL_IGNORE);
	SDL_EventState(SDL_KEYUP, SDL_IGNORE);

	/* Set the surface pixels and refresh! */
	for ( i=0; i<256; ++i ) {
		palette[i].r = 255-i;
		palette[i].g = 255-i;
		palette[i].b = 255-i;
	}
	SDL_SetColors(screen, palette, 0, 256);
	if ( SDL_LockSurface(screen) < 0 ) {
		fprintf(stderr, "Couldn't lock display surface: %s\n",
							SDL_GetError());
		exit(2);
	}
	buffer = (Uint8 *)screen->pixels;
	for ( i=0; i<screen->h; ++i ) {
		memset(buffer,(i*255)/screen->h,
				screen->w*screen->format->BytesPerPixel);
		buffer += screen->pitch;
	}
	SDL_UnlockSurface(screen);
	SDL_UpdateRect(screen, 0, 0, 0, 0);

	/* Loop, waiting for QUIT */
	while ( SDL_WaitEvent(&event) ) {
		switch (event.type) {
			case SDL_QUIT:
				printf("Bye bye..\n");
				exit(0);
			default:
				/* This should never happen */
				printf("Warning: Event %d wasn't filtered\n",
								event.type);
				break;
		}
	}
	printf("SDL_WaitEvent() error: %s\n", SDL_GetError());
	exit(255);
}
