/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_x11video.c,v 1.1 1999/11/23 23:46:25 hercules Exp $";
#endif

/* X11 based SDL video driver implementation.
   Note:  This implementation does not currently need X11 thread locking,
          since the event thread uses a separate X connection and any
          additional locking necessary is handled internally.  However,
          if full locking is neccessary, take a look at XInitThreads().
*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#ifdef MTRR_SUPPORT
#include <asm/mtrr.h>
#include <sys/fcntl.h>
#endif

#include "SDL.h"
#include "SDL_error.h"
#include "SDL_timer.h"
#include "SDL_video.h"
#include "SDL_mouse.h"
#include "SDL_endian.h"
#include "SDL_sysvideo.h"
#include "SDL_pixels_c.h"
#include "SDL_events_c.h"
#include "SDL_x11wm_c.h"
#include "SDL_x11mouse_c.h"
#include "SDL_x11events_c.h"
#include "blank_cursor.h"

#define X_HIDDEN_SIZE	32		/* starting hidden window size */


/* Initialization/Query functions */
static int X11_VideoInit(_THIS, SDL_PixelFormat *vformat);
static SDL_Rect **X11_ListModes(_THIS, SDL_PixelFormat *format, Uint32 flags);
static SDL_Surface *X11_SetVideoMode(_THIS, SDL_Surface *current, int width, int height, int bpp, Uint32 flags);
static int X11_SetColors(_THIS, int firstcolor, int ncolors);
static void X11_VideoQuit(_THIS);

/* Hardware surface functions */
static int X11_AllocHWSurface(_THIS, SDL_Surface *surface);
static int X11_LockHWSurface(_THIS, SDL_Surface *surface);
static void X11_UnlockHWSurface(_THIS, SDL_Surface *surface);
static int X11_FlipHWSurface(_THIS, SDL_Surface *surface);
static void X11_FreeHWSurface(_THIS, SDL_Surface *surface);

/* X11 driver bootstrap functions */

static int X11_Available(void)
{
	Display *display;

	display = XOpenDisplay(NULL);
	if ( display != NULL ) {
		XCloseDisplay(display);
	}
	return(display != NULL);
}

static void X11_DeleteDevice(SDL_VideoDevice *device)
{
	free(device->hidden);
	free(device);
}

static SDL_VideoDevice *X11_CreateDevice(int devindex)
{
	SDL_VideoDevice *device;

	/* Initialize all variables that we clean on shutdown */
	device = (SDL_VideoDevice *)malloc(sizeof(SDL_VideoDevice));
	if ( device ) {
		memset(device, 0, (sizeof *device));
		device->hidden = (struct SDL_PrivateVideoData *)
				malloc((sizeof *device->hidden));
	}
	if ( (device == NULL) || (device->hidden == NULL) ) {
		SDL_OutOfMemory();
		if ( device ) {
			free(device);
		}
		return(0);
	}
	memset(device->hidden, 0, (sizeof *device->hidden));

	/* Set the function pointers */
	device->VideoInit = X11_VideoInit;
	device->ListModes = X11_ListModes;
	device->SetVideoMode = X11_SetVideoMode;
	device->SetColors = X11_SetColors;
	device->UpdateRects = NULL;
	device->VideoQuit = X11_VideoQuit;
	device->AllocHWSurface = X11_AllocHWSurface;
	device->CheckHWBlit = NULL;
	device->FillHWRect = NULL;
	device->SetHWColorKey = NULL;
	device->SetHWAlpha = NULL;
	device->LockHWSurface = X11_LockHWSurface;
	device->UnlockHWSurface = X11_UnlockHWSurface;
	device->FlipHWSurface = X11_FlipHWSurface;
	device->FreeHWSurface = X11_FreeHWSurface;
	device->SetIcon = X11_SetIcon;
	device->SetCaption = X11_SetCaption;
	device->GetWMInfo = X11_GetWMInfo;
	device->FreeWMCursor = X11_FreeWMCursor;
	device->CreateWMCursor = X11_CreateWMCursor;
	device->ShowWMCursor = X11_ShowWMCursor;
	device->WarpWMCursor = X11_WarpWMCursor;
	device->InitOSKeymap = X11_InitOSKeymap;
	device->PumpEvents = X11_PumpEvents;

	device->free = X11_DeleteDevice;

	return device;
}

VideoBootStrap X11_bootstrap = {
	"x11", X11_Available, X11_CreateDevice
};

/* Shared memory information */
extern int XShmQueryExtension(Display *dpy);	/* Not in X11 headers */

/* Shared memory error handler routine */
static int shm_error;
static int (*X_handler)(Display *, XErrorEvent *) = NULL;
static int shm_errhandler(Display *d, XErrorEvent *e)
{
        if ( e->error_code == BadAccess ) {
        	++shm_error;
        	return(0);
        } else
		return(X_handler(d,e));
}

/* X11 I/O error handler routine */
static int (*XIO_handler)(Display *) = NULL;
static int xio_errhandler(Display *d)
{
	/* Ack!  Lost X11 connection! */

	/* We will crash if we try to clean up our display */
	if ( current_video->hidden->Ximage ) {
		SDL_VideoSurface->pixels = NULL;
	}
	current_video->hidden->X11_Display = NULL;

	/* Continue with the standard X11 error handler */
	return(XIO_handler(d));
}

/* Used to sort video modes largest to smallest */
static int CompareModes(const void *A, const void *B)
{
	const SDL_Rect *a = *(SDL_Rect **)A;
	const SDL_Rect *b = *(SDL_Rect **)B;

	return( (int)(b->w*b->h) - (int)(a->w*a->h) );
}

static void X11_UpdateVideoInfo(_THIS)
{
#ifdef XFREE86_DGA
	if ( this->hidden->XF86_DGA ) {
		int unused_int;
		char *unused_ptr;
		int total_mem;

		this->info.hw_available = 1;
		seteuid(0);		/* Try to set root privileges */
		XF86DGAGetVideo(SDL_Display, SDL_Screen,
			&unused_ptr, &unused_int, &unused_int, &total_mem);
		seteuid(getuid());	/* Get rid of root privileges */
		this->info.video_mem = total_mem;
	}
#endif
}

static int X11_VideoInit(_THIS, SDL_PixelFormat *vformat)
{
	char *display;
	int EventBase, ErrorBase;
	int MajorVersion, MinorVersion;
	int flags;
	int local_x11;
	XGCValues gcv;

	/* Open the X11 display */
	display = NULL;		/* Get it from DISPLAY environment variable */
	if ( strncmp(XDisplayName(display), ":", 1) == 0 ) {
		local_x11 = 1;
	} else {
		local_x11 = 0;
	}
	SDL_Display = XOpenDisplay(display);
	if ( SDL_Display == NULL ) {
		SDL_SetError("Couldn't open X11 display");
		return(-1);
	}

	/* Create an alternate X display for graphics updates -- allows us
	   to do graphics updates in a separate thread from event handling.
	   Thread-safe X11 doesn't seem to exist.
	 */
	GFX_Display = XOpenDisplay(display);
	if ( GFX_Display == NULL ) {
		SDL_SetError("Couldn't open X11 display");
		return(-1);
	}

	/* Set the error handler if we lose the X display */
	XIO_handler = XSetIOErrorHandler(xio_errhandler);

	/* Register the "delete window" protocol */
	SDL_DELWIN = XInternAtom(SDL_Display, "WM_DELETE_WINDOW", False);

	/* Determine the screen depth */
	{
		XPixmapFormatValues *pix_format;
		int i, num_formats;

		vformat->BitsPerPixel = DefaultDepth(SDL_Display, SDL_Screen);
		pix_format = XListPixmapFormats(SDL_Display, &num_formats);
		if ( pix_format == NULL ) {
			SDL_SetError("Couldn't determine screen formats");
			return(-1);
		}
		for ( i=0; i<num_formats; ++i ) {
			if ( vformat->BitsPerPixel == pix_format[i].depth )
				break;
		}
		if ( i != num_formats )
			vformat->BitsPerPixel = pix_format[i].bits_per_pixel;
		XFree((char *)pix_format);
	}
	if ( vformat->BitsPerPixel > 8 ) {
		vformat->Rmask = SDL_Visual->red_mask;
	  	vformat->Gmask = SDL_Visual->green_mask;
	  	vformat->Bmask = SDL_Visual->blue_mask;
	}

	/* Check for MIT shared memory extension */
	if ( local_x11 ) {
		this->hidden->use_mitshm = XShmQueryExtension(SDL_Display);
	} else {
		this->hidden->use_mitshm = 0;
	}

#ifdef XFREE86_DGA
	/* Check for XFree86 Direct Graphics Access extension
	   Note:  Only root can perform Direct Graphics Access
	 */
	this->hidden->XF86_DGA = 0;
	this->hidden->switchable_modes = 0;
	seteuid(0);		/* Try to set root privileges */
	if ( local_x11 &&
	     ((geteuid() == 0) || (access("/dev/mem", R_OK|W_OK) == 0)) &&
	     XF86DGAQueryExtension(SDL_Display, &EventBase, &ErrorBase) &&
	     XF86DGAQueryVersion(SDL_Display, &MajorVersion, &MinorVersion) &&
	     XF86DGAQueryDirectVideo(SDL_Display, SDL_Screen, &flags) && 
					(flags & XF86DGADirectPresent) ) {
		int buggy_X11;
		int i, nmodes;
		XF86VidModeModeInfo **modes;

		this->hidden->XF86_DGA = 1;
		/* Metro-X 4.3.0 and earlier has a broken implementation of
		   XF86VidModeGetAllModeLines() - it hangs the client.
		 */
		buggy_X11 = 0;
		if ( strcmp(ServerVendor(SDL_Display),
					"Metro Link Incorporated") == 0 ) {
			FILE *metro_fp;

			metro_fp =
				fopen("/usr/X11R6/lib/X11/Metro/.version", "r");
			if ( metro_fp != NULL ) {
				int major, minor, patch, version;
				major = 0; minor = 0; patch = 0;
				fscanf(metro_fp, "%d.%d.%d",
						&major, &minor, &patch);
				version = major*100+minor*10+patch;
				if ( version < 431 ) {
					buggy_X11 = 1;
				}
				fclose(metro_fp);
			}
		}
#if defined(__alpha__) || defined(__powerpc__)
		/* The alpha and PPC XFree86 servers are also buggy */
		buggy_X11 = 1;
#endif
		/* Enumerate the available fullscreen modes */
		if ( ! buggy_X11 &&
		     XF86VidModeQueryExtension(SDL_Display,
					&EventBase, &ErrorBase) &&
		     XF86VidModeQueryVersion(SDL_Display,
					&MajorVersion, &MinorVersion) &&
		     XF86VidModeGetAllModeLines(SDL_Display, SDL_Screen,
							&nmodes, &modes) ) {
			SDL_modelist = (SDL_Rect **)
					malloc((nmodes+1)*sizeof(SDL_Rect *));
			if ( SDL_modelist ) {
				for ( i=0; i<nmodes; ++i ) {
					SDL_modelist[i] = (SDL_Rect *)
						malloc(sizeof(SDL_Rect));
					if ( SDL_modelist[i] == NULL ) {
						SDL_OutOfMemory();
						seteuid(getuid());
						return(-1);
					}
					SDL_modelist[i]->x = 0;
					SDL_modelist[i]->y = 0;
					SDL_modelist[i]->w = modes[i]->hdisplay;
					SDL_modelist[i]->h = modes[i]->vdisplay;
				}
				SDL_modelist[i] = NULL;

				/* Sort modes largest to smallest */
				qsort(SDL_modelist, nmodes, sizeof(SDL_Rect *),
								CompareModes);
			}
			this->hidden->switchable_modes = 1;
		} else {
			SDL_modelist = (SDL_Rect **)
					malloc((1+1)*sizeof(SDL_Rect *));
			if ( SDL_modelist ) {
				SDL_modelist[0] = (SDL_Rect *)
						malloc(sizeof(SDL_Rect));
				if ( SDL_modelist[0] == NULL ) {
					SDL_OutOfMemory();
					seteuid(getuid());
					return(-1);
				}
				SDL_modelist[0]->x = 0;
				SDL_modelist[0]->y = 0;
				SDL_modelist[0]->w = WidthOfScreen(DefaultScreenOfDisplay(SDL_Display));
				SDL_modelist[0]->h = HeightOfScreen(DefaultScreenOfDisplay(SDL_Display));
				SDL_modelist[1] = NULL;
			}
		}
#ifdef DEBUG_XF86
		if ( SDL_modelist ) {
			int i;

			fprintf(stderr, "XF86 mode list:\n");
			for ( i=0; SDL_modelist[i]; ++i ) {
				fprintf(stderr, "\t%dx%d\n",
					SDL_modelist[i]->w, SDL_modelist[i]->h);
			}
		}
#endif
#ifdef MTRR_SUPPORT
		/* Try to set up "Memory Type Range Register" access */
		this->hidden->MTRR_fd = open("/proc/mtrr", O_WRONLY, 0);
		if ( this->hidden->MTRR_fd >= 0 ) {
			struct mtrr_sentry entry;
			int entry_base;
			int unused;
			int entry_size;

			/* Get the base and length of the video framebuffer */
			XF86DGAGetVideoLL(SDL_Display, SDL_Screen,
				&entry_base, &unused, &unused, &entry_size);

			/* Set the entry values */
			entry.base = (unsigned long)entry_base;
			entry.size = (unsigned long)entry_size;
			entry.size *= 1024;
			entry.type = MTRR_TYPE_WRCOMB;

			/* Magic code to satisfy kernel requirements */
			if ( entry.size < 0x100000 ) {
				entry.size = 0x100000;
			} else
			if ( entry.size % 0x400000 ) {
				entry.size += (0x400000-1);
				entry.size &= ~(0x400000-1);
			}

			/* Add the framebuffer region to the control list */
			ioctl(this->hidden->MTRR_fd,MTRRIOC_ADD_ENTRY,&entry);
		}
#endif
	}
#ifdef VERBOSE_WARNINGS
	/* Warn of buffer overflows if this program is set-uid */
	if ( geteuid() != getuid() ) {
		fprintf(stderr,
		"Warning: This set-uid program may be a security risk\n");
	}
#endif
	seteuid(getuid());	/* Temporarily get rid of root privileges */

#ifdef DEBUG_XF86
	fprintf(stderr, "XFree86 DGA is %s\n", this->hidden->XF86_DGA?"enabled":"disabled");
#endif
#endif /* XFREE86_DGA */

	/* Create (or use) the X11 window */
	SDL_windowid = getenv("SDL_WINDOWID");
	if ( SDL_windowid ) {
		SDL_Window = atol(SDL_windowid);
	} else {
		SDL_Window = XCreateSimpleWindow(SDL_Display,
				DefaultRootWindow(SDL_Display), 0, 0,
				X_HIDDEN_SIZE, X_HIDDEN_SIZE, 1,
				BlackPixel(SDL_Display, SDL_Screen),
				BlackPixel(SDL_Display, SDL_Screen));
	}

	/* Set the class hints so we can get an icon (AfterStep) */
	{
		XClassHint *classhints;

		classhints = XAllocClassHint();
		if ( classhints != NULL ) {
			classhints->res_name = "SDL_App";
			classhints->res_class = "SDL_App";
			XSetClassHint(SDL_Display, SDL_Window, classhints);
			XFree(classhints);
		}
	}

	/* Allow the window to be deleted by the window manager */
	XSetWMProtocols(SDL_Display, SDL_Window, &SDL_DELWIN, 1);

	/* Cache the window in the server, when possible */
	{
		unsigned long mask;
		Screen *xscreen;
		XSetWindowAttributes a;

		xscreen = DefaultScreenOfDisplay(SDL_Display);
		if ( DoesBackingStore(xscreen) != NotUseful ) {
			mask = CWBackingStore;
			a.backing_store = DoesBackingStore(xscreen);
		} else {
			mask = CWSaveUnder;
			a.save_under = DoesSaveUnders(xscreen);
		}
		XChangeWindowAttributes(SDL_Display, SDL_Window, mask, &a);
	}

	/* Create the graphics context */
	gcv.graphics_exposures = False;
	SDL_GC = XCreateGC(SDL_Display, SDL_Window, GCGraphicsExposures, &gcv);
	if ( ! SDL_GC ) {
		SDL_SetError("Couldn't create graphics context");
		return(-1);
	}

	/* If the surface is palettized, create the colormap */
	if ( vformat->BitsPerPixel == 8 ) {
		int     ncolors;

		/* Get the default display colormap (used by icons) */
		SDL_DisplayColormap = DefaultColormap(SDL_Display, SDL_Screen);
		SDL_XColorMap = SDL_DisplayColormap;

		/* Allocate the pixel flags */
		ncolors = (1<<vformat->BitsPerPixel);
		SDL_XPixels = (int *)malloc(ncolors*sizeof(int));
		if ( SDL_XPixels == NULL ) {
			SDL_OutOfMemory();
			return(-1);
		}
		memset(SDL_XPixels, 0, ncolors*sizeof(*SDL_XPixels));
	}

	/* Create the blank cursor */
	SDL_BlankCursor = this->CreateWMCursor(this, blank_cdata, blank_cmask,
					BLANK_CWIDTH, BLANK_CHEIGHT,
						BLANK_CHOTX, BLANK_CHOTY);

	/* Set the events that we look for */
	if ( SDL_windowid == NULL ) {
		XSelectInput(SDL_Display, SDL_Window,
				( FocusChangeMask | PropertyChangeMask
				| EnterWindowMask | LeaveWindowMask
				| KeyPressMask | KeyReleaseMask
				| ButtonPressMask | ButtonReleaseMask
				| PointerMotionMask
				| ExposureMask | StructureNotifyMask ) );
	}

	/* See whether or not we need to swap pixels */
	this->hidden->swap_pixels = 0;
	if ( SDL_BYTEORDER == SDL_LIL_ENDIAN ) {
		if ( XImageByteOrder(SDL_Display) == MSBFirst ) {
			this->hidden->swap_pixels = 1;
		}
	} else {
		if ( XImageByteOrder(SDL_Display) == LSBFirst ) {
			this->hidden->swap_pixels = 1;
		}
	}

	/* Fill in some window manager capabilities */
	this->info.wm_available = 1;

	/* Fill in our hardware acceleration capabilities */
	X11_UpdateVideoInfo(this);

	/* We're done! */
	XSync(SDL_Display, False);
	return(0);
}

static SDL_Rect **X11_ListModes(_THIS, SDL_PixelFormat *format, Uint32 flags)
{
	if ( format->BitsPerPixel == this->screen->format->BitsPerPixel ) {
		if ( (flags & SDL_FULLSCREEN) && this->hidden->XF86_DGA ) {
			return(SDL_modelist);
		} else {
			return((SDL_Rect **)-1);
		}
	} else {
		return((SDL_Rect **)0);
	}
}

/* Various screen update functions available */
static void X11_NormalUpdate(_THIS, int numrects, SDL_Rect *rects);
static void X11_MITSHMUpdate(_THIS, int numrects, SDL_Rect *rects);
#ifdef XFREE86_DGA
static void X11_DirectUpdate(_THIS, int numrects, SDL_Rect *rects);
static void X11_BankedUpdate(_THIS, int numrects, SDL_Rect *rects);
#endif

static int X11_SetupWindow(_THIS, SDL_Surface *current, Uint32 flags)
{
	/* Allocate shared memory if possible */
	if ( this->hidden->use_mitshm ) {
		shminfo.shmid = shmget(IPC_PRIVATE, current->h*current->pitch,
								IPC_CREAT|0777);
		if ( shminfo.shmid >= 0 ) {
			shminfo.shmaddr = (char *)shmat(shminfo.shmid, 0, 0);
			shminfo.readOnly = False;
			if ( shminfo.shmaddr != (char *)-1 ) {
				shm_error = False;
				X_handler = XSetErrorHandler(shm_errhandler);
				XShmAttach(SDL_Display, &shminfo);
				XSync(SDL_Display, True);
				XSetErrorHandler(X_handler);
				if ( shm_error == True )
					shmdt(shminfo.shmaddr);
			} else {
				shm_error = True;
			}
			shmctl(shminfo.shmid, IPC_RMID, NULL);
		} else {
			shm_error = True;
		}
		if ( shm_error == True )
			this->hidden->use_mitshm = 0;
	}
	if ( this->hidden->use_mitshm ) {
		current->pixels = shminfo.shmaddr;
	} else {
		current->pixels = malloc(current->h*current->pitch);
	}
	if ( current->pixels == NULL ) {
		SDL_OutOfMemory();
		return(-1);
	}

	if ( this->hidden->use_mitshm ) {
		SDL_Ximage = XShmCreateImage(SDL_Display, SDL_Visual,
			DefaultDepth(SDL_Display, SDL_Screen),
			ZPixmap, shminfo.shmaddr, &shminfo, 
						current->w, current->h);
	} else {
		SDL_Ximage = XCreateImage(SDL_Display, SDL_Visual,
			DefaultDepth(SDL_Display, SDL_Screen),
			ZPixmap, 0, (char *)current->pixels, 
			current->w, current->h,
			((current->format)->BytesPerPixel == 3) ? 32 :
				(current->format)->BytesPerPixel*8, 0);
	}
	if ( SDL_Ximage == NULL ) {
		SDL_SetError("Couldn't create XImage");
		if ( this->hidden->use_mitshm ) {
			XShmDetach(SDL_Display, &shminfo);
			XSync(SDL_Display, False);
			shmdt(shminfo.shmaddr);
			current->pixels = NULL;
		}
		return(-1);
	}

	/* Determine what blit function to use */
	if ( this->hidden->use_mitshm ) {
		this->UpdateRects = X11_MITSHMUpdate;
	} else {
		this->UpdateRects = X11_NormalUpdate;
	}
	return(0);
}

static int X11_SetupDirect(_THIS, SDL_Surface *current, Uint32 flags)
{
#ifdef XFREE86_DGA
	int banked;
	int vram;

	/* Set the requested video mode */
	if ( this->hidden->switchable_modes ) {
		int i, nmodes;
		XF86VidModeModeInfo **modes;

		/* Save the original video mode settings */
		if ( ! x11_w || ! x11_h ) {
			XF86DGAGetViewPortSize(SDL_Display, SDL_Screen,
							&x11_w, &x11_h);
		}

		/* Get the mode we need to set, and set it */
		XF86VidModeGetAllModeLines(SDL_Display, SDL_Screen,
						&nmodes, &modes);
		for ( i=0; i<nmodes; ++i ) {
			if ( (current->w == modes[i]->hdisplay) && 
			     (current->h == modes[i]->vdisplay) ) {
				break;
			}
		}
		if ( i == nmodes ) {
			/* This shouldn't happen */
			SDL_SetError("Requested video mode not available");
			return(-1);
		}
		XF86VidModeSwitchToMode(SDL_Display, SDL_Screen, modes[i]);
	}

	/* Get the parameters for the video card */
	seteuid(0);		/* Try to set root privileges */
	XF86DGAGetVideo(SDL_Display, SDL_Screen,
			&vinfo.base[0], &vinfo.width, &vinfo.page, &vram);
	vinfo.width *= current->format->BytesPerPixel;
        vinfo.width = (vinfo.width + 3) & ~3;       /* 4-byte aligning */
	vinfo.lines = (vinfo.page/vinfo.width);
	vinfo.w = current->w;
	vinfo.h = current->h;
	vinfo.flip = 0;
	vinfo.base[1] = vinfo.base[0]+vinfo.width*vinfo.h;
	banked = (vinfo.lines < current->h);
#ifdef DEBUG_XF86
	fprintf(stderr,
"Video: ram = %dK, pagesize = %d, w = %d, h = %d, lines = %d, pitch = %d\n",
		vram, vinfo.page, vinfo.w, vinfo.h, vinfo.lines, vinfo.width);
#endif

	/* Arggghh.  The XFree86 3.3.2 implementation of XF86DGAGetVideo()
	   sets atexit() and signal handler functions to the DGA cleanup 
	   function which calls _exit() internally.  This blows away any 
	   atexit() call we might have made previously, and causes a 
	   segmentation fault on exit if we, as a shared library, are 
	   dynamically unloaded.  Hack, hack, hack.
	*/
	atexit(SDL_Quit);

	/* Set direct graphics access */
	XF86DGASetViewPort(SDL_Display, SDL_Screen, 0, 0);
	XF86DGADirectVideo(SDL_Display, SDL_Screen,
#ifdef NO_DIRECTMOUSE
		XF86DGADirectGraphics|XF86DGADirectKeyb);
#else
		XF86DGADirectGraphics|XF86DGADirectMouse|XF86DGADirectKeyb);
#endif
	XF86DGASetVidPage(SDL_Display, SDL_Screen, 0);
	seteuid(getuid());	/* Temporarily get rid of root privileges */

	/* Work around a bug in the Voodoo 3 X server - turn off cursor */
	if ( SDL_BlankCursor != NULL ) {
		XDefineCursor(SDL_Display, SDL_Window,
				X11_GetWMXCursor(SDL_BlankCursor));
	}

	/* Make the input as raw as possible */
	XSetInputFocus(SDL_Display, SDL_Window,
					RevertToPointerRoot, CurrentTime);
	XGrabKeyboard(SDL_Display, SDL_Window, True, GrabModeAsync,
						GrabModeAsync, CurrentTime);
	XGrabPointer(SDL_Display, SDL_Window, True, 
			(ButtonPressMask|ButtonReleaseMask|PointerMotionMask),
			GrabModeAsync, GrabModeAsync, None, None, CurrentTime);

	/* Fill in the current video surface */
	if ( banked ) {
		current->flags |= SDL_SWSURFACE;
		current->pixels = malloc(current->h*current->pitch);
	} else {
		current->flags |= SDL_HWSURFACE;
		current->pitch = vinfo.width;
		if ( ((flags & SDL_DOUBLEBUF) == SDL_DOUBLEBUF) &&
					(vinfo.lines >= vinfo.h*2) ) {
			current->flags |= SDL_DOUBLEBUF;
			current->pixels = vinfo.base[!vinfo.flip];
		} else {
			current->pixels = vinfo.base[0];
		}
	}
	if ( current->pixels == NULL ) {
		SDL_OutOfMemory();
		return(-1);
	}
	current->flags |= SDL_FULLSCREEN;

	/* Determine what blit function to use */
	if ( banked ) {
		this->UpdateRects = X11_BankedUpdate;
	} else {
		this->UpdateRects = X11_DirectUpdate;
	}
	vinfo.active = 1;
	return(0);
#else
	SDL_SetError("Logic error: Fullscreen requested without DGA");
	return(-1);
#endif /* XFREE86_DGA */
}

static void X11_ReleaseMode(_THIS, SDL_Surface *screen)
{
	if ( SDL_Ximage ) {
		XDestroyImage(SDL_Ximage);
		if ( this->hidden->use_mitshm ) {
			XShmDetach(SDL_Display, &shminfo);
			XSync(SDL_Display, False);
			shmdt(shminfo.shmaddr);
		}
		screen->pixels = NULL;
		SDL_Ximage = NULL;
	}
#ifdef XFREE86_DGA
	if ( vinfo.active ) {
		/* Release the direct input and graphics */
		XUngrabPointer(SDL_Display, CurrentTime);
		XUngrabKeyboard(SDL_Display, CurrentTime);
		XF86DGADirectVideo(SDL_Display, SDL_Screen, 0);

		/* Set the original video mode */
		if ( this->hidden->switchable_modes ) {
			XF86VidModeModeInfo **modes;
			int i, nmodes;

			XF86VidModeGetAllModeLines(SDL_Display, SDL_Screen,
							&nmodes, &modes);
			for ( i=0; i<nmodes; ++i ) {
				if ( (x11_w == modes[i]->hdisplay) && 
				     (x11_h == modes[i]->vdisplay) ) {
					break;
				}
			}
			if ( i != nmodes ) {
				XF86VidModeSwitchToMode(SDL_Display,
						SDL_Screen, modes[i]);
			}
		}
		if ( (screen->flags & SDL_HWSURFACE) == SDL_SWSURFACE ) {
			free(screen->pixels);
		}
		vinfo.active = 0;
	}
#endif
}

static int X11_ShowWindow(_THIS, int width, int height, Uint32 flags)
{
	XSizeHints *sizehints;
	XWindowAttributes xwinattr;
	XEvent event;

	/* Resize the window */
	if ( flags & SDL_FULLSCREEN ) {
		XSetWindowAttributes xsetattr;
		Window u1; int u2; unsigned int u3;

		/* If we were not fullscreen, unmap the window */
		XGetWindowAttributes(SDL_Display, SDL_Window, &xwinattr);
		if ( xwinattr.override_redirect != True ) {
			if ( xwinattr.map_state != IsUnmapped ) {
				XUnmapWindow(SDL_Display, SDL_Window);
				do {
					XNextEvent(SDL_Display, &event);
				} while ( event.type != UnmapNotify );
			}
			/* Save original x,y position */
		}
		XMoveWindow(SDL_Display, SDL_Window, 0, 0);

		/* Prevent window manager from grabbing us on placement */
		xsetattr.override_redirect = True;
		XChangeWindowAttributes(SDL_Display, SDL_Window,
					CWOverrideRedirect, &xsetattr);

		/* Make our window cover the entire screen */
		XGetGeometry(SDL_Display, DefaultRootWindow(SDL_Display),
				&u1, &u2, &u2,
				(unsigned int *)&width,
				(unsigned int *)&height,
				&u3, &u3);
	} else {
		XSetWindowAttributes xsetattr;

		/* If we were fullscreen, unmap the window */
		XGetWindowAttributes(SDL_Display, SDL_Window, &xwinattr);
		if ( xwinattr.override_redirect == True ) {
			if ( xwinattr.map_state != IsUnmapped ) {
				XUnmapWindow(SDL_Display, SDL_Window);
				do {
					XNextEvent(SDL_Display, &event);
				} while ( event.type != UnmapNotify );
			}
		}

		/* Allow window manager to grab us and place the window */
		xsetattr.override_redirect = False;
		XChangeWindowAttributes(SDL_Display, SDL_Window,
					CWOverrideRedirect, &xsetattr);
	}
	sizehints = XAllocSizeHints();
	if ( sizehints == NULL ) {
		SDL_SetError("Couldn't allocate size hints");
		return(-1);
	}
	sizehints->min_width   = width;
	sizehints->min_height  = height;
	sizehints->max_width   = width;
	sizehints->max_height  = height;
	sizehints->flags       = (PMinSize | PMaxSize);
	XSetWMNormalHints(SDL_Display, SDL_Window, sizehints);
	XFree(sizehints);
	XResizeWindow(SDL_Display, SDL_Window, width, height);

	/* Map the window onto the display, if needed */
	XGetWindowAttributes(SDL_Display, SDL_Window, &xwinattr);
	if ( xwinattr.map_state == IsUnmapped ) {
		XEvent event;

		XMapRaised(SDL_Display, SDL_Window);
		do {
			XNextEvent(SDL_Display, &event);
		} while ( event.type != MapNotify );
	}
	return(0);
}

static SDL_Surface *X11_SetVideoMode(_THIS, SDL_Surface *current,
				int width, int height, int bpp, Uint32 flags)
{
	/* Lock the event thread, in multi-threading environments */
	SDL_Lock_EventThread();

	/* Check the combination of flags we were passed */
	if ( flags & SDL_FULLSCREEN ) {
		/* Clear fullscreen flag if not supported */
		if ( ! this->hidden->XF86_DGA || (SDL_windowid != NULL) ) {
			flags &= ~SDL_FULLSCREEN;
		}
#if defined(XFREE86_DGA) && defined(VERBOSE_WARNINGS)
		if ( geteuid() != 0 ) {
			fprintf(stderr,
				"You must be root to set a fullscreen mode\n");
		}
#endif
	}
	if ( (flags&SDL_FULLSCREEN) != SDL_FULLSCREEN ) {
		/* There's no windowed double-buffering */
		flags &= ~SDL_DOUBLEBUF;
	}
	if ( (flags&SDL_DOUBLEBUF) == SDL_DOUBLEBUF ) {
		/* Use hardware surfaces when double-buffering */
		flags |= SDL_HWSURFACE;
	}

	/* Release any previously set mode */
	X11_ReleaseMode(this, current);

	/* Show the window, if we own it */
	if ( SDL_windowid == NULL ) {
		if ( X11_ShowWindow(this, width, height, flags) < 0 ) {
			current = NULL;
			goto finished;
		}
	}

	/* Set up the new mode framebuffer */
	current->flags = 0;		/* Clear flags */
	current->w = width;
	current->h = height;
	current->pitch = SDL_CalculatePitch(current);
	if ( (flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
		if ( X11_SetupDirect(this, current, flags) == 0 ) {
			flags |= SDL_HWPALETTE;
		} else {
			/* Failed, fall back to window */
			flags &= ~SDL_FULLSCREEN;
		}
	}
	if ( (flags & SDL_FULLSCREEN) != SDL_FULLSCREEN ) {
		if ( X11_SetupWindow(this, current, flags) < 0 ) {
			current = NULL;
			goto finished;
		}
	}

	/* Set the appropriate colormap */
	if ( current->format->palette != NULL ) {
		if ( flags & SDL_HWPALETTE ) {
			current->flags |= SDL_HWPALETTE;
			if ( SDL_PrivateColormap == 0 ) {
				SDL_PrivateColormap = XCreateColormap(SDL_Display, SDL_Window, SDL_Visual, AllocAll);
			}
			SDL_XColorMap = SDL_PrivateColormap;
		} else {
			/* Free any currently allocated hidden colormap */
			if ( SDL_PrivateColormap != 0 ) {
				XFreeColormap(SDL_Display, SDL_PrivateColormap);
				SDL_PrivateColormap = 0;
			}
			SDL_XColorMap = SDL_DisplayColormap;
		}
		XSetWindowColormap(SDL_Display, SDL_Window, SDL_XColorMap);
	}

	/* We're done */
finished:
	XSync(SDL_Display, False);
	SDL_Unlock_EventThread();
	return(current);
}

/* We don't actually allow hardware surfaces other than the main one */
static int X11_AllocHWSurface(_THIS, SDL_Surface *surface)
{
	return(-1);
}
static void X11_FreeHWSurface(_THIS, SDL_Surface *surface)
{
	return;
}

/* We need to wait for vertical retrace on page flipped displays */
static int X11_LockHWSurface(_THIS, SDL_Surface *surface)
{
#ifdef XFREE86_DGA
	if ( (surface->flags & SDL_DOUBLEBUF) == SDL_DOUBLEBUF ) {
		while ( ! XF86DGAViewPortChanged(GFX_Display,
				DefaultScreen(GFX_Display), 2) ) {
			SDL_Delay(1);
		}
	}
#endif
	return(0);
}
static void X11_UnlockHWSurface(_THIS, SDL_Surface *surface)
{
	return;
}

/* We can flip DGA memory! */
static int X11_FlipHWSurface(_THIS, SDL_Surface *surface)
{
#ifdef XFREE86_DGA
	/* Choose your page (double-buffered for now) */
	vinfo.flip = !vinfo.flip;

	/* Set it up */
	XF86DGASetViewPort(GFX_Display, DefaultScreen(GFX_Display),
					0, vinfo.flip*vinfo.h);
	surface->pixels = vinfo.base[!vinfo.flip];
#endif
	return(0);
}

#ifdef DGA_BROKEN_VIEWPORT
#define FIXED_VIEWPORT	0x80
void DGA_FixViewPort(_THIS)
{
	if ( vinfo.active && !(vinfo.active&FIXED_VIEWPORT) ) {
		XF86DGASetViewPort(GFX_Display, DefaultScreen(GFX_Display),
						0, vinfo.flip*vinfo.h);
		vinfo.active |= FIXED_VIEWPORT;
	}
}
#endif /* DGA_BROKEN_VIEWPORT */

/* Byte-swap the pixels in the display image */
static void SDL_SwapPixels(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
	int i;
	int x, minx, maxx;
	int y, miny, maxy;

	switch (screen->format->BytesPerPixel) {
	    case 2: {
		Uint16 *spot;
		for ( i=0; i<numrects; ++i ) {
			minx = rects[i].x;
			maxx = rects[i].x+rects[i].w;
			miny = rects[i].y;
			maxy = rects[i].y+rects[i].h;
			for ( y=miny; y<maxy; ++y ) {
				spot = (Uint16 *) ((Uint8 *)screen->pixels +
						y * screen->pitch + minx * 2);
				for ( x=minx; x<maxx; ++x, ++spot ) {
					*spot = SDL_Swap16(*spot);
				}
			}
		}
	    }
	    break;

	    case 4: {
		Uint32 *spot;
		for ( i=0; i<numrects; ++i ) {
			minx = rects[i].x;
			maxx = rects[i].x+rects[i].w;
			miny = rects[i].y;
			maxy = rects[i].y+rects[i].h;
			for ( y=miny; y<maxy; ++y ) {
				spot = (Uint32 *) ((Uint8 *)screen->pixels +
						y * screen->pitch + minx * 4);
				for ( x=minx; x<maxx; ++x, ++spot ) {
					*spot = SDL_Swap32(*spot);
				}
			}
		}
	    }
	    break;

	    default:
		/* should never get here */
		break;
	}
}

static void X11_NormalUpdate(_THIS, int numrects, SDL_Rect *rects)
{
	int i;

	/* Check for endian-swapped X server, swap if necessary (VERY slow!) */
	if ( this->hidden->swap_pixels &&
	     ((this->screen->format->BytesPerPixel%2) == 0) ) {
		SDL_SwapPixels(this->screen, numrects, rects);
		for ( i=0; i<numrects; ++i ) {
			XPutImage(GFX_Display, SDL_Window, SDL_GC, SDL_Ximage,
				rects[i].x, rects[i].y,
				rects[i].x, rects[i].y, rects[i].w, rects[i].h);
		}
		SDL_SwapPixels(this->screen, numrects, rects);
	} else {
		for ( i=0; i<numrects; ++i ) {
			XPutImage(GFX_Display, SDL_Window, SDL_GC, SDL_Ximage,
				rects[i].x, rects[i].y,
				rects[i].x, rects[i].y, rects[i].w, rects[i].h);
		}
	}
	XSync(GFX_Display, False);
}

static void X11_MITSHMUpdate(_THIS, int numrects, SDL_Rect *rects)
{
	int i;

	for ( i=0; i<numrects; ++i ) {
		XShmPutImage(GFX_Display, SDL_Window, SDL_GC, SDL_Ximage,
				rects[i].x, rects[i].y,
				rects[i].x, rects[i].y, rects[i].w, rects[i].h,
									False);
	}
	XSync(GFX_Display, False);
}

static void X11_DirectUpdate(_THIS, int numrects, SDL_Rect *rects)
{
}

/* This is very slow.  Ideally, you would organize all the updates into
   lists of bank-sorted updates, and perform all updates for each bank
   in turn.  In the general case, I don't think it's worth the effort.
*/
static void X11_BankedUpdate(_THIS, int numrects, SDL_Rect *rects)
{
#ifdef XFREE86_DGA
	int i, w, h;
	Uint8 *src, *dst;
	int xoffset;
	int bankleft;
	int bank;
	int wl, wr;

	/* Fixed by Alex Jones */
	for ( i=0; i<numrects; ++i ) {
		xoffset = rects[i].x * this->screen->format->BytesPerPixel;
		src = (Uint8 *)this->screen->pixels +
			rects[i].y * this->screen->pitch + xoffset;
		h = rects[i].h;
		w = rects[i].w * this->screen->format->BytesPerPixel;

		/* Get the starting bank and offset */
		bank = 0;
		dst = vinfo.base[0] + xoffset;
		bankleft = vinfo.page - xoffset;

		bankleft -= vinfo.width * rects[i].y;
		dst += vinfo.width * rects[i].y;
		if ( bankleft <= 0 ) {
			bank = -bankleft / vinfo.page + 1;
			bankleft += bank * vinfo.page;
			dst -= bank * vinfo.page;
		}

 		XF86DGASetVidPage(GFX_Display,DefaultScreen(GFX_Display),bank);
		while ( h-- ) {

			if ( w <= bankleft ) {
				wl = w;
				wr = 0;
			} else {
				wl = bankleft;
				wr = w - bankleft;
			}

			memcpy(dst, src, wl);

			bankleft -= vinfo.width;
			dst += vinfo.width;
			if ( bankleft <= 0 ) {
				XF86DGASetVidPage(GFX_Display,
					DefaultScreen(GFX_Display), ++bank);
				bankleft += vinfo.page;
				dst -= vinfo.page;
				memcpy(vinfo.base[0], src+wl, wr);
			}
			src += this->screen->pitch;
		}
	}
#endif /* XFREE86_DGA */
}

static int X11_SetColors(_THIS, int firstcolor, int ncolors)
{
	SDL_Color *colors;
	int      alloct_all;
	int      i;
	XColor  *xcmap;

	alloct_all = 1;
	colors = this->screen->format->palette->colors;
	ncolors = this->screen->format->palette->ncolors;
	xcmap = (XColor *)malloc(ncolors*sizeof(*xcmap));
	if ( xcmap == NULL ) {
		return(0);
	}

	/* It's easy if we have a hidden colormap */
	if ( (this->screen->flags & SDL_HWPALETTE) == SDL_HWPALETTE ) {
		for ( i=0; i<ncolors; ++i ) {
			xcmap[i].pixel = i;
			xcmap[i].red   = (colors[i].r<<8)|colors[i].r;
			xcmap[i].green = (colors[i].g<<8)|colors[i].g;
			xcmap[i].blue  = (colors[i].b<<8)|colors[i].b;
			xcmap[i].flags = (DoRed|DoGreen|DoBlue);
		}
		XStoreColors(GFX_Display, SDL_XColorMap, xcmap, ncolors);
#ifdef XFREE86_DGA
		if ( (this->screen->flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
			XF86DGAInstallColormap(SDL_Display, SDL_Screen,
							SDL_XColorMap);
		}
#endif
	} else {
		unsigned long pixel;
		XColor        wanted;
		XColor        color;

		/* Free existing allocated colors */
		for ( pixel=0; pixel<ncolors; ++pixel ) {
			while ( SDL_XPixels[pixel] > 0 ) {
				XFreeColors(GFX_Display, SDL_XColorMap, &pixel,
									1, 0);
				--SDL_XPixels[pixel];
			}
		}

		/* Try to allocate all the colors */
		for ( i=0; i<ncolors; ++i ) {
			color.pixel = i;
			color.red   = (colors[i].r<<8);
			color.green = (colors[i].g<<8);
			color.blue  = (colors[i].b<<8);
			color.flags = (DoRed|DoGreen|DoBlue);
			if ( alloct_all ) {
				memcpy(&wanted, &color, sizeof(wanted));
				if (XAllocColor(GFX_Display,
						SDL_XColorMap, &color)) {
					++SDL_XPixels[color.pixel];
				}
				if ( memcmp(&color, &wanted, sizeof(wanted)) )
					alloct_all = 0;
			} else {
				if (XAllocColor(GFX_Display,
						SDL_XColorMap, &color)) {
					++SDL_XPixels[color.pixel];
				}
			}
		}
		if ( ! alloct_all ) {
			/* Copy the colors from the colormap to our palette */
			for ( i=0; i<ncolors; ++i ) {
				xcmap[i].pixel = i;
				xcmap[i].flags = (DoRed|DoGreen|DoBlue);
			}
			XQueryColors(GFX_Display,SDL_XColorMap,xcmap,ncolors);
			for ( i=0; i<ncolors; ++i ) {
				colors[i].r = (xcmap[i].red>>8);
				colors[i].g = (xcmap[i].green>>8);
				colors[i].b = (xcmap[i].blue>>8);
			}
		}
	}
	free(xcmap);
	return(alloct_all);
}

/* Note:  If we are terminated, this could be called in the middle of
   another SDL video routine -- notably UpdateRects.
*/
void X11_VideoQuit(_THIS)
{
	/* Shutdown everything that's still up */
	/* The event thread should be done, so we can touch SDL_Display */
	if ( SDL_Display != NULL ) {

		/* Start shutting down the windows */
		X11_ReleaseMode(this, this->screen);
		if ( SDL_Window && (SDL_windowid == NULL) ) {
			XDestroyWindow(SDL_Display, SDL_Window);
		}
		if ( SDL_modelist ) {
			int i;

			for ( i=0; SDL_modelist[i]; ++i ) {
				free(SDL_modelist[i]);
			}
			free(SDL_modelist);
			SDL_modelist = NULL;
		}
		if ( SDL_PrivateColormap != 0 ) {
			XFreeColormap(SDL_Display, SDL_PrivateColormap);
			SDL_PrivateColormap = 0;
		}
		if ( SDL_XPixels ) {
			int numcolors;
			unsigned long pixel;

			numcolors = this->screen->format->palette->ncolors;
			for ( pixel=0; pixel<numcolors; ++pixel ) {
				while ( SDL_XPixels[pixel] > 0 ) {
					XFreeColors(GFX_Display,
						SDL_DisplayColormap,&pixel,1,0);
					--SDL_XPixels[pixel];
				}
			}
			free(SDL_XPixels);
			SDL_XPixels = NULL;
		} 
		if ( SDL_iconcolors ) {
			unsigned long pixel;
			int numcolors =
				((this->screen->format)->palette)->ncolors;
			for ( pixel=0; pixel<numcolors; ++pixel ) {
				while ( SDL_iconcolors[pixel] > 0 ) {
					XFreeColors(SDL_Display,
						SDL_DisplayColormap,&pixel,1,0);
					--SDL_iconcolors[pixel];
				}
			}
			free(SDL_iconcolors);
			SDL_iconcolors = NULL;
		} 
		/* Free that blank cursor */
		if ( SDL_BlankCursor != NULL ) {
			this->FreeWMCursor(this, SDL_BlankCursor);
			SDL_BlankCursor = NULL;
		}

		/* Close the X11 graphics connection */
		if ( GFX_Display != NULL ) {
			XCloseDisplay(GFX_Display);
			GFX_Display = NULL;
		}

		/* Free any MTRR control connections */
		if ( this->hidden->MTRR_fd >= 0 ) {
			close(this->hidden->MTRR_fd);
			this->hidden->MTRR_fd = -1;
		}

		/* Close the X11 display connection */
		XCloseDisplay(SDL_Display);
		SDL_Display = NULL;
	}
	if ( this->screen && (this->screen->flags & SDL_HWSURFACE) ) {
		/* Direct screen access, no memory buffer */
		this->screen->pixels = NULL;
	}
}
