/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_timer.c,v 1.2 1999/11/23 19:01:41 hercules Exp $";
#endif

#include <windows.h>
#include <mmsystem.h>

#include "SDL_timer.h"
#include "SDL_error.h"

#define ROUND_RESOLUTION(X)	\
	(((X+TIMER_RESOLUTION-1)/TIMER_RESOLUTION)*TIMER_RESOLUTION)
#define TIME_WRAP_VALUE	(~(DWORD)0)

/* The first ticks value of the application */
static DWORD start;

void SDL_StartTicks(void)
{
	/* Set first ticks value */
	start = GetTickCount();
}

Uint32 SDL_GetTicks(void)
{
	DWORD now, ticks;

	now = GetTickCount();
	if ( now < start ) {
		ticks = (TIME_WRAP_VALUE-start) + now;
	} else {
		ticks = (now - start);
	}
	return(ticks);
}

void SDL_Delay(Uint32 ms)
{
	Sleep(ms);
}


/* Data to handle a single periodic alarm */
static UINT timer;
static SDL_TimerCallback alarm_callback = NULL;
static Uint32 alarm_interval;
static Uint32 alarm_pending;

static void CALLBACK HandleAlarm(UINT uID,  UINT uMsg, DWORD dwUser,
							DWORD dw1, DWORD dw2)
{
	Uint32 ms;

	if ( alarm_pending ) {
		--alarm_pending;
		if ( ! alarm_pending ) {
			ms = (*alarm_callback)(alarm_interval);
			if ( ms != alarm_interval ) {
				alarm_interval = ROUND_RESOLUTION(ms);
			}
			alarm_pending = (alarm_interval/10);
		}
	}
}

int SDL_SetTimer(Uint32 ms, SDL_TimerCallback callback)
{
	alarm_callback = callback;
	alarm_interval = ROUND_RESOLUTION(ms);
	alarm_pending = (alarm_interval/10);
	if ( ! timer ) {
		/* Allow 1 ms of drift so we don't chew on CPU */
		timer = timeSetEvent(TIMER_RESOLUTION, 1,
					HandleAlarm, 0, TIME_PERIODIC);
		if ( ! timer ) {
			SDL_SetError("timeSetEvent() failed");
			return(-1);
		}
	}
	return(0);
}

int SDL_TimerInit(void)
{
	MMRESULT result;

	/* Set timer resolution */
	result = timeBeginPeriod(TIMER_RESOLUTION);
	if ( result != TIMERR_NOERROR ) {
		SDL_SetError("Warning: Can't set %s ms timer resolution",
							TIMER_RESOLUTION);
	}
	return(0);
}

void SDL_TimerQuit(void)
{
	if ( timer ) {
		timeKillEvent(timer);
	}
	timeEndPeriod(TIMER_RESOLUTION);
}
