/* $Id: visual.c,v 1.30 1999/04/30 19:51:55 marcus Exp $
******************************************************************************

   SVGAlib target: initialization

   Copyright (C) 1998 Steve Cheng	[steve@ggi-project.org]
   Copyright (C) 1999 Marcus Sundberg	[marcus@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/svgalib.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif
#if defined(HAVE_SYS_VT_H) || defined(HAVE_LINUX_VT_H)
# ifdef HAVE_SYS_VT_H
# include <sys/vt.h>
# else
# include <linux/vt.h>
# endif
# define HAVE_VTSTUFF
#endif


static int usagecounter = 0;

#define RELSIG	SIGUSR1
#define ACQSIG	SIGUSR2
/* SVGAlib's signal handlers */
static const int vga_signals[] = {
	RELSIG, ACQSIG,
	SIGHUP, SIGINT, SIGQUIT, SIGILL, SIGABRT, /* == SIGIOT */
#ifdef SIGTRAP
	SIGTRAP,
#endif
#ifdef SIGBUS
	SIGBUS,
#endif
	SIGFPE,	SIGSEGV, SIGPIPE, SIGALRM, SIGTERM,
#ifdef SIGXCPU
	SIGXCPU,
#endif
#ifdef SIGXFSZ
	SIGXFSZ,
#endif
#ifdef SIGVTALRM
	SIGVTALRM,
#endif
#ifdef SIGPROF
	SIGPROF,
#endif
#ifdef SIGPWR
	SIGPWR
#endif
};
#define NUMVGASIGS	(sizeof(vga_signals)/sizeof(int))


void _GGI_svga_freedbs(ggi_visual *vis) {
	int i;

	for (i=LIBGGI_APPLIST(vis)->num-1; i >= 0; i--) {
		_ggi_db_free(LIBGGI_APPBUFS(vis)[i]);
		_ggi_db_del_buffer(LIBGGI_APPLIST(vis), i);
	}
}


static int _GGIchecksvgamodes(ggi_visual *vis)
{
	int modes = 0;
	int i;
	svga_priv *priv = LIBGGI_PRIVATE(vis);

	for (i=1; i <= vga_lastmodenumber(); i++) {
		if (vga_hasmode(i)) {
			vga_modeinfo *modeinfo;
			ggi_modelistmode *sgmode;
			int bpp, size;
			modeinfo = vga_getmodeinfo(i);
			switch(modeinfo->colors) {
			case 2:
				continue; /* 1 bpp doesn't work */
				bpp = size = 1;
				break;
			case 16:
				continue; /* 4 bpp doesn't work */
				bpp = size = 4;
				break;
			case 256:
				bpp = size = 8;
				break;
			case 1<<15:
				bpp = size = 15;
				break;
			case 1<<16:
				bpp = size = 16;
				break;
			case 1<<24:
				bpp = 24;
				if (modeinfo->bytesperpixel==3) {
					size = 24;
				} else {
					size = 32;
				}
				break;
			default:
				continue;
			}
			sgmode = priv->availmodes + modes;
			modes++;
			sgmode->x = modeinfo->width;
			sgmode->y = modeinfo->height;
			sgmode->bpp = bpp;
			sgmode->gt = GT_CONSTRUCT(bpp, (bpp <= 8) ?
			      GT_PALETTE : GT_TRUECOLOR, size);
			if ((modeinfo->flags & IS_MODEX))
				sgmode->ismodex = 1;
			else
				sgmode->ismodex = 0;
		}
	}

	if (modes == 0) {
		return -1;
	} else {
		priv->availmodes = realloc(priv->availmodes,
					   (modes+1)*sizeof(ggi_modelistmode));
		priv->availmodes[modes].bpp = 0;
	}

	return 0;
}


#if 0
static void 
save_palette(ggi_visual *vis)
{
	svga_priv *priv = LIBGGI_PRIVATE(vis);
	ggi_graphtype gt = LIBGGI_MODE(vis)->graphtype;
	int len;

	if (GT_SCHEME(gt) != GT_PALETTE) return;
	switch (GT_DEPTH(gt)) {
	case 4:
		len = 16;
		break;
	case 8:
		len = 256;
		break;
	default:
		return;
	}
	
	priv->savepalette = malloc(sizeof(int)*(len*3+1));
	if (priv->savepalette == NULL) {
		return;
	}
	*priv->savepalette = len;

	vga_getpalvec(0, len, priv->savepalette+1);
}
#endif


static void 
restore_palette(ggi_visual *vis)
{
	ggi_graphtype gt = LIBGGI_MODE(vis)->graphtype;
	int *vgapal, *palptr;
	int len, i;

	if (GT_SCHEME(gt) != GT_PALETTE) return;
	switch (GT_DEPTH(gt)) {
	case 4:
		len = 16;
		break;
	case 8:
		len = 256;
		break;
	default:
		return;
	}
	
	vgapal = malloc(sizeof(int)*(len*3+1));
	if (vgapal == NULL) return;

	palptr = vgapal;
	for (i = 0; i < len; i++) {
		palptr[i*3]   = vis->palette[i].r >> 10;
		palptr[i*3+1] = vis->palette[i].g >> 10;
		palptr[i*3+2] = vis->palette[i].b >> 10;
	}
		
	vga_setpalvec(0, len, vgapal);
	
	free(vgapal);
}


static void
really_switch(ggi_visual *vis)
{
	svga_priv *priv = LIBGGI_PRIVATE(vis);

#if 0
	save_palette(vis);
#endif
	priv->ismapped = 0;
	priv->gfxmode = vga_getcurrentmode();
	_ggi_svgalib_setmode(TEXT);
}

static void
switched_away(void *arg)
{
	ggi_visual *vis = arg;
	svga_priv *priv = LIBGGI_PRIVATE(vis);

	GGIDPRINT_MISC("display-svga: switched_away() called\n");
	if (!priv->autoswitch && !priv->switchpending) {
		gii_event ev;
		ggi_cmddata_switchrequest *data;
		
		GGIDPRINT_MISC("display-svga: queueing GGICMD_REQUEST_SWITCH"
			       " event\n");

		_giiEventBlank(&ev, sizeof(gii_cmd_event));

		data = (void *)ev.cmd.data;

		ev.size   = sizeof(gii_cmd_event);
		ev.cmd.type = evCommand;
		ev.cmd.code = GGICMD_REQUEST_SWITCH;
		data->request = GGI_REQSW_UNMAP;

		_giiSafeAdd(vis->input, &ev);
		
		priv->switchpending = 1;
	} else if (priv->autoswitch) {
		really_switch(vis);
	}
}

static void
switched_back(void *arg)
{
	ggi_visual *vis = arg;
	gii_event ev;

	GGIDPRINT_MISC("display-svga: switched_back() called\n");

	_giiEventBlank(&ev, sizeof(gii_expose_event));

	ev.any.size   = sizeof(gii_expose_event);
	ev.any.type   = evExpose;

	ev.expose.x = ev.expose.y = 0;
	ev.expose.w = LIBGGI_VIRTX(vis);
	ev.expose.h = LIBGGI_VIRTY(vis);

	_giiSafeAdd(vis->input, &ev);
	GGIDPRINT_MISC("svga: EXPOSE sent.\n");

	_ggi_svgalib_setmode(SVGA_PRIV(vis)->gfxmode);
	restore_palette(vis);
	SVGA_PRIV(vis)->ismapped = 1;
}


static int 
GGI_svga_sendevent(ggi_visual *vis, gii_event *ev)
{
	svga_priv *priv = LIBGGI_PRIVATE(vis);

	GGIDPRINT_MISC("GGI_svga_sendevent() called\n");

	if (ev->any.type != evCommand) {
		return GGI_EEVUNKNOWN;
	}
	switch (ev->cmd.code) {
	case GGICMD_ACKNOWLEDGE_SWITCH:
		GGIDPRINT_MISC("display-svga: switch acknowledge\n");
		if (priv->switchpending) {
			priv->switchpending = 0;
			priv->doswitch(vis);
			really_switch(vis);
			return 0;
		} else {
			/* No switch pending */
			return GGI_EEVNOTARGET;
		}
		break;
	case GGICMD_NOHALT_ON_UNMAP:
		GGIDPRINT_MISC("display-svga: nohalt on\n");
		priv->dohalt = 0;
		priv->autoswitch = 0;
		break;
	case GGICMD_HALT_ON_UNMAP:
		GGIDPRINT_MISC("display-svga: halt on\n");
		priv->dohalt = 1;
		priv->autoswitch = 1;
		if (priv->switchpending) {
			/* Do switch and halt */
			priv->switchpending = 0;
			priv->doswitch(vis);
			really_switch(vis);
			pause();
		}
		break;
	}
	
	return GGI_EEVUNKNOWN;
}


/* This is exported from svgalib, we use it to check if
   we're called from GGI's svgalib wrapper. */
#define GSW_MAGIC	(-4711)
extern int __svgalib_tty_fd;

int GGIdlinit(ggi_visual *vis,const char *args,void *argptr)
{
	svga_priv *priv;
	struct sigaction old_signals[NUMVGASIGS];
#ifdef HAVE_VTSTUFF
	struct vt_mode temp_vtmode;
#endif
	int i;
	
	if (__svgalib_tty_fd == GSW_MAGIC) {
		ggiPanic("SVGAlib target called from the SVGAlib wrapper!"
			 " Terminating.\n");
	}
	
	ggLock(_ggi_global_lock); /* Entering protected section */
	if (usagecounter > 0) {
		ggUnlock(_ggi_global_lock);
		fprintf(stderr, "display-svga: You can only open this target "
			"once in an application.\n");
		return GGI_DL_ERROR;
	}
	usagecounter++;
	ggUnlock(_ggi_global_lock); /* Exiting protected section */

	/* Save original signal handlers because SVGAlib will set its own */
	for(i = 0; i < NUMVGASIGS; i++) {
		sigaction(vga_signals[i], NULL, old_signals+i);
	}

	if (!_ggiDebugState) vga_disabledriverreport();

	if (vga_init()) {
		fprintf(stderr, "display-SVGAlib: vga_init() failed\n");
		usagecounter--;
		return GGI_DL_ERROR;
	}

#ifdef HAVE_VTSTUFF
	/* Save VT mode */
	ioctl(__svgalib_tty_fd, VT_GETMODE, &temp_vtmode);
#endif

	/* Trigger the setup code in SVGAlib */
	_ggi_svgalib_setmode(TEXT);

#if 0
	/* Save SVGAlib VT handlers for our own use */
	do {
		struct sigaction sa;

		sigaction(RELSIG, NULL, &sa);
		release = sa.sa_handler;
		sigaction(ACQSIG, NULL, &sa);
		acquire = sa.sa_handler;
	} while(0);
#endif

#ifdef HAVE_VTSTUFF
	/* Restore VT mode */
	ioctl(__svgalib_tty_fd, VT_SETMODE, &temp_vtmode);
#endif

	/* Get rid of SVGAlib's signal handlers */
	for (i = 0; i < NUMVGASIGS; i++) {
		sigaction(vga_signals[i], old_signals+i, NULL);
	}

	ggRegisterCleanup((ggcleanup_func *)GGIdlcleanup, vis);
	ggCleanupForceExit();
    
	LIBGGI_GC(vis) = malloc(sizeof(ggi_gc));
	if (LIBGGI_GC(vis) == NULL) {
		usagecounter--;
		return GGI_DL_ERROR;
	}
	LIBGGI_PRIVATE(vis) = malloc(sizeof(struct svga_priv));
	if (LIBGGI_PRIVATE(vis) == NULL) {
		GGIdlcleanup(vis);
		return GGI_DL_ERROR;
	}
	priv = LIBGGI_PRIVATE(vis);

	priv->availmodes = 
		malloc(vga_lastmodenumber()*sizeof(ggi_modelistmode));
	if (priv->availmodes == NULL || _GGIchecksvgamodes(vis) != 0) {
		GGIdlcleanup(vis);
		return GGI_DL_ERROR;
	}
	priv->inputs = INP_KBD | INP_MOUSE | INP_VT;
	priv->dohalt = 1;
	priv->autoswitch = 1;
	priv->switchpending = 0;
	priv->ismapped = 1;

	/* Open keyboard and mouse input */
	if (priv->inputs & INP_KBD) {
		ggi_linvtsw_arg vtswarg;
		int  vtnum = -1;
		char strbuf[64];
		char *inputstr = "input-linux-kbd";

		/* Fbcon handles the hw on console switch */
		vtswarg.switchaway = switched_away;
		vtswarg.switchback = switched_back;
		vtswarg.funcarg = vis;

		vtswarg.dohalt     = &priv->dohalt;
		vtswarg.autoswitch = &priv->autoswitch;
		vtswarg.onconsole = 1;
		if (getenv("GGI_NEWVT")) {
			vtswarg.forcenew = 1;
		} else {
			vtswarg.forcenew = 0;
		}

		if (_ggiAddDL(vis, "helper-linux-vtswitch", NULL, &vtswarg, 0)
		    != NULL) {
			vtnum = vtswarg.vtnum;
			priv->doswitch = vtswarg.doswitch;
		} else {
			priv->doswitch = NULL;
		}
		if (vtnum != -1) {
			sprintf(strbuf, "linux-kbd:/dev/tty%d", vtnum);
			inputstr = strbuf;
		}

		vis->input = giiOpen(inputstr, NULL);
		if (vis->input == NULL) {
			if (vtnum != -1) {
				sprintf(strbuf, "linux-kbd:/dev/vc/%d", vtnum);
				vis->input = giiOpen(inputstr, NULL);
			}
			if (vis->input == NULL) {
				fprintf(stderr,
"display-svga: Unable to open linux-kbd, trying stdin input.\n");
				/* We're on the Linux console so we want
				   ansikey. */
				vis->input = giiOpen("stdin:ansikey", NULL);
				if (vis->input == NULL) {
					fprintf(stderr,
"display-svga: Unable to open stdin input, try running with '-nokbd'.\n");
					GGIdlcleanup(vis);
					return GGI_DL_ERROR;
				}
			}
		}
	}
	if (priv->inputs & INP_MOUSE) {
		gii_input *inp;
		if ((inp = giiOpen("linux-mouse:auto", &args, NULL)) != NULL) {
			vis->input = giiJoinInputs(vis->input, inp);
		}
	}

	/* Has mode management */
	vis->opdisplay->flush		= GGI_svga_flush;
	vis->opdisplay->getmode		= GGI_svga_getmode;
	vis->opdisplay->setmode		= GGI_svga_setmode;
	vis->opdisplay->getapi		= GGI_svga_getapi;
	vis->opdisplay->checkmode	= GGI_svga_checkmode;
	vis->opdisplay->setflags	= GGI_svga_setflags;
	vis->opdisplay->sendevent	= GGI_svga_sendevent;

	return GGI_DL_OPDISPLAY;
}


int GGIdlcleanup(ggi_visual *vis)
{
	svga_priv *priv = LIBGGI_PRIVATE(vis);

	/* Restore to text mode */
	_ggi_svgalib_setmode(TEXT);

	_GGI_svga_freedbs(vis);

	if (vis->input != NULL) {
		giiClose(vis->input);
		vis->input = NULL;
	}
	if (priv) {
		if (SVGA_PRIV(vis)->availmodes != NULL) {
			free(SVGA_PRIV(vis)->availmodes);
		}
		free(priv);
	}
	if(vis->palette) free(vis->palette);
	free(LIBGGI_GC(vis));

#ifndef ALWAYS_SEGFAULT_ON_EXIT
	if(!_ggiDebugState)
		signal(SIGSEGV,_exit);
#endif

	ggUnregisterCleanup((ggcleanup_func *)GGIdlcleanup, vis);
	
	usagecounter--;

	return 0;
}

#include <ggi/internal/ggidlinit.h>
