/********************************************************************************
*                                                                               *
*                             B i t m a p    O b j e c t                        *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXBitmap.cpp,v 1.6 1999/11/04 20:46:38 jeroen Exp $                      *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXVisual.h"
#include "FXDrawable.h"
#include "FXBitmap.h"
#include "FXVisual.h"
#include "FXDC.h"
#include "FXDCWindow.h"

/*
  Note:
  - Make this work.
  - Try eliminate temp copy:- slap pixels into XImage directly, if possible...
  - Perhaps enforce system-native padding necessary for the above.
  - Our bitmap data is 01234567, i.e. LS-BIT first; byte order ditto.
  - Issue: should FXBitmap return the DC for drawing onto the X-Server resident
    pixmap, or the client-side image bits?
    
    My idea is it should be the latter:
    
      - Allows even richer set of drawing primitives, as everything is
        drawn in software.
      - Very useful to generate off-screen renderings, e.g. during printing.
      - Allows for building and running true-color drawing programs on
        low-end graphics hardware.
      - The only drawback I can see is it will be a fairly large implementation
        effort...
      
*/

/*******************************************************************************/


// Object implementation
FXIMPLEMENT(FXBitmap,FXDrawable,NULL,0)


// For deserialization
FXBitmap::FXBitmap(){
  data=NULL;
  options=0;
  }


// Initialize
FXBitmap::FXBitmap(FXApp* a,const void *pix,FXuint opts,FXint w,FXint h):FXDrawable(a,w,h){
  FXTRACE((100,"FXBitmap::FXBitmap %08x\n",this));
  visual=getApp()->getMonoVisual();
  if(pix) opts&=~BITMAP_OWNED;
  data=(FXuchar*)pix;
  options=opts;
  }


// Create bitmap
void FXBitmap::create(){
  if(!xid){
    FXTRACE((100,"%s::create %08x\n",getClassName(),this));
    
#ifndef FX_NATIVE_WIN32
    
    // App should exist
    if(!getApp()->display){ fxerror("%s::create: trying to create image before opening display.\n",getClassName()); }
  
    // Initialize visual
    visual->init();
    
    // Make pixmap
    xid=XCreatePixmap(getApp()->display,XDefaultRootWindow(getApp()->display),width,height,1);
    if(!xid){ fxerror("%s::create: unable to create image.\n",getClassName()); }
  
#else
  
    // Initialize visual
    visual->init();

    // Create a memory DC compatible with the display
    HDC hdc=::GetDC(GetDesktopWindow());
    xid=CreateCompatibleDC(hdc);
    ::ReleaseDC(GetDesktopWindow(),hdc);
    if(!xid){ fxerror("%s::create: unable to create image.\n",getClassName()); }

#endif

    // Render pixels
    render();

    // Zap data
    if(!(options&BITMAP_KEEP) && (options&BITMAP_OWNED)){
      options&=~BITMAP_OWNED;
      FXFREE(&data);
      }
    }
  }


// Detach bitmap
void FXBitmap::detach(){
  if(xid){
    FXTRACE((100,"%s::detach %08x\n",getClassName(),this));
    xid=0;
    }
  }


// Destroy bitmap
void FXBitmap::destroy(){
  if(xid){
    FXTRACE((100,"%s::destroy %08x\n",getClassName(),this));
#ifndef FX_NATIVE_WIN32
    XFreePixmap(getApp()->display,xid);
#else
    DeleteObject(xid);
#endif
    xid=0;
    }
  }


#ifndef FX_NATIVE_WIN32

// Render bits
void FXBitmap::render_bits(XImage *xim,FXuchar *img){
  register int size=xim->bytes_per_line*height;
  register FXuchar *pix=(FXuchar*)xim->data;
  register int i;
  
  // Most significant bit first
  if(xim->bitmap_bit_order==MSBFirst){
    for(i=0; i<size; i++) pix[i]=fxbitreverse[img[i]];
    }
  
  // Least significant bit first
  else{
    memcpy(pix,img,size);
    }
  }

#endif

// Render into pixmap
void FXBitmap::render(){
  
  FXTRACE((100,"%s::render bitmap %0x8\n",getClassName(),this));
  
  // Do it for X-Windows
#ifndef FX_NATIVE_WIN32
  
  // XImage xim;
  register XImage *xim=NULL;
  register Visual *vis;
  XGCValues values;
  GC gc;

  // Can not render before creation
  if(!xid){ fxerror("%s::render: trying to render bitmap before it has been created.\n",getClassName()); }

  // Check for legal size
  if(width<2 || height<2){ fxerror("%s::render: illegal bitmap size.\n",getClassName()); }

  // Make GC
  values.foreground=0xffffffff;
  values.background=0;
  gc=XCreateGC(getApp()->display,xid,GCForeground|GCBackground,&values);
 
  // Just leave if black if no data
  if(data){
    
    // Get Visual
    vis=visual->visual;

    //xim=XCreateImage(getApp()->display,DefaultVisual(getApp()->display,DefaultScreen(getApp()->display)),1,XYBitmap,0,NULL,width,height,8,(width+7)>>3);
    xim=XCreateImage(getApp()->display,vis,1,XYBitmap,0,NULL,width,height,8,(width+7)>>3);
    if(!xim){ fxerror("%s::render: unable to render image.\n",getClassName()); }

    // Try create temp pixel store
    xim->data=(char*)malloc(xim->bytes_per_line*height);

    // Failed completely
    if(!xim->data){ fxerror("%s::render: unable to allocate memory.\n",getClassName()); }
    
    // Render bits into server-formatted bitmap
    render_bits(xim,data);
    
    // Blast the image
    XPutImage(getApp()->display,xid,gc,xim,0,0,0,0,width,height);
#ifndef WIN32
    //// Need to use something other than malloc for WIN32....
    XDestroyImage(xim);
#endif
    }
  
  // No data, fill with black
  else{
    XFillRectangle(getApp()->display,xid,gc,0,0,width,height);
    }
 
  // We're done
  XFreeGC(getApp()->display,gc);
  
#else
  register FXuchar *widedata,*p,*q;
  register FXint nb,i;
  HBITMAP hbmp;
  
  // Can not render before creation
  if(!xid){ fxerror("%s::render: trying to render bitmap before it has been created.\n",getClassName()); }

  // Check for legal size
  if(width<2 || height<2){ fxerror("%s::render: illegal bitmap size.\n",getClassName()); }

  // Just leave if black if no data
  if(data){
    nb=(width+7)>>3;
    if(nb&1){
      FXMALLOC(&widedata,FXuchar,height*(nb+1));
      for(i=0,p=widedata,q=data; i<height; i++){
        memcpy(p,q,nb); 
        p[nb]=0;
        p+=nb+1;
        q+=nb;
        }
      hbmp=CreateBitmap(width,height,1,1,widedata);
      FXFREE(&widedata);
      }
    else{
      hbmp=CreateBitmap(width,height,1,1,data);
      }
    if(!hbmp){ fxerror("%s: unable to create bitmap.\n",getClassName()); }
    SelectObject((HDC)xid,hbmp);
    DeleteObject(hbmp);
    }
  else{
    hbmp=CreateBitmap(width,height,1,1,NULL);
    if(!hbmp){ fxerror("%s: unable to create bitmap.\n",getClassName()); }
    SelectObject((HDC)xid,hbmp);
    DeleteObject(hbmp);
    }

#endif
  }


#ifdef FX_NATIVE_WIN32

// Get the image's device context
HDC FXBitmap::GetDC() const { return (HDC)xid; }

// Release it (no-op)
int FXBitmap::ReleaseDC(HDC) const { return 1; }

#endif

// Save pixel data only
void FXBitmap::savePixels(FXStream& store) const {
  FXuint size=height*((width+7)>>3);
  store.save(data,size);
  }


// Load pixel data only
void FXBitmap::loadPixels(FXStream& store){
  FXuint size=height*((width+7)>>3);
  if(options&BITMAP_OWNED) FXFREE(&data);
  FXMALLOC(&data,FXuchar,size);
  store.load(data,size);
  options|=BITMAP_OWNED;
  }


// Save data
void FXBitmap::save(FXStream& store) const {
  FXuchar haspixels=(data!=NULL);
  FXDrawable::save(store);
  store << options;
  store << haspixels;
  if(haspixels) savePixels(store);
  }


// Load data
void FXBitmap::load(FXStream& store){
  FXuchar haspixels;
  FXDrawable::load(store);
  store >> options;
  store >> haspixels;
  if(haspixels) loadPixels(store);
  }


// Clean up
FXBitmap::~FXBitmap(){
  FXTRACE((100,"FXBitmap::~FXBitmap %08x\n",this));
  if(xid){
#ifndef FX_NATIVE_WIN32
    XFreePixmap(getApp()->display,xid);
#else
    DeleteDC((HDC)xid);
#endif
    }
  if(options&BITMAP_OWNED){FXFREE(&data);}
  data=(FXuchar*)-1;
  }
  
